/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// S580Application.cpp                                                     //
//                                                                         //
// Implements methods in class S580AppController only                      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// ver. 1.0.0 of Fri 09-Jan-2009 @ 5:10am EST                              //
//                                                                         //
//     this is the version used to compile to the 1.0pr1 distributable     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Copyright (c) 2008-2009 Lucas Stephen Beeler. All rights reserved.      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "S580Application.h"
#include "S580MainWindow.h"
#include "Brushes.h"
#include "NewDialog.h"
#include "EraserOptionsDialog.h"
#include "Erasers.h"
#include "BrushOptionsDialog.h"
#include "resource.h"

S580AppController::S580AppController( )
    : fCurrentPainting(0)
{
}



S580AppController::~S580AppController( )
{
    if (isInstalled( ))
        Application::instance( ).installController(
            * ApplicationController::nullController( ));
}



void  S580AppController::installed(Application& sender)
{
}



void  S580AppController::deinstalled(Application& sender)
{
    uDestroyChildren( );
}



void  S580AppController::applicationStarted(Application& sender)
{
    /* On application start, we display the splash screen and, while the
       splash screen is displayed, we do all the time-consuming startup
       work (loading the brush library, etc). Because of this approach,
       no actual startup work is done in this method. Instead, we do it
       all in thingyInteractionStarted( ) (this is because the splash
       screen is implemented as a dialog box, and thingyInteractionStarted( )
       is called when that dialog box is first displayed */

    DialogThingy  splashscreen(kSplashDialogID);
    delete & splashscreen.installController(*this);
    splashscreen.startInteraction( );
}



void  S580AppController::applicationStopped(Application& sender)
{
}



bool  S580AppController::applicationStopRequested(Application& sender)
{
    if (isPaintingOpen( ))
        if (currentPainting( ).isMutated( ))
            closePainting( );

    std::string  libpath = Application::instance( ).startupDirectory( ) +
        "\\Library";
    std::string  brushdbpath = libpath + "\\" + "Brushes.db";
    brushManager( ).save(brushdbpath);

    return true;
}



bool  S580AppController::isInstalled( ) const
{
    if (this == (& Application::instance( ).controller( )))
        return true;
    else
        return false;
}



void  S580AppController::uCreateChildren( )
{
    std::string  libpath = Application::instance( ).startupDirectory( ) +
        "\\Library";
    std::string  brushdbpath = libpath + "\\" + "Brushes.db";
    fBrushManager.reset(new BrushManager(*this, brushdbpath));

    fCurrentEraser.reset(new Eraser( ));

    fMainWinCtlr.reset(new S580MainWinController(*this));

    /* now that we have a rendering canvas, link its GPUInterface to the
       eraser */
    eraser( ).link(fMainWinCtlr->gpuInterface( ));

    fNewDialogCtlr.reset(new NewDialogController(*this));
    fEraserOptsDialog.reset(new EraserOptionsDialog(*this));
    fBrushOptsDialog.reset(new BrushOptionsDialog(*this));

    fUnivBrush.reset(new Brush(brushManager( ).systemBrush( )));
    fUnivBrush->setName("UniversalBrush");
    fUnivBrush->link(fMainWinCtlr->gpuInterface( ));
    fUnivBrush->cloneAttributesFrom(brushManager( ).activeBrush( ));
}



void  S580AppController::uDestroyChildren( )
{
    S580MainWinController* mainwin = fMainWinCtlr.release( );
    delete mainwin;
}



BrushManager&  S580AppController::brushManager( )
{
    return *fBrushManager;
}



const ToolSpecifier&  S580AppController::currentTool( ) const
{
    return fCurrentTool;
}



void  S580AppController::setCurrentTool(const ToolSpecifier& t)
{
    fCurrentTool = t;
    fMainWinCtlr->setSelectedTool(t);
}



void  S580AppController::newPainting( )
{
    if (isPaintingOpen( )) {

        closePainting( );
    }

    Vector2Di  painting_dimension;
    bool  wascaptured = fNewDialogCtlr->runInteraction(painting_dimension);

    if (! wascaptured)
        return;

    fCurrentPainting = new PaintingDocument(painting_dimension);
    fMainWinCtlr->notifyDocumentChanged( );
}



void  S580AppController::openPainting( )
{
    if (isPaintingOpen( )) {

        closePainting( );
    }

    OpenFileChooser  ofc;
    ofc.setTitle("Open Painting...");
    ofc.addFiletypeMask("Truevision TARGA (*.tga)", "*.tga");
    std::string  filepath = ofc.runFileChoiceInteraction(mainWindowView( ));

    if (filepath == "")
        return;

    fCurrentPainting = new PaintingDocument(filepath);
    fMainWinCtlr->notifyDocumentChanged( );

}



ApplicationWindowThingy&  S580AppController::mainWindowView( )
{
    return fMainWinCtlr->view( );
}



bool  S580AppController::isPaintingOpen( ) const
{
    if (fCurrentPainting)
        return true;
    else
        return false;
}



PaintingDocument&  S580AppController::currentPainting( )
{
    return *fCurrentPainting;
}



const PaintingDocument&  S580AppController::currentPainting( ) const
{
    return *fCurrentPainting;
}



void  S580AppController::closePainting( )
{
    if (fCurrentPainting->isMutated( )) {

        int  btnid = MessageBox(mainWindowView( ).adapter( ).primitive( ),
            "Your current painting has changed. Do you want to save it?",
            "Studio580", MB_YESNO);

        if (btnid == IDYES) {

            if (fCurrentPainting->isMapped( ))
                fCurrentPainting->save( );
            else
                fMainWinCtlr->runSaveInteraction( );
        }
    }

    delete fCurrentPainting;
    fCurrentPainting = 0;

    fMainWinCtlr->notifyDocumentChanged( );
}



void  S580AppController::changeEraserOptions( )
{
    bool waschanged = fEraserOptsDialog->runInteraction(eraser( ));

    if (waschanged) {

        Eraser::saveAsDefault(eraser( ));
        fMainWinCtlr->notifyEraserChanged( );
    }
}



Eraser&  S580AppController::eraser( )
{
    return *fCurrentEraser;
}



const Eraser&  S580AppController::eraser( ) const
{
    return *fCurrentEraser;
}



void  S580AppController::changeBrushOptions( )
{
    mainWindowView( ).wait( );

    fBrushOptsDialog->runInteraction( );

    /* after the interaction is complete, the active brush may have changed,
       so re-clone the user's brush attributes into the universal brush,
       so that the changes are reflected when the user paints */
    universalBrush( ).cloneAttributesFrom(brushManager( ).activeBrush( ));

    mainWindowView( ).unwait( );
}



void  S580AppController::notifyBrushChanged( )
{
    fMainWinCtlr->notifyBrushChanged( );
}



void  S580AppController::notifyDocumentStateChanged( )
{
    fMainWinCtlr->notifyDocumentStateChanged( );
}



void  S580AppController::thingyItemActivated(DialogThingy&, int)
{
}



void  S580AppController::thingyClosed(DialogThingy&)
{
}



void  S580AppController::thingyInteractionStarted(DialogThingy& sender)
{
    /* All time consuming application startup work is done here (because
       the splash screen is implemented as a dialog box, and 
       thingyInteractionStarted( ) is called when that dialog box is first
       displayed); for more info, see the comment in applicationStarted( ) */

    /* calling show( ) on the sending DialogThingy and the Win32 API function
       UpdateWindow in this sequence is necessary to force the splash dialog
       to display. Indeed, PowerTools wouldn't normally show the sender until
       after this method returns. We need to do this before doing time
       consuming startup work */
    sender.show( );
    UpdateWindow(sender.adapter( ).primitive( ));

    /* the actual meat of application startup -- here we create the
       application's "child objects" (the main window, the brush manager,
       etc). */
    uCreateChildren( );
    setCurrentTool(kBrushTool);

    mainWindowView( ).show( );

    sender.stopInteraction( );
}



void  S580AppController::thingyInteractionStopped(DialogThingy&)
{
}



void  S580AppController::layoutSubthingies(CompositeThingy& sender)
{
}



void  S580AppController::enableSubthingies(CompositeThingy& sender)
{
}



void  S580AppController::disableSubthingies(CompositeThingy& sender)
{
}



Brush&  S580AppController::universalBrush( )
{
    return *fUnivBrush;
}



const Brush&  S580AppController::universalBrush( ) const
{
    return *fUnivBrush;
}
