/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Erasers.cpp                                                             //
//                                                                         //
// Implements methods in class Eraser only                                 //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  <unversioned module>                                                   //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Copyright (c) 2008 Lucas Stephen Beeler. All rights reserved.           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "Erasers.h"
#include <stdexcept>
#include <sstream>
#include <fstream>
#include <cmath>

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  Eraser                                                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const std::string  Eraser::kPrefsFilename = "Eraser.prefs";

Eraser*            Eraser::sDefaultEraser = 0;
RGBAImage*         Eraser::sNoiseTexImage = 0;


void  Eraser::sInitSharedState( )
{
    int  cswidth, blendpix;
    bool isnoisy;

    std::string prefspath = Application::instance( ).startupDirectory( ) +
        "\\Library\\" + kPrefsFilename;

    std::string  prefstext =
        Win32Tools::filesystem( ).mapTextFile(prefspath);

    StructuredReader  prefsreader(prefstext);

    std::string  prefstype = prefsreader.readFiletypeID( );

    if (prefstype != "ERASERPREFS")
        throw std::runtime_error("Eraser: preferences file isn't of "
            "correct type");

    std::string  curtok = prefsreader.readIdentifier( );

    if (curtok != "defaultEraser")
        throw std::runtime_error("Eraser: malformed preferences file");

    prefsreader.readGroupOpen( );

    bool  gotwidth = false, gotblend = false, gotnoise = false;
    while ((!gotwidth) || (!gotblend) || (!gotnoise)) {

        curtok = prefsreader.readIdentifier( );

        if (curtok == "crossSecWidth") {

            prefsreader.readAssignmentOperator( );
            cswidth = prefsreader.readInteger( );
            if (! uValidateWidth(cswidth))
                throw std::runtime_error("Eraser: cross section width value "
                    "in prefs file out of range");
            else
                gotwidth = true;
        }
        else if (curtok == "blendRegionSize") {

            if (!gotwidth)
                throw std::runtime_error("Eraser: cross section width "
                    "specification must precede blend region size "
                    "specification in prefs file");

            prefsreader.readAssignmentOperator( );
            blendpix = prefsreader.readInteger( );
            if (!uValidateBlendPix(blendpix, cswidth))
                throw std::runtime_error("Eraser: blend region size value "
                    "in preferences file is out of range");
            else
                gotblend = true;
        }
        else if (curtok == "isNoiseEnabled") {

            prefsreader.readAssignmentOperator( );
            isnoisy = prefsreader.readBoolean( );
            gotnoise = true;
        }
    } /* while we haven't got all necessary info */

    std::string  ipd =
        Application::instance( ).startupDirectory( ) + "\\Library";
    std::string  noiseimgpath = ipd + "\\GrayNoise_256.tga";

    sNoiseTexImage =
        RGBAImage::createMappedImage(RGBAImage::kTargaImageFormat,
        noiseimgpath);

    sDefaultEraser = new Eraser(cswidth, blendpix, isnoisy);
}



const Eraser&  Eraser::defaultEraser( )
{
    if (!sDefaultEraser)
        sInitSharedState( );

    return *sDefaultEraser;
}



void  Eraser::saveAsDefault(const Eraser& deferaser)
{
    std::string prefspath = Application::instance( ).startupDirectory( ) +
        "\\Library\\" + kPrefsFilename;

    std::ofstream  prefsstream(prefspath.c_str( ));

    prefsstream << "(# ERASERPREFS)\n\ndefaultEraser {\n\n";

    prefsstream << "    crossSecWidth    := " << deferaser.width( ) << "\n";
    prefsstream << "    blendRegionSize  := " << deferaser.blendPixels( ) <<
        "\n";
    prefsstream << "    isNoiseEnabled   := ";
    if (deferaser.isEdgeNoiseEnabled( ))
        prefsstream << "true\n";
    else
        prefsstream << "false\n";

    prefsstream << "}\n";

    prefsstream.close( );
}



Eraser::Eraser(int cswidth, int blendpix, bool isnoisy)
    : fWidth(cswidth), fBlendPixels(blendpix), fIsEdgeNoiseEnabled(isnoisy),
      fNumGeomVerts(0), fGPU(0)
{
    uGenerateGeometry( );
}



Eraser::Eraser( )
    : fGPU(0)
{
    if (! sDefaultEraser)
        sInitSharedState( );

    fWidth = sDefaultEraser->fWidth;
    fBlendPixels = sDefaultEraser->fBlendPixels;
    fIsEdgeNoiseEnabled = sDefaultEraser->fIsEdgeNoiseEnabled;

    uGenerateGeometry( );
}



Eraser::Eraser(const Eraser& src)
    : fGPU(0), fWidth(src.fWidth), fBlendPixels(src.fBlendPixels),
      fIsEdgeNoiseEnabled(src.fIsEdgeNoiseEnabled),
      fNumGeomVerts(0), fVertexProgID(0), fFragmentProgID(0),
      fProgramSetID(0)
{
    uGenerateGeometry( );
}



Eraser::~Eraser( )
{
}



const Eraser&  Eraser::operator=(const Eraser& src)
{
    fWidth = src.fWidth;
    fBlendPixels = src.fBlendPixels;
    fIsEdgeNoiseEnabled = src.fIsEdgeNoiseEnabled;

    uGenerateGeometry( );

    return *this;
}



int  Eraser::width( ) const
{
    return fWidth;
}



void  Eraser::setWidth(int width)
{
    if (! uValidateWidth(width))
        throw std::logic_error("Eraser: setWidth( ): width value out of "
            "range");
    else
        fWidth = width;

    uGenerateGeometry( );
}



int  Eraser::blendPixels( ) const
{
    return fBlendPixels;
}



void  Eraser::setBlendPixels(int blendpix)
{
    if (! uValidateBlendPix(blendpix, fWidth))
        throw std::logic_error("Eraser: setBlendPixels( ): blend pixel "
            "count out of range");
    else
        fBlendPixels = blendpix;
}



bool  Eraser::isEdgeNoiseEnabled( ) const
{
    return fIsEdgeNoiseEnabled;
}



void  Eraser::enableEdgeNoise( )
{
    fIsEdgeNoiseEnabled = true;
}



void  Eraser::disableEdgeNoise( )
{
    fIsEdgeNoiseEnabled = false;
}



std::string  Eraser::description( ) const
{
    std::ostringstream  descstream;

    descstream << "(" << fWidth << " pixels, " << fBlendPixels <<
        " blended, ";

    if (fIsEdgeNoiseEnabled)
        descstream << "noise enabled";
    else
        descstream << "noise disabled";
    descstream << ")";

    return descstream.str( );
}



bool  Eraser::uValidateWidth(int width)
{
    if ((width < kMinWidth) || (width > kMaxWidth))
        return false;
    else
        return true;
}



bool  Eraser::uValidateBlendPix(int blendpix, int width)
{
    int maxblendpix = width / 4;

    if ((blendpix < kMinBlendPix) || (blendpix > maxblendpix))
        return false;
    else
        return true;
}



void    Eraser::uGenerateGeometry( )
{
    const float  numwedgesf = (static_cast<float>(fWidth) * kPi) / 2.0f;
    const int    numwedges = static_cast<int>(numwedgesf);

    float  thetastep = (2.0f * kPi) / static_cast<float>(numwedges);

    fGeometryData[0] = 0.0f;
    fGeometryData[1] = 0.0f;

    float  theta = 0.0f;
    float  radius = static_cast<float>(fWidth) / 2.0f;
    int    slotticker = 2;

    for (int i = 0; i <= numwedges; i++) {

        fGeometryData[slotticker++] = radius * std::cos(theta);
        fGeometryData[slotticker++] = radius * std::sin(theta);

        theta += thetastep;
    }

    fNumGeomVerts = slotticker / 2;
}



void  Eraser::erase(const Vector2Di& pt)
{
    if (!isLinked( ))
        throw std::logic_error("Eraser: can't erase: this Eraser isn't "
            "linked");

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix( );
    glLoadIdentity( );
    glTranslatef(static_cast<float>(pt.x), static_cast<float>(pt.y), 0.0f);

    glColor4f(1.0f, 1.0f, 1.0f, 1.0f);

    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(2, GL_FLOAT, 0, fGeometryData);
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, fNoiseTexID);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glUseProgram(fProgramSetID);
    glUniform1f(fRadiusUnifLoc, static_cast<float>(fWidth / 2));
    glUniform1f(fBlendPixUnifLoc, static_cast<float>(fBlendPixels));
    glUniform1i(fNoiseSamplerUnifLoc, 0);
    if (fIsEdgeNoiseEnabled)
        glUniform1i(fEnableNoiseUnifLoc, 1);
    else
        glUniform1i(fEnableNoiseUnifLoc, 0);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glDrawArrays(GL_TRIANGLE_FAN, 0, fNumGeomVerts);
    glDisable(GL_BLEND);
    glUseProgram(0);
    glDisableClientState(GL_VERTEX_ARRAY);

    glPopMatrix( );
}



bool  Eraser::isLinked( ) const
{
    if (fGPU)
        return true;
    else
        return false;
}



GPUInterface&  Eraser::linkedGPU( ) const
{
    if (!isLinked( ))
        throw std::logic_error("Eraser: can't get linked GPU reference: "
            "this eraser isn't linked");

    return *fGPU;
}



void  Eraser::link(GPUInterface& gpu)
{
    if (isLinked( ))
        throw std::logic_error("Eraser: can't link: this eraser is already"
            " linked");

    std::string  ipd = Application::instance( ).startupDirectory( ) +
        "\\Library";

    gpu.setImagingProgramsDirectory(ipd);

    fVertexProgID = gpu.prepareVertexProgram("EraserV.glsl");
    fFragmentProgID = gpu.prepareFragmentProgram("EraserF.glsl");

    GLuint  programs[2] = { fVertexProgID, fFragmentProgID };

    fProgramSetID = gpu.prepareProgramSet(programs, 2);

    glUseProgram(fProgramSetID);
    fRadiusUnifLoc = glGetUniformLocation(fProgramSetID, "Radius");
    fNoiseSamplerUnifLoc = glGetUniformLocation(fProgramSetID, "NoiseTex");
    fEnableNoiseUnifLoc = glGetUniformLocation(fProgramSetID,
        "IsNoiseEnabled");
    fBlendPixUnifLoc = glGetUniformLocation(fProgramSetID, "NumBlendPixels");
    glUseProgram(0);

    glActiveTexture(GL_TEXTURE0);
    fNoiseTexID = gpu.downloadTexture(*sNoiseTexImage);

    fGPU = &gpu;
}



void  Eraser::unlink( )
{
    if (!isLinked( ))
        throw std::logic_error("Eraser: can't unlink: this eraser isn't "
            "linked");

    glDeleteShader(fVertexProgID);
    glDeleteShader(fFragmentProgID);
    glDeleteProgram(fProgramSetID);
    glDeleteTextures(1, &fNoiseTexID);

    fGPU = 0;
}
//////////////////////////////////////////////////////////////////////////////
// END  Eraser                                                              //
//////////////////////////////////////////////////////////////////////////////
