/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// BrushOptionsDialog.cpp                                                  //
//                                                                         //
// Implements methods in class BrushOptionsDialog only                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  <unversioned module>                                                   //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Copyright (c) 2008 Lucas Stephen Beeler. All rights reserved.           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "BrushOptionsDialog.h"
#include "S580Application.h"
#include "Brushes.h"
#include "resource.h"
#include <sstream>
#include <cmath>
#include <fstream>
#include <iostream>
#include <iomanip>

BrushOptionsDialog::BrushOptionsDialog(S580AppController& parent)
    : fParent(&parent), fIsEditInProgress(false)
{
    fView.reset(new DialogThingy(parent.mainWindowView( ),
        kBrushOptionsDialogID));
    delete & fView->installController(*this);
}



BrushOptionsDialog::~BrushOptionsDialog( )
{
}



void  BrushOptionsDialog::layoutSubthingies(CompositeThingy& sender)
{
}



void  BrushOptionsDialog::enableSubthingies(CompositeThingy& sender)
{
}



void  BrushOptionsDialog::disableSubthingies(CompositeThingy& sender)
{
}



void  BrushOptionsDialog::thingyItemActivated(DialogThingy& sender, int itemid)
{
    if (itemid == kBODoneButtonID)
        uProcessDoneButtonClick( );
    else if (itemid == kBOBrushListID)
        uProcessListBoxSelChange( );
    else if (itemid == kBOCrossSectionSliderID)
        uProcessWidthSliderAction( );
    else if (itemid == kBOBlendPixSliderID)
        uProcessBlendSliderAction( );
    else if (itemid == kBOSourceImageSetButtonID)
        uProcessSetSourceClick( );
    else if (itemid == kBOSimMapSetButtonID)
        uProcessSetSimMapClick( );
    else if (itemid == kBONicknameEditID)
        uProcessNameEditAction( );
    else if (itemid == kBORenameButtonID)
        uProcessRenameButtonClick( );
    else if (itemid == kBOEnableNoiseCheckboxID)
        uProcessNoiseCheckboxClick( );
    else if (itemid == kBONewButtonID)
        uProcessNewButtonClick( );
    else if (itemid == kBODelButtonID)
        uProcessDeleteButtonClick( );
    else if (itemid == kBOCloneButtonID)
        uProcessCloneButtonClick( );
    else if (itemid == kBOJumpParamSliderID)
        uProcessJumpParamSliderAction( );
}



void  BrushOptionsDialog::thingyClosed(DialogThingy& sender)
{
    sender.stopInteraction( );
}



void  BrushOptionsDialog::thingyInteractionStarted(DialogThingy& sender)
{
    sender.themeItem(kBOBrushListID);
    sender.themeItem(kBONewButtonID);
    sender.themeItem(kBODelButtonID);
    sender.themeItem(kBOCloneButtonID);
    sender.themeItem(kBODoneButtonID);
    sender.themeItem(kBOSourceImageSetButtonID);
    sender.themeItem(kBOSimMapSetButtonID);
    sender.themeItem(kBOCrossSectionSliderID);
    sender.themeItem(kBOBlendPixSliderID);
    sender.themeItem(kBOEnableNoiseCheckboxID);
    sender.themeItem(kBOSourceImageGroupID);
    sender.themeItem(kBOSimMapGroupID);
    sender.themeItem(kBOCrossSectionGroupID);
    sender.themeItem(kBOBlendPixGroupID);
    sender.themeItem(kBOAppearGroupID);
    sender.themeItem(kBOSourceImageReadoutID);
    sender.themeItem(kBOSimMapReadoutID);
    sender.themeItem(kBOCrossSectionReadoutID);
    sender.themeItem(kBOBlendPixReadoutID);
    sender.themeItem(kBONicknameGroupID);
    sender.themeItem(kBONicknameEditID);
    sender.themeItem(kBORenameButtonID);
    sender.themeItem(kBOJumpParamGroupID);
    sender.themeItem(kBOJumpParamReadoutID);
    sender.themeItem(kEnterKeyTextResID);

    uPopulateBrushListBox( );

    fFocusBrush.reset(new Brush(parent( ).brushManager( ).activeBrush( )));
    fFocusBrushSrcKey = parent( ).brushManager( ).activeBrush( ).name( );

    uInitWidthAssembly( );
    uInitBlendPixAssembly( );
    uInitJumpParamAssembly( );

    uDisableRenameButton( );

    uSelectBrushListItem(fFocusBrushSrcKey);
    uSyncDisplayPaneToSelection( );

    if (parent( ).brushManager( ).numBrushes( ) < 2)
        uDisableDeleteButton( );
}



void  BrushOptionsDialog::thingyInteractionStopped(DialogThingy&)
{
    Brush*  focus = fFocusBrush.release( );
    delete focus;
    fFocusBrush.reset(0);
}



void  BrushOptionsDialog::runInteraction( )
{
    fView->startInteraction( );
}



S580AppController&  BrushOptionsDialog::parent( )
{
    return *fParent;
}



void  BrushOptionsDialog::uAddBrushListItem(const std::string& itemtext)
{
    HWND  listwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBrushListID);
    SendMessage(listwin, LB_INSERTSTRING, -1, (LPARAM)(itemtext.c_str( )));
}



int  BrushOptionsDialog::uGetBrushListItemNumber(const std::string& name)
{
    HWND  listwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBrushListID);

    LRESULT  getresult = SendMessage(listwin, LB_FINDSTRINGEXACT,
        (WPARAM) -1, (LPARAM) name.c_str( ));

    if (getresult == LB_ERR)
        throw std::logic_error("BrushOptionsDialog: can't get brush list "
            "item number of '" + name + "': string not present in list");

    return (int)(getresult);
}



void  BrushOptionsDialog::uSelectBrushListItem(const std::string& name)
{
    int  itemnum = uGetBrushListItemNumber(name);
    uSelectBrushListItem(itemnum);
}



void  BrushOptionsDialog::uSelectBrushListItem(int itemnum)
{
    HWND  listwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBrushListID);

    LRESULT  setresult = SendMessage(listwin, LB_SETCURSEL, itemnum, 0);

    if (setresult == LB_ERR)
        throw std::logic_error("BrushOptionsDialog: can't set brush list "
            "selection by number: item number is out of range");
}



void  BrushOptionsDialog::uSetSrcImgReadoutText(const std::string& text)
{
    HWND  readoutwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOSourceImageReadoutID);

    Win32Tools::setWindowTextString(readoutwin, text);
}



void  BrushOptionsDialog::uSetSimMapReadoutText(const std::string& text)
{
    HWND  readoutwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOSimMapReadoutID);

    Win32Tools::setWindowTextString(readoutwin, text);
}



void  BrushOptionsDialog::uSetNameEditText(const std::string& text)
{
    HWND  editwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBONicknameEditID);

    Win32Tools::setWindowTextString(editwin, text);
}



std::string  BrushOptionsDialog::uGetBrushListSelection( ) const
{
    HWND  listwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBrushListID);

    LRESULT  selresult = SendMessage(listwin, LB_GETCURSEL, 0, 0);

    if (selresult == LB_ERR)
        throw std::logic_error("BrushOptionsDialog: can't get brush list "
            "selection: no item is selected");

    return uGetBrushListItemText(selresult);
}



std::string  BrushOptionsDialog::uGetBrushListItemText(int itemnum) const
{
    HWND  listwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBrushListID);

    char  getbuffer [256];

    LRESULT  getresult = SendMessage(listwin, LB_GETTEXT, itemnum,
        (LPARAM) getbuffer);

    if (getresult == LB_ERR)
        throw std::logic_error("BrushOptionsDialog: can't get brush list "
            "item text by index number: index is invalid");

    return std::string(getbuffer);
}



void  BrushOptionsDialog::uSyncDisplayPaneToSelection( )
{
    uSetNameEditText(fFocusBrush->name( ));

    std::string  srcname =
        Win32Tools::simplifyFilename(fFocusBrush->sourceImagePath( ));
    uSetSrcImgReadoutText(srcname);

    std::string  simname =
        Win32Tools::simplifyFilename(fFocusBrush->similarityMapPath( ));
    uSetSimMapReadoutText(simname);

    uSetWidthAssemblyValue(fFocusBrush->width( ));

    uSetBlendPixRangeMax(fFocusBrush->width( ) / 4);

    uSetBlendPixAssemblyValue(fFocusBrush->blendPixels( ));

    uSetEdgeNoiseCheckState(fFocusBrush->isEdgeNoiseEnabled( ));

    uSetJumpParamAssemblyValue(fFocusBrush->jumpParameter( ));
}




void  BrushOptionsDialog::uSetWidthReadoutValue(int width)
{
    HWND  readoutwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOCrossSectionReadoutID);

    std::ostringstream  widthstream;

    widthstream << width << " px";

    Win32Tools::setWindowTextString(readoutwin, widthstream.str( ));
}




void  BrushOptionsDialog::uInitWidthAssembly( )
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOCrossSectionSliderID);

    SendMessage(sliderwin, TBM_SETRANGEMIN, (WPARAM) TRUE,
        (LPARAM) 2);
    SendMessage(sliderwin, TBM_SETRANGEMAX, (WPARAM) TRUE,
        (LPARAM) 64);
    SendMessage(sliderwin, TBM_SETTICFREQ, (WPARAM) 2, 0);
}



void  BrushOptionsDialog::uSetWidthAssemblyValue(int val)
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOCrossSectionSliderID);

    SendMessage(sliderwin, TBM_SETPOS, (WPARAM) TRUE,
        (LPARAM) val / 2);

    uSetWidthReadoutValue(val);
}




void   BrushOptionsDialog::uSetBlendPixReadoutValue(int val)
{
    HWND  readoutwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBlendPixReadoutID);

    std::ostringstream  valstream;

    valstream << val << " px";

    Win32Tools::setWindowTextString(readoutwin, valstream.str( ));
}



void  BrushOptionsDialog::uSetBlendPixAssemblyValue(int val)
{
    uSetBlendPixSliderValue(val);
    uSetBlendPixReadoutValue(val);
}



void  BrushOptionsDialog::uSetBlendPixRangeMax(int rmax)
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBlendPixSliderID);

    SendMessage(sliderwin, TBM_SETRANGEMAX, (WPARAM) TRUE,
        (LPARAM) rmax);

    if (rmax <= 1)
        EnableWindow(sliderwin, FALSE);
    else
        EnableWindow(sliderwin, TRUE);
}




void  BrushOptionsDialog::uSetEdgeNoiseCheckState(bool state)
{
    HWND  checkwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOEnableNoiseCheckboxID);

    if (state)
        SendMessage(checkwin, BM_SETCHECK, (WPARAM) BST_CHECKED, 0);
    else
        SendMessage(checkwin, BM_SETCHECK, (WPARAM) BST_UNCHECKED, 0);
}



void  BrushOptionsDialog::uProcessListBoxSelChange( )
{
    uSaveFocusBrushData( );

    std::string  newselname = uGetBrushListSelection( );

    Brush*  oldfocus = fFocusBrush.release( );
    fFocusBrush.reset(
        new Brush(parent( ).brushManager( ).find(newselname)));
    delete oldfocus;

    fFocusBrushSrcKey = newselname;

    uSyncDisplayPaneToSelection( );
}



void  BrushOptionsDialog::uProcessWidthSliderAction( )
{
    int sliderval = uGetWidthSliderValue( );

    uSetWidthReadoutValue(sliderval);
    fFocusBrush->setWidth(sliderval);
    
    int  newblendmax = sliderval / 4;

    int  oldblendpix = uGetBlendPixSliderValue( );
    int  newblendpix = clamp(oldblendpix, 1, newblendmax);

    if (oldblendpix != newblendpix) {

        uSetBlendPixSliderValue(newblendpix);
        uSetBlendPixReadoutValue(newblendpix);
        fFocusBrush->setBlendPixels(newblendpix);
    }

    uSetBlendPixRangeMax(newblendmax);
}



void  BrushOptionsDialog::uProcessBlendSliderAction( )
{
    int sliderval = uGetBlendPixSliderValue( );

    uSetBlendPixReadoutValue(sliderval);

    fFocusBrush->setBlendPixels(sliderval);
}




int  BrushOptionsDialog::uGetWidthSliderValue( ) const
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOCrossSectionSliderID);

    return 2 * ((int) SendMessage(sliderwin, TBM_GETPOS, 0, 0));
}



int  BrushOptionsDialog::uGetBlendPixSliderValue( ) const
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBlendPixSliderID);

    return ((int) SendMessage(sliderwin, TBM_GETPOS, 0, 0));
}



void  BrushOptionsDialog::uSetBlendPixSliderValue(int val)
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBlendPixSliderID);

    SendMessage(sliderwin, TBM_SETPOS, (WPARAM) TRUE,
        (LPARAM) val);
}



void  BrushOptionsDialog::uInitBlendPixAssembly( )
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBlendPixSliderID);

    SendMessage(sliderwin, TBM_SETRANGEMIN, (WPARAM) TRUE,
        (LPARAM) 1);
}



void  BrushOptionsDialog::uProcessSetSourceClick( )
{
    OpenFileChooser  ofc;
    ofc.addFiletypeMask("Truevision TARGA (*.tga, *.tpic)", "*.tga");
    ofc.addFiletypeMask("Truevision TARGA (*.tga, *.tpic)", "*.tpic");

    ofc.setTitle("Select a Source Image");

    std::string  imgfn = ofc.runFileChoiceInteraction(*fView);

    HWND  dpeer = fView->adapter( ).primitive( );

    /* if the image filename 'imgfn' returned is the null string, then the
       user clicked 'Cancel' in the 'Open...' dialog box before choosing an
       image file to open, so just return */
    if (imgfn == "")
        return;

    /* try to load image data from the the file the user selected */
    RGBAImage*  userimg = 0;
    try {

        userimg = RGBAImage::createMappedImage(RGBAImage::kTargaImageFormat,
            imgfn);
    }
    catch (std::runtime_error&) {

        const std::string&  appname = Application::instance( ).name( );
        MessageBox(dpeer, TEXT("The file selected does not contain "
            "valid TARGA image data"), appname.c_str( ), MB_ICONERROR);

        delete userimg;
        return;
    }

    /* if execution reaches this point, then we were able read image data
       from the file chosen by the user sucessfully. But we're still not in
       the clear, since the image setting interaction can still fail if the
       image chosen by the user doesn't meet the constraints specified for
       *this, so let's check it against these constraints now */
    static  const int    kMinWidthConstraint = 16;
    static  const int    kMaxWidthConstraint = 128;
    static  const int    kMinHeightConstraint = 16;
    static  const int    kMaxHeightConstraint = 128;
    static  const float  kAspectRatioConstraint = 1.0f;
    static  const bool   kIsPowerOf2ConstEnabled = true;

    if ((userimg->width( ) < kMinWidthConstraint) ||
        (userimg->width( ) > kMaxWidthConstraint)) {

        const std::string&  appname = Application::instance( ).name( );
        std::ostringstream  msgstream;
        msgstream << "The image you've chosen is too wide.\n" <<
            "Valid source images have widths between " <<
            kMinWidthConstraint << " pixels and " << kMaxWidthConstraint <<
            " pixels, inclusive.";

        MessageBox(dpeer, msgstream.str( ).c_str( ), appname.c_str( ),
            MB_ICONERROR);

        delete userimg;
        return;
    }

    if ((userimg->height( ) < kMinHeightConstraint) ||
        (userimg->height( ) > kMaxHeightConstraint)) {

        const std::string&  appname = Application::instance( ).name( );
        std::ostringstream  msgstream;
        msgstream << "The image you've chosen is too tall.\n" <<
            "Valid source images have heights between " <<
            kMinHeightConstraint << " pixels and " << kMaxHeightConstraint
            << " pixels, inclusive.";

        MessageBox(dpeer, msgstream.str( ).c_str( ), appname.c_str( ),
            MB_ICONERROR);

        delete userimg;
        return;
    }

    float imgwf = static_cast<float>(userimg->width( ));
    float imghf = static_cast<float>(userimg->height( ));
    float rat = (imgwf / imghf);

    if (rat != kAspectRatioConstraint) {

        const std::string&  appname = Application::instance( ).name( );
        std::ostringstream  msgstream;
        msgstream << "The image you've chosen has the wrong aspect " <<
            "ratio.\nValid source images must have an aspect ratio (" <<
            "width / height) of " << kAspectRatioConstraint << ".";

        MessageBox(dpeer, msgstream.str( ).c_str( ), appname.c_str( ),
            MB_ICONERROR);

        delete userimg;
        return;
    }

    if (kIsPowerOf2ConstEnabled) {

        double lgwidth = std::log(static_cast<double>(userimg->width( ))) /
            std::log(2.0);
        double lgheight = std::log(static_cast<double>(userimg->height( ))) /
            std::log(2.0);

        double fracwidth;
        std::modf(lgwidth, &fracwidth);
        fracwidth = lgwidth - fracwidth;

        double fracheight;
        std::modf(lgheight, &fracheight);
        fracheight = lgheight - fracheight;

        if (fracwidth != 0.0) {

            const std::string&  appname = Application::instance( ).name( );
            std::ostringstream  msgstream;
            msgstream << "The image you've chosen has a width that is not "
                "a power of two.\nValid source images must have widths and "
                "heights that are powers of two.";

            MessageBox(dpeer, msgstream.str( ).c_str( ), appname.c_str( ),
                MB_ICONERROR);

            delete userimg;
            return;
        }

        if (fracheight != 0.0) {

            const std::string&  appname = Application::instance( ).name( );
            std::ostringstream  msgstream;
            msgstream << "The image you've chosen has a height that is not "
                "a power of two.\nValid source images must have widths and "
                "heights that are powers of two.";

            MessageBox(dpeer, msgstream.str( ).c_str( ), appname.c_str( ),
                MB_ICONERROR);

            delete userimg;
            return;
        }
    }

    std::string  simplefn = Win32Tools::simplifyFilename(imgfn);
    uSetSrcImgReadoutText(simplefn);
    fFocusBrush->setSourceImagePath(imgfn);
    delete userimg;
}



void  BrushOptionsDialog::uProcessSetSimMapClick( )
{
    OpenFileChooser  ofc;
    ofc.addFiletypeMask("Image Similarity Map (*.simset)", "*.simset");

    ofc.setTitle("Select a Similarity Map");

    std::string  simmapfn = ofc.runFileChoiceInteraction(*fView);

    if (simmapfn == "")
        return;

    HWND  dpeer = fView->adapter( ).primitive( );

    std::ifstream  simstream(simmapfn.c_str( ));

    if (!simstream) {

        const std::string&  appname = Application::instance( ).name( );
        MessageBox(dpeer, "You don't have permission to open the file "
            "that you selected,\nor the file does not exist. Please choose "
            "another.", appname.c_str( ), MB_ICONERROR);
        simstream.close( );
        return;
    }

    simstream.close( );

    std::string  simplesimmapfn = Win32Tools::simplifyFilename(simmapfn);
    uSetSimMapReadoutText(simplesimmapfn);
    fFocusBrush->setSimilarityMapPath(simmapfn);
}



void  BrushOptionsDialog::uDisableRenameButton( )
{
    HWND  buttonwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBORenameButtonID);

    EnableWindow(buttonwin, FALSE);
}



void  BrushOptionsDialog::uEnableRenameButton( )
{
    HWND  buttonwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBORenameButtonID);

    EnableWindow(buttonwin, TRUE);
}



void  BrushOptionsDialog::uMakeDoneButtonDefault( )
{
    HWND  renamewin = GetDlgItem(fView->adapter( ).primitive( ),
        kBORenameButtonID);
    HWND  donewin = GetDlgItem(fView->adapter( ).primitive( ),
        kBODoneButtonID);
    HWND  dialogwin = fView->adapter( ).primitive( );

    SendMessage(renamewin, BM_SETSTYLE, BS_PUSHBUTTON, MAKELPARAM(TRUE, 0));
    SendMessage(dialogwin, DM_SETDEFID, kBODoneButtonID, 0);
}



void  BrushOptionsDialog::uMakeRenameButtonDefault( )
{
    HWND  renamewin = GetDlgItem(fView->adapter( ).primitive( ),
        kBORenameButtonID);
    HWND  donewin = GetDlgItem(fView->adapter( ).primitive( ),
        kBODoneButtonID);
    HWND  dialogwin = fView->adapter( ).primitive( );

    SendMessage(donewin, BM_SETSTYLE, BS_PUSHBUTTON, MAKELPARAM(TRUE, 0));
    SendMessage(dialogwin, DM_SETDEFID, kBORenameButtonID, 0);
}



void  BrushOptionsDialog::uProcessNameEditAction( )
{
    HWND  editwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBONicknameEditID);

    if (editwin == GetFocus( )) {

        uEnableRenameButton( );
        uMakeRenameButtonDefault( );
        fIsEditInProgress = true;
    }
    else {

        if (fIsEditInProgress) {

            uMakeDoneButtonDefault( );
            uDisableRenameButton( );

            fView->setItemText(kBONicknameEditID, fFocusBrush->name( ));
        }

        fIsEditInProgress = false;
    } /* else the edit window doesn't have keyboard focus */
}



void  BrushOptionsDialog::uProcessRenameButtonClick( )
{
    HWND  dialogwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBODoneButtonID);

    std::string  edittext = fView->itemText(kBONicknameEditID);
    if (edittext != fFocusBrush->name( )) {

        fFocusBrush->setName(edittext);
    }

    SetFocus(dialogwin);

    uSaveFocusBrushData( );
}



void  BrushOptionsDialog::uSaveFocusBrushData( )
{
    parent( ).brushManager( ).remove(fFocusBrushSrcKey);
    Brush*  focuscopy = new Brush(*fFocusBrush);
    parent( ).brushManager( ).install(focuscopy);

    if (fFocusBrushSrcKey != fFocusBrush->name( )) {

        uClearBrushListBox( );
        uPopulateBrushListBox( );
        uSelectBrushListItem(fFocusBrush->name( ));

        parent( ).brushManager( ).setActiveBrush(fFocusBrush->name( ));

        fFocusBrushSrcKey = fFocusBrush->name( );
    }
}



void  BrushOptionsDialog::uClearBrushListBox( )
{
    HWND  brushlistwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBrushListID);

    SendMessage(brushlistwin, LB_RESETCONTENT, 0, 0);
}



void  BrushOptionsDialog::uProcessNoiseCheckboxClick( )
{
    HWND  checkwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOEnableNoiseCheckboxID);

    LRESULT checkstate = SendMessage(checkwin, BM_GETCHECK, 0, 0);

    if (checkstate == BST_CHECKED)
        fFocusBrush->enableEdgeNoise( );
    else if (checkstate == BST_UNCHECKED)
        fFocusBrush->disableEdgeNoise( );
}



void  BrushOptionsDialog::uProcessDoneButtonClick( )
{
    uSaveFocusBrushData( );
    parent( ).brushManager( ).setActiveBrush(fFocusBrush->name( ));
    fView->stopInteraction( );
}



void  BrushOptionsDialog::uProcessNewButtonClick( )
{
    /* create a new brush */
    Brush&  newbrush = parent( ).brushManager( ).newBrush( );

    /* save the existing focus brush information */
    parent( ).brushManager( ).remove(fFocusBrushSrcKey);
    Brush*  focuscopy = new Brush(*fFocusBrush);
    parent( ).brushManager( ).install(focuscopy);

    /* make the new brush the focus brush */
    *fFocusBrush = newbrush;
    fFocusBrushSrcKey = newbrush.name( );

    /* clear and repopulate the brush list box, selecting the new
       focus brush */
    uClearBrushListBox( );
    uPopulateBrushListBox( );

    uSelectBrushListItem(fFocusBrush->name( ));
    uSyncDisplayPaneToSelection( );

    if (parent( ).brushManager( ).numBrushes( ) > 1)
        uEnableDeleteButton( );
}



void  BrushOptionsDialog::uProcessDeleteButtonClick( )
{
    HWND  listwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOBrushListID);

    LRESULT  selnum =  (int) SendMessage(listwin, LB_GETCURSEL, 0, 0);

    if (selnum == LB_ERR)
        throw std::logic_error("BrushOptionsDialog: can't get brush list "
            "selected item for deletion: no item is selected");

    int lastbrush = parent( ).brushManager( ).numBrushes( ) - 1;

    if (selnum == lastbrush)
        selnum--;

    parent( ).brushManager( ).remove(uGetBrushListSelection( ));

    uClearBrushListBox( );
    uPopulateBrushListBox( );
    
    SendMessage(listwin, LB_SETCURSEL, (WPARAM)(selnum), 0);

    *fFocusBrush =
        parent( ).brushManager( ).find(uGetBrushListSelection( ));
    fFocusBrushSrcKey = fFocusBrush->name( );

    uSyncDisplayPaneToSelection( );

    if (parent( ).brushManager( ).numBrushes( ) < 2)
        uDisableDeleteButton( );
}



void  BrushOptionsDialog::uPopulateBrushListBox( )
{
    std::vector<std::string>  brmanifest =
        parent( ).brushManager( ).manifest( );
    for (int i = 0; i < brmanifest.size( ); i++)
        uAddBrushListItem(brmanifest[i]);
}



void  BrushOptionsDialog::uEnableDeleteButton( )
{
    HWND  delbutwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBODelButtonID);

    EnableWindow(delbutwin, TRUE);
}



void  BrushOptionsDialog::uDisableDeleteButton( )
{
    HWND  delbutwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBODelButtonID);

    EnableWindow(delbutwin, FALSE);
}



void  BrushOptionsDialog::uProcessCloneButtonClick( )
{
    /* save the existing focus brush information */
    parent( ).brushManager( ).remove(fFocusBrushSrcKey);
    Brush*  focuscopy = new Brush(*fFocusBrush);
    parent( ).brushManager( ).install(focuscopy);

    /* create a new brush */
    Brush&  newbrush =
        parent( ).brushManager( ).cloneBrush(fFocusBrush->name( ));

    /* make the new brush the focus brush */
    *fFocusBrush = newbrush;
    fFocusBrushSrcKey = newbrush.name( );

    /* clear and repopulate the brush list box, selecting the new
       focus brush */
    uClearBrushListBox( );
    uPopulateBrushListBox( );

    uSelectBrushListItem(fFocusBrush->name( ));
    uSyncDisplayPaneToSelection( );

    if (parent( ).brushManager( ).numBrushes( ) > 1)
        uEnableDeleteButton( );
}



void  BrushOptionsDialog::uInitJumpParamAssembly( )
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOJumpParamSliderID);

    SendMessage(sliderwin, TBM_SETRANGEMIN, (WPARAM) TRUE,
        (LPARAM) 0);
    SendMessage(sliderwin, TBM_SETRANGEMAX, (WPARAM) TRUE,
        (LPARAM) 66);
    SendMessage(sliderwin, TBM_SETTICFREQ, (WPARAM) 2, 0);
}




void  BrushOptionsDialog::uProcessJumpParamSliderAction( )
{
    float  val = uGetJumpParamSliderValue( );

    uSetJumpParamReadoutValue(val);

    fFocusBrush->setJumpParameter(val);
}



float  BrushOptionsDialog::uGetJumpParamSliderValue( ) const
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOJumpParamSliderID);

    int slidervali = ((int) SendMessage(sliderwin, TBM_GETPOS, 0, 0));

    float slidervalf = (static_cast<float>(slidervali) * 0.025f) + 0.85f;

    return slidervalf;
}




void  BrushOptionsDialog::uSetJumpParamReadoutValue(float p)
{
    std::ostringstream  valstream;

    valstream << std::fixed << std::setprecision(3) << p;

    fView->setItemText(kBOJumpParamReadoutID, valstream.str( ));
}




void  BrushOptionsDialog::uSetJumpParamAssemblyValue(float val)
{
    HWND  sliderwin = GetDlgItem(fView->adapter( ).primitive( ),
        kBOJumpParamSliderID);

    int vali = static_cast<int>(((val - 0.85f) / 0.025f) + 0.5f);

    SendMessage(sliderwin, TBM_SETPOS, (WPARAM) TRUE, (LPARAM) vali);

    uSetJumpParamReadoutValue(uGetJumpParamSliderValue( ));
}
