//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// WorkPane.cpp                                                             //
//                                                                          //
// Implements methods in the WorkPaneView class and WorkPaneController      //
// interface. Defines and implements methods in the NullWorkPaneController  //
// class.                                                                   //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 2.1.6 of Fri 3 May 2008 @ 7:56pm EDT                                //
//                                                                          //
//     changed all references of class template Vertex2D to Vector2D; see   //
//     the AffineGeometry module's header for more info                     //
//                                                                          //
// ver. 2.1.5 of Wed 26-Mar-2008 @ 5:48pm EDT                               //
//                                                                          //
//    Added UIComponent enable/disable management support to WorkPane views //
//    and controllers.                                                      //
//                                                                          //
// ver. 2.1.0 of Fri 07-Mar-2008 @ 3:31pm EST                               //
//                                                                          //
//    Added the new InsetWorkPaneView class. Class WorkPaneView modified to //
//    support configurable background coloring.                             //
//                                                                          //
// ver. 2.0.0 of Fri 16-Feb-2008 @ 12:27am EST                              //
//                                                                          //
//    This file has been around for months and has seen several evolutions  //
//    through CVS commit cycles in projects GPUISupport, PyramidBuilder,    //
//    and Synthesizer. However this current version 2.0.0 is the first      //
//    formal "version" of the file and is considered stable.                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008 Lucas Stephen Beeler. All Rights Reserved.       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "WorkPane.h"
#include <typeinfo>

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  NullWorkPaneController                                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class  NullWorkPaneController : public virtual WorkPaneController {

private:

    bool  fIsInstalled;

public:

    NullWorkPaneController( );
    virtual  ~NullWorkPaneController( );

    void  installed(WorkPaneView& sender);
    void  deinstalled(WorkPaneView& sender);
    void  layout(WorkPaneView& sender);
    void  commandPropagated(WorkPaneView& sender, HWND commandSource,
        unsigned short commandID);
    bool  isInstalled( ) const;
    void  viewEnabled(UIComponentView& sender);
    void  viewDisabled(UIComponentView& sender);
};




NullWorkPaneController::NullWorkPaneController( )
    : fIsInstalled(false)
{
}




NullWorkPaneController::~NullWorkPaneController( )
{
}




void  NullWorkPaneController::installed(WorkPaneView& sender)
{
    fIsInstalled = true;
}




void  NullWorkPaneController::deinstalled(WorkPaneView& sender)
{
    fIsInstalled = false;
}




void  NullWorkPaneController::layout(WorkPaneView& sender)
{
    ;
}




void  NullWorkPaneController::commandPropagated(WorkPaneView& sender,
    HWND commandSource, unsigned short commandID)
{
    ;
}




bool  NullWorkPaneController::isInstalled( ) const
{
    return fIsInstalled;
}




void  NullWorkPaneController::viewEnabled(UIComponentView& sender)
{
}




void  NullWorkPaneController::viewDisabled(UIComponentView& sender)
{
}
//////////////////////////////////////////////////////////////////////////////
// END  NullWorkPaneController                                              //
//////////////////////////////////////////////////////////////////////////////







//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  WorkPaneView                                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
bool  WorkPaneView::sIsWindowClassRegistered = false;
const std::string  WorkPaneView::kWindowClassName = "WorkPaneView";
const Vector2Di  WorkPaneView::kDefaultPosition = Vector2Di( );
WorkPaneView*  WorkPaneView::sObjAwaitingInit = 0;
std::map<HWND, WorkPaneView*>  WorkPaneView::sObjectMap;


void  WorkPaneView::sRegisterWindowClass( )
{
    WNDCLASS    workPaneClass;

    workPaneClass.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
    workPaneClass.lpfnWndProc = WorkPaneView::sWindowProc;
    workPaneClass.cbClsExtra = 0;
    workPaneClass.cbWndExtra = sizeof(WorkPaneView*);
    workPaneClass.hInstance = NULL;
    workPaneClass.hIcon = NULL;
    workPaneClass.hCursor = LoadCursor(NULL, IDC_ARROW);
    workPaneClass.hbrBackground = NULL;
    workPaneClass.lpszMenuName = NULL;
    workPaneClass.lpszClassName = kWindowClassName.c_str( );

    ATOM  registrationResult = RegisterClass(&workPaneClass);
    if (!registrationResult)
        throw std::runtime_error("WorkPaneView: sRegisterWindowClass( )"
            " failed to register window class");
}




LRESULT WINAPI WorkPaneView::sWindowProc(HWND targetWindow, UINT messageCode,
    WPARAM wParam, LPARAM lParam)
{
    WorkPaneView*   obj = 0;

    if (sObjectMap.find(targetWindow) != sObjectMap.end( ))
        return  sObjectMap[targetWindow]->uProcessWindowMessage(targetWindow,
            messageCode, wParam, lParam);

    switch (messageCode) {

        case WM_CREATE:
            if (sObjAwaitingInit) {

                sObjectMap[targetWindow] = sObjAwaitingInit;
                sObjAwaitingInit = 0;
                return 0;
            }
            else {

                throw std::logic_error("WorkPaneView: got a WM_CREATE "
                    "message, but no object is awaiting init");
            }
        break;
    
        default:
            return DefWindowProc(targetWindow, messageCode, wParam, lParam);
        break;
    }
}




WorkPaneView::WorkPaneView(HWND parent, unsigned short childID)
    : fWindowPrimitive(0), fController(new NullWorkPaneController( )),
      fBackgroundBrush(GetSysColorBrush(COLOR_BTNFACE)),
      fBackground(kStandardControlBackground),
      fBackgroundSolid(GetSysColor(COLOR_BTNFACE)) 
{
    if (!sIsWindowClassRegistered) {

        sRegisterWindowClass( );
        sIsWindowClassRegistered = true;
    }

    sObjAwaitingInit = this;

    fWindowPrimitive = CreateWindow(kWindowClassName.c_str( ), TEXT(""),
        WS_CHILD | WS_VISIBLE, kDefaultPosition.x, kDefaultPosition.y,
        kDefaultWidth, kDefaultHeight, parent, (HMENU)(childID), NULL, 0);
    if (!fWindowPrimitive)
        throw std::runtime_error("WorkPaneView: WorkPaneView( ) failed to "
            "create native peer window");
}




WorkPaneView::~WorkPaneView( )
{
    if (typeid(*fController) == typeid(NullWorkPaneController))
        delete fController;
    else
        fController->deinstalled(*this);

    Win32Tools::destroyPrimitive(fWindowPrimitive);
}




LRESULT  WorkPaneView::uProcessWindowMessage(HWND targetWindow,
    UINT messageCode, WPARAM wParam, LPARAM lParam)
{
    char            workBuffer[1024];
    std::string     workString;
    RECT            rect;
    PAINTSTRUCT     paintinfo;
    HDC             targetdc;

    switch (messageCode) {
    
        case WM_SIZE:
            fController->layout(*this);
            return 0;
        break;

        case WM_CTLCOLORSTATIC:
            GetClassName((HWND)(lParam), workBuffer, 1023);
            workString = workBuffer;
            if (workString == "Edit") {
                
                SetBkColor((HDC) wParam, RGB(255, 255, 255));
                return (LRESULT) GetStockObject(WHITE_BRUSH);
            }
            else if (workString == "Static") {

                if (IsWindowEnabled((HWND) lParam) == TRUE)
                    SetTextColor((HDC) wParam, GetSysColor(COLOR_BTNTEXT));
                else
                    SetTextColor((HDC) wParam, GetSysColor(COLOR_GRAYTEXT));

                SetBkColor((HDC) wParam, fBackgroundSolid);
                return (LRESULT) fBackgroundBrush;
            }
            else
                return DefWindowProc(targetWindow, messageCode, wParam,
                    lParam);
        break;

        case WM_CTLCOLORBTN:
            return (LRESULT) fBackgroundBrush;
        break;

        case WM_PAINT:
            targetdc = BeginPaint(targetWindow, &paintinfo);
            GetClientRect(targetWindow, &rect);
            FillRect(targetdc, &rect, fBackgroundBrush);
            EndPaint(targetWindow, &paintinfo);
            return 0;
        break;

        case WM_HSCROLL:
        case WM_COMMAND:
            fController->commandPropagated(*this, (HWND)(lParam),
                static_cast<unsigned short>(HIWORD(wParam)));
            return 0;
        break;

        case WM_DESTROY:
            sObjectMap.erase(targetWindow);
            return 0;
        break;

        default:
            return DefWindowProc(targetWindow, messageCode, wParam, lParam);
        break;
    }
}




unsigned short  WorkPaneView::width( ) const
{
    return Win32Tools::getWindowWidth(fWindowPrimitive);
}




void  WorkPaneView::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fWindowPrimitive, w);
}




unsigned short  WorkPaneView::height( ) const
{
    return Win32Tools::getWindowHeight(fWindowPrimitive);
}




void  WorkPaneView::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fWindowPrimitive, h);
}




unsigned short  WorkPaneView::clientWidth( ) const
{
    return Win32Tools::getClientAreaWidth(fWindowPrimitive);
}




unsigned short  WorkPaneView::clientHeight( ) const
{
    return Win32Tools::getClientAreaHeight(fWindowPrimitive);
}




Vector2Di  WorkPaneView::position( ) const
{
    RECT  paneBoundsRect;
    GetWindowRect(fWindowPrimitive, &paneBoundsRect);

    POINT  paneOrigin;

    paneOrigin.x = paneBoundsRect.left;
    paneOrigin.y = paneBoundsRect.top;

    ScreenToClient(GetParent(fWindowPrimitive), &paneOrigin);

    return Vector2Di(paneOrigin.x, paneOrigin.y);
}




void  WorkPaneView::setPosition(const Vector2Di& pos)
{
    unsigned short w = this->width( );
    unsigned short h = this->height( );

    MoveWindow(fWindowPrimitive, pos.x, pos.y, w, h, TRUE);
}




void  WorkPaneView::setPosition(unsigned short x, unsigned short y)
{
    this->setPosition(Vector2Di(x, y));
}




HWND  WorkPaneView::peerHandle( )
{
    return fWindowPrimitive;
}




void  WorkPaneView::invalidate( )
{
    RECT  clientRect;
    GetClientRect(fWindowPrimitive, &clientRect);
    
    InvalidateRect(fWindowPrimitive, &clientRect, FALSE);
    UpdateWindow(fWindowPrimitive);
}




void  WorkPaneView::show( )
{
    ShowWindow(fWindowPrimitive, SW_SHOWNORMAL);
    invalidate( );
}




void  WorkPaneView::hide( )
{
    ShowWindow(fWindowPrimitive, SW_HIDE);
}




bool  WorkPaneView::isVisible( ) const
{
    if(IsWindowVisible(fWindowPrimitive))
        return true;
    else
        return false;
}




WorkPaneController&  WorkPaneView::controller( )
{
    return *fController;
}




WorkPaneController&  WorkPaneView::installController(WorkPaneController& c)
{
    WorkPaneController*  oldController = fController;

    oldController->deinstalled(*this);

    fController = &c;

    fController->installed(*this);
    fController->layout(*this);

    return *oldController;
}




void  WorkPaneView::enable( )
{
    if (! this->isEnabled( )) {

        EnableWindow(fWindowPrimitive, TRUE);

        std::set<HWND>::iterator it;
        for (it = fAutoEnableSet.begin( ); it != fAutoEnableSet.end( ); it++)
            EnableWindow(*it, TRUE);

        fAutoEnableSet.clear( );
    }

    fController->viewEnabled(*this);
}




void  WorkPaneView::disable( )
{
    if (this->isEnabled( )) {

        fAutoEnableSet.clear( );

        std::set<HWND>::iterator it;
        for (it = fChildSet.begin( ); it != fChildSet.end( ); it++) {

            if (IsWindowEnabled(*it)) {

                fAutoEnableSet.insert(*it);
                EnableWindow(*it, FALSE);
            }
        } /* for */

        EnableWindow(fWindowPrimitive, FALSE);
    }

    fController->viewDisabled(*this);
}




bool  WorkPaneView::isEnabled( ) const
{
    if (IsWindowEnabled(fWindowPrimitive))
        return true;
    else
        return false;
}




void  WorkPaneView::manageChild(HWND c)
{
    fChildSet.insert(c);
}




void  WorkPaneView::unmanageChild(HWND c)
{
    if (isChildManaged(c))
        fChildSet.erase(c);
}




bool  WorkPaneView::isChildManaged(HWND c) const
{
    if (fChildSet.find(c) != fChildSet.end( ))
        return true;
    else
        return false;
}




WindowBackgroundKind  WorkPaneView::backgroundKind( ) const
{
    return fBackground;
}




void  WorkPaneView::setBackgroundKind(WindowBackgroundKind bgk)
{
    if (bgk == kStandardControlBackground) {

        fBackgroundBrush = GetSysColorBrush(COLOR_BTNFACE);
        fBackgroundSolid = GetSysColor(COLOR_BTNFACE);
    }
    else if (bgk == kWhiteBackground) {

        fBackgroundBrush = (HBRUSH) GetStockObject(WHITE_BRUSH);
        fBackgroundSolid = RGB(255, 255, 255);
    }
    else {

        throw std::invalid_argument("WorkPaneView: unrecognized background "
            "kind");
    }

    fBackground = bgk;
}
//////////////////////////////////////////////////////////////////////////////
// END  WorkPaneView                                                        //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  InsetWorkPaneView                                                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
InsetWorkPaneView::InsetWorkPaneView(HWND parent, unsigned short childID)
{
    fController = new NullWorkPaneController( );

    if (!sIsWindowClassRegistered) {

        sRegisterWindowClass( );
        sIsWindowClassRegistered = true;
    }

    WorkPaneView::sObjAwaitingInit = this;

    fWindowPrimitive = CreateWindowEx(WS_EX_CLIENTEDGE,
        kWindowClassName.c_str( ), TEXT(""), WS_CHILD | WS_VISIBLE,
        kDefaultPosition.x, kDefaultPosition.y, kDefaultWidth, kDefaultHeight,
        parent, (HMENU)(childID), NULL, 0);
    if (!fWindowPrimitive)
        throw std::runtime_error("InsetWorkPaneView: InsetWorkPaneView( ) "
            "failed to create native peer window");
}




InsetWorkPaneView::~InsetWorkPaneView( )
{
    Win32Tools::destroyPrimitive(fWindowPrimitive);
}
//////////////////////////////////////////////////////////////////////////////
// END  WorkPaneView                                                        //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  WorkPaneController                                                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
WorkPaneController::WorkPaneController( )
{
}




WorkPaneController::~WorkPaneController( )
{
}




WorkPaneController*  WorkPaneController::nullController( )
{
    return new NullWorkPaneController( );
}
//////////////////////////////////////////////////////////////////////////////
// END  WorkPaneController                                                  //
//////////////////////////////////////////////////////////////////////////////
