//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Win32Support.cpp                                                         //
//                                                                          //
// Implements methods in class Win32Tools, and nested classes               //
// Win32Tools::Filesystem and Win32Tools::WidgetFactory                     //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 2.10.0 of Sat 18 Oct 2008 @ 9:02pm EDT                              //
//                                                                          //
//     Added method doesFileExist( ) to class Win32Tools::Filesystem        //
//                                                                          //
// ver. 2.9.1 of Tue 20 May 2008 @ 2:38pm EDT                               //
//                                                                          //
//     makeBoldLabel( ) in Win32Tools WidgetFactory now creates a hidden    //
//     widget (i.e. the WS_VISIBLE style flag is no longer set).            //
//                                                                          //
//  older versions:                                                         //
//                                                                          //
//      older change history elided; check out an older CVS rev to get it   //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "Win32Support.h"
#include "ImageRepresentation.h"
#include <sstream>
#include <sys/types.h>
#include <sys/stat.h>
#include <cstring>
#include <fstream>

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  Win32Tools                                                       //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

HFONT                       Win32Tools::sControlFaceFont = 0;
HFONT                       Win32Tools::sBoldLabelFont = 0;
Win32Tools::Filesystem*     Win32Tools::sFSInstance = 0;
Win32Tools::WidgetFactory*  Win32Tools::sWidgetFacInstance = 0;




Win32Tools::Win32Tools( )
{
}




Win32Tools::~Win32Tools( )
{
}




std::string  Win32Tools::getWindowTextString(HWND w)
{
    unsigned  len = GetWindowTextLength(w);
    
    if (len > 0) {
    
        char*  buffer = new char[len + 1]; // len + 1 since len may not count
                                           // terminal '\0'
        
        GetWindowText(w, buffer, len + 1);
        
        std::string result = buffer;
        
        delete [ ] buffer;
        
        return result;
    }
    else {
    
        return std::string("");
    }
}




void  Win32Tools::setWindowTextString(HWND w, const std::string& s)
{
    SetWindowText(w, s.c_str( ));
}




HFONT  Win32Tools::getControlFaceFont( )
{
    if (!sControlFaceFont) {

        NONCLIENTMETRICS  env_desc;
        env_desc.cbSize = sizeof(NONCLIENTMETRICS);
        SystemParametersInfo(SPI_GETNONCLIENTMETRICS,
            sizeof(NONCLIENTMETRICS), &env_desc, 0);

        sControlFaceFont = CreateFontIndirect(&env_desc.lfMessageFont);
    }

    return sControlFaceFont;
}




void  Win32Tools::applyControlFaceProperties(HWND w)
{
    SendMessage(w, WM_SETFONT, (LPARAM)(getControlFaceFont( )), TRUE);
}




HFONT  Win32Tools::getBoldLabelFont( )
{
    if (!sBoldLabelFont) {


        NONCLIENTMETRICS  env_desc;
        env_desc.cbSize = sizeof(NONCLIENTMETRICS);
        SystemParametersInfo(SPI_GETNONCLIENTMETRICS,
            sizeof(NONCLIENTMETRICS), &env_desc, 0);

        env_desc.lfMessageFont.lfWeight = FW_BOLD;

        sBoldLabelFont = CreateFontIndirect(&env_desc.lfMessageFont);
    }

    return sBoldLabelFont;
}




void  Win32Tools::applyBoldLabelProperties(HWND w)
{
    SendMessage(w, WM_SETFONT, (LPARAM)(getBoldLabelFont( )), TRUE);
}




void  Win32Tools::initializeGLClientState( )
{
    glewInit( );
}




Win32Tools::WidgetFactory&  Win32Tools::widgetFactory( )
{
    if (!sWidgetFacInstance)
        sWidgetFacInstance = new WidgetFactory( );

    return *sWidgetFacInstance;
}




Win32Tools::Filesystem&  Win32Tools::filesystem( )
{
    if (!sFSInstance)
        sFSInstance = new Filesystem( );

    return *sFSInstance;
}




unsigned short  Win32Tools::getWindowHeight(HWND w)
{
    RECT  r;
    BOOL  gotRectOk = GetWindowRect(w, &r);

    if (!gotRectOk)
        throw std::runtime_error("Win32Tools: getWindowHeight( ): "
            "execution of GetWindowRect( ) failed");

    return static_cast<unsigned short>(r.bottom - r.top);
}




unsigned short  Win32Tools::getWindowWidth(HWND w)
{
    RECT  r;
    BOOL  gotRectOk = GetWindowRect(w, &r);

    if (!gotRectOk)
        throw std::runtime_error("Win32Tools: getWindowWidth( ): "
            "execution of GetWindowRect( ) failed");

    return static_cast<unsigned short>(r.right - r.left);
}




unsigned short  Win32Tools::getClientAreaHeight(HWND w)
{
    RECT  r;
    BOOL  gotRectOk = GetClientRect(w, &r);

    if (!gotRectOk)
        throw std::runtime_error("Win32Tools: getWindowHeight( ): "
            "execution of GetClientRect( ) failed");

    return static_cast<unsigned short>(r.bottom - r.top);
}




unsigned short  Win32Tools::getClientAreaWidth(HWND w)
{
    RECT  r;
    BOOL  gotRectOk = GetClientRect(w, &r);

    if (!gotRectOk)
        throw std::runtime_error("Win32Tools: getWindowHeight( ): "
            "execution of GetClientRect( ) failed");

    return static_cast<unsigned short>(r.right - r.left);
}




void  Win32Tools::setWindowOrigin(HWND win, unsigned short x,
        unsigned short y)
{
    int wid = getWindowWidth(win);
    int ht = getWindowHeight(win);

    MoveWindow(win, x, y, wid, ht, TRUE);
}




Vector2Di  Win32Tools::getWindowOrigin(HWND w)
{
    RECT  childrect;
    BOOL  gotRectOk = GetWindowRect(w, &childrect);
    if (!gotRectOk)
        throw std::runtime_error("Win32Tools: getWindowOrigin( ): "
            "execution of GetWindowRect( ) failed");

    POINT  parentogn;
    parentogn.x = 0;
    parentogn.y = 0;
    ClientToScreen(GetParent(w), &parentogn);

    return Vector2Di(childrect.left - parentogn.x, childrect.top -
        parentogn.y);
}




void  Win32Tools::setWindowWidth(HWND win, unsigned short wd)
{
    int  ht = getWindowHeight(win);

    SetWindowPos(win, 0, 0, 0, wd, ht, SWP_NOACTIVATE | SWP_NOMOVE |
        SWP_NOOWNERZORDER | SWP_NOZORDER);
}




void  Win32Tools::setWindowHeight(HWND win, unsigned short ht)
{
    int  wd = getWindowWidth(win);

    SetWindowPos(win, 0, 0, 0, wd, ht, SWP_NOACTIVATE | SWP_NOMOVE |
        SWP_NOOWNERZORDER | SWP_NOZORDER);
}




int  Win32Tools::getWindowTextIntegral(HWND w)
{
    std::string  wintext_string = getWindowTextString(w);

    std::istringstream  wintext_stream(wintext_string);

    int  wintext_int;
    wintext_stream >> wintext_int;

    return wintext_int;
}




bool  Win32Tools::isWindowTextIntegral(HWND w)
{
    const int kMaxDigits = 9;

    std::string  wintext = getWindowTextString(w);

    /* check to see if the window text is the null string, if so, it's not a
       number */
    if (wintext == "")
        return false;

    /* check to see if the window text contains any characters other than
       numbers or the negative sign; if it does, it's not a number */
    if (wintext.find_first_not_of("-1234567890") != std::string::npos)
        return false;

    /* if the negative sign appears, it must appear first; otherwise, the
       window text is not a number */
    const size_t  firstneg = wintext.find_first_of("-");
    if ((firstneg != std::string::npos) && (firstneg != 0))
        return false;

    /* if the negative sign appears, only one may appear */
    const size_t  lastneg = wintext.find_last_of("-");
    if (firstneg != lastneg)
        return false;

    /* if the negative sign appears, something else must also appear */
    const size_t  firstdigit = wintext.find_first_not_of("-");
    if (firstdigit == std::string::npos)
        return false;

    /* the number of significant figures, not including leading zeros must be
       kMaxDigits or less */
    const size_t  firstnonzero = wintext.find_first_of("123456789",
        firstdigit);
    const size_t  lastdigit = wintext.find_last_of("0123456789");
    if ((lastdigit - firstnonzero + 1) > kMaxDigits)
        return false;

    /* if all of the above negative tests failed, then the window text is a
       number */
    return true;
}




void  Win32Tools::setWindowTextIntegral(HWND w, int val)
{
    std::ostringstream  wintext_stream;
    wintext_stream << val;

    setWindowTextString(w, wintext_stream.str( ));
}




void  Win32Tools::destroyPrimitive(HWND& win)
{
    if (win == 0)
        return;

    BOOL  destroyedOk = FALSE;

    destroyedOk = DestroyWindow(win);

    if (! destroyedOk)
        throw std::runtime_error("Win32Tools: destroyPrimitive( ) couldn't "
            "delete Win32 object");

    win = 0;
}




void  Win32Tools::checkButton(HWND btn)
{
    SendMessage(btn, (UINT)(BM_SETCHECK), (WPARAM)(BST_CHECKED), 0);
}




void  Win32Tools::uncheckButton(HWND btn)
{
    SendMessage(btn, (UINT)(BM_SETCHECK), (WPARAM)(BST_UNCHECKED), 0);
}



std::string  Win32Tools::simplifyFilename(const std::string& filename)
{
    std::string result;

    int lastslashpos = filename.find_last_of("\\/");

    if (lastslashpos != std::string::npos)
        result = filename.substr(lastslashpos + 1,
            filename.length( ) - lastslashpos - 1);
    else
        result = filename;

    return result;
}




RGBColorNative  Win32Tools::toNative(const RGBColor& incol)
{
    BYTE  r_comp = static_cast<BYTE>(incol.r * 255.0f);
    BYTE  g_comp = static_cast<BYTE>(incol.g * 255.0f);
    BYTE  b_comp = static_cast<BYTE>(incol.b * 255.0f);

    return RGB(r_comp, g_comp, b_comp);
}




RGBColor  Win32Tools::fromNative(const RGBColorNative& incol)
{
    return RGBColor(static_cast<float>(GetRValue(incol)) / 255.0f,
                    static_cast<float>(GetGValue(incol)) / 255.0f,
                    static_cast<float>(GetBValue(incol)) / 255.0f);
}




PBITMAPINFO  Win32Tools::toNative(const RGBAImage& img)
{
    /* as of right now, we only support converting 32 bit-per-pixel
       RGBAImage objects to Win32 DIBs */
    if (img.bitsPerPixel( ) != 32)
        throw std::logic_error("Win32Tools: can't convert RGBAImage "
            "object to Windows DIB: input image isn't 32-bpp");

    const int kBitsPerPixel = 32;
    const int kBytesPerPixel = kBitsPerPixel / 8;
    const int kBufferBytes = kBytesPerPixel * img.width( ) * img.height( );

    BYTE*  packedDIBDataSegment = new BYTE [sizeof(BITMAPINFO) +
        kBufferBytes];

    PBITMAPINFOHEADER  headerPtr =
        reinterpret_cast<PBITMAPINFOHEADER>(packedDIBDataSegment);

    unsigned char*  dibBufferPtr =
        reinterpret_cast<unsigned char*>(packedDIBDataSegment +
        sizeof(BITMAPINFO));

    /* setup the DIB info structure */
    headerPtr->biSize = sizeof(BITMAPINFOHEADER);
    headerPtr->biWidth = img.width( );
    headerPtr->biHeight = img.height( );
    headerPtr->biPlanes = 1;
    headerPtr->biBitCount = kBitsPerPixel;
    headerPtr->biCompression = BI_RGB;
    headerPtr->biSizeImage = 0;
    headerPtr->biXPelsPerMeter = 2836;
    headerPtr->biYPelsPerMeter = 2836;
    headerPtr->biClrUsed = 0;
    headerPtr->biClrImportant = 0;

    /* transfer the input image's pixel buffer data into the output DIB's
       pixel buffer */
    const int numpixels = img.width( ) * img.height( );
    const unsigned char*  source_it = img.imageBuffer( );
    unsigned char*  dest_it = dibBufferPtr;
    for (int i = 0; i < numpixels; i++) {

        dest_it[0] = source_it[2];
        dest_it[1] = source_it[1];
        dest_it[2] = source_it[0];
        dest_it[3] = 0;

        dest_it += 4;
        source_it += 4;
    }

    return reinterpret_cast<PBITMAPINFO>(packedDIBDataSegment);
}
/////////////////////////////////////////////////////////////////////////////
// END  Win32Tools                                                         //
/////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  Win32Tools::Filesystem                                           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

Win32Tools::Filesystem::Filesystem( )
{
}




Win32Tools::Filesystem::~Filesystem( )
{
}




bool  Win32Tools::Filesystem::doesDirectoryExist(const std::string& dir)
{
    WIN32_FILE_ATTRIBUTE_DATA  dirdata;

    if (GetFileAttributesEx(dir.c_str( ), GetFileExInfoStandard, &dirdata))
        if (dirdata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
            return true;

    return false;
}



bool  Win32Tools::Filesystem::doesFileExist(const std::string& file)
{
    WIN32_FILE_ATTRIBUTE_DATA  filedata;

    if (GetFileAttributesEx(file.c_str( ), GetFileExInfoStandard, &filedata))
        if (! (filedata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
            return true;

    return false;
}



void  Win32Tools::Filesystem::createDirectory(const std::string& dir)
{
    if (doesDirectoryExist(dir))
        throw std::logic_error("Win32Tools::Filesystem: can't create the "
            "requested directory because it already exists");

    int createdOk = CreateDirectory(dir.c_str( ), NULL);

    if (!createdOk)
        throw std::runtime_error("Win32Tools::Filesystem: couldn't create "
            "requested directory");
}




time_t  Win32Tools::Filesystem::modificationTime(const std::string& path)
{
    const int kNoError = 0;

    struct stat  filedata;

    int  errorcode = stat(path.c_str( ), &filedata);

    if (errorcode != kNoError)
        throw std::runtime_error("modificationTime( ): target path does "
            "not exist or is not accessible");

    return filedata.st_mtime;
}




std::string  Win32Tools::Filesystem::cwd( )
{
    int bufsize =                             // we add 1 because
        GetCurrentDirectory(NULL, NULL) + 1;  // GetCurrentDirectory( )
                                              // isn't smart enough to
                                              // account for the terminal
                                              // '\0'

    char*  buffer = new char[bufsize];
    GetCurrentDirectory(bufsize, buffer);
    std::string  result = buffer;
    delete [ ] buffer;

    return result;
}




std::string  Win32Tools::Filesystem::mapTextFile(const std::string& path,
    int maxlines)
{
    std::ifstream  instream(path.c_str( ), std::ios::in);

    if (!instream)
        throw std::runtime_error("Win32Tools::Filesystem: couldn't map "
            "text: source doesn't exist or can't be opened for reading");

    if (maxlines < 1) maxlines = INT_MAX;

    std::string  accum;
    std::string  currline;
    int          currline_num = 0;
    while (instream && (currline_num < maxlines)) {

        std::getline(instream, currline);
        accum += (currline + "\n");

        currline_num++;
    }

    instream.close( );

    return accum;
}




std::string  Win32Tools::Filesystem::extractDirectory(const std::string& path)
{
    size_t  lastslash = path.find_last_of("\\");

    if (lastslash == std::string::npos)
        return ".";
    else
        return path.substr(0, lastslash);
}




std::string  Win32Tools::Filesystem::extractFilename(const std::string& path)
{
    size_t  lastslash = path.find_last_of("\\");

    std::string nodir;
    if (lastslash == std::string::npos)
        nodir = path;
    else
        nodir = path.substr(lastslash + 1, path.length( ) - lastslash - 1);

    size_t lastdot = nodir.find_last_of(".");
    if (lastdot == std::string::npos)
        return nodir;
    else
        return nodir.substr(0, lastdot);
}
/////////////////////////////////////////////////////////////////////////////
// END  Win32Tools::Filesystem                                             //
/////////////////////////////////////////////////////////////////////////////







/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  Win32Tools::WidgetFactory                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
const std::string
    Win32Tools::WidgetFactory::kSeparatorClassName = "LDXSeparator";

Win32Tools::WidgetFactory::WidgetFactory( )
{
    WNDCLASS    sepclass;

    sepclass.style = CS_HREDRAW | CS_VREDRAW;
    sepclass.lpfnWndProc = DefWindowProc;
    sepclass.cbClsExtra = 0;
    sepclass.cbWndExtra = 0;
    sepclass.hInstance = NULL;
    sepclass.hIcon = NULL;
    sepclass.hCursor = LoadCursor(NULL, IDC_ARROW);
    sepclass.hbrBackground = (HBRUSH)(COLOR_BTNFACE + 1);
    sepclass.lpszMenuName = NULL;
    sepclass.lpszClassName = kSeparatorClassName.c_str( );

    ATOM  sepresult = RegisterClass(&sepclass);
    if (!sepresult)
        throw std::runtime_error("WidgetFactory: failed to register "
            "separator window class");
}




Win32Tools::WidgetFactory::~WidgetFactory( )
{
}




HWND  Win32Tools::WidgetFactory::makeHorizontalSeparator(HWND parent,
    long long id)
{
    HWND  result = CreateWindowEx(WS_EX_STATICEDGE,
        kSeparatorClassName.c_str( ), TEXT(""), WS_CHILD, 0, 0,
        Win32Tools::kHorizSeparatorWidth, Win32Tools::kHorizSeparatorHeight,
        parent, (HMENU) id, NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeHorizontalSeparator( ) "
            "could not create Win32 object");

    return result;
}




HWND  Win32Tools::WidgetFactory::makePushButton(HWND parent, long long id)
{
    HWND  result = CreateWindow(TEXT("BUTTON"), TEXT(""), BS_PUSHBUTTON |
        WS_CHILD | WS_CLIPSIBLINGS, 0, 0, Win32Tools::kDialogPushButtonWidth,
        Win32Tools::kPushButtonHeight, parent, (HMENU) id, NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makePushButton( ) could "
            "not create Win32 object");

    Win32Tools::applyControlFaceProperties(result);

    return result;
}




HWND  Win32Tools::WidgetFactory::makeDefaultPushButton(HWND parent,
    long long id)
{
    HWND  result = CreateWindow(TEXT("BUTTON"), TEXT(""), BS_DEFPUSHBUTTON |
        WS_VISIBLE | WS_CHILD, 0, 0, Win32Tools::kDialogPushButtonWidth,
        Win32Tools::kPushButtonHeight, parent, (HMENU) id, NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeDefaultPushButton( ) "
            "could not create Win32 object");

    Win32Tools::applyControlFaceProperties(result);

    return result;
}




HWND  Win32Tools::WidgetFactory::makeRadioButton(HWND parent, long long id)
{
    HWND  result = CreateWindow(TEXT("button"), TEXT("Ay"), BS_RADIOBUTTON |
        BS_VCENTER | WS_VISIBLE | WS_CHILD, 0, 0,
        Win32Tools::kDialogPushButtonWidth, Win32Tools::kRadioButtonHeight,
        parent, (HMENU) id, NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeRadioButton( ) could "
            "not create Win32 object");

    Win32Tools::applyControlFaceProperties(result);
    Win32Tools::uncheckButton(result);

    return result;
}




HWND  Win32Tools::WidgetFactory::makeEditArea(HWND parent, long long id)
{
    HWND  result = CreateWindowEx(WS_EX_CLIENTEDGE, TEXT("edit"), TEXT(""),
        WS_CHILD | ES_LEFT, 0, 0, Win32Tools::kDialogPushButtonWidth,
        Win32Tools::kEditAreaHeight, parent, (HMENU)(id), NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeEditArea( ) could "
            "not create Win32 object");

    Win32Tools::applyControlFaceProperties(result);

    return result;
}




HWND  Win32Tools::WidgetFactory::makeBoldLabel(HWND parent, long long id)
{
    HWND  result = CreateWindowA(TEXT("static"), TEXT(""), WS_CHILD |
        SS_LEFT, 0, 0, Win32Tools::kDialogPushButtonWidth,
        Win32Tools::kBoldLabelHeight, parent, (HMENU)(id), NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeBoldLabel( ) could "
            "not create Win32 object");

    Win32Tools::applyBoldLabelProperties(result);

    return result;
}




HWND  Win32Tools::WidgetFactory::makeLabel(HWND parent, long long id)
{
    HWND  result = CreateWindowA(TEXT("static"), TEXT(""), WS_CHILD |
        SS_LEFT, 0, 0, Win32Tools::kDialogPushButtonWidth,
        Win32Tools::kLabelHeight, parent, (HMENU)(id), NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeLabel( ) could "
            "not create Win32 object");

    Win32Tools::applyControlFaceProperties(result);

    return result;
}




HWND  Win32Tools::WidgetFactory::makeRightAlignedLabel(HWND parent,
    long long id)
{
    HWND  result = CreateWindowA(TEXT("static"), TEXT(""), WS_CHILD |
        WS_VISIBLE | SS_RIGHT, 0, 0,
        Win32Tools::kDialogPushButtonWidth, Win32Tools::kLabelHeight,
        parent, (HMENU)(id), NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeRightAlignedLabel( ) "
            "could not create Win32 object");

    Win32Tools::applyControlFaceProperties(result);

    return result;
}




HWND  Win32Tools::WidgetFactory::makeHorizontalTrackbar(HWND parent,
    long long id)
{
    HWND  result = CreateWindowEx(0, TRACKBAR_CLASS,
        TEXT("TrackerThingy_trackbar"), WS_CHILD | WS_VISIBLE | TBS_HORZ |
        TBS_AUTOTICKS, 0, 0, 64, kTrackbarMinorAxis, parent,
        (HMENU)(id), NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeHorizontalTrackbar( ) "
            "could not create Win32 object");

    Win32Tools::applyControlFaceProperties(result);
    SendMessage(result, PBM_SETRANGE, 0, MAKELPARAM(0, 100));

    return result;
}



HWND  Win32Tools::WidgetFactory::makeProgressBar(HWND parent, long long id)
{
    HWND  result = CreateWindowEx(0, PROGRESS_CLASS, TEXT(""), WS_CHILD,
        0, 0, 130, Win32Tools::kProgressBarHeight, parent,
        (HMENU)(id), NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeProgressBar( ) could "
            "not create Win32 object");

    return result;
}




HWND  Win32Tools::WidgetFactory::makeGroupBox(HWND parent, long long id)
{
    HWND  result = CreateWindowEx(0, WC_BUTTON, TEXT("Group"), WS_CHILD |
        BS_GROUPBOX | WS_CLIPCHILDREN | WS_CLIPSIBLINGS, 0, 0, 72, 48,
        parent, (HMENU)(id), NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeGroupBox( ) could "
            "not create Win32 object");
    Win32Tools::applyControlFaceProperties(result);

    return result;
}




HWND  Win32Tools::WidgetFactory::makeCheckBox(HWND parent, long long id)
{
    HWND  result = CreateWindow(TEXT("button"), TEXT("checkBox"),
        BS_CHECKBOX | BS_VCENTER | WS_CHILD, 0, 0,
        Win32Tools::kDialogPushButtonWidth, Win32Tools::kRadioButtonHeight,
        parent, (HMENU) id, NULL, 0);
    if (!result)
        throw std::runtime_error("WidgetFactory: makeCheckBox( ) could "
            "not create Win32 object");

    Win32Tools::applyControlFaceProperties(result);
    Win32Tools::uncheckButton(result);

    return result;
}
/////////////////////////////////////////////////////////////////////////////
// END  Win32Tools::WidgetFactory                                          //
/////////////////////////////////////////////////////////////////////////////
