//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// TranscriptionServer.h                                                    //
//                                                                          //
// Packages the interface definition of class TranscriptionServer.          //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 1.2 of Tue 04-Sep-2007 @ 1:09am EDT                                 //
//                                                                          //
//     Significantly improved handling of the transcription output          //
//     filestream. Specifically, whereas stream opening and closing used    //
//     to be done in an ad-hoc fashion in the c-tor and d-tor, I've now     //
//     added the private utility methods uOpenOutputStream( ),              //
//     uCloseOutputStream( ), and uIsOutputStreamOpen( ) to make stream     //
//     handling more explicit and callable from several different places.   //
//     Also, I've added the notion of "transcript directory", which is      //
//     the directory on disk where transcript files will be stored. Right   //
//     now this path is configured programmatically in the                  //
//     uGetDefaultTXDirectory( ) method, but it would be trivial now to     //
//     write accessor/mutator methods to let a client programmer configure  //
//     it -- this is a big step over where we were before, when             //
//     no directory notion even existed, and transcript files just got      //
//     shoved into the executing thread's current working directory (this   //
//     was relatively annoying in that tended to clutter up the IDE-created //
//     project directories, among other things).                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007, Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef TRANSCRIPTION_SERVER__HXX
#define TRANSCRIPTION_SERVER__HXX

#include <Win32Support.h>
#include <string>
#include <fstream>

class TranscriptionServer {

private:

    static  std::string  uGetDefaultTXDirectory( );

    std::string     fApplicationName;
    std::ofstream*  fOutputStream;
    std::string     fLineAccumulator;

    /* (UTILITY) returns the output filestream object connected to this
       TranscriptionServer's transcript file; if the output filestream
       hasn't yet been opened when this method is invoked (i.e.
       uIsOutputStreamOpen( ) == false), then try open it; note that
       opening it may cause a runtime_error exception to be propogated
       if the stream can't be opened (for file permissions reasons, etc.) */
    std::ofstream&  uStream( );

    /* (UTILITY) returns the current system time as a short
       "timestamp" string whose format is appropriate for the user's
        current system locale. */
    std::string     uTimestamp( );

    void            uOpenOutputStream( );
    void            uCloseOutputStream( );
    bool            uIsOutputStreamOpen( );

public:

    /* creates a new TranscriptionServer instance configured to provide
       runtime transcription services to the application having the logical
       name 'appname'. Note that 'appname' is a logical application name,
       not a physical filename or executable image name, and thus it should
       be something intelligible to the whoever will ultimately read
       transcript files (ex: "PaintShop" is superior to "psp.exe"). */
    TranscriptionServer(const std::string& appname);

    /* closes the transcript file connected to this TranscriptionServer
       and deallocates storage for its file handle */
    virtual ~TranscriptionServer( );

    /* put 's' in the transcription buffer, but do not actually timestamp
      's' and write it into the transcript file. Timestamping and permanent
      write occur only when writeln( ) or flushln( ) are called */
    void  write(const std::string& s);

    /* Timestamp 's' and write its contents immediately into the transcript
       file. If the transcription buffer is non-empty, its contents are
       first flushed into the transcript file before 's' is written */
    void  writeln(const std::string& s);

    /* if the transcription buffer is non-empty, timestamp its contents and
       flush them permanently into the transcript file */
    void  flushln( );

    /* timestamp 'data' and stream it into the transcription file, followed
       by a newline */
    template<typename TStreamable>
    TranscriptionServer&  operator<<(const TStreamable& data)
    {
        uStream( ) << "[" << uTimestamp( ) << "]: ";
        uStream( ) << data;
        uStream( ) << std::endl;

        return *this;
    }
};

#endif
