//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// TranscriptionServer.cpp                                                  //
//                                                                          //
// Implements member functions of class TranscriptionServer.                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 1.2 of Tue 04-Sep-2007 @ 1:09am EDT                                 //
//                                                                          //
//     Significantly improved handling of the transcription output          //
//     filestream. Specifically, whereas stream opening and closing used    //
//     to be done in an ad-hoc fashion in the c-tor and d-tor, I've now     //
//     added the private utility methods uOpenOutputStream( ),              //
//     uCloseOutputStream( ), and uIsOutputStreamOpen( ) to make stream     //
//     handling more explicit and callable from several different places.   //
//     Also, I've added the notion of "transcript directory", which is      //
//     the directory on disk where transcript files will be stored. Right   //
//     now this path is configured programmatically in the                  //
//     uGetDefaultTXDirectory( ) method, but it would be trivial now to     //
//     write accessor/mutator methods to let a client programmer configure  //
//     it -- this is a big step over where we were before, when             //
//     no directory notion even existed, and transcript files just got      //
//     shoved into the executing thread's current working directory (this   //
//     was relatively annoying in that tended to clutter up the IDE-created //
//     project directories, among other things).                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007, Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include <TranscriptionServer.h>
#include <ctime>
#include <cstdio>

/* line 41 below stops the MSVC 2003 & 2005 compilers from propogating
   annoying warnings about security risks present in the ANSI C standard
   library */
#pragma warning(disable : 4996)



std::string  TranscriptionServer::uGetDefaultTXDirectory( )
{
    const std::string  varkey = "%UserProfile%";
    const size_t       kPathBufferSize = 1024;
    char               pathBuffer[kPathBufferSize];

    ExpandEnvironmentStrings(varkey.c_str( ), pathBuffer, kPathBufferSize);

    if (varkey != pathBuffer)
        return  std::string(pathBuffer) + "\\Transcripts";
    else
        throw std::runtime_error("TranscriptionServer: uGetDefaultTXDirectory"
        "( ): couldn't expand environment string");
}



/* creates a new TranscriptionServer instance configured to provide
   runtime transcription services to the application having the logical
   name 'appname'. Note that 'appname' is a logical application name,
   not a physical filename or executable image name, and thus it should
   be something intelligible to the whoever will ultimately read
   transcript files (ex: "PaintShop" is superior to "psp.exe"). */
TranscriptionServer::TranscriptionServer(const std::string& appname)
    : fApplicationName(appname), fOutputStream(0)
{
}



/* closes the transcript file connected to this TranscriptionServer
   and deallocates storage for its file handle */
TranscriptionServer::~TranscriptionServer( )
{
    if (uIsOutputStreamOpen( ))
        uCloseOutputStream( );
}



/* put 's' in the transcription buffer, but do not actually timestamp
  's' and write it into the transcript file. Timestamping and permanent
  write occur only when writeln( ) or flushln( ) are called */
void  TranscriptionServer::write(const std::string& s)
{
    fLineAccumulator += s;
}



/* Timestamp 's' and write its contents immediately into the transcript
   file. If the transcription buffer is non-empty, its contents are
   first flushed into the transcript file before 's' is written */
void  TranscriptionServer::writeln(const std::string& s)
{
    uStream( ) << "\n[" << uTimestamp( ) << "]: ";
    
    if (fLineAccumulator != "") {

        uStream( ) << fLineAccumulator << " " << s << std::endl;
    
        fLineAccumulator = "";
    }
    else {
    
        uStream( ) << s << std::endl;
    }
}



/* (UTILITY) returns the current system time as a short
   "timestamp" string whose format is appropriate for the user's
   current system locale. */
std::string  TranscriptionServer::uTimestamp( )
{
    const time_t  systemTimecode = std::time(0);
    
    std::tm*  systemTimeStruct = std::localtime(&systemTimecode);
    
    char*  timetext = asctime(systemTimeStruct);
    
    timetext[24] = '\0';
    
    return std::string(timetext);
}



/* if the transcription buffer is non-empty, timestamp its contents and
   write them permanently into the transcript file */
void  TranscriptionServer::flushln( )
{
    if (fLineAccumulator != "") {
    
        uStream( ) << "\n[" << uTimestamp( ) << "]: ";
        
        uStream( ) << fLineAccumulator << std::endl;
        
        fLineAccumulator = "";
    }
}


/* (UTILITY) returns the output filestream object connected to this
   TranscriptionServer's transcript file; if the output filestream
   hasn't yet been opened when this method is invoked (i.e.
   uIsOutputStreamOpen( ) == false), then try to open it; note that
   attemption to open the stream may cause a runtime_error exception
   to be propogated if file permissions reasons, etc. prevent stream
   opening */
std::ofstream&  TranscriptionServer::uStream( )
{
    if (!uIsOutputStreamOpen( ))
        uOpenOutputStream( );

    return *fOutputStream;
}




void  TranscriptionServer::uOpenOutputStream( )
{
    if (uIsOutputStreamOpen( ))
        throw std::logic_error("TranscriptionServer: uOpenOutputStream( ) "
            "was invoked but the output stream is already open");

    // if the default transcript directory doesn't exist, then create it;
    // if we can't create it for whatever reason (permissions, etc.),
    // then Win32Tools::Filesystem::createDirectory( ) will throw a
    // runtime_error
    const std::string  txDirectory = uGetDefaultTXDirectory( );
    if (!Win32Tools::filesystem( ).doesDirectoryExist(txDirectory))
        Win32Tools::filesystem( ).createDirectory(txDirectory);

    // use system time functions to generate a timecode string that
    // will become part of the filename of the transcript file; this
    // ensures that one single transcript file doesn't just keep
    // getting overwritten every time the program is run during a
    // long day of testing programs
    const time_t  systemTimecode = std::time(0);
    char  timecodeStringBuffer[255];
    std::sprintf(timecodeStringBuffer, "%d",
        static_cast<unsigned long>(systemTimecode));
    std::string  timecodeString = timecodeStringBuffer;

    // compose the application name and timecode string into a unique,
    // intelligible filename, then get the full path associated with this
    // filename by concatenating it with the transcript directory
    // name we got earlier
    const std::string  txFilename = "transcript-" + fApplicationName + "-" +
        timecodeString + ".text";
    const std::string  txFullPath = txDirectory + "\\" + txFilename;
    fOutputStream = new std::ofstream(txFullPath.c_str( ), std::ios::out);
    
    // attempt to open the transcript filestream: if it opens o.k., then
    // write the transcript header into it; if we can't open it, then throw
    // a runtime_error
    if (!(*fOutputStream)) {
    
        delete fOutputStream;
        
        throw std::runtime_error("ERROR: TranscriptionServer: unable to "
            "open transcript file for writing");
    }
    uStream( ) << "Execution Transcript for " << fApplicationName <<
        " Session" << std::endl;
}




void  TranscriptionServer::uCloseOutputStream( )
{
    if (!uIsOutputStreamOpen( ))
        throw std::logic_error("TranscriptionServer: uCloseOutputStream( ) "
            "was invoked but the output stream isn't open");

    if (fLineAccumulator != "")
        flushln( );

    writeln("transcript session explicitly closed.");

    fOutputStream->close( );
    delete fOutputStream;
    fOutputStream = 0;
}




bool  TranscriptionServer::uIsOutputStreamOpen( )
{
    if (fOutputStream)
        return true;
    else
        return false;
}
