//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// TextureSynthesis.h                                                       //
//                                                                          //
// Packages interfaces definitions for classes ImagePyramid,                //
// PixelNeighborhood, and ImageSimilarityMap.                               //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver 3.0.2 of Mon 23 June 2008 @ 10:56pm EDT                              //
//                                                                          //
//     the EdgingMode enumerated type was made global; previously, it was   //
//     nested in class PixelNeighborhood                                    //
//                                                                          //
// ver 3.0.1 of Fri 3 May 2008 @ 7:43pm EDT                                 //
//                                                                          //
//     changed all references of class template Vertex2D to Vector2D; see   //
//     the AffineGeometry module's header for more info                     //
//                                                                          //
//  ver 3.0.0 of Thu 02-May-2008 @ 12:12pm EDT                              //
//                                                                          //
//      PixelNeighborhoods now use pre-computed weights in distance( ) so   //
//      we don't have to evaluate a transcendental function. Added class    //
//      ImageSimilarityMap and removed class PyramidalTextureCache. Tested  //
//      as part of Similaris and SimilarisReader and considered stable.     //
//                                                                          //
//  older versions:                                                         //
//                                                                          //
//      older change history elided; check out an older CVS rev to get it   //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008, Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef TEXTURE_SYNTHESIS__HXX
#define TEXTURE_SYNTHESIS__HXX

#include <ImageRepresentation.h>
#include <UserInteraction.h>
#include <stdexcept>
#include <map>
#include <set>
#include <cmath>
#include <ctime>


//--------------------------------------------------------------------------//
// TYPE  EdgingMode                                                         //
//--------------------------------------------------------------------------//
enum  EdgingMode { kReflectMode = 16, kWrapMode = 17 };
//--------------------------------------------------------------------------//




//--------------------------------------------------------------------------//
// CLASS  PixelNeighborhood                                                 //
//--------------------------------------------------------------------------//
// SYNOPSIS: a pixel neighborhood is a 2D matrix of pixels, where each      //
//           pixel is a triple of RGB intensity values. The size of a       //
//           neighborhood is described by its radius, which bounds the      //
//           the coordinate system used to index the individual pixels that //
//           make up the neighborhood. If a neighborhood is of radius r,    //
//           then it contains (2r + 1)^2 pixels, indexed from pixel         //
//           (-r, -r) in the neighborhood's lower-left hand corner to pixel //
//           (r, r) in its upper-right. There is a distinguished origin     //
//           pixel having coordinates (0, 0), and each PixelNeighborhood    //
//           instance can be seen as characerizing a local environment      //
//           around its origin.                                             //
//--------------------------------------------------------------------------//
class  PixelNeighborhood {

private:

    static  const float*   sWeights[9];
    static  bool           sAreWeightsInited;

    float*           fPixels;
    int              fRadius;

    mutable  RGBColorNative*  fDrawableStore;

    Vector2Di  uEdgePixel(const RGBAImage& img, const Vector2Di& loc,
        EdgingMode mode);
    void  uInitWeightsPointers(const float** ptrarray);

 public:

    static  const int    kMaxRadius = 8;
    static  const float  kNHWeights0[ ];
    static  const float  kNHWeights1[ ];
    static  const float  kNHWeights2[ ];
    static  const float  kNHWeights3[ ];
    static  const float  kNHWeights4[ ];
    static  const float  kNHWeights5[ ];
    static  const float  kNHWeights6[ ];
    static  const float  kNHWeights7[ ];
    static  const float  kNHWeights8[ ];

    /** create a new pixel neigborhood of radius 'rad' with backing data
        store 'pixdata'. The caller allocates storage for an array of
        (2*rad + 1)^2 floats, populates it with values, and passes a pointer
        to it in 'pixdata'. The elements of 'pixdata' are per-pixel RGB
        intensity values that must be in the range [0, 1]. 'pixdata' is
        indexed such that elements pixdata[0], pixdata[1], and pixdata[2]
        correspond the R, G, and B intensities of the new neighborhood's
        (-rad, -rad) pixel; the elements continue in row major order. Caller
        passes ownership of 'pixdata' to the newly created object. If 'rad'
        exceeds kMaxRadius or is negative, an invalid_argument exception is
        thrown */
    PixelNeighborhood(int rad, float* pixdata);

    /** create a new pixel neigborhood of radius 'rad' centered about pixel
        location ('xpos', 'ypos') in image 'img'. If construction of the
        new neighborhood samples pixels outside of the bounds of 'img', then
        adjust the neighborhood sample points to be within the bounds, either
        by reflecting them back inside the image boundaries or by wrapping
        them around the boundaries, depending on 'mode'. If 'rad' exceeds
        kMaxRadius or is negative, an invalid_argument exception is thrown */
    PixelNeighborhood(int rad, const RGBAImage& img, int x, int y,
        EdgingMode mode = kReflectMode);

    PixelNeighborhood(const PixelNeighborhood& source);

    virtual  ~PixelNeighborhood( );

    const PixelNeighborhood&  operator=(const PixelNeighborhood& source);

    /** compute the perceptual distance between this neighborhood and
        'other'; this and other must be of the same size, if they
        aren't, an invalid_argument exception is thrown */
    float  distance(const PixelNeighborhood& other) const;

    /** draw the pixel neighborhood into the selected OpenGL graphics
        context at the current raster position */
    void  draw( ) const;
};
//--------------------------------------------------------------------------//








//--------------------------------------------------------------------------//
// CLASS  ImagePyramid                                                      //
//--------------------------------------------------------------------------//

/* An ImagePyramid encapuslates a Gaussian image pyramid. It enables multi-
   scale image representation and progressive smoothing of high-frequency
   information in sampled images. An ImagePyramid consists of a base image
   and zero or more images created by progressively transforming the base
   image. Clients create an ImagePyramid by specifying its base image and
   number of levels that will make up the pyramid. The base image always
   corresponds to level 0. Levels greater than zero are constructed as
   follows: for any level n, n > 0, build level n from level (n - 1) by
   downsampling level (n - 1) by half, employing a 2D Gaussian kernel during
   reconstruction. If level (n - 1) has size m x m pixels, then level n is
   of size (m / 2) x (m / 2) pixels and the downsampling transformation from
   level (n - 1) to level n discards the highest-frequency information. */
class  ImagePyramid
    : public virtual InterstitialActionSender {

private:

    class  MosaicTile {

    private:

        int         fDimension;
        int         fSeqNum;
        Vector2Di   fOrigin;

    public:

        MosaicTile(int seqnum, const Vector2Di& origin, int dim);
        static  MosaicTile  nullTile( );

        bool  operator==(const MosaicTile& rhs);
        bool  operator!=(const MosaicTile& rhs);
        MosaicTile  next( ) const;
        const Vector2Di&  origin( ) const;
        int  dimension( ) const;
        bool  isNullTile( ) const;
    };

    int                                    fNumLevels;
    float                                  fSampleRadius;
    std::set<InterstitialActionReceiver*>  fReceiverSet;
    std::vector<Rectangle2Di>              fTileset;
    RGBAImage*                             fMosaicImage;

    void  uCheckPixelRange(int lev, int x, int y) const;

    void  uDownsampleBuffer(RGBColorNative* dest, int destdims,
        const RGBColorNative* src, int srcdims);

    void  uDownsampleBuffer(RGBColorNative* dest, int destdims,
        const RGBColorNative* src, int srcdims, const int& totalpix,
        int& processedpix);

    void  uBroadcastUniformIAE(int pctprog);

    RGBAImage*  uMixPyramidMosaic(RGBColorNative** imagebuffers,
        int* leveldims, int numlevs);

    void  uConstructorHelper(const RGBAImage& base, int numlevels);

    inline  float  uEvalKernel(float x, float y) const
    {
        return (1.0f / (2.0f*kPi)) * exp(-(x*x + y*y) /
            (2.0f));
    }

public:

    static  const int    kMinXCoord = 0;
    static  const int    kMinYCoord = 0;
    static  const float  kDefaultSampleRadius;

    virtual  ~ImagePyramid( );

    /* create a new ImagePyramid with base image 'base' and having
       'numlevels' total levels (including the base level). If
       'base' isn't a square image or if 'numlevels' is less than
       1 or greater than lg(basedim) + 1 (where 'basedim' is the pixel
       length of an edge of the base image), an invalid_argument
       exception is thrown */
    ImagePyramid(const RGBAImage& base, int numlevels);

    /* create a new ImagePyramid with base image 'base' and having
       'numlevels' total levels (including the base level). Since pyramid
       construction is a computationally expensive operation that may
       require several seconds of CPU time, 'iar' will be sent an
       interstitial action event about every (1/60)th of a second during
       pyramid construction. The caller should build 'iar' to run an
       event loop and/or provide feedback to the user so that the
       application remains responsive during the construction process. If
       'base' isn't a square image or if 'numlevels' is less than
       1 or greater than lg(basedim) + 1 (where 'basedim' is the pixel
       length of an edge of the base image), an invalid_argument
       exception is thrown */
    ImagePyramid(const RGBAImage& base, int numlevels,
        InterstitialActionReceiver& iar);

    /* create a new ImagePyramid with base image 'base' having
       lg(basedim) + 1 total levels (including the base level), where
       'basedim' is the pixel length of an edge of the base image. If
       'base' isn't a square image, an invalid_argument exception is
       thrown */
    ImagePyramid(const RGBAImage& base);

    /* create a new ImagePyramid with base image 'base' having
       lg(basedim) + 1 total levels (including the base level), where
       'basedim' is the pixel length of an edge of the base image. Since
       pyramid construction is a computationally expensive operation that
       may require several seconds of CPU time, 'iar' will be sent an
       interstitial action event about every (1/60)th of a second during
       pyramid construction. The caller should build 'iar' to run an
       event loop and/or provide feedback to the user so that the
       application remains responsive during the construction process. If
       'base' isn't a square image, an invalid_argument exception is
       thrown */
    ImagePyramid(const RGBAImage& base, InterstitialActionReceiver& iar);

    /* return the color value of the pixel at coordinates ('x', 'y') in
       level 'lev'. If 'lev' is less than zero or greater than
       numLevels( ) - 1, an invalid_argument exception is thrown. If
       'x' or 'y' are less than zero or greater than
       levelDimension('lev') - 1, an invalid_argument exception is thrown */
    const RGBColor  pixel(int lev, int x, int y) const;

    /* return the color value of the pixel at coordinates ('x', 'y') in
       level 'lev' in the packed, native color representation format of
       the operating system. If 'lev' is less than zero or greater than
       numLevels( ) - 1, an invalid_argument exception is
       thrown. If 'x' or 'y' are less than zero or greater than
       levelDimension('lev') - 1, an invalid_argument exception is thrown */
    const RGBColorNative  nativePixel(int lev, int x, int y) const;

    /* return a pixel neighborhood of radius 'rad' whose origin is the
       pixel at coordinates ('x', 'y') in level 'lev' If 'lev' is less
       than zero or greater than numLevels( ) - 1, an invalid_argument
       exception is thrown. If ('x', 'y') is specified such that any of
       the pixels in the neighborhood to be created fall outside of the
       dimensions of 'lev', an invalid_argument exception is thrown.
       If 'rad' is less than zero or greater than
       PixelNeighborhood::kMaxRadius, an invalid_argument exception is
       thrown */
    PixelNeighborhood  createNeighborhood(int lev, int x, int y,
                                          int rad) const;

    /* return the number of levels in this, including the base level */
    int  numLevels( ) const;

    /* return the pixel dimension of level 'lev'. Since all the levels of
       an ImagePyramid are square images, the quantity returned is both the
       width and height (in pixels) of the image at level 'lev' of the
       pyramid. If 'lev' is less than zero or greater than numLevels( ) - 1,
        an invalid_argument exception is thrown */
    int  levelDimension(int lev) const;

    /* draws all the pixels in level 'lev' into the selected OpenGL
       graphics context at the current raster position; if 'lev' is less
       than zero or greater than numLevels( ) - 1, an invalid_argument
       exception is thrown */
    void  draw(int lev) const;

    /* draws a non-empty rectangular subset of the pixels in level 'lev'
       into the selected OpenGL graphics context at the current raster
       position; if 'lev' is less than zero or greater than
       numLevels( ) - 1, an invalid_argument exception is thrown;
       if left >= right or bot >= top, the target subset is considered
       empty and an invalid_argument exception is thrown; if any of the
       bounds are negative or exceed the dimensions of 'lev', an
       invalid_argument exception is thrown */
    void  draw(int lev, int left, int bot, int right, int top) const;

    /* copies all the pixels in level 'lev' into the RGBAImage 'dest'.
       The point 'where' names the lower-left corner (in 'dest' coordinates)
       of the rectangle on the surface of 'dest' into which the pixels of
       'lev' are copied. If 'lev' is less than zero or greater than
       numLevels( ) - 1, an invalid_argument exception is thrown. If
       'where' is such that not all of the pixels of 'lev' can be copied
       into 'dest' without overflowing its bounds, an invalid_argument
       exception is thrown */
    void  copy(RGBAImage& dest, int lev, const Vector2Di& where) const;

    /* interstitial action management methods mandated by the                     
       InterstitialActionSender interface. */
    void  registerReceiver(InterstitialActionReceiver& recv);
    void  deregisterReceiver(InterstitialActionReceiver& recv);
    void  cancelAction(InterstitialActionReceiver& recv);

    /* returns all levels of the pyramid packed into a single flat image;
       this is a good format when pyramid level data needs to be downloaded
       into texture memory for use with programmable shaders. To get the
       locations of the individual pyramid levels within the returned mosaic
       image, call mosaicTiles( ) */
    const  RGBAImage&  mosaicImage( ) const;

    /* returns a collection of 2D rectangles encoding the locations and
       sizes of individual pyramid levels within the packed mosaic
       returned by mosaicImage( ). The location & extent rectangle for
       some pyramid level 'lev' is stored as mosaicTiles( )[lev] */
    const std::vector<Rectangle2Di>&  mosaicTiles( ) const;
};
//--------------------------------------------------------------------------//




//--------------------------------------------------------------------------//
// CLASS  ImageSimilarityMap                                                //
//--------------------------------------------------------------------------//
class ImageSimilarityMap :
    public virtual InterstitialActionSender {

private:

    typedef  std::map<float, Vector2Di>  DistanceMap;


    RGBAImage*                      fEncodedImage;
    InterstitialActionReceiverSet   fReceiverSet;
    int                             fEntriesPerPix;
    bool                            fIsBoosted;
    float                           fBoostSepFrac;
    int                             fWidth;
    int                             fHeight;
    bool                            fDidUserCancel;
    clock_t                         fInstrLastDispatch;
    double                          fInstrTotalPix;
    double                          fInstrDonePix;
    int                             fNumEdgeExclPix;

    void  uConstructorHelper(const RGBAImage& srcimg, int numentries,
        bool isboosted, float boostfrac);

    void  uConstructorHelper(const std::string& infile);

    void  uBroadcastUniformIAE(int pctprog);

    inline  float  uCartesianDist(int x1, int y1, int x2, int y2)
    {
        float deltax = static_cast<float>(x2 - x1);
        float deltay = static_cast<float>(y2 - y1);

        return std::sqrt((deltax * deltax) + (deltay * deltay));
    }

    void  uFindSimilarBoosted(const RGBAImage& basisimg, int refx,
        int refy, DistanceMap& outmap);

    void  uFindSimilarUnboosted(const RGBAImage& basisimg, int refx,
        int refy, DistanceMap& outmap);

    void  uFindOthersNearby(const DistanceMap& inset, DistanceMap& outset,
        int refx, int refy, float critdist);

public:

    static  const unsigned short  kNeighborhoodRadius = 2;
    static  const unsigned int    kNotifyTicks;
    static  const unsigned short  kDefaultEdgeExclusionPixels = 2;

    ImageSimilarityMap(const RGBAImage& srcimg, int numentries,
        bool isboosted, float boostfrac);
    ImageSimilarityMap(const RGBAImage& srcimg, int numentries,
        bool isboosted, float boostfrac, InterstitialActionReceiver& iar);
    ImageSimilarityMap(const std::string& infile);
    ImageSimilarityMap(const std::string& infile,
        InterstitialActionReceiver& iar);
    virtual  ~ImageSimilarityMap( );

    virtual  std::vector<Vector2Di>  pixelsSimilarTo(
        const Vector2Di& p) const;
    virtual  std::vector<Vector2Di>  pixelsSimilarTo(int x, int y) const;
    virtual  void  write(const std::string& outfile) const;
    virtual  const RGBAImage&  encodedImage( ) const;
    virtual  int  width( ) const;
    virtual  int  height( ) const;
    virtual  int  entriesPerPixel( ) const;
    void  registerReceiver(InterstitialActionReceiver& recv);
    void  deregisterReceiver(InterstitialActionReceiver& recv);
    void  cancelAction(InterstitialActionReceiver& recv);
};
//--------------------------------------------------------------------------//





//--------------------------------------------------------------------------//
// CLASS  PyramidalSimilarityMap                                            //
//--------------------------------------------------------------------------//
class  PyramidalSimilarityMap
    : public virtual InterstitialActionSender,
      public virtual InterstitialActionReceiver {

private:

    RGBAImage*                      fMosaicImage;
    InterstitialActionReceiverSet   fReceiverSet;
    int                             fEntriesPerPix;
    int                             fNumLevels;
    bool                            fIsBoosted;
    float                           fBoostSepFrac;
    std::vector<Rectangle2Di>       fTileset;
    std::vector<float>              fInstFracPerLev;
    int                             fInstrCurrLev;
    bool                            fDidUserCancel;

    void  uBroadcastUniformIAE(int pctprog);
    void  uConstructorHelper(const ImagePyramid& basispyr, int maxlev,
        int numentries, bool isboosted, float boostfrac);
    void  uConstructorHelper(const std::string& infile);
    void  uComputeTileset(int basewd, int baseht, int numlevs, int numents);
    void  uPrepareMosaicImage(const std::vector<Rectangle2Di>& tileset);

public:

    PyramidalSimilarityMap(const ImagePyramid& basispyr, int maxlev,
        int numentries, bool isboosted, float boostfrac);
    PyramidalSimilarityMap(const ImagePyramid& basispyr, int maxlev,
        int numentries, bool isboosted, float boostfrac,
        InterstitialActionReceiver& iar);
    PyramidalSimilarityMap(const std::string& infile);
    PyramidalSimilarityMap(const std::string& infile,
        InterstitialActionReceiver& iar);

    virtual  ~PyramidalSimilarityMap( );

    virtual  std::vector<Vector2Di>  pixelsSimilarTo(int lev,
        const Vector2Di& p) const;
    virtual  std::vector<Vector2Di>  pixelsSimilarTo(int lev, int x,
        int y) const;
    virtual  const std::vector<Rectangle2Di>&  mosaicTiles( ) const;
    virtual  const RGBAImage&  mosaicImage( ) const;
    virtual  void  write(const std::string& outfile) const;
    virtual  int  entriesPerPixel( ) const;

    void  registerReceiver(InterstitialActionReceiver& recv);
    void  deregisterReceiver(InterstitialActionReceiver& recv);
    void  cancelAction(InterstitialActionReceiver& recv);
    void  interstitialActionEvent(InterstitialActionSender& sender,
        const InterstitialActionToken& evt);
};
//--------------------------------------------------------------------------//

#endif
