//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// TextureSynthesis.cpp                                                     //
//                                                                          //
// Implements methods in classes ImagePyramid, PixelNeighborhood, and       //
// ImageSimilarityMap                                                       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver 3.0.2 of Mon 23 June 2008 @ 10:56pm EDT                              //
//                                                                          //
//     the EdgingMode enumerated type was made global; previously, it was   //
//     nested in class PixelNeighborhood                                    //
//                                                                          //
// ver 3.0.1 of Fri 3 May 2008 @ 7:43pm EDT                                 //
//                                                                          //
//     changed all references of class template Vertex2D to Vector2D; see   //
//     the AffineGeometry module's header for more info                     //
//                                                                          //
//  ver 3.0.0 of Thu 02-May-2008 @ 12:12pm EDT                              //
//                                                                          //
//      PixelNeighborhoods now use pre-computed weights in distance( ) so   //
//      we don't have to evaluate a transcendental function. Added class    //
//      ImageSimilarityMap and removed class PyramidalTextureCache. Tested  //
//      as part of Similaris and SimilarisReader and considered stable.     //
//                                                                          //
//  older versions:                                                         //
//                                                                          //
//      older change history elided; check out an older CVS rev to get it   //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008, Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "TextureSynthesis.h"
#include "StructuredIO.h"
#include <string>
#include <sstream>
#include <cstring>
#include <cmath>
#include <ctime>
#include <fstream>

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  PixelNeighborhood                                                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
// NOTE: initializations of the static neighborhood weights arrays have     //
//       been placed at the end of this file because of their length        //
//////////////////////////////////////////////////////////////////////////////

bool  PixelNeighborhood::sAreWeightsInited = false;
const float*   PixelNeighborhood::sWeights[9];


PixelNeighborhood::PixelNeighborhood(const PixelNeighborhood& source)
    : fDrawableStore(0)
{
    int numelts = 3 * (2 * source.fRadius + 1) * (2 * source.fRadius + 1);
    
    fPixels = new float[numelts];
    std::memcpy(fPixels, source.fPixels, numelts * sizeof(float));

    if (source.fDrawableStore) {

        int storesize = (2 * source.fRadius + 1) * (2 * source.fRadius + 1);

        fDrawableStore = new RGBColorNative [storesize];

        std::memcpy(fDrawableStore, source.fDrawableStore,
            storesize * sizeof(RGBColorNative));
    }

    fRadius = source.fRadius;

    if (!sAreWeightsInited) {

        uInitWeightsPointers(sWeights);
        sAreWeightsInited = true;
    }
}




PixelNeighborhood::PixelNeighborhood(int rad, float* pixdata)
    : fDrawableStore(0)
{
    if ((rad < 0) || (rad > kMaxRadius)) {

        std::ostringstream  msgstream;

        msgstream << "PixelNeighborood: specified radius " << rad <<
            "does not fall within permissible range [0, " << kMaxRadius <<
            "]";
        
        throw std::invalid_argument(msgstream.str( ));
    }

    fPixels = pixdata;
    fRadius = rad;

    if (!sAreWeightsInited) {

        uInitWeightsPointers(sWeights);
        sAreWeightsInited = true;
    }
}




PixelNeighborhood::PixelNeighborhood(int rad, const RGBAImage& img, int x,
    int y, EdgingMode mode)
        : fRadius(rad), fDrawableStore(0)
{
    if ((rad < 0) || (rad > kMaxRadius)) {

        std::ostringstream  msgstream;

        msgstream << "PixelNeighborood: specified radius " << rad <<
            "does not fall within permissible range [0, " << kMaxRadius <<
            "]";
        
        throw std::invalid_argument(msgstream.str( ));
    }

    int numelts = 3 * (2 * rad + 1) * (2 * rad + 1);

    fPixels = new float[numelts];

    int pixticker = 0;

    for (int j = -rad; j <= rad; j++) {
        for (int i = -rad; i <= rad; i++) {

            int  currpixloc_x = x + i;
            int  currpixloc_y = y + j;

            Vector2Di edgedloc =
                uEdgePixel(img, Vector2Di(currpixloc_x, currpixloc_y), mode);

            RGBAImage::const_fragment  currpix =
                img.fragmentAt(static_cast<unsigned short>(edgedloc.x),
                static_cast<unsigned short>(edgedloc.y));

            fPixels[pixticker++] = currpix.color( ).r;
            fPixels[pixticker++] = currpix.color( ).g;
            fPixels[pixticker++] = currpix.color( ).b;

        } /* for all pixels in the current row */
    } /* for all rows in the neighborhood */

    if (!sAreWeightsInited) {

        uInitWeightsPointers(sWeights);
        sAreWeightsInited = true;
    }
}




PixelNeighborhood::~PixelNeighborhood( )
{
    delete [ ] fPixels;

    if (fDrawableStore)
        delete [ ] fDrawableStore;
}




const PixelNeighborhood&  PixelNeighborhood::operator=(
    const PixelNeighborhood& source)
{
    if (&source == this)
        return *this;

    int numelts = 3 * (2 * source.fRadius + 1) * (2 * source.fRadius + 1);
    
    float* copy = new float[numelts];
    std::memcpy(copy, source.fPixels, numelts * sizeof(float));

    delete fPixels;
    fPixels = copy;

    if (fDrawableStore) {

        delete [ ] fDrawableStore;
        fDrawableStore = 0;
    }

    if (source.fDrawableStore) {

        int storesize = (2 * source.fRadius + 1) * (2 * source.fRadius + 1);

        fDrawableStore = new RGBColorNative [storesize];

        std::memcpy(fDrawableStore, source.fDrawableStore,
            storesize * sizeof(RGBColorNative));
    }

    fRadius = source.fRadius;

    return *this;
}




float  PixelNeighborhood::distance(const PixelNeighborhood& other) const
{
    if (fRadius != other.fRadius)
        throw std::logic_error("PixelNeighborhood: can't compute perceptual "
            "distance between two neighborhoods of differing radii");

    int numelts = 3 * (2 * fRadius + 1) * (2 * fRadius + 1);

    float accum = 0.0f;

    int nhticker = 0;
    for (int i = 0; i < numelts; i += 3) {

        float rawval = std::sqrt(
            ((this->fPixels[i] - other.fPixels[i]) *
                (this->fPixels[i] - other.fPixels[i])) +
            ((this->fPixels[i + 1] - other.fPixels[i + 1]) *
                (this->fPixels[i + 1] - other.fPixels[i + 1])) +
            ((this->fPixels[i + 2] - other.fPixels[i + 2]) *
                (this->fPixels[i + 2] - other.fPixels[i + 2])));

        accum += rawval * sWeights[fRadius][nhticker];
        
        nhticker++;
    }

    return accum;
}




void  PixelNeighborhood::draw( ) const
{
    if (!fDrawableStore) {

        int storesize = (2 * fRadius + 1) * (2 * fRadius + 1);

        fDrawableStore = new RGBColorNative [storesize];
    
        int numcomps = 3 * (2 * fRadius + 1) * (2 * fRadius + 1);

        RGBColorNative* storeiter = & fDrawableStore[0];

        for (int i = 0; i < numcomps; i += 3) {

            *storeiter = Win32Tools::toNative(RGBColor(fPixels[i],
                fPixels[i + 1], fPixels[i + 2]));

            storeiter++;
        } /* for */
    } /* if */

    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);

    glDrawPixels((2 * fRadius) + 1, (2 * fRadius) + 1, GL_RGBA,
        GL_UNSIGNED_INT_8_8_8_8_REV, fDrawableStore);
}




Vector2Di  PixelNeighborhood::uEdgePixel(const RGBAImage& img,
    const Vector2Di& loc, EdgingMode mode)
{
    if ((mode != kReflectMode) && (mode != kWrapMode))
        throw std::invalid_argument("PixelNeighborhood: invalid edging mode");

    Vector2Di bounds = Vector2Di(img.width( ), img.height( )) -
        Vector2Di(1, 1);
    Vector2Di result;

    if (loc.x < 0) {

        if (mode == kReflectMode)
            result.x = -loc.x;
        else
            result.x = img.width( ) + loc.x;
    }
    else if (loc.x > bounds.x) {

        if (mode == kReflectMode)
            result.x = (2 * bounds.x) - loc.x;
        else
            result.x = loc.x - img.width( );
    }
    else {

        result.x = loc.x;
    }


    if (loc.y < 0) {

        if (mode == kReflectMode)
            result.y = -loc.y;
        else
            result.y = img.height( ) + loc.y;
    }
    else if (loc.y > bounds.y) {

        if (mode == kReflectMode)
            result.y = (2 * bounds.y) - loc.y;
        else
            result.y = loc.y - img.height( );
    }
    else {

        result.y = loc.y;
    }

    return result;
}




void  PixelNeighborhood::uInitWeightsPointers(const float** ptrarray)
{
    ptrarray[0] = kNHWeights0;
    ptrarray[1] = kNHWeights1;
    ptrarray[2] = kNHWeights2;
    ptrarray[3] = kNHWeights3;
    ptrarray[4] = kNHWeights4;
    ptrarray[5] = kNHWeights5;
    ptrarray[6] = kNHWeights6;
    ptrarray[7] = kNHWeights7;
    ptrarray[8] = kNHWeights8;
}
//////////////////////////////////////////////////////////////////////////////
// END  PixelNeighborhood                                                   //
//////////////////////////////////////////////////////////////////////////////









//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ImagePyramid                                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const float  ImagePyramid::kDefaultSampleRadius = 2.1f;

void  ImagePyramid::uCheckPixelRange(int lev, int x, int y) const
{
    if ((x < 0) || (x >= levelDimension(lev)) || (y < 0) ||
        (y >= levelDimension(lev)))
            throw std::out_of_range("ImagePyramid: pixel out of range");
}




void  ImagePyramid::uDownsampleBuffer(RGBColorNative* dest, int destdim,
    const RGBColorNative* src, int srcdim)
{
    float src_to_dest = static_cast<float>(destdim) /
        static_cast<float>(srcdim);
    float dest_to_src = static_cast<float>(srcdim) /
        static_cast<float>(destdim);

    int numdestpix = destdim * destdim;

    /* iterate over all the pixels in the destination buffer: for each
       pixel in the destination buffer, compute the location to which it
       corresponds in the source buffer and sample & blend together the
       values of source pixels in a neighborhood surrounding this location
       to compute a destination pixel value */
    
    for (int cdp = 0; cdp < numdestpix; cdp++) { // "cdp" := "current
                                                 //  destination pixel"
        float src_x = dest_to_src * static_cast<float>(cdp % destdim);
        float src_y = dest_to_src * static_cast<float>(cdp / destdim);

        float src_x_min = std::ceil(src_x - fSampleRadius);
        if (src_x_min < 0.0f)
            src_x_min = 0.0f;

        float src_y_min = std::ceil(src_y - fSampleRadius);
        if (src_y_min < 0.0f)
            src_y_min = 0.0f;
        
        float src_x_max = std::floor(src_x + fSampleRadius);
        if (src_x_max > static_cast<float>(srcdim - 1))
            src_x_max = static_cast<float>(srcdim - 1);
        
        float src_y_max = std::floor(src_y + fSampleRadius);
        if (src_y_max > static_cast<float>(srcdim - 1))
            src_y_max = static_cast<float>(srcdim - 1);

        float    weight_accum = 0.0f;
        RGBColor color_accum(0.0f, 0.0f, 0.0f);

        for (float iy = src_y_min; iy <= src_y_max; iy += 1.0f) {
            for (float ix = src_x_min; ix <= src_x_max; ix += 1.0f) {

                float curweight = uEvalKernel(ix - src_x, iy - src_y);

                int csp = static_cast<int>(ix) + srcdim *
                    static_cast<int>(iy);

                color_accum += curweight * Win32Tools::fromNative(src[csp]);
                weight_accum += curweight;
            }
        }

        color_accum = (1.0f / weight_accum) * color_accum;

        dest[cdp] = Win32Tools::toNative(color_accum);
    }
}




void  ImagePyramid::uDownsampleBuffer(RGBColorNative* dest, int destdim,
    const RGBColorNative* src, int srcdim, const int& totalpix,
    int& processedpix)
{
    static  const float  kNotifyTicks_float =
        static_cast<float>(CLOCKS_PER_SEC) * (1.0f / 60.0f);
    static  const int    kNotifyTicks = static_cast<int>(kNotifyTicks_float);
    static  clock_t      lasttime = 0;

    float src_to_dest = static_cast<float>(destdim) /
        static_cast<float>(srcdim);
    float dest_to_src = static_cast<float>(srcdim) /
        static_cast<float>(destdim);

    int numdestpix = destdim * destdim;

    /* iterate over all the pixels in the destination buffer: for each
       pixel in the destination buffer, compute the location to which it
       corresponds in the source buffer and sample & blend together the
       values of source pixels in a neighborhood surrounding this location
       to compute a destination pixel value */

    for (int cdp = 0; cdp < numdestpix; cdp++) { // "cdp" := "current
                                                 //  destination pixel"
        float src_x = dest_to_src * static_cast<float>(cdp % destdim);
        float src_y = dest_to_src * static_cast<float>(cdp / destdim);

        float src_x_min = std::ceil(src_x - fSampleRadius);
        if (src_x_min < 0.0f)
            src_x_min = 0.0f;

        float src_y_min = std::ceil(src_y - fSampleRadius);
        if (src_y_min < 0.0f)
            src_y_min = 0.0f;
        
        float src_x_max = std::floor(src_x + fSampleRadius);
        if (src_x_max > static_cast<float>(srcdim - 1))
            src_x_max = static_cast<float>(srcdim - 1);
        
        float src_y_max = std::floor(src_y + fSampleRadius);
        if (src_y_max > static_cast<float>(srcdim - 1))
            src_y_max = static_cast<float>(srcdim - 1);

        float    weight_accum = 0.0f;
        RGBColor color_accum(0.0f, 0.0f, 0.0f);

        for (float iy = src_y_min; iy <= src_y_max; iy += 1.0f) {
            for (float ix = src_x_min; ix <= src_x_max; ix += 1.0f) {

                float curweight = uEvalKernel(ix - src_x, iy - src_y);

                int csp = static_cast<int>(ix) + srcdim *
                    static_cast<int>(iy);

                color_accum += curweight * Win32Tools::fromNative(src[csp]);
                weight_accum += curweight;
            }
       }

        color_accum = (1.0f / weight_accum) * color_accum;

        dest[cdp] = Win32Tools::toNative(color_accum);
 
        processedpix++;

        if ((std::clock( ) - lasttime) > kNotifyTicks) {

            float pctdone_float = (static_cast<float>(processedpix) /
                static_cast<float>(totalpix)) * 100.0f;

            uBroadcastUniformIAE(static_cast<int>(pctdone_float));

            lasttime = std::clock( );
        } /* if */ 
    } /* for all pixels in destination buffer */
} /* uDownsampleBuffer( ) */




void  ImagePyramid::uConstructorHelper(const RGBAImage& base, int numlevels)
{
    fNumLevels = numlevels;
    RGBColorNative** imagebuffers = new RGBColorNative*[numlevels];
    int* leveldims = new int [numlevels];

    leveldims[0] = base.width( );

    int num_basepix = base.width( ) * base.height( );

    imagebuffers[0] = new RGBColorNative[num_basepix];

    std::memcpy(imagebuffers[0],
        reinterpret_cast<const RGBColorNative*>(base.imageBuffer( )),
        num_basepix * sizeof(RGBColorNative));

    /* declare and initialize donepix and pixtodo, but actually compute
       their values only if we need to send out an IAE */
    int  donepix = 0;
    int  pixtodo = 0;
    int  divisor = 4;
    if (!fReceiverSet.empty( )) {

        for (int i = 1; i < numlevels; i++) {

            int levelpix = ((num_basepix / divisor) > 0) ?
                (num_basepix / divisor) : 1;

            pixtodo += levelpix;
            divisor *= 4;
        }
    } /* if receiver set is non-empty */

    for (int lev = 1; lev < numlevels; lev++) {

        leveldims[lev] = leveldims[lev - 1] / 2;

        if (leveldims[lev] == 0)
            leveldims[lev] = 1;

        int curlev_numpix = leveldims[lev] * leveldims[lev];

        imagebuffers[lev] = new RGBColorNative [curlev_numpix];

        if ((leveldims[lev] == 1) && (leveldims[lev - 1] == 1)) {

            imagebuffers[lev][0] = imagebuffers[lev - 1][0];
            donepix++;
        }
        else {

            /* only if the receiver set is non-empty to we need to use the
               instrumented version of uDownsampleBuffer( ) */
            if (!fReceiverSet.empty( ))
                uDownsampleBuffer(imagebuffers[lev], leveldims[lev],
                    imagebuffers[lev - 1], leveldims[lev - 1], pixtodo,
                    donepix);
            else
                uDownsampleBuffer(imagebuffers[lev], leveldims[lev],
                    imagebuffers[lev - 1], leveldims[lev - 1]);
        }
    } /* for */

    fMosaicImage = uMixPyramidMosaic(imagebuffers, leveldims,
        fNumLevels);

    delete [ ] leveldims;
}




RGBAImage*  ImagePyramid::uMixPyramidMosaic(RGBColorNative** imagebuffers,
    int* leveldims, int numlevs)
{
    const int result_width = leveldims[0] + (leveldims[0] / 2) + 3;
    const int result_height = leveldims[0] + 2;

    RGBAImage*  result = new RGBAImage(result_width,result_height);

    RGBAImage::fragment f = result->firstFragment( );
    while (f <= result->lastFragment( )) {

        static  const RGBColor  backblue = RGBColor(0.8f, 0.93f, 0.95f);

        f.setColor(backblue);

        f++;
    }

    fTileset.clear( );

    MosaicTile  basetile = MosaicTile(0, Vector2Di(1, 1), leveldims[0]);
    fTileset.push_back(Rectangle2Di(basetile.origin( ), basetile.dimension( ),
        basetile.dimension( )));

    MosaicTile currtile = basetile.next( );

    while (currtile != MosaicTile::nullTile( )) {

        fTileset.push_back(Rectangle2Di(currtile.origin( ),
            currtile.dimension( ), currtile.dimension( )));
        currtile = currtile.next( );
    };

    for (int tnum = 0; tnum < fTileset.size( ); tnum++) {

        const int srcrowlen = fTileset[tnum].width;
        const int destrowlen = result->width( );
        const int srcnumrows = fTileset[tnum].height;
        const int destnumrows = result->height( );
        const int deststartrow = fTileset[tnum].origin.y;
        const int deststartentry = fTileset[tnum].origin.x;
        const int bytesPerPix = 4;

        const unsigned char*  srcbuffer =
            reinterpret_cast<const unsigned char*>(imagebuffers[tnum]);
        unsigned char*  destbuffer = result->imageBuffer( );
        for (int i = 0; i < srcnumrows; i++) {

            size_t  srcoffset = i * srcrowlen * bytesPerPix;
            size_t  destoffset = (((deststartrow + i) * destrowlen) +
                deststartentry) * bytesPerPix;

            std::memcpy(&destbuffer[destoffset], &srcbuffer[srcoffset],
                srcrowlen * bytesPerPix);
        }   
    }

    return result;
}




ImagePyramid::ImagePyramid(const RGBAImage& base, int numlevels)
    : fSampleRadius(kDefaultSampleRadius)
{
    if (base.width( ) != base.height( ))
        throw std::invalid_argument("ImagePyramid: base image isn't square");

    const float maxlevs_float =
        std::log(static_cast<float>(base.width( ))) / std::log(2.0f);

    const int maxlevs = static_cast<int>(maxlevs_float) + 1;

    if ((numlevels < 1) || (numlevels > maxlevs))
        throw std::out_of_range("ImagePyramid: number of levels out of "
            "range");

    uConstructorHelper(base, numlevels);
}




ImagePyramid::ImagePyramid(const RGBAImage& base, int numlevels,
    InterstitialActionReceiver& iar)
        : fSampleRadius(kDefaultSampleRadius)
{
    if (base.width( ) != base.height( ))
        throw std::invalid_argument("ImagePyramid: base image isn't square");

    const float maxlevs_float =
        std::log(static_cast<float>(base.width( ))) / std::log(2.0f);

    const int maxlevs = static_cast<int>(maxlevs_float) + 1;

    if ((numlevels < 1) || (numlevels > maxlevs))
        throw std::out_of_range("ImagePyramid: number of levels out of "
            "range");

    fReceiverSet.insert(&iar);

    uConstructorHelper(base, numlevels);
}




ImagePyramid::ImagePyramid(const RGBAImage& base)
    : fSampleRadius(kDefaultSampleRadius)

{
    if (base.width( ) != base.height( ))
        throw std::invalid_argument("ImagePyramid: base image isn't square");

    const float numlevels_float =
        std::log(static_cast<float>(base.width( ))) / std::log(2.0f);

    const int numlevels = static_cast<int>(numlevels_float) + 1;

    uConstructorHelper(base, numlevels);
}




ImagePyramid::ImagePyramid(const RGBAImage& base,
    InterstitialActionReceiver& iar)
        : fSampleRadius(kDefaultSampleRadius)
{
    if (base.width( ) != base.height( ))
        throw std::invalid_argument("ImagePyramid: base image isn't square");

    const float numlevels_float =
        std::log(static_cast<float>(base.width( ))) / std::log(2.0f);

    const int numlevels = static_cast<int>(numlevels_float) + 1;

    fReceiverSet.insert(&iar);

    uConstructorHelper(base, numlevels);
}




ImagePyramid::~ImagePyramid( )
{
    delete fMosaicImage;
}




const RGBColor  ImagePyramid::pixel(int lev, int x, int y) const
{
    RGBColorNative  natpix = nativePixel(lev, x, y);

    return Win32Tools::fromNative(natpix);
}




const RGBColorNative  ImagePyramid::nativePixel(int lev, int x, int y) const
{
    uCheckPixelRange(lev, x, y);

    const int  pix_x = fTileset[lev].origin.x + x;
    const int  pix_y = fTileset[lev].origin.y + y;
    const int  image_wd = fMosaicImage->width( );

    const int  pix_idx = (pix_y * image_wd) + pix_x;

    RGBColorNative*  buffer =
        reinterpret_cast<RGBColorNative*>(fMosaicImage->imageBuffer( ));

    return buffer[pix_idx];
}




PixelNeighborhood  ImagePyramid::createNeighborhood(int lev, int x, int y,
                            int rad) const
{
    uCheckPixelRange(lev, x, y);

    if ((rad < 0) || (rad > PixelNeighborhood::kMaxRadius)) {

        std::ostringstream  msgstream;

        msgstream << "ImagePyramid: can't create a neighborhood of " <<
            "radius " << rad << ": permissible range is 0..." <<
            PixelNeighborhood::kMaxRadius;

        throw std::invalid_argument(msgstream.str( ));

    }

    const int  maxcol = fTileset[lev].width - 1;
    const int  maxrow = maxcol;

    if ( ((x - rad) < 0) || ((x + rad) > maxcol) || ((y - rad) < 0) ||
         ((y + rad) > maxrow) ) {

             std::ostringstream  msgstream;

             msgstream << "ImagePyramid: can't create a neighborhood of " <<
                 "radius " << rad << " at (" << x << ", " << y <<
                 ") on level " << lev << ": neighborhood includes " <<
                 "out of bounds pixels";

             throw std::invalid_argument(msgstream.str( ));
    }

    const int  rowlen = (2 * rad) + 1;
    const int  numrows = rowlen;
    const int  beginpix_x = x - rad;
    const int  endpix_x = x + rad;
    const int  beginpix_y = y - rad;
    const int  endpix_y = y + rad;

    float* resultbuf = new float [3 * rowlen * numrows];
    float* bufiter = resultbuf;

    for (int j = beginpix_y; j <= endpix_y; j++) {
        for (int i = beginpix_x; i <= endpix_x; i++) {

            RGBColor pixval = pixel(lev, i, j);

            *bufiter = pixval.r;
            bufiter++;

            *bufiter = pixval.g;
            bufiter++;

            *bufiter = pixval.b;
            bufiter++;
        }
    };

    return PixelNeighborhood(rad, resultbuf);
}




int  ImagePyramid::numLevels( ) const
{
    return fNumLevels;
}




int  ImagePyramid::levelDimension(int lev) const
{
    if ((lev < 0) || (lev >= fNumLevels))
        throw std::logic_error("ImagePyramid: level doesn't exist");
    else
        return fTileset[lev].width;
}




void  ImagePyramid::draw(int lev) const
{
    if ((lev < 0) || (lev >= fNumLevels))
        throw std::logic_error("ImagePyramid: level doesn't exist");

    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, fMosaicImage->width( ));
    glPixelStorei(GL_UNPACK_SKIP_ROWS, fTileset[lev].origin.y);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, fTileset[lev].origin.x);

    glDrawPixels(fTileset[lev].width, fTileset[lev].height, GL_RGBA,
        GL_UNSIGNED_INT_8_8_8_8_REV, fMosaicImage->imageBuffer( ));
}




void  ImagePyramid::draw(int lev, int left, int bot, int right, int top) const
{
    if ((lev < 0) || (lev >= fNumLevels))
        throw std::logic_error("ImagePyramid: level doesn't exist");

    int subwidth = right - left;
    int subheight = top - bot;

    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, fMosaicImage->width( ));
    glPixelStorei(GL_UNPACK_SKIP_ROWS, fTileset[lev].origin.y + bot);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, fTileset[lev].origin.x + left);

    glDrawPixels(subwidth, subheight, GL_RGBA, GL_UNSIGNED_INT_8_8_8_8_REV,
        fMosaicImage->imageBuffer( ));
}




void  ImagePyramid::registerReceiver(InterstitialActionReceiver& recv)
{
    fReceiverSet.insert(&recv);
}




void  ImagePyramid::deregisterReceiver(InterstitialActionReceiver& recv)
{
    if (fReceiverSet.find(&recv) != fReceiverSet.end( ))
        fReceiverSet.erase(&recv);
}




void  ImagePyramid::cancelAction(InterstitialActionReceiver& recv)
{
}




void  ImagePyramid::uBroadcastUniformIAE(int pctprog)
{
    static const std::string  kIAEStatusMessage =
        "Building image pyramid...";

    InterstitialActionToken  out_iae(kIAEStatusMessage, pctprog);

    std::set<InterstitialActionReceiver*>::iterator it;
    for (it = fReceiverSet.begin( ); it != fReceiverSet.end( ); it++)
        (*it)->interstitialActionEvent(*this, out_iae);
}




void  ImagePyramid::copy(RGBAImage& dest, int lev,
                         const Vector2Di& where) const
{
    if ((lev < 0) || (lev >= fNumLevels)) {

        std::ostringstream  msgstream;
        msgstream << "ImagePyramid: can't copy pixels from level " << lev <<
            ": level doesn't exist";

        throw std::invalid_argument(msgstream.str( ));
    }

    RGBAImage::transferPixels(*fMosaicImage, dest, fTileset[lev].origin.x,
        fTileset[lev].origin.y, where.x, where.y, levelDimension(lev),
        levelDimension(lev));
}




const  RGBAImage&  ImagePyramid::mosaicImage( ) const
{
    return *fMosaicImage;
}




const std::vector<Rectangle2Di>&  ImagePyramid::mosaicTiles( ) const
{
    return fTileset;
}
//////////////////////////////////////////////////////////////////////////////
// END  ImagePyramid                                                        //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// NESTED CLASS  ImagePyramid :: MosaicTile                                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
ImagePyramid::MosaicTile::MosaicTile(int seqnum, const Vector2Di& origin,
    int dim) : fSeqNum(seqnum), fOrigin(origin), fDimension(dim)
{
}




ImagePyramid::MosaicTile  ImagePyramid::MosaicTile::nullTile( )
{
    return MosaicTile(-1, Vector2Di(0, 0), -1);
}




ImagePyramid::MosaicTile  ImagePyramid::MosaicTile::next( ) const
{
    if (isNullTile( ))
        throw std::logic_error("MosaicTile: can't get next tile: tile "
            "is null");

    if (fDimension == 1)
        return  nullTile( );

    int  nextdim = fDimension / 2;

    if ((fSeqNum % 2) == 0) {

        Vector2Di  nextogn = fOrigin;
        nextogn.x += (fDimension + 1);

        return MosaicTile(fSeqNum + 1, nextogn, nextdim);
    }
    else {

        Vector2Di  nextogn = fOrigin;
        nextogn.y += (fDimension + 1);

        return MosaicTile(fSeqNum + 1, nextogn, nextdim);
    }
}




const Vector2Di&  ImagePyramid::MosaicTile::origin( ) const
{
    if (isNullTile( ))
        throw std::logic_error("MosaicTile: can't get tile origin: tile "
            "is null");

    return fOrigin;
}




int  ImagePyramid::MosaicTile::dimension( ) const
{
    if (isNullTile( ))
        throw std::logic_error("MosaicTile: can't get tile dimension: tile "
            "is null");

    return fDimension;
}




bool  ImagePyramid::MosaicTile::isNullTile( ) const
{
    if (fSeqNum < 0)
        return true;
    else
        return false;
}




bool  ImagePyramid::MosaicTile::operator==(
    const ImagePyramid::MosaicTile& rhs)
{
    if (this->isNullTile( ) && rhs.isNullTile( ))
        return true;

    if (fSeqNum == rhs.fSeqNum)
        if (fOrigin == rhs.fOrigin)
            if (fDimension == rhs.fDimension)
                return true;

    return false;
}




bool  ImagePyramid::MosaicTile::operator!=(
    const ImagePyramid::MosaicTile& rhs)
{
    return (! (*this == rhs));
}
//////////////////////////////////////////////////////////////////////////////
// END  MosaicTile                                                          //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ImageSimilarityMap                                                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const unsigned int  ImageSimilarityMap::kNotifyTicks =
        static_cast<unsigned int>(static_cast<float>(CLOCKS_PER_SEC) * 
        (1.0f / 60.0f));


ImageSimilarityMap::ImageSimilarityMap(const RGBAImage& srcimg,
    int numentries, bool isboosted, float boostfrac)
        : fDidUserCancel(false), fIsBoosted(isboosted),
          fBoostSepFrac(boostfrac), fInstrLastDispatch(0),
          fInstrTotalPix(0.0), fInstrDonePix(0.0),
          fNumEdgeExclPix(kDefaultEdgeExclusionPixels)
{
    uConstructorHelper(srcimg, numentries, isboosted, boostfrac);
}




ImageSimilarityMap::ImageSimilarityMap(const RGBAImage& srcimg,
    int numentries, bool isboosted, float boostfrac,
    InterstitialActionReceiver& iar)
        : fDidUserCancel(false), fIsBoosted(isboosted),
          fBoostSepFrac(boostfrac), fInstrLastDispatch(0),
          fInstrTotalPix(0.0), fInstrDonePix(0.0),
          fNumEdgeExclPix(kDefaultEdgeExclusionPixels)
{
    fReceiverSet.insert(&iar);
    uConstructorHelper(srcimg, numentries, isboosted, boostfrac);
}




ImageSimilarityMap::ImageSimilarityMap(const std::string& infile)
    : fDidUserCancel(false), fIsBoosted(false), fBoostSepFrac(0.0f),
      fInstrLastDispatch(0), fInstrTotalPix(0.0), fInstrDonePix(0.0),
      fNumEdgeExclPix(kDefaultEdgeExclusionPixels)
{
    uConstructorHelper(infile);
}




ImageSimilarityMap::ImageSimilarityMap(const std::string& infile,
    InterstitialActionReceiver& iar)
    : fDidUserCancel(false), fIsBoosted(false), fBoostSepFrac(0.0f),
      fInstrLastDispatch(0), fInstrTotalPix(0.0), fInstrDonePix(0.0),
      fNumEdgeExclPix(kDefaultEdgeExclusionPixels)
{
    fReceiverSet.insert(&iar);
    uConstructorHelper(infile);
}




void  ImageSimilarityMap::uBroadcastUniformIAE(int pctprog)
{
    static const std::string  kIAEStatusMessage =
        "Building image similarity map...";

    InterstitialActionToken  out_iae(kIAEStatusMessage, pctprog);

    std::set<InterstitialActionReceiver*>::iterator it;
    for (it = fReceiverSet.begin( ); it != fReceiverSet.end( ); it++)
        (*it)->interstitialActionEvent(*this, out_iae);
}




void  ImageSimilarityMap::uConstructorHelper(const RGBAImage& basisimg,
    int numentries, bool isboosted, float boostfrac)
{
    fInstrTotalPix = static_cast<double>(basisimg.width( )) *
        static_cast<double>(basisimg.height( )) *
        static_cast<double>(basisimg.width( )) *
        static_cast<double>(basisimg.height( ));

    fEntriesPerPix = numentries;
    fWidth = basisimg.width( );
    fHeight = basisimg.height( );
    fEncodedImage = new RGBAImage(numentries * basisimg.width( ),
        basisimg.height( ));

    fInstrLastDispatch = std::clock( );

    for (int j = 0; j < basisimg.height( ); j++) {
        for (int i = 0; i < basisimg.width( ); i++) {

            std::map<float, Vector2Di> distmap;

            if (fIsBoosted)
                uFindSimilarBoosted(basisimg, i, j, distmap);
            else
                uFindSimilarUnboosted(basisimg, i, j, distmap);

            std::map<float, Vector2Di>::const_iterator it = distmap.begin( );
            RGBAImage::fragment mapentrypix =
                fEncodedImage->fragmentAt(fEntriesPerPix * i, j);
            for ( ; it != distmap.end( ); it++) {

                RGBColor coordcolor =
                    RGBColor(static_cast<float>(it->second.x) / 255.0f,
                    static_cast<float>(it->second.y) / 255.0f,
                    0.0f);

                mapentrypix.setColor(coordcolor);

                mapentrypix++;
            } /* for all perceptually similar pixels in the perceptual
                 distance map */

        } /* for all pixels in row */

    } /* for all rows in basisimg */
}




void  ImageSimilarityMap::uConstructorHelper(const std::string& infile)
{
    std::string indata = Win32Tools::filesystem( ).mapTextFile(infile);

    StructuredReader inreader(indata);

    /* make sure the user gave us the right type of file  */
    std::string filekind = inreader.readFiletypeID( );
    if (filekind != "SIMSET")
    throw std::runtime_error("ImageSimilarityMap: input file is of "
            "incorrect kind");

    /* read the file header  */
    std::string curtok;
    curtok = inreader.readIdentifier( );
    if (curtok != "simset")
        throw std::runtime_error("ImageSimilarityMap: expected header group");
    inreader.readGroupOpen( );
    bool gotWidth = false, gotHeight = false, gotEntries = false;
    for (int i = 0; i < 3; i++) {

        curtok = inreader.readIdentifier( );
        if (curtok == "horizPixels") {
    
            inreader.readAssignmentOperator( );
            fWidth = inreader.readInteger( );
            gotWidth = true;
        }
        else if (curtok == "vertPixels") {
    
            inreader.readAssignmentOperator( );
            fHeight = inreader.readInteger( );
            gotHeight = true;
        }
        else if (curtok == "entriesPerPixel") {
    
            inreader.readAssignmentOperator( );
            fEntriesPerPix = inreader.readInteger( );
            gotEntries = true;
        }
        else {
    
            throw std::runtime_error("ImageSimilarityMap: bad token in file "
                   "header");
        }
    } /* for all 3 required header fields */
    if (! (gotWidth && gotHeight && gotEntries))
        throw std::runtime_error("ImageSimilarityMap: one or more required "
            "header fields not present");
    inreader.readGroupClose( );

    /* create the encoded image and read the data into it */
    fEncodedImage = new RGBAImage(fWidth * fEntriesPerPix, fHeight);
    int numpixbuckets = fWidth * fHeight;
    for (int i = 0; i < numpixbuckets; i++) {

        /* read the target bucket's location */
        Vector2Di targloc = inreader.readVector2Di( );

        /* read the group open brace */
        inreader.readGroupOpen( );

        /* read the target pixel's entries */
        for (int k = 0; k < fEntriesPerPix; k++) {

            RGBAImage::fragment f =
            fEncodedImage->fragmentAt(fEntriesPerPix * targloc.x + k,
                    targloc.y);

            Vector2Di ent = inreader.readVector2Di( );

            RGBColor fval;
            fval.r = static_cast<float>(ent.x) / 255.0f;
            fval.g = static_cast<float>(ent.y) / 255.0f;
            fval.b = 0.0f;
            
            f.setColor(fval);
        } /* for all k similarity entries  */

        /* read the close group brace */
        inreader.readGroupClose( );

    } /* for all i buckets */
}




ImageSimilarityMap::~ImageSimilarityMap( )
{
    delete fEncodedImage;
    fEncodedImage = 0;
}




void  ImageSimilarityMap::uFindSimilarBoosted(const RGBAImage& basisimg,
    int refx, int refy, DistanceMap& outmap)
{
    const float critdist =
        fBoostSepFrac * static_cast<float>(basisimg.width( ));

    PixelNeighborhood basisnh =
        PixelNeighborhood(kNeighborhoodRadius, basisimg, refx, refy,
        kReflectMode);

    outmap.clear( );
    for (int i = 0; i < fEntriesPerPix; i++)
        outmap[1.0e31 - 1.0e30*i] = Vector2Di(0, 0);

    const int  startx = fNumEdgeExclPix;
    const int  stopx = basisimg.width( ) - fNumEdgeExclPix;
    const int  starty = fNumEdgeExclPix;
    const int  stopy = basisimg.height( ) - fNumEdgeExclPix;
    for (int j = starty; j < stopy; j++) {
        for (int i = startx; i < stopx; i++) {

            float ptdelta = uCartesianDist(refx, refy, i, j);

            if (critdist > ptdelta)
                continue;

            PixelNeighborhood candidnh =
                PixelNeighborhood(kNeighborhoodRadius, basisimg, i,
                j, kReflectMode);

            /* nhdelta is perceptual distance between neighborhoods -- not
               Cartesian distance */
            float nhdelta = candidnh.distance(basisnh);
 
            /* if the current perceptual distance is less than at least one in
               the output map */
            if (outmap.upper_bound(nhdelta) != outmap.end( )) {

                DistanceMap  nearset;
                uFindOthersNearby(outmap, nearset, i, j, critdist);

                /* if no pixels in the output map are geometrically close (as
                   Cartesian distance) to the candidate */
                if (nearset.empty( )) {

                    if (outmap.size( ) == fEntriesPerPix)
                        outmap.erase(--outmap.end( ));

                    outmap.insert(std::make_pair(nhdelta, Vector2Di(i, j)));
                }
                else {

                    float nearset_worst = ((--nearset.end( ))->first);

                    if (nhdelta < nearset_worst) {

                        outmap.erase((--nearset.end( ))->first);
                        outmap.insert(std::make_pair(nhdelta,
                            Vector2Di(i, j)));
                    }

                } /* else some pixels in the output map are geometrically 
                     close to the candidate */

            } /* if the current perceptual distance is less than at least one in
                 the output map */


            /* do this section only if the client needs instrumentation */
            if (! fReceiverSet.empty( )) {

                fInstrDonePix += 1.0;

                if ((std::clock( ) - fInstrLastDispatch) > kNotifyTicks) {

                    double pctdone_dbl = static_cast<float>(fInstrDonePix
                        / fInstrTotalPix) * 100.0;

                    uBroadcastUniformIAE(
                        static_cast<int>(pctdone_dbl));

                    fInstrLastDispatch = std::clock( );
                } /* if enough time */ 
            } /* if our client needs instrumentation */

            if (fDidUserCancel)
                return;

        } /* for all pixels in row */
    } /* for all rows in basisimg */
}




void  ImageSimilarityMap::uFindOthersNearby(const DistanceMap& inset,
    DistanceMap& outset, int refx, int refy, float critdist)
{
    outset.clear( );

    DistanceMap::const_iterator iter = inset.begin( );
    for ( ; iter != inset.end( ); iter++) {

        float d = uCartesianDist(refx, refy, iter->second.x, iter->second.y);

        if (d <= critdist)
            outset.insert(*iter);
    }
}




void  ImageSimilarityMap::uFindSimilarUnboosted(const RGBAImage& basisimg,
    int refx, int refy, DistanceMap& outmap)
{
    PixelNeighborhood basisnh =
        PixelNeighborhood(kNeighborhoodRadius, basisimg, refx, refy,
        kReflectMode);

    outmap.clear( );
    outmap[static_cast<float>(INT_MAX)] = Vector2Di(-1, -1);

    for (int j = 0; j < basisimg.height( ); j++) {
        for (int i = 0; i < basisimg.width( ); i++) {

            if ((refx == i) && (refy == j))
                continue;

            PixelNeighborhood candidnh =
                PixelNeighborhood(kNeighborhoodRadius, basisimg, i,
                j, kReflectMode);

            float nhdelta = candidnh.distance(basisnh);
 
            /* if the current perceptual distance is less than at least one in
               the output map */
            if (outmap.upper_bound(nhdelta) != outmap.end( )) {

                if (outmap.size( ) == fEntriesPerPix)
                    outmap.erase(--outmap.end( ));

                outmap.insert(std::make_pair(nhdelta, Vector2Di(i, j)));
            }

            /* do this section only if the client needs instrumentation */
            if (! fReceiverSet.empty( )) {

                fInstrDonePix += 1.0;

                if ((std::clock( ) - fInstrLastDispatch) > kNotifyTicks) {

                    double pctdone_dbl = static_cast<float>(fInstrDonePix
                        / fInstrTotalPix) * 100.0;

                    uBroadcastUniformIAE(
                        static_cast<int>(pctdone_dbl));

                    fInstrLastDispatch = std::clock( );
                } /* if enough time */ 
            } /* if our client needs instrumentation */

            if (fDidUserCancel)
                return;

        } /* for all pixels in row */
    } /* for all rows in basisimg */
}




std::vector<Vector2Di>  ImageSimilarityMap::pixelsSimilarTo(
    const Vector2Di& p) const
{
    std::vector<Vector2Di>  result(fEntriesPerPix);

    for (int i = 0; i < fEntriesPerPix; i++) {

        RGBAImage::const_fragment f =
            fEncodedImage->fragmentAt(fEntriesPerPix * p.x + i, p.y);
    
        result[i].x = static_cast<int>(f.color( ).r * 255.0f);
        result[i].y = static_cast<int>(f.color( ).g * 255.0f);
    }

    return result;
}




std::vector<Vector2Di>  ImageSimilarityMap::pixelsSimilarTo(int x,
    int y) const
{
    return pixelsSimilarTo(Vector2Di(x, y));
}




void  ImageSimilarityMap::write(const std::string& outfile) const
{
    std::ofstream  os(outfile.c_str( ), std::ios::out);
    if (!os)
        throw std::runtime_error("ImageSimilarityMap: unable to open output "
            "file for writing");

    std::string        linepref = "";
    const std::string  kIndent = "    ";

    /* write the file type identifier */
    os << "(# SIMSET)" << "\n\n";

    /* write the information header */
    os << "simset {" << "\n";
    linepref += kIndent;
    os << linepref << "horizPixels     := " << width( ) << "\n";
    os << linepref << "vertPixels      := " << height( ) << "\n";
    os << linepref << "entriesPerPixel := " << entriesPerPixel( ) << "\n";
    linepref = "";
    os << "}\n\n";

    /* write the similarity data */
    for (int j = 0; j < height( ); j++) {
        for (int i = 0; i < width( ); i++) {
    
            int x = fEntriesPerPix * i;
            
            os << "\n";
            os << "(" << i << ", " << j << ")  {\n";
    
            linepref += kIndent;
    
            for (int k = 0; k < fEntriesPerPix; k++) {
    
            RGBAImage::const_fragment f =
                fEncodedImage->fragmentAt(x + k, j);
    
            int entr = static_cast<int>(f.color( ).r * 255.0f);
            int entg = static_cast<int>(f.color( ).g * 255.0f);
    
            os << linepref << "(" << entr << ", " << entg << ")\n";
            } /* for all k entries in the current pixel bucket  */
    
            linepref = "";
            os << "}\n";
    
        } /* for all i pixel buckets in the current row  */
    } /* for all j rows of pixel buckets  */
    
    os.close( );
}




const RGBAImage&  ImageSimilarityMap::encodedImage( ) const
{
    return *fEncodedImage;
}




int  ImageSimilarityMap::width( ) const
{
    return fWidth;
}




int  ImageSimilarityMap::height( ) const
{
    return fHeight;
}




int  ImageSimilarityMap::entriesPerPixel( ) const
{
    return fEntriesPerPix;
}




void  ImageSimilarityMap::registerReceiver(InterstitialActionReceiver& recv)
{
    fReceiverSet.insert(&recv);
}




void  ImageSimilarityMap::deregisterReceiver(InterstitialActionReceiver& recv)
{
    fReceiverSet.erase(&recv);
}



void  ImageSimilarityMap::cancelAction(InterstitialActionReceiver& recv)
{
    fDidUserCancel = true;
}
//////////////////////////////////////////////////////////////////////////////
// END  ImageSimilarityMap                                                  //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  PyramidalSimilarityMap                                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
PyramidalSimilarityMap::PyramidalSimilarityMap(const ImagePyramid& basispyr,
    int maxlev, int numentries, bool isboosted, float boostfrac)
        : fDidUserCancel(false), fIsBoosted(isboosted),
          fBoostSepFrac(boostfrac)
{
    uConstructorHelper(basispyr, maxlev, numentries, isboosted, boostfrac);
}




PyramidalSimilarityMap::PyramidalSimilarityMap(const ImagePyramid& basispyr,
    int maxlev, int numentries, bool isboosted, float boostfrac,
    InterstitialActionReceiver& iar)
        : fDidUserCancel(false), fIsBoosted(isboosted),
          fBoostSepFrac(boostfrac)
{
    fReceiverSet.insert(&iar);
    uConstructorHelper(basispyr, maxlev, numentries, isboosted, boostfrac);
}




PyramidalSimilarityMap::PyramidalSimilarityMap(const std::string& infile)
        : fDidUserCancel(false)
{
    uConstructorHelper(infile);
}




PyramidalSimilarityMap::PyramidalSimilarityMap(const std::string& infile,
    InterstitialActionReceiver& iar)
        : fDidUserCancel(false)
{
    fReceiverSet.insert(&iar);
    uConstructorHelper(infile);
}




void  PyramidalSimilarityMap::uBroadcastUniformIAE(int pctprog)
{
    static const std::string  kIAEStatusMessage =
        "Building image similarity map...";

    InterstitialActionToken  out_iae(kIAEStatusMessage, pctprog);

    std::set<InterstitialActionReceiver*>::iterator it;
    for (it = fReceiverSet.begin( ); it != fReceiverSet.end( ); it++)
        (*it)->interstitialActionEvent(*this, out_iae);
}




void  PyramidalSimilarityMap::uConstructorHelper(const ImagePyramid& basispyr,
    int maxlev, int numentries, bool isboosted, float boostfrac)
{
    fEntriesPerPix = numentries;
    fNumLevels = maxlev + 1;

    uComputeTileset(basispyr.levelDimension(0), basispyr.levelDimension(0),
        fNumLevels, fEntriesPerPix);

    /* for instrumentation only */
    float totalpix = 0.0f;
    for (int i = 0; i < fNumLevels; i++) {

        float pixthislev = static_cast<float>(basispyr.levelDimension(i)) *
            static_cast<float>(basispyr.levelDimension(i));
        fInstFracPerLev.push_back(pixthislev);
        totalpix += pixthislev;
    }
    for (int i = 0; i < fInstFracPerLev.size( ); i++)
        fInstFracPerLev[i] /= totalpix;
    /* end instrumentation */

    uPrepareMosaicImage(fTileset);

    for (int i = 0; i < fNumLevels; i++) {

        RGBAImage* workimg = new RGBAImage(basispyr.mosaicTiles( )[i].width,
            basispyr.mosaicTiles( )[i].height);

        basispyr.copy(*workimg, i, Vector2Di(0, 0));

        fInstrCurrLev = i;
        ImageSimilarityMap* workmap;
        if (! fReceiverSet.empty( ))
            workmap = new ImageSimilarityMap(*workimg, fEntriesPerPix,
                isboosted, boostfrac, *this);
        else
            workmap = new ImageSimilarityMap(*workimg, fEntriesPerPix,
                isboosted, boostfrac);

        if (fDidUserCancel) {

            delete workimg;
            delete workmap;
            return;
        }

        RGBAImage::transferPixels(workmap->encodedImage( ), *fMosaicImage,
            0, 0, fTileset[i].origin.x, fTileset[i].origin.y,
            fTileset[i].width, fTileset[i].height);

        delete workmap;
        delete workimg;
    }
}




void  PyramidalSimilarityMap::uConstructorHelper(const std::string& infile)
{
    std::string indata = Win32Tools::filesystem( ).mapTextFile(infile);

    StructuredReader inreader(indata);

    /* make sure the user gave us the right type of file  */
    std::string filekind = inreader.readFiletypeID( );
    if (filekind != "PYRSIM")
        throw std::runtime_error("PyramidalSimilarityMap: input file is of "
            "incorrect kind");

    /* read the file header  */
    std::string curtok;
    curtok = inreader.readIdentifier( );
    if (curtok != "pyramidalSimset")
        throw std::runtime_error("PyramidalSimilarityMap: expected header "
            "group");
    inreader.readGroupOpen( );
    int basewidth, baseheight;
    bool gotBWid = false, gotBHt = false, gotEnts = false, gotLevs = false,
        gotIsBoosted = false, gotBoostFrac = false;
    for (int i = 0; i < 6; i++) {

        curtok = inreader.readIdentifier( );
        if (curtok == "baseHorizPixels") {
    
            inreader.readAssignmentOperator( );
            basewidth = inreader.readInteger( );
            gotBWid = true;
        }
        else if (curtok == "baseVertPixels") {
    
            inreader.readAssignmentOperator( );
            baseheight = inreader.readInteger( );
            gotBHt = true;
        }
        else if (curtok == "numLevels") {
    
            inreader.readAssignmentOperator( );
            fNumLevels = inreader.readInteger( );
            gotLevs = true;
        }
        else if (curtok == "entriesPerPix") {
    
            inreader.readAssignmentOperator( );
            fEntriesPerPix = inreader.readInteger( );
            gotEnts = true;
        }
        else if (curtok == "isVarBoosted") {
    
            inreader.readAssignmentOperator( );
            fIsBoosted = inreader.readBoolean( );
            gotIsBoosted = true;
        }
        else if (curtok == "boostSepFrac") {
    
            inreader.readAssignmentOperator( );
            fBoostSepFrac = inreader.readFloat( );
            gotBoostFrac = true;
        }
        else {
    
            throw std::runtime_error("PyramidalSimilarityMap: bad token in "
                "file header");
        }
    } /* for all 6 required header fields */
    if (! (gotBWid && gotBHt && gotEnts && gotLevs && gotIsBoosted &&
           gotBoostFrac)) {

            throw std::runtime_error("PyramidalSimilarityMap: one or "
                    "required header fields not present");
    }
    inreader.readGroupClose( );

    /* use header data to build the tileset & prep the mosaic image*/
    uComputeTileset(basewidth, baseheight, fNumLevels, fEntriesPerPix);
    uPrepareMosaicImage(fTileset);

    /* read the level data */
    for (int lev = 0; lev < fNumLevels; lev++) {

        /* read the level number and abort if it's not in order */
        int filelev = inreader.readInteger( );
        if (filelev != lev)
            throw std::runtime_error("PyramidalSimilarityMap: unexpected "
                "level number in input file");

        /* read the current level open brace */
        inreader.readGroupOpen( );

        /* scan through all the pixel buckets on the current level */
        int numpixbuckets = (fTileset[lev].width / fEntriesPerPix) *
            fTileset[lev].height;
        for (int p = 0; p < numpixbuckets; p++) {

            /* read the bucket location vector */
            Vector2Di ploc = inreader.readVector2Di( );
            ploc.x *= fEntriesPerPix;
            ploc = ploc + fTileset[lev].origin;

            /* read the bucket entry group open brace */
            inreader.readGroupOpen( );

            RGBAImage::fragment f = fMosaicImage->fragmentAt(ploc.x, ploc.y);

            for (int e = 0; e < fEntriesPerPix; e++) {

                Vector2Di eval = inreader.readVector2Di( );

                RGBColor evalrgb;
                evalrgb.r = static_cast<float>(eval.x) / 255.0f;
                evalrgb.g = static_cast<float>(eval.y) / 255.0f;
                evalrgb.b = 0.0f;

                f.setColor(evalrgb);

                f++;
            } /* for all entries in the current bucket */

            /* read the bucket entry group close brace */
            inreader.readGroupClose( );

        } /* for all pixel buckets on the current level */

        /* read the current level close brace */
        inreader.readGroupClose( );

    } /* for all levels in the input file */
}




PyramidalSimilarityMap::~PyramidalSimilarityMap( )
{
    delete fMosaicImage;
}





std::vector<Vector2Di>  PyramidalSimilarityMap::pixelsSimilarTo(int lev,
    const Vector2Di& p) const
{
    Vector2Di floc = p + fTileset[lev].origin;

    RGBAImage::const_fragment f = fMosaicImage->fragmentAt(floc.x, floc.y);

    std::vector<Vector2Di>  result(fEntriesPerPix);
    for (int i = 0; i < fEntriesPerPix; i++) {

        Vector2Di fval;
        fval.x = static_cast<int>(f.color( ).r * 255.0f);
        fval.y = static_cast<int>(f.color( ).g * 255.0f);
        result[i] = fval;

        f++;
    }

    return result;
}




std::vector<Vector2Di>  PyramidalSimilarityMap::pixelsSimilarTo(int lev,
    int x, int y) const
{
    return pixelsSimilarTo(lev, Vector2Di(x, y));
}




const std::vector<Rectangle2Di>&
    PyramidalSimilarityMap::mosaicTiles( ) const
{
    return fTileset;
}




const RGBAImage&  PyramidalSimilarityMap::mosaicImage( ) const
{
    return *fMosaicImage;
}





void  PyramidalSimilarityMap::registerReceiver(
    InterstitialActionReceiver& recv)
{
}




void  PyramidalSimilarityMap::deregisterReceiver(
    InterstitialActionReceiver& recv)
{
}




void  PyramidalSimilarityMap::cancelAction(InterstitialActionReceiver& recv)
{
    fDidUserCancel = true;
}




void  PyramidalSimilarityMap::interstitialActionEvent(
    InterstitialActionSender&sender, const InterstitialActionToken& evt)
{
    if (fDidUserCancel) {

        sender.cancelAction(*this);
    }
    if (! fReceiverSet.empty( )) {

        float progprelevs = 0.0f;
        for (int i = 0; i < fInstrCurrLev; i++)
            progprelevs += fInstFracPerLev[i];
        progprelevs *= 100.0f;

        float progthislev = static_cast<float>(evt.percentProgress( )) *
            fInstFracPerLev[fInstrCurrLev];

        float overallprog = progprelevs + progthislev;

        uBroadcastUniformIAE(static_cast<int>(overallprog));
    }
}




void  PyramidalSimilarityMap::write(const std::string& outfile) const
{
    /* attempt to open 'outfile' for writing */
    std::ofstream  outstream(outfile.c_str( ), std::ios::out);
    if (!outstream)
        throw std::runtime_error("PyramidalSimilarityMap: can't write file: "
            "unable to open output file for writing");

    /* write the file magic number */
    outstream << "(# PYRSIM)\n\n";

    /* write the file header */
    std::string indent = "    ";
    outstream << "pyramidalSimset {\n\n";
    outstream << indent << "baseHorizPixels := " << (fTileset[0].width /
        fEntriesPerPix) << "\n";
    outstream << indent << "baseVertPixels  := " << fTileset[0].height
        << "\n";
    outstream << indent << "numLevels       := " << fTileset.size( ) << "\n";
    outstream << indent << "entriesPerPix   := " << fEntriesPerPix << "\n";
    outstream << indent << "isVarBoosted    := " << std::boolalpha <<
        fIsBoosted << "\n";
    outstream << indent << "boostSepFrac    := " << fBoostSepFrac << "\n";
    outstream << "}\n";

    for (int lev = 0; lev < fTileset.size( ); lev++) {

        outstream << "\n" << lev << " {\n\n";

        for (int j = 0; j < fTileset[lev].height; j++) {
            for (int i = 0; i < fTileset[lev].width; i += fEntriesPerPix) {

                int x = (i / fEntriesPerPix);
                indent = "    ";
                outstream << indent << "(" << x << ", " << j << ") {\n\n";
                indent = "        ";

                for (int k = 0; k < fEntriesPerPix; k++) {

                    RGBAImage::const_fragment f =
                        fMosaicImage->fragmentAt(fTileset[lev].origin.x + i +
                        k, fTileset[lev].origin.y + j);

                    int r = static_cast<int>(f.color( ).r * 255.0f);
                    int g = static_cast<int>(f.color( ).g * 255.0f);

                    outstream << indent << "(" << r << ", " << g << ")\n";
                } /* for all pixels in the current bucket */

                indent = "    ";
                outstream << indent << "}\n";

            } /* for all bucket-start pixels in the current row */
        } /* for all pixel rows on the current level */

        outstream << "}\n\n";

    } /* for all levels in the pyramid */
}




void  PyramidalSimilarityMap::uComputeTileset(int basewd, int baseht,
    int numlevs, int numents)
{
    Rectangle2Di currect = Rectangle2Di(1, 1, basewd, baseht);
    currect.width *= numents;
    fTileset.push_back(currect);

    for (int i = 1; i < numlevs; i++) {

        if ((i % 2) == 0)  /* shift even tiles horizontally */
            currect.origin.x += (currect.width + 1);
        else               /* shift odd tiles vertically */
            currect.origin.y += (currect.height + 1);

        currect.width /= 2;
        currect.height /= 2;
        fTileset.push_back(currect);
    }
}




void  PyramidalSimilarityMap::uPrepareMosaicImage(
    const std::vector<Rectangle2Di>& tileset)
{
    int mosaicwd = tileset[0].width + 2;
    
    int mosaicht;
    if (tileset.size( ) > 1)
        mosaicht = tileset[0].height + tileset[1].height + 3;
    else
        mosaicht = tileset[0].height + 2;

    fMosaicImage = new RGBAImage(mosaicwd, mosaicht);

    RGBAImage::fragment f = fMosaicImage->firstFragment( );
    while (f <= fMosaicImage->lastFragment( )) {

        static  const RGBColor  backblue = RGBColor(0.8f, 0.93f, 0.95f);
        f.setColor(backblue);
        f++;
    }
}




int  PyramidalSimilarityMap::entriesPerPixel( ) const
{
    return fEntriesPerPix;
}
//////////////////////////////////////////////////////////////////////////////
// END  PyramidalSimilarityMap                                              //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// INITIALIZATIONS FOR LARGE STATICS IN  PixelNeighborhood                  //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const float  PixelNeighborhood::kNHWeights0[ ] = {

      1.000000000000f,
};


const float  PixelNeighborhood::kNHWeights1[ ] = {

      0.058549832553f,
      0.096532352269f,
      0.058549832553f,
      0.096532352269f,
      0.159154936671f,
      0.096532352269f,
      0.058549832553f,
      0.096532352269f,
      0.058549832553f
};


const float  PixelNeighborhood::kNHWeights2[ ] = {

      0.010769639164f,
      0.022799327970f,
      0.029274916276f,
      0.022799327970f,
      0.010769639164f,
      0.022799327970f,
      0.048266176134f,
      0.061974994838f,
      0.048266176134f,
      0.022799327970f,
      0.029274916276f,
      0.061974994838f,
      0.079577468336f,
      0.061974994838f,
      0.029274916276f,
      0.022799327970f,
      0.048266176134f,
      0.061974994838f,
      0.048266176134f,
      0.022799327970f,
      0.010769639164f,
      0.022799327970f,
      0.029274916276f,
      0.022799327970f,
      0.010769639164f
};


const float  PixelNeighborhood::kNHWeights3[ ] = {

      0.002641285770f,
      0.006077534985f,
      0.010020161979f,
      0.011837422848f,
      0.010020161979f,
      0.006077534985f,
      0.002641285770f,
      0.006077534985f,
      0.013984262012f,
      0.023056149483f,
      0.027237623930f,
      0.023056149483f,
      0.013984262012f,
      0.006077534985f,
      0.010020161979f,
      0.023056149483f,
      0.038013163954f,
      0.044907249510f,
      0.038013163954f,
      0.023056149483f,
      0.010020161979f,
      0.011837422848f,
      0.027237623930f,
      0.044907249510f,
      0.053051646799f,
      0.044907249510f,
      0.027237623930f,
      0.011837422848f,
      0.010020161979f,
      0.023056149483f,
      0.038013163954f,
      0.044907249510f,
      0.038013163954f,
      0.023056149483f,
      0.010020161979f,
      0.006077534985f,
      0.013984262012f,
      0.023056149483f,
      0.027237623930f,
      0.023056149483f,
      0.013984262012f,
      0.006077534985f,
      0.002641285770f,
      0.006077534985f,
      0.010020161979f,
      0.011837422848f,
      0.010020161979f,
      0.006077534985f,
      0.002641285770f
};


const float  PixelNeighborhood::kNHWeights4[ ] = {

      0.000728756131f,
      0.001748195034f,
      0.003266058164f,
      0.004752086941f,
      0.005384819582f,
      0.004752086941f,
      0.003266058164f,
      0.001748195034f,
      0.000728756131f,
      0.001748195034f,
      0.004193701781f,
      0.007834866643f,
      0.011399663985f,
      0.012917511165f,
      0.011399663985f,
      0.007834866643f,
      0.004193701781f,
      0.001748195034f,
      0.003266058164f,
      0.007834866643f,
      0.014637458138f,
      0.021297376603f,
      0.024133088067f,
      0.021297376603f,
      0.014637458138f,
      0.007834866643f,
      0.003266058164f,
      0.004752086941f,
      0.011399663985f,
      0.021297376603f,
      0.030987497419f,
      0.035113435239f,
      0.030987497419f,
      0.021297376603f,
      0.011399663985f,
      0.004752086941f,
      0.005384819582f,
      0.012917511165f,
      0.024133088067f,
      0.035113435239f,
      0.039788734168f,
      0.035113435239f,
      0.024133088067f,
      0.012917511165f,
      0.005384819582f,
      0.004752086941f,
      0.011399663985f,
      0.021297376603f,
      0.030987497419f,
      0.035113435239f,
      0.030987497419f,
      0.021297376603f,
      0.011399663985f,
      0.004752086941f,
      0.003266058164f,
      0.007834866643f,
      0.014637458138f,
      0.021297376603f,
      0.024133088067f,
      0.021297376603f,
      0.014637458138f,
      0.007834866643f,
      0.003266058164f,
      0.001748195034f,
      0.004193701781f,
      0.007834866643f,
      0.011399663985f,
      0.012917511165f,
      0.011399663985f,
      0.007834866643f,
      0.004193701781f,
      0.001748195034f,
      0.000728756131f,
      0.001748195034f,
      0.003266058164f,
      0.004752086941f,
      0.005384819582f,
      0.004752086941f,
      0.003266058164f,
      0.001748195034f,
      0.000728756131f
};


const float  PixelNeighborhood::kNHWeights5[ ] = {

      0.000214475513f,
      0.000527524680f,
      0.001062304014f,
      0.001751443255f,
      0.002364201704f,
      0.002612846671f,
      0.002364201704f,
      0.001751443255f,
      0.001062304014f,
      0.000527524680f,
      0.000214475513f,
      0.000527524680f,
      0.001297501149f,
      0.002612846671f,
      0.004307855852f,
      0.005814996548f,
      0.006426565815f,
      0.005814996548f,
      0.004307855852f,
      0.002612846671f,
      0.001297501149f,
      0.000527524680f,
      0.001062304014f,
      0.002612846671f,
      0.005261627026f,
      0.008674956858f,
      0.011709966697f,
      0.012941514142f,
      0.011709966697f,
      0.008674956858f,
      0.005261627026f,
      0.002612846671f,
      0.001062304014f,
      0.001751443255f,
      0.004307855852f,
      0.008674956858f,
      0.014302584343f,
      0.019306469709f,
      0.021336948499f,
      0.019306469709f,
      0.014302584343f,
      0.008674956858f,
      0.004307855852f,
      0.001751443255f,
      0.002364201704f,
      0.005814996548f,
      0.011709966697f,
      0.019306469709f,
      0.026061009616f,
      0.028801869601f,
      0.026061009616f,
      0.019306469709f,
      0.011709966697f,
      0.005814996548f,
      0.002364201704f,
      0.002612846671f,
      0.006426565815f,
      0.012941514142f,
      0.021336948499f,
      0.028801869601f,
      0.031830988824f,
      0.028801869601f,
      0.021336948499f,
      0.012941514142f,
      0.006426565815f,
      0.002612846671f,
      0.002364201704f,
      0.005814996548f,
      0.011709966697f,
      0.019306469709f,
      0.026061009616f,
      0.028801869601f,
      0.026061009616f,
      0.019306469709f,
      0.011709966697f,
      0.005814996548f,
      0.002364201704f,
      0.001751443255f,
      0.004307855852f,
      0.008674956858f,
      0.014302584343f,
      0.019306469709f,
      0.021336948499f,
      0.019306469709f,
      0.014302584343f,
      0.008674956858f,
      0.004307855852f,
      0.001751443255f,
      0.001062304014f,
      0.002612846671f,
      0.005261627026f,
      0.008674956858f,
      0.011709966697f,
      0.012941514142f,
      0.011709966697f,
      0.008674956858f,
      0.005261627026f,
      0.002612846671f,
      0.001062304014f,
      0.000527524680f,
      0.001297501149f,
      0.002612846671f,
      0.004307855852f,
      0.005814996548f,
      0.006426565815f,
      0.005814996548f,
      0.004307855852f,
      0.002612846671f,
      0.001297501149f,
      0.000527524680f,
      0.000214475513f,
      0.000527524680f,
      0.001062304014f,
      0.001751443255f,
      0.002364201704f,
      0.002612846671f,
      0.002364201704f,
      0.001751443255f,
      0.001062304014f,
      0.000527524680f,
      0.000214475513f
};


const float  PixelNeighborhood::kNHWeights6[ ] = {

      0.000065750944f,
      0.000164439130f,
      0.000348117639f,
      0.000623827567f,
      0.000946282060f,
      0.001215050346f,
      0.001320642885f,
      0.001215050346f,
      0.000946282060f,
      0.000623827567f,
      0.000348117639f,
      0.000164439130f,
      0.000065750944f,
      0.000164439130f,
      0.000411252549f,
      0.000870621414f,
      0.001560155535f,
      0.002366594505f,
      0.003038767492f,
      0.003302849131f,
      0.003038767492f,
      0.002366594505f,
      0.001560155535f,
      0.000870621414f,
      0.000411252549f,
      0.000164439130f,
      0.000348117639f,
      0.000870621414f,
      0.001843105536f,
      0.003302849131f,
      0.005010080989f,
      0.006433071103f,
      0.006992131006f,
      0.006433071103f,
      0.005010080989f,
      0.003302849131f,
      0.001843105536f,
      0.000870621414f,
      0.000348117639f,
      0.000623827567f,
      0.001560155535f,
      0.003302849131f,
      0.005918711424f,
      0.008978073485f,
      0.011528074741f,
      0.012529911473f,
      0.011528074741f,
      0.008978073485f,
      0.005918711424f,
      0.003302849131f,
      0.001560155535f,
      0.000623827567f,
      0.000946282060f,
      0.002366594505f,
      0.005010080989f,
      0.008978073485f,
      0.013618811965f,
      0.017486901954f,
      0.019006581977f,
      0.017486901954f,
      0.013618811965f,
      0.008978073485f,
      0.005010080989f,
      0.002366594505f,
      0.000946282060f,
      0.001215050346f,
      0.003038767492f,
      0.006433071103f,
      0.011528074741f,
      0.017486901954f,
      0.022453624755f,
      0.024404935539f,
      0.022453624755f,
      0.017486901954f,
      0.011528074741f,
      0.006433071103f,
      0.003038767492f,
      0.001215050346f,
      0.001320642885f,
      0.003302849131f,
      0.006992131006f,
      0.012529911473f,
      0.019006581977f,
      0.024404935539f,
      0.026525823399f,
      0.024404935539f,
      0.019006581977f,
      0.012529911473f,
      0.006992131006f,
      0.003302849131f,
      0.001320642885f,
      0.001215050346f,
      0.003038767492f,
      0.006433071103f,
      0.011528074741f,
      0.017486901954f,
      0.022453624755f,
      0.024404935539f,
      0.022453624755f,
      0.017486901954f,
      0.011528074741f,
      0.006433071103f,
      0.003038767492f,
      0.001215050346f,
      0.000946282060f,
      0.002366594505f,
      0.005010080989f,
      0.008978073485f,
      0.013618811965f,
      0.017486901954f,
      0.019006581977f,
      0.017486901954f,
      0.013618811965f,
      0.008978073485f,
      0.005010080989f,
      0.002366594505f,
      0.000946282060f,
      0.000623827567f,
      0.001560155535f,
      0.003302849131f,
      0.005918711424f,
      0.008978073485f,
      0.011528074741f,
      0.012529911473f,
      0.011528074741f,
      0.008978073485f,
      0.005918711424f,
      0.003302849131f,
      0.001560155535f,
      0.000623827567f,
      0.000348117639f,
      0.000870621414f,
      0.001843105536f,
      0.003302849131f,
      0.005010080989f,
      0.006433071103f,
      0.006992131006f,
      0.006433071103f,
      0.005010080989f,
      0.003302849131f,
      0.001843105536f,
      0.000870621414f,
      0.000348117639f,
      0.000164439130f,
      0.000411252549f,
      0.000870621414f,
      0.001560155535f,
      0.002366594505f,
      0.003038767492f,
      0.003302849131f,
      0.003038767492f,
      0.002366594505f,
      0.001560155535f,
      0.000870621414f,
      0.000411252549f,
      0.000164439130f,
      0.000065750944f,
      0.000164439130f,
      0.000348117639f,
      0.000623827567f,
      0.000946282060f,
      0.001215050346f,
      0.001320642885f,
      0.001215050346f,
      0.000946282060f,
      0.000623827567f,
      0.000348117639f,
      0.000164439130f,
      0.000065750944f
};


const float  PixelNeighborhood::kNHWeights7[ ] = {

      0.000020732932f,
      0.000052472780f,
      0.000115123927f,
      0.000218955014f,
      0.000360995764f,
      0.000515949505f,
      0.000639249454f,
      0.000686580373f,
      0.000639249454f,
      0.000515949505f,
      0.000360995764f,
      0.000218955014f,
      0.000115123927f,
      0.000052472780f,
      0.000020732932f,
      0.000052472780f,
      0.000132802918f,
      0.000291366043f,
      0.000554151251f,
      0.000913641008f,
      0.001305812038f,
      0.001617870876f,
      0.001737660263f,
      0.001617870876f,
      0.001305812038f,
      0.000913641008f,
      0.000554151251f,
      0.000291366043f,
      0.000132802918f,
      0.000052472780f,
      0.000115123927f,
      0.000291366043f,
      0.000639249454f,
      0.001215793076f,
      0.002004503738f,
      0.002864917275f,
      0.003549565561f,
      0.003812380368f,
      0.003549565561f,
      0.002864917275f,
      0.002004503738f,
      0.001215793076f,
      0.000639249454f,
      0.000291366043f,
      0.000115123927f,
      0.000218955014f,
      0.000554151251f,
      0.001215793076f,
      0.002312325174f,
      0.003812380368f,
      0.005448806100f,
      0.006750944071f,
      0.007250793278f,
      0.006750944071f,
      0.005448806100f,
      0.003812380368f,
      0.002312325174f,
      0.001215793076f,
      0.000554151251f,
      0.000218955014f,
      0.000360995764f,
      0.000913641008f,
      0.002004503738f,
      0.003812380368f,
      0.006285552401f,
      0.008983563632f,
      0.011130424216f,
      0.011954536662f,
      0.011130424216f,
      0.008983563632f,
      0.006285552401f,
      0.003812380368f,
      0.002004503738f,
      0.000913641008f,
      0.000360995764f,
      0.000515949505f,
      0.001305812038f,
      0.002864917275f,
      0.005448806100f,
      0.008983563632f,
      0.012839668430f,
      0.015908047557f,
      0.017085904256f,
      0.015908047557f,
      0.012839668430f,
      0.008983563632f,
      0.005448806100f,
      0.002864917275f,
      0.001305812038f,
      0.000515949505f,
      0.000639249454f,
      0.001617870876f,
      0.003549565561f,
      0.006750944071f,
      0.011130424216f,
      0.015908047557f,
      0.019709700719f,
      0.021169032902f,
      0.019709700719f,
      0.015908047557f,
      0.011130424216f,
      0.006750944071f,
      0.003549565561f,
      0.001617870876f,
      0.000639249454f,
      0.000686580373f,
      0.001737660263f,
      0.003812380368f,
      0.007250793278f,
      0.011954536662f,
      0.017085904256f,
      0.021169032902f,
      0.022736418992f,
      0.021169032902f,
      0.017085904256f,
      0.011954536662f,
      0.007250793278f,
      0.003812380368f,
      0.001737660263f,
      0.000686580373f,
      0.000639249454f,
      0.001617870876f,
      0.003549565561f,
      0.006750944071f,
      0.011130424216f,
      0.015908047557f,
      0.019709700719f,
      0.021169032902f,
      0.019709700719f,
      0.015908047557f,
      0.011130424216f,
      0.006750944071f,
      0.003549565561f,
      0.001617870876f,
      0.000639249454f,
      0.000515949505f,
      0.001305812038f,
      0.002864917275f,
      0.005448806100f,
      0.008983563632f,
      0.012839668430f,
      0.015908047557f,
      0.017085904256f,
      0.015908047557f,
      0.012839668430f,
      0.008983563632f,
      0.005448806100f,
      0.002864917275f,
      0.001305812038f,
      0.000515949505f,
      0.000360995764f,
      0.000913641008f,
      0.002004503738f,
      0.003812380368f,
      0.006285552401f,
      0.008983563632f,
      0.011130424216f,
      0.011954536662f,
      0.011130424216f,
      0.008983563632f,
      0.006285552401f,
      0.003812380368f,
      0.002004503738f,
      0.000913641008f,
      0.000360995764f,
      0.000218955014f,
      0.000554151251f,
      0.001215793076f,
      0.002312325174f,
      0.003812380368f,
      0.005448806100f,
      0.006750944071f,
      0.007250793278f,
      0.006750944071f,
      0.005448806100f,
      0.003812380368f,
      0.002312325174f,
      0.001215793076f,
      0.000554151251f,
      0.000218955014f,
      0.000115123927f,
      0.000291366043f,
      0.000639249454f,
      0.001215793076f,
      0.002004503738f,
      0.002864917275f,
      0.003549565561f,
      0.003812380368f,
      0.003549565561f,
      0.002864917275f,
      0.002004503738f,
      0.001215793076f,
      0.000639249454f,
      0.000291366043f,
      0.000115123927f,
      0.000052472780f,
      0.000132802918f,
      0.000291366043f,
      0.000554151251f,
      0.000913641008f,
      0.001305812038f,
      0.001617870876f,
      0.001737660263f,
      0.001617870876f,
      0.001305812038f,
      0.000913641008f,
      0.000554151251f,
      0.000291366043f,
      0.000132802918f,
      0.000052472780f,
      0.000020732932f,
      0.000052472780f,
      0.000115123927f,
      0.000218955014f,
      0.000360995764f,
      0.000515949505f,
      0.000639249454f,
      0.000686580373f,
      0.000639249454f,
      0.000515949505f,
      0.000360995764f,
      0.000218955014f,
      0.000115123927f,
      0.000052472780f,
      0.000020732932f
};


const float  PixelNeighborhood::kNHWeights8[ ] = {

      0.000006673817f,
      0.000017042188f,
      0.000038405164f,
      0.000076377786f,
      0.000134047194f,
      0.000207616351f,
      0.000283777918f,
      0.000342301500f,
      0.000364378066f,
      0.000342301500f,
      0.000283777918f,
      0.000207616351f,
      0.000134047194f,
      0.000076377786f,
      0.000038405164f,
      0.000017042188f,
      0.000006673817f,
      0.000017042188f,
      0.000043518754f,
      0.000098071017f,
      0.000195037501f,
      0.000342301500f,
      0.000530166901f,
      0.000724652316f,
      0.000874097517f,
      0.000930471928f,
      0.000874097517f,
      0.000724652316f,
      0.000530166901f,
      0.000342301500f,
      0.000195037501f,
      0.000098071017f,
      0.000043518754f,
      0.000017042188f,
      0.000038405164f,
      0.000098071017f,
      0.000221006456f,
      0.000439523807f,
      0.000771388295f,
      0.001194749610f,
      0.001633029082f,
      0.001969809178f,
      0.002096850891f,
      0.001969809178f,
      0.001633029082f,
      0.001194749610f,
      0.000771388295f,
      0.000439523807f,
      0.000221006456f,
      0.000098071017f,
      0.000038405164f,
      0.000076377786f,
      0.000195037501f,
      0.000439523807f,
      0.000874097517f,
      0.001534088864f,
      0.002376043471f,
      0.003247666173f,
      0.003917433321f,
      0.004170085769f,
      0.003917433321f,
      0.003247666173f,
      0.002376043471f,
      0.001534088864f,
      0.000874097517f,
      0.000439523807f,
      0.000195037501f,
      0.000076377786f,
      0.000134047194f,
      0.000342301500f,
      0.000771388295f,
      0.001534088864f,
      0.002692409791f,
      0.004170085769f,
      0.005699831992f,
      0.006875309162f,
      0.007318729069f,
      0.006875309162f,
      0.005699831992f,
      0.004170085769f,
      0.002692409791f,
      0.001534088864f,
      0.000771388295f,
      0.000342301500f,
      0.000134047194f,
      0.000207616351f,
      0.000530166901f,
      0.001194749610f,
      0.002376043471f,
      0.004170085769f,
      0.006458755583f,
      0.008828071877f,
      0.010648688301f,
      0.011335469782f,
      0.010648688301f,
      0.008828071877f,
      0.006458755583f,
      0.004170085769f,
      0.002376043471f,
      0.001194749610f,
      0.000530166901f,
      0.000207616351f,
      0.000283777918f,
      0.000724652316f,
      0.001633029082f,
      0.003247666173f,
      0.005699831992f,
      0.008828071877f,
      0.012066544034f,
      0.014555029571f,
      0.015493748710f,
      0.014555029571f,
      0.012066544034f,
      0.008828071877f,
      0.005699831992f,
      0.003247666173f,
      0.001633029082f,
      0.000724652316f,
      0.000283777918f,
      0.000342301500f,
      0.000874097517f,
      0.001969809178f,
      0.003917433321f,
      0.006875309162f,
      0.010648688301f,
      0.014555029571f,
      0.017556717619f,
      0.018689028919f,
      0.017556717619f,
      0.014555029571f,
      0.010648688301f,
      0.006875309162f,
      0.003917433321f,
      0.001969809178f,
      0.000874097517f,
      0.000342301500f,
      0.000364378066f,
      0.000930471928f,
      0.002096850891f,
      0.004170085769f,
      0.007318729069f,
      0.011335469782f,
      0.015493748710f,
      0.018689028919f,
      0.019894367084f,
      0.018689028919f,
      0.015493748710f,
      0.011335469782f,
      0.007318729069f,
      0.004170085769f,
      0.002096850891f,
      0.000930471928f,
      0.000364378066f,
      0.000342301500f,
      0.000874097517f,
      0.001969809178f,
      0.003917433321f,
      0.006875309162f,
      0.010648688301f,
      0.014555029571f,
      0.017556717619f,
      0.018689028919f,
      0.017556717619f,
      0.014555029571f,
      0.010648688301f,
      0.006875309162f,
      0.003917433321f,
      0.001969809178f,
      0.000874097517f,
      0.000342301500f,
      0.000283777918f,
      0.000724652316f,
      0.001633029082f,
      0.003247666173f,
      0.005699831992f,
      0.008828071877f,
      0.012066544034f,
      0.014555029571f,
      0.015493748710f,
      0.014555029571f,
      0.012066544034f,
      0.008828071877f,
      0.005699831992f,
      0.003247666173f,
      0.001633029082f,
      0.000724652316f,
      0.000283777918f,
      0.000207616351f,
      0.000530166901f,
      0.001194749610f,
      0.002376043471f,
      0.004170085769f,
      0.006458755583f,
      0.008828071877f,
      0.010648688301f,
      0.011335469782f,
      0.010648688301f,
      0.008828071877f,
      0.006458755583f,
      0.004170085769f,
      0.002376043471f,
      0.001194749610f,
      0.000530166901f,
      0.000207616351f,
      0.000134047194f,
      0.000342301500f,
      0.000771388295f,
      0.001534088864f,
      0.002692409791f,
      0.004170085769f,
      0.005699831992f,
      0.006875309162f,
      0.007318729069f,
      0.006875309162f,
      0.005699831992f,
      0.004170085769f,
      0.002692409791f,
      0.001534088864f,
      0.000771388295f,
      0.000342301500f,
      0.000134047194f,
      0.000076377786f,
      0.000195037501f,
      0.000439523807f,
      0.000874097517f,
      0.001534088864f,
      0.002376043471f,
      0.003247666173f,
      0.003917433321f,
      0.004170085769f,
      0.003917433321f,
      0.003247666173f,
      0.002376043471f,
      0.001534088864f,
      0.000874097517f,
      0.000439523807f,
      0.000195037501f,
      0.000076377786f,
      0.000038405164f,
      0.000098071017f,
      0.000221006456f,
      0.000439523807f,
      0.000771388295f,
      0.001194749610f,
      0.001633029082f,
      0.001969809178f,
      0.002096850891f,
      0.001969809178f,
      0.001633029082f,
      0.001194749610f,
      0.000771388295f,
      0.000439523807f,
      0.000221006456f,
      0.000098071017f,
      0.000038405164f,
      0.000017042188f,
      0.000043518754f,
      0.000098071017f,
      0.000195037501f,
      0.000342301500f,
      0.000530166901f,
      0.000724652316f,
      0.000874097517f,
      0.000930471928f,
      0.000874097517f,
      0.000724652316f,
      0.000530166901f,
      0.000342301500f,
      0.000195037501f,
      0.000098071017f,
      0.000043518754f,
      0.000017042188f,
      0.000006673817f,
      0.000017042188f,
      0.000038405164f,
      0.000076377786f,
      0.000134047194f,
      0.000207616351f,
      0.000283777918f,
      0.000342301500f,
      0.000364378066f,
      0.000342301500f,
      0.000283777918f,
      0.000207616351f,
      0.000134047194f,
      0.000076377786f,
      0.000038405164f,
      0.000017042188f,
      0.000006673817f
};
//////////////////////////////////////////////////////////////////////////////
// END  INITIALIZATIONS                                                     //
//////////////////////////////////////////////////////////////////////////////
