//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// StructuredIO.h                                                           //
//                                                                          //
// Implements methods in class StructuredReader only.                       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver 2.0.0 of Fri 09-Jan-2009 @ 5:30am EST                                //
//                                                                          //
//     added support to install (define) macros that are automatically      //
//     expanded in quoted strings.                                          //
//                                                                          //
// ver 1.2.0 of Sun 26-Oct-2008 @ 7:21pm EDT                                //
//                                                                          //
//     added readQuotedString( ) to class StructuredReader                  //
//                                                                          //
// ver 1.1.0 of Thu 15-May-2008 @ 1:34pm EDT                                //
//                                                                          //
//     added readBoolean( ) and readFloat( ) to class Structured reader;    //
//     both methods were tested as part of MapPickerTester and are          //
//     considered stable                                                    //
//                                                                          //
// prior change history elided; check out an older CVS rev to get it        //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008-2009, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "StructuredIO.h"
#include <cctype>
#include <cstdlib>

StructuredReader::StructuredReader(const std::string& target)
    : fTarget(target), fCursor(0)
{
}




StructuredReader::~StructuredReader( )
{
}




std::string  StructuredReader::readIdentifier( )
{
    uSkipWhitespace( );

    std::string  accum;
    char         curchar;

    /* read the initial letter */
    curchar = fTarget[fCursor];
    if (!isalpha(curchar))
        throw std::runtime_error("StructuredReader: identifiers must begin "
            "with a letter");
    
    accum += fTarget[fCursor++];

    /* read any remaining letters, numbers and underscores */
    while(true) {

        curchar = fTarget[fCursor];

        if (! (std::isalnum(curchar) || (curchar == '_')))
            break;

        accum += fTarget[fCursor++];
    }

    return accum;
}




bool  StructuredReader::readBoolean( )
{
    uSkipWhitespace( );

    std::string  accum;
    char         curchar;

    while(true) {

        curchar = fTarget[fCursor];

        if (! std::isalpha(curchar))
            break;

        accum += fTarget[fCursor++];
    }

    if (accum == "true")
        return true;
    else if (accum == "false")
        return false;
    else
        throw std::runtime_error("StructuredReader: expected a boolean");
}




void  StructuredReader::readGroupOpen( )
{
    uSkipWhitespace( );

    char curchar = fTarget[fCursor];
    if (curchar != '{')
        throw std::runtime_error("StructuredReader: expected a '{'");
    else
        fCursor++;
}




void  StructuredReader::readGroupClose( )
{
    uSkipWhitespace( );

    char curchar = fTarget[fCursor];
    if (curchar != '}')
        throw std::runtime_error("StructuredReader: expected a '}'");
    else
        fCursor++;
}




float  StructuredReader::readFloat( )
{
    uSkipWhitespace( );

    std::string  accum;
    char         curchar;

    bool gotDecimal = false;
    bool gotNum = false;
    while(true) {

        curchar = fTarget[fCursor];

        if (curchar == '.') {

            if (!gotDecimal)
                gotDecimal = true;
            else
                throw std::runtime_error("StructuredReader: floating point "
                    "values can have only one decimal point");
        }
        else if (std::isdigit(curchar)) {

            gotNum = true;
        }
        else {

            break;
        }

        accum += fTarget[fCursor++];
    }

    if (! (gotDecimal && gotNum))
        throw std::runtime_error("StructuredReader: floating point values "
            "must contain exactly one decimal point and at least one digit");

    return static_cast<float>(std::atof(accum.c_str( )));
}




int  StructuredReader::readInteger( )
{
    uSkipWhitespace( );

    std::string  accum;

    /* if we don't have at least one number, we've got a problem */
    char curchar = fTarget[fCursor];
    if (!std::isdigit(curchar))
        throw std::runtime_error("StructuredReader: expected a digit");

    /* read numbers until we run out of them */
    while(std::isdigit(fTarget[fCursor]))
        accum += fTarget[fCursor++];

    return std::atoi(accum.c_str( ));
}




Vector2Di  StructuredReader::readVector2Di( )
{
    uSkipWhitespace( );

    Vector2Di    result;
    std::string  accum;

    /* read the opening parenthesis */
    if (fTarget[fCursor] != '(')
        throw std::runtime_error("StructuredReader: expected a '('");
    else
        fCursor++;

    /* read the x-coordinate */
    result.x = this->readInteger( );

    /* read the ',' ignoring whitespace */
    uSkipWhitespace( );
    if (fTarget[fCursor] != ',')
        throw std::runtime_error("StructuredReader: expected a ','");
    else
        fCursor++;

    /* read the y-coordinate */
    result.y = this->readInteger( );

    /* read the closing parenthesis */
    if (fTarget[fCursor] != ')')
        throw std::runtime_error("StructuredReader: expected a ')'");
    else
        fCursor++;

    return result;
}




void  StructuredReader::readAssignmentOperator( )
{
    uSkipWhitespace( );

    /* read the colon */
    if (fTarget[fCursor] != ':')
        throw std::runtime_error("StructuredReader: expected a ':='");
    else
        fCursor++;

    /* read the equals sign */
    if (fTarget[fCursor] != '=')
        throw std::runtime_error("StructuredReader: expected a ':='");
    else
        fCursor++;
}




int  StructuredReader::cursorPosition( ) const
{
    return fCursor;
}




int  StructuredReader::targetLength( ) const
{
    return static_cast<int>(fTarget.length( ));
}




void  StructuredReader::uSkipWhitespace( )
{
    while (std::isspace(fTarget[fCursor]))
        fCursor++;
}




std::string  StructuredReader::readFiletypeID( )
{
    /* read the opening parenthesis & hash pair */
    if (fTarget[fCursor] != '(')
        throw std::runtime_error("StructuredReader: expected a '('");
    fCursor++;
    if (fTarget[fCursor] != '#')
        throw std::runtime_error("StructuredReader: expected a '#'");
    fCursor++;

    /* read the type identifier */
    std::string result = readIdentifier( );

    /* read the close parenthesis */
    if (fTarget[fCursor] != ')')
        throw std::runtime_error("StructuredReader: expected a ')'");
    fCursor++;

    return result;
}




std::string   StructuredReader::readQuotedString( )
{
    uSkipWhitespace( );

    std::string  accum;
    char         curchar;

    /* read the open double quotation mark */
    curchar = fTarget[fCursor];
    if (curchar != '"')
        throw std::runtime_error("StructuredReader: expected a '\"'");

    fCursor++;
    
    /* read until we reach the closing double quotation mark */
    while(true) {

        curchar = fTarget[fCursor];

        /* if the current character is a '$' then it signals the
           opening of a macro reference that we need to expand */
        if (curchar == '$') {

            accum += uReadMacro( );
        }
        else if (curchar == '"') {

            fCursor++;
            break;
        }
        else {

            accum += fTarget[fCursor++];
        }
    } /* infinite loop */

    return accum;
}



void  StructuredReader::installMacro(const std::string& key,
    const std::string& value)
{
    fMacros[key] = value;
}



void  StructuredReader::removeMacro(const std::string& key)
{
    fMacros.erase(key);
}



bool  StructuredReader::isMacroInstalled(const std::string& key) const
{
    if (fMacros.find(key) != fMacros.end( ))
        return true;
    else
        return false;
}



std::string  StructuredReader::uResolveMacro(const std::string& key) const
{
    if (! isMacroInstalled(key))
        throw std::logic_error("StructuredReader: can't resolve macro: "
            "macro isn't installed");

    return fMacros.find(key)->second;
}



std::string  StructuredReader::uReadMacro( )
{
    std::string  accum;

    /* read the dollar sign */
    if (fTarget[fCursor] != '$')
        throw std::runtime_error("StructuredReader: macro references must "
            "begin with a '$'");
    else
        fCursor++;

    /* read the open paren */
    if (fTarget[fCursor] != '(')
        throw std::runtime_error("StructuredReader: macro names must be "
            "enclosed in parentheses");
    else
        fCursor++;

    /* read the macro name */
    std::string macro = readIdentifier( );

    /* read the close parenthesis */
    if (fTarget[fCursor] != ')')
        throw std::runtime_error("StructuredReader: macro references must "
            "end with a ')'");
    else
        fCursor++;

    return uResolveMacro(macro);
}
