//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ModalDialog.cpp                                                          //
//                                                                          //
// Implements methods in class ModalDialogView; defines class               //
// NullModalDialogController and implments its methods                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// version 1.1.3 of Fri 3 May 2008 @ 7:40pm EDT                             //
//                                                                          //
//     changed all references of class template Vertex2D to Vector2D; see   //
//     the AffineGeometry module's header for more info                     //
//                                                                          //
// version 1.1.2 of Wednedsday 26 March 2008 @ 5:39pm EDT                   //
//                                                                          //
//     Added methods stubs for UIComponent-style enable/disable management  //
//     support to ModalDialog views and controllers. Note that these are    //
//     only stubs; they don't actually do anything, since it's unclear why  //
//     one would want to disable a *modal* dialog. Nevertheless, the        //
//     stubs need to be there so that ModalDialog views and controllers are //
//     interface-compliant with their UIComponent superclasses.             //
//                                                                          //
// older change history elided; check out an older CVS rev to get it        //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "Application.h"
#include "ModalDialog.h"
#include <stdexcept>

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  NullModalDialogController                                         //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class  NullModalDialogController
    : public virtual ModalDialogController {

private:

    bool  fIsInstalled;

public:

    NullModalDialogController( );
    virtual  ~NullModalDialogController( );

    void  installed(ModalDialogView& sender);
    void  deinstalled(ModalDialogView& sender);
    void  layout(ModalDialogView& sender);
    void  image(ModalDialogView& sender);
    void  commandPropagated(ModalDialogView& sender,
        HWND commandSource, unsigned short commandID);
    void  interactionStarted(ModalDialogView& sender);
    void  interactionFinished(ModalDialogView& sender);
    void  dialogClosed(ModalDialogView& sender);
    bool  isInstalled( ) const;
    void  viewEnabled(UIComponentView& sender);
    void  viewDisabled(UIComponentView& sender);
};




NullModalDialogController::NullModalDialogController( )
    : fIsInstalled(false)
{
}




NullModalDialogController::~NullModalDialogController( )
{
}




void  NullModalDialogController::installed(ModalDialogView& sender)
{
    fIsInstalled = true;
}




void  NullModalDialogController::deinstalled(ModalDialogView& sender)
{
    fIsInstalled = false;
}




void  NullModalDialogController::layout(ModalDialogView& sender)
{
}




void  NullModalDialogController::image(ModalDialogView& sender)
{
}




void  NullModalDialogController::commandPropagated(ModalDialogView& sender,
    HWND commandSource, unsigned short commandID)
{
}




void  NullModalDialogController::interactionStarted(ModalDialogView& sender)
{
}




void  NullModalDialogController::interactionFinished(ModalDialogView& sender)
{
}




void  NullModalDialogController::dialogClosed(ModalDialogView& sender)
{
}




bool  NullModalDialogController::isInstalled( ) const
{
    return fIsInstalled;
}




void  NullModalDialogController::viewEnabled(UIComponentView& sender)
{
}




void  NullModalDialogController::viewDisabled(UIComponentView& sender)
{
}
//////////////////////////////////////////////////////////////////////////////
// END  NullModalDialogController                                           //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ModalDialogView                                                   //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
ModalDialogView*  ModalDialogView::sDialogAwaitingInit = 0;
std::map<HWND, ModalDialogView*>  ModalDialogView::sObjectMap;
const Vector2Di  ModalDialogView::kDefaultPosition = Vector2Di(100, 100);
const std::string  ModalDialogView::kDefaultTitle = "ModalDialogView";
const bool  ModalDialogView::kDefaultAutoPositionState = true;


INT_PTR  CALLBACK  ModalDialogView::sDialogProc(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    INT_PTR     result = FALSE;
    WINDOWPOS*  posinfo = (WINDOWPOS*)(lparam);

    switch (msgcode) {

        case  WM_INITDIALOG:
            /* if no dialog is awaiting initialization, and we get a
               WM_INITDIALOG message, then a key property of ModalDialogs
               has somewhere been violated; likewise, if we get a
               WM_INITDIALOG message for a dialog that has already been
               init'd (and we know it's already been init'd if its handle
               is present in the object store), then something's gone
               terribly wrong. */
            if (!sDialogAwaitingInit)
                throw std::logic_error("ModalDialogView: got an init message "
                    "but no dialog is awaiting initialization");
            if (sObjectMap.find(target) != sObjectMap.end( ))
                throw std::logic_error("ModalDialogView: tried to create "
                    "new primitive-to-object mapping, but primitive key is "
                    "already present.");

            if (sDialogAwaitingInit->fIsInteractionCacheOnly) {

                sDialogAwaitingInit->fTitleCache =
                    Win32Tools::getWindowTextString(target);
                sDialogAwaitingInit->fWidthCache =
                    Win32Tools::getWindowWidth(target);
                sDialogAwaitingInit->fHeightCache =
                    Win32Tools::getWindowHeight(target);
                sDialogAwaitingInit->fDialogPrimitive = target;
                sDialogAwaitingInit->uAutoPosition( );

                sDialogAwaitingInit->fIsInteractionCacheOnly = false;
                sDialogAwaitingInit->fDialogPrimitive = 0;
                sDialogAwaitingInit = 0;

                EndDialog(target, 0);
                return 0;
            }

            sDialogAwaitingInit->fDialogPrimitive = target;
            sObjectMap[target] = sDialogAwaitingInit;
            sDialogAwaitingInit = 0;

            result = sObjectMap[target]->uProcessDialogMessage(target,
                msgcode, wparam, lparam);
        break;

        case  WM_CLOSE:
            if (sObjectMap.find(target) == sObjectMap.end( ))
                throw std::logic_error("ModalDialogView: got a close message "
                    "but primitive key isn't present in primitive-to-object "
                    "mapping");
            result = sObjectMap[target]->uProcessDialogMessage(target,
                msgcode, wparam, lparam);
        break;

        case  WM_DESTROY:
            if (sObjectMap.find(target) != sObjectMap.end( ))
                result = sObjectMap[target]->uProcessDialogMessage(target,
                    msgcode, wparam, lparam);
            else
                result = 0;
        break;

        case  WM_WINDOWPOSCHANGED:
            if (sObjectMap.find(target) != sObjectMap.end( )) {

                result = sObjectMap[target]->uProcessDialogMessage(target,
                    msgcode, wparam, lparam);
            }
            else {

                posinfo = (WINDOWPOS*)(lparam);

                sDialogAwaitingInit->fRealPos = Vector2Di(posinfo->x,
                    posinfo->y);

                return 0;
            }
        break;

        case  WM_COMMAND:
            if (sObjectMap.find(target) == sObjectMap.end( ))
                throw std::logic_error("ModalDialogView: got a command "
                    "propagated message, but primitive key isn't present in "
                    "primitive-to-object mapping");
            result = sObjectMap[target]->uProcessDialogMessage(target,
                msgcode, wparam, lparam);
        break;

        case  WM_SIZE:
            if (sObjectMap.find(target) == sObjectMap.end( ))
                throw std::logic_error("ModalDialogView: got a primitive "
                    "resized message, but primitive key isn't present in "
                    "primitive-to-object mapping");
            result = sObjectMap[target]->uProcessDialogMessage(target,
                msgcode, wparam, lparam);
        break;

        case  WM_GETMINMAXINFO:
            if (sObjectMap.find(target) != sObjectMap.end( )) {

                result = sObjectMap[target]->uProcessDialogMessage(target,
                    msgcode, wparam, lparam);
            }
        break;

        default:
            /* returning false means we didn't process the message here,
               so the default behavior provided by the Windows API should
               be invoked */
            result = FALSE;
        break;
    }

    /* if this message concerns a previously init'd dialog, then we
       can look up the dialog handle in the object store to get its
       associated object; if the associated object has its interaction
       stopping flag set (fIsInteractionStopping == true, then we'll call
       EndDialog( ) to remove the target dialog from the screen (i.e., to
       stop the user's interaction with the dialog. */
    if (sObjectMap.find(target) != sObjectMap.end( )) {   // has been init'd?
        if (sObjectMap[target]->fIsInteractionStopping) { // is stop flag set?

            EndDialog(target, 0);
            sObjectMap[target]->fIsInteractionStopping = false;
        }
    }

    return  result;
}




ModalDialogView::ModalDialogView(HWND parent, int resid)
    : fDialogPrimitive(0), fIsInteractionStopping(false),
      fIsAutoPosEnabled(kDefaultAutoPositionState), fTitleCache(kDefaultTitle),
      fWidthCache(0), fHeightCache(0),
      fPosCache(kDefaultPosition), fRealPos(0, 0),
      fController(ModalDialogController::nullController( )),
      fTemplateResID(resid), fParentWindow(parent),
      fIsInteractionCacheOnly(false),
      fMinHorizTrackSize(kDefaultMinHorizTrackSize),
      fMinVerticalTrackSize(kDefaultMinVerticalTrackSize)
{
    uCachePeerState( );
}




ModalDialogView::~ModalDialogView( )
{
    if (isInteractionRunning( ))
        Win32Tools::destroyPrimitive(fDialogPrimitive);

    fController->deinstalled(*this);

    if (typeid(*fController) == typeid(NullModalDialogController))
        delete fController;
}




INT_PTR  ModalDialogView::uProcessDialogMessage(HWND target, UINT msgcode,
        WPARAM wparam, LPARAM lparam)
{
    if (msgcode == WM_CLOSE) {

        fController->dialogClosed(*this);
        return 0;
    }
    else if (msgcode == WM_DESTROY) {

        fDialogPrimitive = 0;
        sObjectMap.erase(target);
        fController->interactionFinished(*this);
        return 0;
    }
    else if (msgcode == WM_WINDOWPOSCHANGED) {

        WINDOWPOS*  posinfo = (WINDOWPOS*)(lparam);
        fRealPos.x = posinfo->x;
        fRealPos.y = posinfo->y;
        return 0;
    }
    else if (msgcode == WM_INITDIALOG) {

        uSyncState( );
        fController->interactionStarted(*this);
        uSyncState( );
        fController->layout(*this);
        return true;
    }
    else if (msgcode == WM_COMMAND) {

        fController->commandPropagated(*this, (HWND)(lparam),
            (unsigned short)(HIWORD(wparam)));
        return 0;
    }
    else if (msgcode == WM_SIZE) {

        fController->layout(*this);
        return 0;
    }
    else if (msgcode == WM_GETMINMAXINFO) {

        MINMAXINFO*  constraints = 0;
        constraints = (MINMAXINFO*) lparam;
        constraints->ptMinTrackSize.x = fMinHorizTrackSize;
        constraints->ptMinTrackSize.y = fMinVerticalTrackSize;
        return 0;
    }

    return false;
}




std::string  ModalDialogView::title( ) const
{
    return fTitleCache;
}




void  ModalDialogView::setTitle(const std::string& s)
{
    fTitleCache = s;

    if (isInteractionRunning( ))
        Win32Tools::setWindowTextString(fDialogPrimitive, fTitleCache);
}




unsigned short  ModalDialogView::height( ) const
{
    return fHeightCache;
}




void  ModalDialogView::setHeight(unsigned short h)
{
    fHeightCache = h;

    if (isInteractionRunning( ))
        Win32Tools::setWindowHeight(fDialogPrimitive, fHeightCache);
}




unsigned short  ModalDialogView::width( ) const
{
    return fWidthCache;
}




void  ModalDialogView::setWidth(unsigned short w)
{
    fWidthCache = w;

    if (isInteractionRunning( ))
        Win32Tools::setWindowWidth(fDialogPrimitive, fWidthCache);
}




Vector2Di  ModalDialogView::position( ) const
{
    if (isAutoPositionEnabled( ))
        return fRealPos;
    else
        return fPosCache;
}




void  ModalDialogView::setPosition(const Vector2Di& p)
{
    fPosCache = p;

    if (isInteractionRunning( ) && (! isAutoPositionEnabled( )))
        Win32Tools::setWindowOrigin(fDialogPrimitive, fPosCache.x,
            fPosCache.y);
}




HWND  ModalDialogView::peerHandle( )
{
    return fDialogPrimitive;
}




void  ModalDialogView::invalidate( )
{
    if (isInteractionRunning( )) {

        InvalidateRect(fDialogPrimitive, NULL, TRUE);
        UpdateWindow(fDialogPrimitive);
    }
}




ModalDialogController&  ModalDialogView::controller( )
{
    return *fController;
}




ModalDialogController&  ModalDialogView::installController(
    ModalDialogController& c)
{
    ModalDialogController*  oldController = fController;

    oldController->deinstalled(*this);

    fController = &c;

    fController->installed(*this);

    return *oldController;
}




void  ModalDialogView::startInteraction( )
{
    if (isInteractionRunning( ))
        throw std::logic_error("ModalDialogView: startInteraction( ) "
            "requested, but interaction is already running");

    fIsInteractionStopping = false;
    sDialogAwaitingInit = this;

    int dbresult =  DialogBox(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(fTemplateResID), fParentWindow,
        ModalDialogView::sDialogProc);

    if (dbresult == -1)
        throw std::runtime_error("ModalDialogView: couldn't start "
            "interaction because dialog resource template doesn't "
            "exist");
}




void  ModalDialogView::finishInteraction( )
{
    if (!isInteractionRunning( ))
        throw std::logic_error("ModalDialogView: finishInteraction( ) "
            "requested, but no interaction is running");

    fIsInteractionStopping = true;

    /* before we "pop down" this dialog box, we need to invalidate the
       region of our parent window that we overlaid -- this is an issue
       with mixed-mode OpenGL/GDI windows */
    RECT  clirect;
    GetClientRect(fDialogPrimitive, &clirect);
    MapWindowPoints(fDialogPrimitive, fParentWindow, (LPPOINT) &clirect, 2);
    clirect.top = clirect.top - 24;
    clirect.left = clirect.left - 24;
    clirect.bottom = clirect.bottom + 24;
    clirect.right = clirect.right + 24;
    ShowWindow(fDialogPrimitive, SW_HIDE);
    InvalidateRect(fParentWindow, &clirect, TRUE);
    UpdateWindow(fParentWindow);
}




bool  ModalDialogView::isInteractionRunning( ) const
{
    if (fDialogPrimitive)
        return true;
    else
        return false;
}




void  ModalDialogView::enableAutoPosition( )
{
    fIsAutoPosEnabled = true;
}




void  ModalDialogView::disableAutoPosition( )
{
    fIsAutoPosEnabled = false;
}




bool  ModalDialogView::isAutoPositionEnabled( ) const
{
    return fIsAutoPosEnabled;
}




void  ModalDialogView::uSyncState( )
{
    Win32Tools::setWindowTextString(fDialogPrimitive, fTitleCache);
    if (fWidthCache != 0)
        Win32Tools::setWindowHeight(fDialogPrimitive, fHeightCache);
    if (fHeightCache != 0)
        Win32Tools::setWindowWidth(fDialogPrimitive, fWidthCache);

    if (isAutoPositionEnabled( ))
        uAutoPosition( );
    else
        Win32Tools::setWindowOrigin(fDialogPrimitive, fPosCache.x,
            fPosCache.y);
}




void  ModalDialogView::uAutoPosition( )
{
    int  this_w = this->width( );
    int  this_h = this->height( );
    int  par_w = Win32Tools::getClientAreaWidth(fParentWindow);
    int  par_h = Win32Tools::getClientAreaHeight(fParentWindow);

    POINT  parentogn = { 0, 0 };
    ClientToScreen(fParentWindow, & parentogn);

    Win32Tools::setWindowOrigin(this->fDialogPrimitive, parentogn.x +
        ((par_w - this_w) / 2), parentogn.y + ((par_h - this_h) / 2));
}




void  ModalDialogView::show( )
{
    this->startInteraction( );
}




void  ModalDialogView::hide( )
{
    this->finishInteraction( );
}




void  ModalDialogView::enable( )
{
}




void  ModalDialogView::disable( )
{
}




bool  ModalDialogView::isEnabled( ) const
{
    return true;
}




bool  ModalDialogView::isVisible( ) const
{
    return this->isInteractionRunning( );
}




void  ModalDialogView::uCachePeerState( )
{
    fIsInteractionCacheOnly = true;
    this->startInteraction( );
}



HWND  ModalDialogView::itemPeer(int resid)
{
    if (! isInteractionRunning( ))
        throw std::logic_error("ModalDialogView: requested item peer, but "
            "no interaction is running");

    HWND  result = GetDlgItem(fDialogPrimitive, resid);

    if (result == NULL)
        throw std::runtime_error("ModalDialogView: requested peer doesn't "
            "exist.");

    return result;
}




unsigned short  ModalDialogView::clientWidth( ) const
{
    if (! isInteractionRunning( ))
        throw std::logic_error("ModalDialogView: requested surface width, "
            "but no interaction is running");

    RECT  cr;
    GetClientRect(fDialogPrimitive, &cr);

    return (cr.right - cr.left);
}




unsigned short  ModalDialogView::clientHeight( ) const
{
    if (! isInteractionRunning( ))
        throw std::logic_error("ModalDialogView: requested surface height, "
            "but no interaction is running");

    RECT  cr;
    GetClientRect(fDialogPrimitive, &cr);

    return (cr.bottom - cr.top);
}




unsigned short  ModalDialogView::minHorizTrackSize( ) const
{
    return fMinHorizTrackSize;
}




void  ModalDialogView::setMinHorizTrackSize(unsigned short mhs)
{
    if (mhs < kDefaultMinHorizTrackSize)
        mhs = kDefaultMinHorizTrackSize;

    fMinHorizTrackSize = mhs;
}




unsigned short  ModalDialogView::minVerticalTrackSize( ) const
{
    return fMinVerticalTrackSize;
}




void  ModalDialogView::setMinVerticalTrackSize(unsigned short mvs)
{
    if (mvs < kDefaultMinVerticalTrackSize)
        mvs = kDefaultMinVerticalTrackSize;

    fMinVerticalTrackSize = mvs;
}
//////////////////////////////////////////////////////////////////////////////
// END  ModalDialogView                                                     //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// INTERFACE  ModalDialogController                                         //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
ModalDialogController*  ModalDialogController::nullController( )
{
    return new NullModalDialogController( );
}




ModalDialogController::ModalDialogController( )
{
}




ModalDialogController::~ModalDialogController( )
{
}
//////////////////////////////////////////////////////////////////////////////
// END  ModalDialogController                                               //
//////////////////////////////////////////////////////////////////////////////
