//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ImageRepresentation.cpp                                                  //
//                                                                          //
// Implements methods in the RGBColor lightweight type. Implements methods  //
// in the RGBAImageClass and its nested classes fragment, const_fragment,   //
// subspace, and const_subspace. Implements methods in the ImageTools       //
// procedure wrapper. Implements methods in the PCTI image convolution      //
// classes Convolver, ConvolverFactory, ContinuousConvolver,                //
// and GaussianContinuousConvolver.                                         //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver 2.2.0 of Sat 18 Oct 2008 @ 10:15pm EDT                               //
//                                                                          //
//     added methods installMapping( ) and write( ) to class RGBAImage;     //
//     added method saveTARGA to class ImageTools                           //
//                                                                          //
// ver 2.1.5 of Mon 12 May 2008 @ 3:24pm EDT                                //
//                                                                          //
//     added method cancelAction( ) to class RGBAImage to make the class    //
//     compliant with the new InterstitialActionSender interface            //
//                                                                          //
// ver 2.1.1 of Fri 3 May 2008 @ 7:38pm EDT                                 //
//                                                                          //
//     changed all references of class template Vertex2D to Vector2D; see   //
//     the AffineGeometry module's header for more info                     //
//                                                                          //
// previous change history elided; check out an older CVS rev to get it     //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008 Lucas Stephen Beeler. All Rights Reserved.       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "ImageRepresentation.h"
#include <stdexcept>
#include <cstdlib>
#include <malloc.h>
#include <cstring>
#include <cmath>
#include <vector>

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  RGBAImage                                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// transferPixels( )                                       (STATIC METHOD) //
//-------------------------------------------------------------------------//
void  RGBAImage::transferPixels(const RGBAImage& src, RGBAImage& dest,
    unsigned short srcx, unsigned short srcy, unsigned short destx,
    unsigned short desty, unsigned short width, unsigned short height)
{
    unsigned  src_start_idx = ((src.fImageWidth * srcy) + srcx) * 4;
    unsigned  dest_start_idx = ((dest.fImageWidth * desty) + destx) * 4;
    unsigned  src_step = src.fImageWidth * 4;
    unsigned  dest_step = dest.fImageWidth * 4;
    unsigned  copylen = width * 4;

    const unsigned char*  srcrow = & src.fImageBuffer[src_start_idx];
    unsigned char*  destrow = & dest.fImageBuffer[dest_start_idx];
    for (int i = 0; i < height; i++) {

        std::memcpy(destrow, srcrow, copylen);
        srcrow += src_step;
        destrow += dest_step;
    }
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// RGBAImage(short, short)                                   (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::RGBAImage(unsigned short width, unsigned short height)
    : fImageWidth(width), fImageHeight(height),
      fImageBufferSize(((unsigned)(width)) * ((unsigned)(height)) * 4),
      fImageKind(kInMemoryImage)
{
    fImageBuffer = (unsigned char*) new DWORD32 [width * height];

    std::memset(fImageBuffer, 255, fImageBufferSize);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// RGBAImage(const RGBAImage&)                          (COPY CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::RGBAImage(const RGBAImage& source)
{
    fImageWidth = source.width( );
    fImageHeight = source.height( );
    fImageBufferSize = fImageWidth * fImageHeight * 4;
    fImageKind = kInMemoryImage;

    fImageBuffer = new unsigned char[fImageBufferSize];

    std::memcpy(fImageBuffer, source.imageBuffer( ), fImageBufferSize);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// ~RGBAImage( )                                              (DESTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::~RGBAImage( )
{
    delete [ ] fImageBuffer;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator=( )                                               (ASSIGNMENT) //
//-------------------------------------------------------------------------//
const RGBAImage&  RGBAImage::operator=(const RGBAImage& source)
{
    delete [ ] fImageBuffer;

    fImageWidth = source.width( );
    fImageHeight = source.height( );
    fImageBufferSize = fImageWidth * fImageHeight * 4;
    fImageKind = kInMemoryImage;

    fImageBuffer = new unsigned char[fImageBufferSize];

    std::memcpy(fImageBuffer, source.imageBuffer( ), fImageBufferSize);

    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// imageBufferSize( )                                                      //
//-------------------------------------------------------------------------//
unsigned  RGBAImage::imageBufferSize( ) const
{
    return fImageBufferSize;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// bitsPerChannel( )                                                       //
//-------------------------------------------------------------------------//
unsigned short  RGBAImage::bitsPerChannel( ) const
{
    return sBitsPerChannel;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// bitsPerPixel( )                                                         //
//-------------------------------------------------------------------------//
unsigned short  RGBAImage::bitsPerPixel( ) const
{
    return sBitsPerPixel;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// channelsPerPixel( )                                                     //
//-------------------------------------------------------------------------//
unsigned short  RGBAImage::channelsPerPixel( ) const
{
    return sChannelsPerPixel;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// pixelConfiguration( )                                                   //
//-------------------------------------------------------------------------//
PixelConfiguration  RGBAImage::pixelConfiguration( ) const
{
    return sPixelConfiguration;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// fragmentAt( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::fragment  RGBAImage::fragmentAt(unsigned short x, unsigned short y)
{
    if ((x >= fImageWidth) || (y >= fImageHeight))
        throw std::logic_error("RGBAImage: fragmentAt( ) index out of range");

    return RGBAImage::fragment(this, x, y);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// fragmentAt( ) const                                                     //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::fragmentAt(unsigned short x,
    unsigned short y) const
{
    if ((x >= fImageWidth) || (y >= fImageHeight))
        throw std::logic_error("RGBAImage: fragmentAt( ) index out of range");

    return RGBAImage::const_fragment(this, x, y);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// firstFragment( )                                                        //
//-------------------------------------------------------------------------//
RGBAImage::fragment  RGBAImage::firstFragment( )
{
    return RGBAImage::fragment(this, 0, 0);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// firstFragment( ) const                                                  //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::firstFragment( ) const
{
    return RGBAImage::const_fragment(this, 0, 0);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// lastFragment( )                                                         //
//-------------------------------------------------------------------------//
RGBAImage::fragment  RGBAImage::lastFragment( )
{
    return RGBAImage::fragment(this, fImageWidth - 1, fImageHeight - 1);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// lastFragment( ) const                                                   //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::lastFragment( ) const
{
    return RGBAImage::const_fragment(this, fImageWidth - 1,
        fImageHeight - 1);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// subspaceAt( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::subspace  RGBAImage::subspaceAt(unsigned short x,
    unsigned short y, unsigned short w, unsigned short h)
{
    if (((x + w) > fImageWidth) || ((y + h) > fImageHeight))
        throw std::logic_error("RGBAImage: subspaceAt( ) subspace "
            "dimensions exceed image bounds");

    return RGBAImage::subspace(this, Vector2Di(x, y), w, h);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// subspaceAt( ) const                                                     //
//-------------------------------------------------------------------------//
RGBAImage::const_subspace  RGBAImage::subspaceAt(unsigned short x,
    unsigned short y, unsigned short w, unsigned short h) const
{
    if (((x + w) > fImageWidth) || ((y + h) > fImageHeight))
        throw std::logic_error("RGBAImage: subspaceAt( ) subspace "
            "dimensions exceed image bounds");

    return RGBAImage::const_subspace(this, Vector2Di(x, y), w, h);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// convolve (const Convolver&)                                             //
//-------------------------------------------------------------------------//
void  RGBAImage::convolve(const Convolver&  mask)
{
    int  w = this->width( );
    int  h = this->height( );
    int  bufsize = this->imageBufferSize( );
    int  rowbytes = w * 4;
    int  r = mask.radius( );
    int  d = mask.sideDimension( );

    unsigned char*  outbuffer = new unsigned char[bufsize];
    const unsigned char*  inbuffer = this->imageBuffer( );

    for (int iter = 0; iter < bufsize; iter += 4) {

        int x = (iter % rowbytes) / 4;
        int y = iter / rowbytes;

        int  const_left = ((x - r) < 0) ? (r - x) : 0;
        int  const_right = ((w - x - 1) < r) ? (r - (w - x - 1)) : 0;
        int  const_bottom = ((y - r) < 0) ? (r - y) : 0;
        int  const_top = ((h - y - 1) < r) ? (r - (h - y - 1)) : 0;

        float  wadjust = 1.0f;

        if (const_bottom || const_left || const_right || const_top) {

            wadjust = 1.0f / mask.subspaceSum(const_left, const_bottom,
                d - const_right - 1, d - const_top - 1);
        }


        unsigned char*  outbyteaddr = & outbuffer[iter];

        float  accum_r = 0.0f;
        float  accum_g = 0.0f;
        float  accum_b = 0.0f;

        for (int j = const_bottom; j < (d - const_top); j++) {
            for (int i = const_left; i < (d - const_right); i++) {

                int hoffset = i - r;
                int voffset = j - r;

                int inbytenum = iter + (voffset * rowbytes) +
                    (hoffset * 4);
                const unsigned char*  inbyteaddr = & inbuffer[inbytenum];

                float  weight = mask.element(i, j);

                accum_r += wadjust * weight * inbyteaddr[0];
                accum_g += wadjust * weight * inbyteaddr[1];
                accum_b += wadjust * weight * inbyteaddr[2];
            }
        }

        outbyteaddr[0] = static_cast<unsigned char>(accum_r);
        outbyteaddr[1] = static_cast<unsigned char>(accum_g);
        outbyteaddr[2] = static_cast<unsigned char>(accum_b);
        outbyteaddr[3] = inbuffer[iter + 3];
    }

    delete [ ] fImageBuffer;

    fImageBuffer = outbuffer;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// convolve (const ContinuousConvolver&)                                   //
//-------------------------------------------------------------------------//
void  RGBAImage::convolve(const ContinuousConvolver&  filter)
{
    int  w = this->width( );
    int  h = this->height( );
    int  bufsize = this->imageBufferSize( );
    int  rowbytes = w * 4;
    int  r = static_cast<int>(floor(filter.radius( )));
    int  d = (2 * r) + 1;

    unsigned char*  outbuffer = new unsigned char[bufsize];
    const unsigned char*  inbuffer = this->imageBuffer( );

    for (int iter = 0; iter < bufsize; iter += 4) {

        int x = (iter % rowbytes) / 4;
        int y = iter / rowbytes;

        int  const_left = ((x - r) < 0) ? (r - x) : 0;
        int  const_right = ((w - x - 1) < r) ? (r - (w - x - 1)) : 0;
        int  const_bottom = ((y - r) < 0) ? (r - y) : 0;
        int  const_top = ((h - y - 1) < r) ? (r - (h - y - 1)) : 0;

        bool                 needs_wadjust = false;
        float                wadjust = 1.0f;
        std::vector<float>   weights;

        if (const_bottom || const_left || const_right || const_top)
            needs_wadjust = true;
        else
            needs_wadjust = false;

        unsigned char*  outbyteaddr = & outbuffer[iter];

        float  accum_r = 0.0f;
        float  accum_g = 0.0f;
        float  accum_b = 0.0f;

        weights.clear( );

        for (int j = const_bottom; j < (d - const_top); j++) {
            for (int i = const_left; i < (d - const_right); i++) {

                int hoffset = i - r;
                int voffset = j - r;

                int inbytenum = iter + (voffset * rowbytes) +
                    (hoffset * 4);
                const unsigned char*  inbyteaddr = & inbuffer[inbytenum];

                float  curr_weight = filter.evaluate(hoffset, voffset);

                accum_r += curr_weight * inbyteaddr[0];
                accum_g += curr_weight * inbyteaddr[1];
                accum_b += curr_weight * inbyteaddr[2];

                weights.push_back(curr_weight);
            }
        }

        wadjust = 0.0f;
        for (int i = 0; i < weights.size( ); i++)
            wadjust += weights[i];

        wadjust = 1.0f / wadjust;

        accum_r = accum_r * wadjust;
        accum_g = accum_g * wadjust;
        accum_b = accum_b * wadjust;

        outbyteaddr[0] = static_cast<unsigned char>(accum_r);
        outbyteaddr[1] = static_cast<unsigned char>(accum_g);
        outbyteaddr[2] = static_cast<unsigned char>(accum_b);
        outbyteaddr[3] = inbuffer[iter + 3];
    }

    delete [ ] fImageBuffer;

    fImageBuffer = outbuffer;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// resample ( )                                                            //
//-------------------------------------------------------------------------//
void  RGBAImage::resample(unsigned short out_w, unsigned short out_h,
    const ContinuousConvolver& filter)
{
    // Collate all the information we'll need about the source image
    // (i.e. what's stored in the image buffer of *this right now) and
    // the convolution filter into some (hopefully) well-named locals
    int                   src_w = this->width( );
    int                   src_h = this->height( );
    int                   src_rowbytes = src_w * 4;
    const unsigned char*  src_buffer = this->imageBuffer( );

    // Collate all the information we'll need about the output image (i.e.
    // what will be stored in the image buffer of *this when we're done) and
    // allocate storage for the output image buffer
    int                   out_bufsize = out_w * out_h * 4;
    int                   out_rowbytes = out_w * 4;
    unsigned char*        out_buffer = new unsigned char[out_bufsize];

    // compute conversion factors that will allow us to map coordinates
    // in output image space to their corresponding coordinates in source
    // image space and vice versa; as we step through the pixels in the
    // output image buffer, we'll use these factors to determine which
    // pixels in the source image should be blended to compute the value
    // of the output pixel we're currently visiting, etc.
    float  to_src_x = static_cast<float>(src_w) / static_cast<float>(out_w);
    float  to_src_y = static_cast<float>(src_h) / static_cast<float>(out_h);
    float  to_out_x = 1.0f / to_src_x;
    float  to_out_y = 1.0f / to_src_y;

    // iterate over and visit each pixel in output image buffer; for
    // each pixel, assemble a corresponding "neighborhood" of pixels
    // in the source image buffer that correspond to the output pixel
    // we're currently visiting, then sample and blend together the values
    // of all the pixels in that neighborhood to compute a value for the
    // output pixel
    for (int curr_outpix = 0; curr_outpix < out_bufsize; curr_outpix += 4) {

        // get the (x, y)-coordinates of the output pixel we're currently
        // visiting
        int  curr_outpix_x = (curr_outpix % out_rowbytes) / 4;
        int  curr_outpix_y = curr_outpix / out_rowbytes;

        // determine the location in source image space that corresponds
        // the output pixel we're currently visiting -- this is the center
        // of the pixel neighborhood that supports the convolution filter
        float  nh_center_x = to_src_x * static_cast<float>(curr_outpix_x);
        float  nh_center_y = to_src_y * static_cast<float>(curr_outpix_y);

        // compute the x and y radii of the source image sampling
        // neighborhood -- these might be different (i.e. the convolution
        // support may be an elliptical or rectangular region as opposted
        // to a round or square one if the image is being sized more along
        // one axis than the other
        float  nh_rad_x = filter.radius( ) * to_src_x;
        float  nh_rad_y = filter.radius( ) * to_src_y;

        // determine the left, bottom, right and top edges of the rectangle
        // bounding the sampling neighborhood in the source image
        int  nh_left = static_cast<int>(ceil(nh_center_x - nh_rad_x));
        int  nh_bottom = static_cast<int>(ceil(nh_center_y - nh_rad_y));
        int  nh_right = static_cast<int>(floor(nh_center_x + nh_rad_x));
        int  nh_top = static_cast<int>(floor(nh_center_y + nh_rad_y));

        // it's possible that the sampling neighborhood may extend beyond the
        // bounds of the source image, so let's adjust these bounds to make
        // sure we don't try to access out-of-range source image data
        if (nh_left < 0)  nh_left = 0;
        if (nh_bottom < 0)  nh_bottom = 0;
        if (nh_right > (src_w - 1))  nh_right = (src_w - 1);
        if (nh_top > (src_h - 1))  nh_top = (src_h - 1);

        // iterate over all the pixels in the sampling neighborhood, get
        // their r-, g-, and b-component values, and weight these values by
        // evaluating the convolver; note, however, that the weights obtained
        // from the convolver aren't guaranteed to sum to 1, so to preserve
        // overall intensity, we adjust the output pixel value we compute by
        // multiplying it by (1.0f / weightsum)
        float  weightsum = 0.0f;
        float  accum_r = 0.0f;
        float  accum_g = 0.0f;
        float  accum_b = 0.0f;
        for (int j = nh_bottom; j <= nh_top; j++) {
            for (int i = nh_left; i <= nh_right; i++) {

                float  weight = filter.evaluate(
                    to_out_x * (nh_center_x - static_cast<float>(i)),
                    to_out_y * (nh_center_y - static_cast<float>(j)));

                weightsum += weight;

                const unsigned char*  curr_srcpix =
                    & src_buffer[(j * src_rowbytes) + (i * 4)];

                accum_r += weight * static_cast<float>(curr_srcpix[0]);
                accum_g += weight * static_cast<float>(curr_srcpix[1]);
                accum_b += weight * static_cast<float>(curr_srcpix[2]);
            }
        }
        accum_r *= (1.0f / weightsum);
        accum_g *= (1.0f / weightsum);
        accum_b *= (1.0f / weightsum);

        // write the blended pixel component values into the output buffer
        out_buffer[curr_outpix] = static_cast<unsigned char>(accum_r);
        out_buffer[curr_outpix + 1] = static_cast<unsigned char>(accum_g);
        out_buffer[curr_outpix + 2] = static_cast<unsigned char>(accum_b);
        out_buffer[curr_outpix + 3] = 255;
    }

    // deallocate storage currently used by the image buffer of (*this)
    delete  [ ] fImageBuffer;

    // update the member variables of (*this) to reflect its new dimensions
    // and buffer data pointer after resampling
    fImageBuffer = out_buffer;
    fImageWidth = out_w;
    fImageHeight = out_h;
    fImageBufferSize = out_bufsize;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// draw ( )                                                                //
//-------------------------------------------------------------------------//
void  RGBAImage::draw( ) const
{
    const RGBColorNative* buffer =
        reinterpret_cast<const RGBColorNative*>(imageBuffer( ));

    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);

    glDrawPixels(width( ), height( ), GL_RGBA, GL_UNSIGNED_INT_8_8_8_8_REV,
        buffer);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// imageKind ( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::ImageKind  RGBAImage::imageKind( ) const
{
    return fImageKind;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// createMappedImage (ImageFileFormat, const string&)                      //
//-------------------------------------------------------------------------//
RGBAImage*  RGBAImage::createMappedImage(ImageFileFormat format,
        const std::string& path)
{
    RGBAImage*  result;

    if (format == kTargaImageFormat)
        result = ImageTools::loadTARGA(path);
    else
        throw std::invalid_argument("RGBAImage: can't create file-mapped "
            " image: invalid format kind");

    result->fMapSourcePath = path;
    result->fMapSourceFormat = format;
    result->fImageKind = kFileMappedImage;

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// createMappedImage (ImageFileFormat, const string&,                      //
//     InterstitalActionReceiverSet& notifyset)                            //
//-------------------------------------------------------------------------//
RGBAImage*  RGBAImage::createMappedImage(ImageFileFormat format,
        const std::string& path, InterstitialActionReceiverSet& notifyset,
        InterstitialActionSender& proxyhost)
{
    RGBAImage*  result;

    if (format == kTargaImageFormat)
        result = ImageTools::loadTARGA(path, notifyset, proxyhost);
    else
        throw std::invalid_argument("RGBAImage: can't create file-mapped "
            " image: invalid format kind");

    result->fMapSourcePath = path;
    result->fMapSourceFormat = format;
    result->fImageKind = kFileMappedImage;

    return result;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// mappingSource ( )                                                       //
//-------------------------------------------------------------------------//
const std::string&  RGBAImage::mappingSource( ) const
{
    if (fImageKind != kFileMappedImage)
        throw std::logic_error("RGBAImage: can't access file mapping "
            "source: this image isn't mapped");

    return fMapSourcePath;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// sourceFormat ( )                                                        //
//-------------------------------------------------------------------------//
RGBAImage::ImageFileFormat  RGBAImage::sourceFormat( ) const
{
    if (fImageKind != kFileMappedImage)
        throw std::logic_error("RGBAImage: can't access file mapping "
            "format: this image isn't mapped");

    return fMapSourceFormat;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// registerReceiver ( )                                                    //
//-------------------------------------------------------------------------//
void  RGBAImage::registerReceiver(InterstitialActionReceiver& recv)
{
    fReceiverSet.insert(& recv);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// deregisterReceiver ( )                                                  //
//-------------------------------------------------------------------------//
void  RGBAImage::deregisterReceiver(InterstitialActionReceiver& recv)
{
    fReceiverSet.erase(& recv);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// cancelAction ( )                                                        //
//-------------------------------------------------------------------------//
void  RGBAImage::cancelAction(InterstitialActionReceiver& recv)
{
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// installMapping ( )                                                      //
//-------------------------------------------------------------------------//
void  RGBAImage::installMapping(ImageFileFormat format,
    const std::string& path)
{
    fImageKind = kFileMappedImage;
    fMapSourceFormat = format;
    fMapSourcePath = path;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// write ( )                                                               //
//-------------------------------------------------------------------------//
void  RGBAImage::write( ) const
{
    if (fImageKind != kFileMappedImage)
        throw std::logic_error("RGBAImage: can't write image to mapping "
            "source file: this image isn't mapped");

    if (fMapSourceFormat == kTargaImageFormat)
        ImageTools::saveTARGA(*this, fMapSourcePath);
    else
        throw std::invalid_argument("RGBAImage: can't write image to "
            "mapping source file: invalid format kind");

}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  RGBAImage                                                          //
/////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  RGBAImage::fragment                                              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// fragment(RGBAImage*, unsigned, unsigned)                  (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::fragment::fragment(RGBAImage* img, unsigned x, unsigned y)
    : fHostSubspace(0), fHostImage(img), fX(x), fY(y)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// fragment(subspace*, unsigned, unsigned)                   (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::fragment::fragment(subspace* sub, unsigned x, unsigned y)
    : fHostSubspace(sub), fHostImage(sub->fHostImage)
{
    fX = sub->fOrigin.x + x;
    fY = sub->fOrigin.y + y;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// fragment(const fragment&)                            (COPY CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::fragment::fragment(const fragment& frag)
    : fHostImage(frag.fHostImage), fX(frag.fX), fY(frag.fY),
      fHostSubspace(frag.fHostSubspace)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator=( )                                               (ASSIGNMENT) //
//-------------------------------------------------------------------------//
const RGBAImage::fragment&  RGBAImage::fragment::operator=(
    const RGBAImage::fragment& rhs)
{
    fHostImage = rhs.fHostImage;
    fHostSubspace = rhs.fHostSubspace;
    fX = rhs.fX;
    fY = rhs.fY;

    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uAddrRed( )                                                             //
//-------------------------------------------------------------------------//
unsigned char*  RGBAImage::fragment::uAddrRed( ) const
{
    unsigned  w = fHostImage->fImageWidth;
    unsigned  pixelIndex = ((fY * w) + fX) * 4;

    return (unsigned char*) & (fHostImage->imageBuffer( )[pixelIndex]);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uAddrGreen( )                                                           //
//-------------------------------------------------------------------------//
unsigned char*  RGBAImage::fragment::uAddrGreen( ) const
{
    return (uAddrRed( ) + 1);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uAddrBlue( )                                                            //
//-------------------------------------------------------------------------//
unsigned char*  RGBAImage::fragment::uAddrBlue( ) const
{
    return (uAddrRed( ) + 2);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==(const fragment&)                                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator==(const RGBAImage::fragment& rhs) const
{
    uCheckSameHosts(rhs);

    if ((fX == rhs.fX) && (fY == rhs.fY))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==(const const_fragment&)                                       //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator==(
    const RGBAImage::const_fragment& rhs) const
{
    uCheckSameHosts(rhs);

    if ((fX == rhs.fX) && (fY == rhs.fY))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator!=(const fragment&)                                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator!=(const RGBAImage::fragment& rhs) const
{
    return !(*this == rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator!=(const const_fragment&)                                       //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator!=(
    const RGBAImage::const_fragment& rhs) const
{
    return !(*this == rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<(const fragment&)                                              //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator<(const RGBAImage::fragment& rhs) const
{
    uCheckSameHosts(rhs);

    if (fY < rhs.fY)
        return true;
    else if ((fY == rhs.fY) && (fX < rhs.fX))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<(const const_fragment&)                                        //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator<(
    const RGBAImage::const_fragment& rhs) const
{
    uCheckSameHosts(rhs);

    if (fY < rhs.fY)
        return true;
    else if ((fY == rhs.fY) && (fX < rhs.fX))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>(const fragment&)                                              //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator>(const RGBAImage::fragment& rhs) const
{
    return !((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>(const const_fragment&)                                        //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator>(
    const RGBAImage::const_fragment& rhs) const
{
    return !((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<=(const fragment&)                                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator<=(const RGBAImage::fragment& rhs) const
{
    return ((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<=(const const_fragment&)                                       //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator<=(
    const RGBAImage::const_fragment& rhs) const
{
    return ((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>=(const fragment&)                                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator>=(const RGBAImage::fragment& rhs) const
{
    return !(*this < rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>=(const const_fragment&)                                       //
//-------------------------------------------------------------------------//
bool  RGBAImage::fragment::operator>=(
    const RGBAImage::const_fragment& rhs) const
{
    return !(*this < rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator++( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::fragment&  RGBAImage::fragment::operator++( )
{
    uAdvance( );
    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator++(int)                                                         //
//-------------------------------------------------------------------------//
RGBAImage::fragment  RGBAImage::fragment::operator++(int)
{
    RGBAImage::fragment  oldVal = *this;

    uAdvance( );

    return oldVal;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// color( )                                                                //
//-------------------------------------------------------------------------//
RGBColor  RGBAImage::fragment::color( ) const
{
    unsigned d = fHostImage->bitsPerChannel( );
    float valcount = static_cast<float>((1 << d) - 1);

    float r = ((float)(*uAddrRed( ))) / valcount;
    float g = ((float)(*uAddrGreen( ))) / valcount;
    float b = ((float)(*uAddrBlue( ))) / valcount;

    return RGBColor(r, g, b);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// setColor( )                                                             //
//-------------------------------------------------------------------------//
void  RGBAImage::fragment::setColor(const RGBColor& c)
{
    unsigned d = fHostImage->bitsPerChannel( );
    float valcount = static_cast<float>((1 << d) - 1);

    unsigned char r = static_cast<unsigned char>(c.r * valcount);
    unsigned char g = static_cast<unsigned char>(c.g * valcount);
    unsigned char b = static_cast<unsigned char>(c.b * valcount);

    (*uAddrRed( )) = r;
    (*uAddrGreen( )) = g;
    (*uAddrBlue( )) = b;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// uCheckSameHosts(fragment&)                                              //
//-------------------------------------------------------------------------//
void   RGBAImage::fragment::uCheckSameHosts(
    const RGBAImage::fragment& other) const
{
    if ((fHostImage != other.fHostImage) ||
        (fHostSubspace != other.fHostSubspace))
        throw std::logic_error("RGBAImage::fragment: attempted to apply a "
            "comparison operator over two image fragments with different "
            "hosts");
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uCheckSameHosts(const_fragment&)                                        //
//-------------------------------------------------------------------------//
void   RGBAImage::fragment::uCheckSameHosts(
    const RGBAImage::const_fragment& other) const
{
    if ((fHostImage != other.fHostImage) ||
        (fHostSubspace != other.fHostSubspace))
        throw std::logic_error("RGBAImage::fragment: attempted to apply a "
            "comparison operator over two image fragments with different "
            "hosts");
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uAdvance( )                                                             //
//-------------------------------------------------------------------------//
void  RGBAImage::fragment::uAdvance( )
{
    unsigned lastcol;
    if (!fHostSubspace)
        lastcol = fHostImage->width( ) - 1;
    else
        lastcol = fHostSubspace->fOrigin.x + fHostSubspace->fWidth - 1;

    unsigned firstcol;
    if (!fHostSubspace)
        firstcol = 0;
    else
        firstcol = fHostSubspace->fOrigin.x;

    if (fX != lastcol)
        fX++;
    else
        fY++, fX = firstcol;
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  RGBAImage::fragment                                                //
/////////////////////////////////////////////////////////////////////////////







/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  RGBAImage::const_fragment                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// const_fragment(const RGBAImage*, unsigned,                (CONSTRUCTOR) //
//                unsigned)                                                //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment::const_fragment(const RGBAImage* img, unsigned x,
    unsigned y) : fHostImage(img), fHostSubspace(0), fX(x), fY(y),
                  fHostConstSubspace(0)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// const_fragment(const RGBAImage::subspace*, unsigned,      (CONSTRUCTOR) //
//                unsigned)                                                //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment::const_fragment(const RGBAImage::subspace* sub,
    unsigned x, unsigned y) : fHostImage(sub->fHostImage), fHostSubspace(sub),
                              fHostConstSubspace(0)
{
    fX = sub->fOrigin.x + x;
    fY = sub->fOrigin.y + y;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// const_fragment(const RGBAImage::const_subspace*,          (CONSTRUCTOR) //
//                unsigned, unsigned)                                      //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment::const_fragment(
    const RGBAImage::const_subspace* csub, unsigned x, unsigned y)
        : fHostImage(csub->fHostImage), fHostSubspace(0),
          fHostConstSubspace(csub)
{
    fX = csub->fOrigin.x + x;
    fY = csub->fOrigin.y + y;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// const_fragment(const RGBAImage::fragment&)                (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment::const_fragment(const fragment& frag)
    : fHostImage(frag.fHostImage), fX(frag.fX), fY(frag.fY),
      fHostSubspace(frag.fHostSubspace), fHostConstSubspace(0)
{
}




//-------------------------------------------------------------------------//
// const_fragment(const RGBAImage::const_fragment&)     (COPY CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment::const_fragment(const const_fragment& frag)
    : fHostImage(frag.fHostImage), fX(frag.fX), fY(frag.fY),
      fHostSubspace(frag.fHostSubspace),
      fHostConstSubspace(frag.fHostConstSubspace)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator=(const RGBAImage::fragment&)                      (ASSIGNMENT) //
//-------------------------------------------------------------------------//
const RGBAImage::const_fragment&  RGBAImage::const_fragment::operator=(
    const RGBAImage::fragment& rhs)
{
    fHostImage = rhs.fHostImage;
    fHostSubspace = rhs.fHostSubspace;
    fHostConstSubspace = 0;
    fX = rhs.fX;
    fY = rhs.fY;

    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator=(const RGBAImage::const_fragment&)                (ASSIGNMENT) //
//-------------------------------------------------------------------------//
const RGBAImage::const_fragment&  RGBAImage::const_fragment::operator=(
    const RGBAImage::const_fragment& rhs)
{
    fHostImage = rhs.fHostImage;
    fHostSubspace = rhs.fHostSubspace;
    fHostConstSubspace = rhs.fHostConstSubspace;
    fX = rhs.fX;
    fY = rhs.fY;

    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uAddrRed( )                                                             //
//-------------------------------------------------------------------------//
const unsigned char*  RGBAImage::const_fragment::uAddrRed( ) const
{
    unsigned  w = fHostImage->fImageWidth;
    unsigned  pixelIndex = ((fY * w) + fX) * 4;

    return & (fHostImage->imageBuffer( )[pixelIndex]);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uAddrGreen( )                                                           //
//-------------------------------------------------------------------------//
const unsigned char*  RGBAImage::const_fragment::uAddrGreen( ) const
{
    return (uAddrRed( ) + 1);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uAddrBlue( )                                                            //
//-------------------------------------------------------------------------//
const unsigned char*  RGBAImage::const_fragment::uAddrBlue( ) const
{
    return (uAddrRed( ) + 2);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==(const RGBAImage::fragment&)                                  //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator==(
    const RGBAImage::fragment& rhs) const
{
    uCheckSameHosts(rhs);

    if ((fX == rhs.fX) && (fY == rhs.fY))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==(const RGBAImage::const_fragment&)                            //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator==(
    const RGBAImage::const_fragment& rhs) const
{
    uCheckSameHosts(rhs);

    if ((fX == rhs.fX) && (fY == rhs.fY))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator!=(const RGBAImage::fragment&)                                  //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator!=(
    const RGBAImage::fragment& rhs) const
{
    return !(*this == rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator!=(const RGBAImage::const_fragment&)                            //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator!=(
    const RGBAImage::const_fragment& rhs) const
{
    return !(*this == rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<(const RGBAImage::fragment&)                                   //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator<(
    const RGBAImage::fragment& rhs) const
{
    uCheckSameHosts(rhs);

    if (fY < rhs.fY)
        return true;
    else if ((fY == rhs.fY) && (fX < rhs.fX))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<(const RGBAImage::const_fragment&)                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator<(
    const RGBAImage::const_fragment& rhs) const
{
    uCheckSameHosts(rhs);

    if (fY < rhs.fY)
        return true;
    else if ((fY == rhs.fY) && (fX < rhs.fX))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>(const RGBAImage::fragment&)                                   //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator>(
    const RGBAImage::fragment& rhs) const
{
    return !((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>(const RGBAImage::const_fragment&)                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator>(
    const RGBAImage::const_fragment& rhs) const
{
    return !((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<=(const RGBAImage::fragment&)                                  //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator<=(
    const RGBAImage::fragment& rhs) const
{
    return ((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<=(const RGBAImage::const_fragment&)                            //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator<=(
    const RGBAImage::const_fragment& rhs) const
{
    return ((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>=(const RGBAImage::fragment&)                                  //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator>=(
    const RGBAImage::fragment& rhs) const
{
    return !(*this < rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>=(const RGBAImage::const_fragment&)                            //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_fragment::operator>=(
    const RGBAImage::const_fragment& rhs) const
{
    return !(*this < rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator++( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment&  RGBAImage::const_fragment::operator++( )
{
    uAdvance( );
    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator++(int)                                                         //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::const_fragment::operator++(int)
{
    RGBAImage::const_fragment  oldVal = *this;

    uAdvance( );

    return oldVal;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// color( )                                                                //
//-------------------------------------------------------------------------//
RGBColor  RGBAImage::const_fragment::color( ) const
{
    unsigned d = fHostImage->bitsPerChannel( );
    float valcount = static_cast<float>((1 << d) - 1);

    float r = ((float)(*uAddrRed( ))) / valcount;
    float g = ((float)(*uAddrGreen( ))) / valcount;
    float b = ((float)(*uAddrBlue( ))) / valcount;

    return RGBColor(r, g, b);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uCheckSameHosts(fragment&)                                              //
//-------------------------------------------------------------------------//
void   RGBAImage::const_fragment::uCheckSameHosts(
    const RGBAImage::fragment& other) const
{
    if ((fHostConstSubspace) ||
        (fHostImage != other.fHostImage) ||
        (fHostSubspace != other.fHostSubspace))
        throw std::logic_error("RGBAImage::fragment: attempted to apply a "
            "comparison operator over two image fragments with different "
            "hosts");
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uCheckSameHosts(const_fragment&)                                        //
//-------------------------------------------------------------------------//
void   RGBAImage::const_fragment::uCheckSameHosts(
    const RGBAImage::const_fragment& other) const
{
    if ((fHostImage != other.fHostImage) ||
        (fHostSubspace != other.fHostSubspace) ||
        (fHostConstSubspace != other.fHostConstSubspace))
        throw std::logic_error("RGBAImage::fragment: attempted to apply a "
            "comparison operator over two image fragments with different "
            "hosts");
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uAdvance( )                                                             //
//-------------------------------------------------------------------------//
void  RGBAImage::const_fragment::uAdvance( )
{
    unsigned lastcol;
    if (fHostSubspace)
        lastcol = fHostSubspace->fOrigin.x + fHostSubspace->fWidth - 1;
    else if (fHostConstSubspace)
        lastcol = fHostConstSubspace->fOrigin.x + fHostConstSubspace->fWidth - 1;
    else
        lastcol = fHostImage->width( ) - 1;

    unsigned firstcol;
    if (fHostSubspace)
        firstcol = fHostSubspace->fOrigin.x;
    else if (fHostConstSubspace)
        firstcol = fHostConstSubspace->fOrigin.x;
    else
        firstcol = 0;

    if (fX != lastcol)
        fX++;
    else
        fY++, fX = firstcol;
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  RGBAImage::fragment                                                //
/////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  RGBAImage::subspace                                              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// subspace(RGBAImage*, const Vector2Di&,                    (CONSTRUCTOR) //
//          unsigned short, unsigned short)                                //
//-------------------------------------------------------------------------//
RGBAImage::subspace::subspace(RGBAImage* img, const Vector2Di& origin,
    unsigned short w, unsigned short h)
        : fHostImage(img), fOrigin(origin), fWidth(w), fHeight(h)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// subspace(const subspace&)                            (COPY CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::subspace::subspace(const RGBAImage::subspace& sub)
    : fHostImage(sub.fHostImage), fOrigin(sub.fOrigin), fWidth(sub.fWidth),
      fHeight(sub.fHeight)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator=( )                                               (ASSIGNMENT) //
//-------------------------------------------------------------------------//
const RGBAImage::subspace&  RGBAImage::subspace::operator=(
    const RGBAImage::subspace& sub)
{
    fHostImage = sub.fHostImage;
    fOrigin = sub.fOrigin;
    fWidth = sub.fWidth;
    fHeight = sub.fHeight;

    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==(const subspace&)                                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::subspace::operator==(const RGBAImage::subspace& rhs) const
{
    if (fHostImage == rhs.fHostImage)
        if (fOrigin == rhs.fOrigin)
            if (fWidth == rhs.fWidth)
                if (fHeight == rhs.fHeight)
                    return true;

    return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==(const const_subspace&)                                       //
//-------------------------------------------------------------------------//
bool  RGBAImage::subspace::operator==(
    const RGBAImage::const_subspace& rhs) const
{
    if (fHostImage == rhs.fHostImage)
        if (fOrigin == rhs.fOrigin)
            if (fWidth == rhs.fWidth)
                if (fHeight == rhs.fHeight)
                    return true;

    return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator!=(const const_subspace&)                                       //
//-------------------------------------------------------------------------//
bool  RGBAImage::subspace::operator!=(
    const RGBAImage::const_subspace& rhs) const
{
    return (!(*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// firstFragment( )                                                        //
//-------------------------------------------------------------------------//
RGBAImage::fragment  RGBAImage::subspace::firstFragment( )
{
    return RGBAImage::fragment(this, 0, 0);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// firstFragment( ) const                                                  //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment   RGBAImage::subspace::firstFragment( ) const
{
    return RGBAImage::const_fragment(this, 0, 0);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// lastFragment( )                                                         //
//-------------------------------------------------------------------------//
RGBAImage::fragment  RGBAImage::subspace::lastFragment( )
{
    return RGBAImage::fragment(this, fWidth - 1, fHeight - 1);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// lastFragment( ) const                                                   //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::subspace::lastFragment( ) const
{
    return RGBAImage::const_fragment(this, fWidth - 1, fHeight - 1);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// fragmentAt( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::fragment  RGBAImage::subspace::fragmentAt(unsigned short x,
    unsigned short y)
{
    if ((x >= fWidth) || (y >= fHeight))
        throw std::logic_error("RGBAImage::subspace: fragmentAt( ) index "
            "out of range");

    return RGBAImage::fragment(this, x, y);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// fragmentAt( ) const                                                     //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::subspace::fragmentAt(unsigned short x,
    unsigned short y) const
{
    if ((x >= fWidth) || (y >= fHeight))
        throw std::logic_error("RGBAImage::subspace: fragmentAt( ) index "
            "out of range");

    return RGBAImage::const_fragment(this, x, y);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// width( )                                                                //
//-------------------------------------------------------------------------//
unsigned short  RGBAImage::subspace::width( ) const
{
    return fWidth;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// height( )                                                               //
//-------------------------------------------------------------------------//
unsigned short  RGBAImage::subspace::height( ) const
{
    return fHeight;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// height( )                                                               //
//-------------------------------------------------------------------------//
const Vector2Di&  RGBAImage::subspace::origin( ) const
{
    return fOrigin;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// makeImage( )                                                            //
//-------------------------------------------------------------------------//
RGBAImage  RGBAImage::subspace::makeImage( ) const
{
    RGBAImage result(fWidth, fHeight);

    RGBAImage::const_fragment  srcIter = this->firstFragment( );
    RGBAImage::fragment        destIter = result.firstFragment( );

    while (srcIter <= this->lastFragment( )) {

        RGBColor c = srcIter.color( );
        destIter.setColor(c);

        srcIter++;
        destIter++;
    }

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// hostImage( )                                                            //
//-------------------------------------------------------------------------//
RGBAImage&  RGBAImage::subspace::hostImage( )
{
    return *fHostImage;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// hostImage( ) const                                                      //
//-------------------------------------------------------------------------//
const RGBAImage&   RGBAImage::subspace::hostImage( ) const
{
    return *fHostImage;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// subspaceAt( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::subspace  RGBAImage::subspace::subspaceAt(unsigned short x,
    unsigned short y, unsigned short w, unsigned short h)
{
    return  RGBAImage::subspace(fHostImage, Vector2Di(fOrigin.x + x,
        fOrigin.y + y), w, h);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// subspaceAt( ) const                                                     //
//-------------------------------------------------------------------------//
RGBAImage::const_subspace  RGBAImage::subspace::subspaceAt(unsigned short x,
    unsigned short y, unsigned short w, unsigned short h) const
{
    return  RGBAImage::const_subspace(fHostImage, Vector2Di(fOrigin.x + x,
        fOrigin.y + y), w, h);
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  RGBAImage::subspace                                                //
/////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  RGBAImage::const_subspace                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// const_subspace(const RGBAImage*, const Vector2Di&,        (CONSTRUCTOR) //
//                unsigned short, unsigned short)                          //
//-------------------------------------------------------------------------//
RGBAImage::const_subspace::const_subspace(const RGBAImage* img,
    const Vector2Di& origin, unsigned short w, unsigned short h)
        : fHostImage(img), fOrigin(origin), fWidth(w), fHeight(h)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// const_subspace(const subspace&)                           (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::const_subspace::const_subspace(const subspace& sub)
    : fHostImage(sub.fHostImage), fOrigin(sub.fOrigin), fWidth(sub.fWidth),
      fHeight(sub.fHeight)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// const_subspace(const const_subspace&)                (COPY CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBAImage::const_subspace::const_subspace(const const_subspace& sub)
    : fHostImage(sub.fHostImage), fOrigin(sub.fOrigin), fWidth(sub.fWidth),
      fHeight(sub.fHeight)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator=(const const_subspace&)                           (ASSIGNMENT) //
//-------------------------------------------------------------------------//
const RGBAImage::const_subspace&  RGBAImage::const_subspace::operator=(
    const RGBAImage::const_subspace& sub)
{
    fHostImage = sub.fHostImage;
    fOrigin = sub.fOrigin;
    fWidth = sub.fWidth;
    fHeight = sub.fHeight;

    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator=(const subspace&)                                 (ASSIGNMENT) //
//-------------------------------------------------------------------------//
const RGBAImage::const_subspace&  RGBAImage::const_subspace::operator=(
    const RGBAImage::subspace& sub)
{
    fHostImage = sub.fHostImage;
    fOrigin = sub.fOrigin;
    fWidth = sub.fWidth;
    fHeight = sub.fHeight;

    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==(const const_subspace&)                                       //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_subspace::operator==(
    const RGBAImage::const_subspace& rhs) const
{
    if (fHostImage == rhs.fHostImage)
        if (fOrigin == rhs.fOrigin)
            if (fWidth == rhs.fWidth)
                if (fHeight == rhs.fHeight)
                    return true;

    return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==(const subspace&)                                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_subspace::operator==(
    const RGBAImage::subspace& rhs) const
{
    if (fHostImage == rhs.fHostImage)
        if (fOrigin == rhs.fOrigin)
            if (fWidth == rhs.fWidth)
                if (fHeight == rhs.fHeight)
                    return true;

    return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator!=(const const_subspace&)                                       //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_subspace::operator!=(
    const RGBAImage::const_subspace& rhs) const
{
    return !(*this == rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator!=(const subspace&)                                             //
//-------------------------------------------------------------------------//
bool  RGBAImage::const_subspace::operator!=(
    const RGBAImage::subspace& rhs) const
{
    return !(*this == rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// firstFragment( )                                                        //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::const_subspace::firstFragment( ) const
{
    return RGBAImage::const_fragment(this, 0, 0);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// lastFragment( )                                                         //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::const_subspace::lastFragment( ) const
{
    return RGBAImage::const_fragment(this, fWidth - 1, fHeight - 1);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// fragmentAt( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::const_fragment  RGBAImage::const_subspace::fragmentAt(
    unsigned short x, unsigned short y) const
{
    if ((x >= fWidth) || (y >= fHeight))
        throw std::logic_error("RGBAImage::const_subspace: fragmentAt( ) "
            "index out of range");

    return RGBAImage::const_fragment(this, x, y);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// width( )                                                                //
//-------------------------------------------------------------------------//
unsigned short  RGBAImage::const_subspace::width( ) const
{
    return fWidth;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// height( )                                                               //
//-------------------------------------------------------------------------//
unsigned short  RGBAImage::const_subspace::height( ) const
{
    return fHeight;
}
//-------------------------------------------------------------------------//








//-------------------------------------------------------------------------//
// height( )                                                               //
//-------------------------------------------------------------------------//
const Vector2Di&  RGBAImage::const_subspace::origin( ) const
{
    return fOrigin;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// makeImage( )                                                            //
//-------------------------------------------------------------------------//
RGBAImage  RGBAImage::const_subspace::makeImage( ) const
{
    RGBAImage result(fWidth, fHeight);

    RGBAImage::const_fragment  srcIter = this->firstFragment( );
    RGBAImage::fragment        destIter = result.firstFragment( );

    while (srcIter <= this->lastFragment( )) {

        RGBColor c = srcIter.color( );
        destIter.setColor(c);

        srcIter++;
        destIter++;
    }

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// hostImage( )                                                            //
//-------------------------------------------------------------------------//
const RGBAImage&   RGBAImage::const_subspace::hostImage( ) const
{
    return *fHostImage;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// subspaceAt( )                                                           //
//-------------------------------------------------------------------------//
RGBAImage::const_subspace  RGBAImage::const_subspace::subspaceAt(
    unsigned short x, unsigned short y, unsigned short w,
    unsigned short h) const
{
    return  RGBAImage::const_subspace(fHostImage, Vector2Di(fOrigin.x + x,
        fOrigin.y + y), w, h);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// draw ( )                                                                //
//-------------------------------------------------------------------------//
void  RGBAImage::const_subspace::draw( ) const
{
    const int  imgwidth = fHostImage->width( );
    const int  imgheight = fHostImage->height( );
    const int  left = fOrigin.x;
    const int  bot = fOrigin.y;

    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, imgwidth);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, bot);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, left);

    const RGBColorNative* buffer =
        reinterpret_cast<const RGBColorNative*>(fHostImage->imageBuffer( ));

    glDrawPixels(fWidth, fHeight, GL_RGBA, GL_UNSIGNED_INT_8_8_8_8_REV,
        buffer);
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  RGBAImage::const_subspace                                          //
/////////////////////////////////////////////////////////////////////////////







/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// TYPE  RGBColor                                                          //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// RGBColor( )                                               (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBColor::RGBColor( ) : r(0.0f), g(0.0f), b(0.0f)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// RGBColor(float, float, float)                             (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
RGBColor::RGBColor(float ur, float ug, float ub)
    : r(ur), g(ug), b(ub)
{
    uClamp( );
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator==( )                                                           //
//-------------------------------------------------------------------------//
bool   RGBColor::operator==(const RGBColor& rhs) const
{
    uClampCheck( );
    rhs.uClampCheck( );

    if (r == rhs.r)
        if (g == rhs.g)
            if (b == rhs.b)
                return true;

    return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator!=( )                                                           //
//-------------------------------------------------------------------------//
bool   RGBColor::operator!=(const RGBColor& rhs) const
{
    return !(*this == rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator<( )                                                            //
//-------------------------------------------------------------------------//
bool   RGBColor::operator<(const RGBColor& rhs) const
{
    uClampCheck( );
    rhs.uClampCheck( );

    if (r < rhs.r)
        return true;
    else if ((r == rhs.r) && (g < rhs.g))
        return true;
    else if ((r == rhs.r) && (g == rhs.g) && (b < rhs.b))
        return true;
    else
        return false;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>( )                                                            //
//-------------------------------------------------------------------------//
bool   RGBColor::operator>(const RGBColor& rhs) const
{
    return !((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>( )                                                            //
//-------------------------------------------------------------------------//
bool   RGBColor::operator<=(const RGBColor& rhs) const
{
    return ((*this < rhs) || (*this == rhs));
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator>=( )                                                           //
//-------------------------------------------------------------------------//
bool   RGBColor::operator>=(const RGBColor& rhs) const
{
    return !(*this < rhs);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator*( )                                                            //
//-------------------------------------------------------------------------//
RGBColor   RGBColor::operator*(float x) const
{
    uClampCheck( );

    RGBColor  result(r * x, g * x, b * x);

    result.uClamp( );

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator/( )                                                            //
//-------------------------------------------------------------------------//
RGBColor   RGBColor::operator/(float x) const
{
    uClampCheck( );

    RGBColor  result(r / x, g / x, b / x);

    result.uClamp( );

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator+( )                                                            //
//-------------------------------------------------------------------------//
RGBColor   RGBColor::operator+(const RGBColor& rhs) const
{
    uClampCheck( );
    rhs.uClampCheck( );

    RGBColor  result(r + rhs.r, g + rhs.g, b + rhs.b);

    result.uClamp( );

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator+=( )                                                           //
//-------------------------------------------------------------------------//
RGBColor   RGBColor::operator+=(const RGBColor& rhs)
{
    uClampCheck( );
    rhs.uClampCheck( );

    r += rhs.r;
    g += rhs.g;
    b += rhs.b;

    uClamp( );

    return *this;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator-( )                                                            //
//-------------------------------------------------------------------------//
RGBColor   RGBColor::operator-(const RGBColor& rhs) const
{
    uClampCheck( );
    rhs.uClampCheck( );

    RGBColor  result(r - rhs.r, g - rhs.g, b - rhs.b);

    result.uClamp( );

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// distance( )                                                             //
//-------------------------------------------------------------------------//
float  RGBColor::distance(const RGBColor& rhs) const
{
    uClampCheck( );
    rhs.uClampCheck( );

    float dR = r - rhs.r;
    float dG = g - rhs.g;
    float dB = b - rhs.b;

    float result = std::sqrt((dR * dR) + (dG * dG) + (dB * dB));

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uClamp( )                                                               //
//-------------------------------------------------------------------------//
void  RGBColor::uClamp( )
{
    /* clamp the red component value */
    if (r > 1.0f)
        r = 1.0f;
    else if (r < 0.0f)
        r = 0.0f;
    else
        ;

    /* clamp the green component value */
    if (g > 1.0f)
        g = 1.0f;
    else if (g < 0.0f)
        g = 0.0f;
    else
        ;

    /* clamp the blue component value */
    if (b > 1.0f)
        b = 1.0f;
    else if (b < 0.0f)
        b = 0.0f;
    else
        ;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// uClamp( )                                                               //
//-------------------------------------------------------------------------//
void  RGBColor::uClampCheck( ) const
{
    if (((r > 1.0f) || (r < 0.0f)) ||
        ((g > 1.0f) || (g < 0.0f)) ||
        ((b > 1.0f) || (b < 0.0f)))
        throw std::logic_error("RGBColor: clamp check failed");
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator*(float, RGBColor)                                (FREE METHOD) //
//-------------------------------------------------------------------------//
RGBColor  operator*(float x, RGBColor c)
{
    return c * x;
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  RGBColor                                                           //
/////////////////////////////////////////////////////////////////////////////







/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  ImageTools                                                       //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// ImageTools( )                                             (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
ImageTools::ImageTools( )
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// ~ImageTools( )                                             (DESTRUCTOR) //
//-------------------------------------------------------------------------//
ImageTools::~ImageTools( )
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// loadTARGA(const string&)                                (STATIC METHOD) //
//-------------------------------------------------------------------------//
RGBAImage*  ImageTools::loadTARGA(const std::string& filename)
{
    int loadedWidth;
    int loadedHeight;

    unsigned char* rawImageData =
        (unsigned char*) tga_load(filename.c_str( ), &loadedWidth,
        &loadedHeight, TGA_TRUECOLOR_32);

    if (!rawImageData)
        throw std::runtime_error("ImageTools: loadTARGA( ) unable to read "
            "image data from disk or image data on disk is not in TrueVision "
            "TARGA format");

    RGBAImage*  result = new RGBAImage(loadedWidth, loadedHeight);

    for (unsigned i = 0; i < result->imageBufferSize( ); i++)
            result->imageBuffer( )[i] = rawImageData[i];

    for (unsigned i = 3; i < result->imageBufferSize( ); i += 4)
            result->imageBuffer( )[i] = (unsigned char)(255);

    free(rawImageData);

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// loadTARGA(const string&, InterstitialActionReceiverSet&,                //
//     InterstitialActionSender& proxyhost)                 (STATIC METOD) //
//-------------------------------------------------------------------------//
RGBAImage*  ImageTools::loadTARGA(const std::string& filename,
    InterstitialActionReceiverSet& notifyset,
    InterstitialActionSender& proxyhost)
{
    int loadedWidth;
    int loadedHeight;

    unsigned char* rawImageData =
        (unsigned char*) tga_load(filename.c_str( ), &loadedWidth,
        &loadedHeight, TGA_TRUECOLOR_32, &notifyset, &proxyhost);

    if (!rawImageData)
        throw std::runtime_error("ImageTools: loadTARGA( ) unable to read "
            "image data from disk or image data on disk is not in TrueVision "
            "TARGA format");

    RGBAImage*  result = new RGBAImage(loadedWidth, loadedHeight);

    for (unsigned i = 0; i < result->imageBufferSize( ); i++)
            result->imageBuffer( )[i] = rawImageData[i];

    for (unsigned i = 3; i < result->imageBufferSize( ); i += 4)
            result->imageBuffer( )[i] = (unsigned char)(255);

    free(rawImageData);

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// saveTARGA( )                                             (STATIC METOD) //
//-------------------------------------------------------------------------//
void  ImageTools::saveTARGA(const RGBAImage& img, const std::string& file)
{
    unsigned char* rawImageData = new unsigned char[img.imageBufferSize( )];

    for (unsigned i = 0; i < img.imageBufferSize( ); i++)
            rawImageData[i] = img.imageBuffer( )[i];

    for (unsigned i = 3; i < img.imageBufferSize( ); i += 4)
            rawImageData[i] = (unsigned char)(255);

    tga_write_raw(file.c_str( ), img.width( ), img.height( ),
        rawImageData, TGA_TRUECOLOR_32);

    delete [ ] rawImageData;
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  ImageTools                                                         //
/////////////////////////////////////////////////////////////////////////////






/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  Convolver                                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// Convolver(unsigned short)                                 (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
Convolver::Convolver(unsigned short radius)
    : fSideDimension(2 * radius + 1)
{
    if ((radius < 1) || (radius > 5))
        throw  std::logic_error("Convolver::Convolver( ): "
            "radius out of range");

    fKernelData = new float[fSideDimension * fSideDimension];

    for (unsigned short y = 0; y < fSideDimension; y++)
        for (unsigned short x = 0; x < fSideDimension; x++)
            this->element(x, y) = 0.0f;

    this->element(radius, radius) = 1.0f;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// Convolver(const Convolver&)                          (COPY CONSTRUCTOR) //
//-------------------------------------------------------------------------//
Convolver::Convolver(const Convolver& src)
    : fSideDimension(src.sideDimension( ))
{
    int nelements = fSideDimension * fSideDimension;

    fKernelData = new float[fSideDimension * fSideDimension];

    std::memcpy(fKernelData, src.fKernelData, sizeof(float) * nelements);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// operator=( )                                               (ASSIGNMENT) //
//-------------------------------------------------------------------------//
const Convolver&  Convolver::operator=(const Convolver& rhs)
{
    if (this == (& rhs))
        return *this;

    delete [ ] fKernelData;

    fSideDimension = rhs.fSideDimension;

    int nelements = fSideDimension * fSideDimension;

    fKernelData = new float[fSideDimension * fSideDimension];

    std::memcpy(fKernelData, rhs.fKernelData, sizeof(float) * nelements);

    return *this;
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  Convolver                                                          //
/////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  ConvolverFactory                                                 //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// STATIC FIELDS                                                           //
//-------------------------------------------------------------------------//
ConvolverFactory*  ConvolverFactory::sSingletonInstance = 0;
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// ConvolverFactory( )                                       (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
ConvolverFactory::ConvolverFactory( )
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// ~ConvolverFactory( )                                       (DESTRUCTOR) //
//-------------------------------------------------------------------------//
ConvolverFactory::~ConvolverFactory( )
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// instance( )                                             (STATIC METHOD) //
//-------------------------------------------------------------------------//
ConvolverFactory&  ConvolverFactory::instance( )
{
    if (!sSingletonInstance)
        sSingletonInstance = new ConvolverFactory( );

    return (* sSingletonInstance);
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// createGaussian( )                                                       //
//-------------------------------------------------------------------------//
Convolver*  ConvolverFactory::createGaussian(unsigned short r)
{
    Convolver* result = new Convolver(r);

    float var = static_cast<float>(r) / 2.0f; var = var * var;

    for (int j = 0; j < result->sideDimension( ); j++) {

        for (int i = 0; i < result->sideDimension( ); i++) {

            float iadj = static_cast<float>(i) - static_cast<float>(r);
            float jadj = static_cast<float>(j) - static_cast<float>(r);

            result->element(i, j) = (1.0f / (2.0f * kPi * var)) *
                (std::exp(-(iadj * iadj) / (2.0f * var))) *
                (std::exp(-(jadj * jadj) / (2.0f * var)));
        }
    }

    result->normalize( );

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// createBox( )                                                            //
//-------------------------------------------------------------------------//
Convolver*  ConvolverFactory::createBox(unsigned short r)
{
    Convolver* result = new Convolver(r);

    int nelements = (result->sideDimension( )) * (result->sideDimension( ));

    float w = 1.0f / static_cast<float>(nelements);

    for (int j = 0; j < result->sideDimension( ); j++)
        for (int i = 0; i < result->sideDimension( ); i++)
            result->element(i, j) = w;

    return result;
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  ConvolverFactory                                                   //
/////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  ContinuousConvolver                                              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// ContinuousConvolver( )                                    (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
ContinuousConvolver::ContinuousConvolver(float radius)
    : fRadius(radius)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// ~ContinuousConvolver( )                                    (DESTRUCTOR) //
//-------------------------------------------------------------------------//
ContinuousConvolver::~ContinuousConvolver( )
{
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  ContinuousConvolver                                                //
/////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  GaussianContinuousConvolver                                      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// GaussianContinuousConvolver(float)                        (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
GaussianContinuousConvolver::GaussianContinuousConvolver(float radius)
    : ContinuousConvolver(radius),
      fVariance((radius / 3.0f) * (radius / 3.0f))
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// GaussianContinuousConvolver(float, float)                 (CONSTRUCTOR) //
//-------------------------------------------------------------------------//
GaussianContinuousConvolver::GaussianContinuousConvolver(float radius,
    float variance)
    : ContinuousConvolver(radius), fVariance(variance)
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// ~GaussianContinuousConvolver( )                            (DESTRUCTOR) //
//-------------------------------------------------------------------------//
GaussianContinuousConvolver::~GaussianContinuousConvolver( )
{
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// evaluate( )                                                             //
//-------------------------------------------------------------------------//
float  GaussianContinuousConvolver::evaluate(float x, float y) const
{
    return (1.0f / (2.0f*kPi*fVariance)) * exp(-(x*x + y*y) /
        (2.0f*fVariance));
}
//-------------------------------------------------------------------------//

/////////////////////////////////////////////////////////////////////////////
// END  GaussianContinuousConvolver                                        //
/////////////////////////////////////////////////////////////////////////////
