//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// GPUInterface.h                                                           //
//                                                                          //
// Packages the interface definitions of classes GPUInterface and           //
// TextureBuffer                                                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// version 3.1.0 of Fri 24 Oct 2008 @ 6:16pm EDT                            //
//                                                                          //
//     added imaging programs directory get/set support to GPUInterfaces    //
//                                                                          //
// version 3.0.0 of Thu 24 July 2008 @ 6:38pm EDT                           //
//                                                                          //
//     added class TextureBuffer, which was tested as part of project       //
//     BufferObjects and is considered stable                               //
//                                                                          //
// version 2.5.0 of Sat 14 June 2008 @ 6:41pm EDT                           //
//                                                                          //
//     added configurable OpenGL API call failure handling                  //
//                                                                          //
// older history elided; check out an older CVS rev to get it               //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef  GPU_INTERFACE__HXX
#define  GPU_INTERFACE__HXX

#include <Win32Support.h>
#include <Application.h>
#include <TranscriptionServer.h>
#include <ImageRepresentation.h>
#include <GL/gl.h>
#include <map>
#include <set>
#include <string>
#include <GPUInterface.h>

//-----------------------------------------------------------------------------
/* CLASS  TextureBuffer */
class TextureBuffer {

private:

    static  bool  sWasExtensionCheckDone;

    GLuint   fHandle;
    GLuint   fTexHandle;
    int      fWidth;
    int      fHeight;

    void  uDeleteObjects( );

public:

    static  const int  kMinDimension = 16;

    TextureBuffer(int width, int height);
    virtual  ~TextureBuffer( );

    int     width( ) const;
    int     height( ) const;
    GLuint  textureHandle( );
    GLuint  handle( );
};
//-----------------------------------------------------------------------------





//-----------------------------------------------------------------------------
/* CLASS  GPUInterface */
class GPUInterface {

private:

    HGLRC         fRenderingContext;
    HDC           fRenderingDC;
    std::string   fImagingProgDir;
    unsigned      fNumTextureUnits;
    unsigned      fNumAuxBuffers;
    bool          fIsUserReportingEnabled;
    int           fCallFailureMode;

    void  uTranscribeGLInfo( );

    std::string  uReadProgramSource(const std::string& filename);
    GLuint       uCompileVertexProgram(const std::string& src);
    GLuint       uCompileFragmentProgram(const std::string& src);
    std::string  uGetShaderInfoLog(GLuint shader);
    std::string  uGetProgramInfoLog(GLuint program);
    GLint        uGetUniformLocation(GLuint program,
                     const std::string& name);

public:

    enum  CallFailureMode { kSilentMode, kExplicitMode };

    GPUInterface(HDC, HGLRC);
    virtual ~GPUInterface( );

    /** methods for compiling and linking GLSL vertex and fragment shader
        programs */

    virtual  GLuint  prepareVertexProgram(const std::string& filename);
    virtual  GLuint  prepareFragmentProgram(const std::string& filename);
    template<typename TIterator>
    GLuint  prepareProgramSet(const TIterator& begin,
        const TIterator& end)
    {

        #ifdef ENABLE_DIAGNOSTICS
        TranscriptionServer&  trans =
            Application::instance( ).transcriptionServer( );
        #endif

        // try to create a new GLSL program object into which we'll
        // link all of the programs named in our argument list; if
        // the new program object can't be created, then we throw a
        // runtime_error
        GLuint  newprog = glCreateProgram( );
        if (!newprog)
            throw  std::runtime_error("GPUInterface: couldn't create GLSL "
                "program object");

        // if execution reaches this point, then we have a new, valid GLSL
        // program object; we will now attach to this program object each
        // of the shader programs
        for (TIterator it = begin; it != end; it++)
            glAttachShader(newprog, *it);
        glLinkProgram(newprog);

        #ifdef ENABLE_DIAGNOSTICS
        trans.write("\n");
        trans.write(">>>>> BEGIN GLSL LINKER INFO LOG >>>>>>>>>>>>>>>>>>>>>"
            ">>>\n\n");
        std::string  infolog = uGetProgramInfoLog(newprog);
        trans.write(infolog);
        trans.write("\n");
        trans.write("<<<<< END GLSL LINKER INFO LOG <<<<<<<<<<<<<<<<<<<<<<<"
            "<<<");
        trans.flushln( );
        #endif

        // determine whether or not our new GLSL program object linked
        // successfully; if it didn't, throw a runtime_error
        int link_ok;
        glGetProgramiv(newprog, GL_LINK_STATUS, &link_ok);
        if (!link_ok) {

            // deallocate the shader object we created above because it's a
            // programmer-managed resource we've alloc'd and we're about
            // throw an exception
            glDeleteProgram(newprog);

            throw std::runtime_error("GPUInterface: linking of activated "
                "programs failed");
        }

        return newprog;
    }


    template<typename TContainer>
    GLuint  GPUInterface::prepareProgramSet(const TContainer& programs)
    {
        return prepareProgramSet(programs.begin( ), programs.end( ));
    }

    GLuint  GPUInterface::prepareProgramSet(const GLuint* programs,
        int numprogs)
    {
        return prepareProgramSet(& programs[0], & programs[numprogs]);
    }

    
    /** methods for setting the values of uniform variables in GLSL
        shader programs */

    virtual  void  setUniform1i(GLuint progset, const std::string& name,
        int val);
    virtual  void  setUniform1f(GLuint progset, const std::string& name,
        float val);
    template<typename TIterator>
    void  setUniform2fv(GLuint progset, const std::string& name,
        const TIterator& begin, const TIterator& end)
    {
        GLint  varloc = uGetUniformLocation(progset, name);

        TIterator it;

        int sizeticker = 0;
        for (it = begin; it != end; it++)
            sizeticker++;

        if (sizeticker == 0)
            return;

        float* transferbuf = new float[2 * sizeticker];

        it = begin;
        for (int i = 0; i < (2 * sizeticker); i += 2, it++) {

            transferbuf[i] = static_cast<float>(it->x);
            transferbuf[i + 1] = static_cast<float>(it->y);
        }

        glUniform2fv(varloc, (2 * sizeticker), transferbuf);

        delete [ ] transferbuf;
    } /* setUniform2fv( ) */

    template<typename TIterator>
    void  setUniform1fv(GLuint progset, const std::string& name,
        const TIterator& begin, const TIterator& end)
    {
        GLint  varloc = uGetUniformLocation(progset, name);

        TIterator it;

        int sizeticker = 0;
        for (it = begin; it != end; it++)
            sizeticker++;

        if (sizeticker == 0)
            return;

        float* transferbuf = new float[sizeticker];

        it = begin;
        for (int i = 0; i < sizeticker; i++, it++)
            transferbuf[i] = static_cast<float>(*it);

        glUniform1fv(varloc, sizeticker, transferbuf);

        delete [ ] transferbuf;
    } /* setUniform1fv */

    virtual  void  setUniform2f(GLuint progset, const::std::string& name,
        const Vector2Df& val);
    virtual  void  setSampler2D(GLuint progset, const std::string& name,
        int val);

    /** methods for texture management */
    virtual  GLuint  downloadTexture(const RGBAImage& texdata);
    virtual  void  updateTexture(GLuint texid, const RGBAImage& texdata);

    /** methods for call failure mode management */
    virtual  CallFailureMode  callFailureMode( ) const;
    virtual  void  setCallFailureMode(CallFailureMode);

    /** methods for imaging programs directory management */
    const  std::string&  imagingProgramsDirectory( ) const;
    void  setImagingProgramsDirectory(const std::string&);

};
//--------------------------------------------------------------------------//

#endif