//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// GPUInterface.cpp                                                         //
//                                                                          //
// Implements methods in classes GPUInterface and TextureBuffer             //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// version 3.1.0 of Fri 24 Oct 2008 @ 6:16pm EDT                            //
//                                                                          //
//     added imaging programs directory get/set support to GPUInterfaces    //
//                                                                          //
// version 3.0.0 of Thu 24 July 2008 @ 6:38pm EDT                           //
//                                                                          //
//     added class TextureBuffer, which was tested as part of project       //
//     BufferObjects and is considered stable                               //
//                                                                          //
// version 2.5.0 of Sat 14 June 2008 @ 6:41pm EDT                           //
//                                                                          //
//     added configurable OpenGL API call failure handling                  //
//                                                                          //
// older history elided; check out an older CVS rev to get it               //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "GPUInterface.h"
#include <vector>
#include <cstdarg>
#include <cstdio>
#include <ctime>
#include <cstring>
#include <sstream>
#include <fstream>


/* line 23 below stops the MSVC 2003 & 2005 compilers from propogating
   annoying warnings about security risks present in the ANSI C standard
   library */
#pragma warning(disable : 4996)

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  GPUInterface                                                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
GPUInterface::GPUInterface(HDC renderDC, HGLRC renderContext)
    : fRenderingDC(renderDC), fRenderingContext(renderContext),
      fNumAuxBuffers(0), fCallFailureMode(kExplicitMode)
{
    #ifdef ENABLE_DIAGNOSTICS
        TranscriptionServer&  trans =
            Application::instance( ).transcriptionServer( );
        trans.write("GPUInterface: interface created on device with "
            "parameters");
        uTranscribeGLInfo( );
    #endif

    // determine the number of texture units on this GPU
    int ntexunits;
    glGetIntegerv(GL_MAX_TEXTURE_UNITS, &ntexunits);
    fNumTextureUnits = ntexunits;

    // determine the number of auxiliary buffers on this GPU
    int formatid = GetPixelFormat(this->fRenderingDC);
    PIXELFORMATDESCRIPTOR  gpu_pfd;
    DescribePixelFormat(this->fRenderingDC, formatid,
        sizeof(PIXELFORMATDESCRIPTOR), &gpu_pfd);
    fNumAuxBuffers = gpu_pfd.cAuxBuffers;

    // set the imaging programs directory to '.\Imaging Programs'
    fImagingProgDir = Win32Tools::filesystem( ).cwd( ) +
        "\\ImagingPrograms";

    #ifdef ENABLE_DIAGNOSTICS
        trans << "GPUInterface: imaging programs directory is '" +
            fImagingProgDir + "'";
    #endif
}




GPUInterface::~GPUInterface( )
{
}




void  GPUInterface::uTranscribeGLInfo( )
{
    const  size_t  kWorkBufferSize = 32;

    TranscriptionServer&  transcript =
        Application::instance( ).transcriptionServer( );

    transcript.write("\n\n");
    
    const char* vendorNameRaw = (const char*) glGetString(GL_VENDOR);
    std::string vendorName = vendorNameRaw;

    const char* versionNameRaw = (const char*) glGetString(GL_VERSION);
    std::string versionName = versionNameRaw;

    int numTexUnitsRaw;
    glGetIntegerv(GL_MAX_TEXTURE_IMAGE_UNITS, &numTexUnitsRaw);
    char  numTexUnits[kWorkBufferSize];
    sprintf(numTexUnits, "%d", numTexUnitsRaw);

    int  numdrawbufs_raw;
    glGetIntegerv(GL_MAX_DRAW_BUFFERS, &numdrawbufs_raw);
    char  numdrawbufs[kWorkBufferSize];
    sprintf(numdrawbufs, "%d", numdrawbufs_raw);

    std::string failmode;
    if (callFailureMode( ) == kExplicitMode)
        failmode = "explicit mode";
    else if (callFailureMode( ) == kSilentMode)
        failmode = "silent mode";
    else
        failmode = "Unknown Mode";

    transcript.write("OpenGL Vendor: \t\t\t" + vendorName + "\n");
    transcript.write("OpenGL Version: \t\t" + versionName + "\n");
    transcript.write("Texture Image Units: \t" +
        std::string(numTexUnits) + "\n");
    transcript.write("Total Draw Buffers: \t" + std::string(numdrawbufs)
        + "\n");
    transcript.write("Call Failure Mode:  \t" + failmode + "\n");


    int formatid = GetPixelFormat(this->fRenderingDC);
    PIXELFORMATDESCRIPTOR  gpu_pfd;
    DescribePixelFormat(this->fRenderingDC, formatid,
        sizeof(PIXELFORMATDESCRIPTOR), &gpu_pfd);

    char  numauxbufs[kWorkBufferSize];
    sprintf(numauxbufs, "%d", gpu_pfd.cAuxBuffers);
    transcript.write("Auxiliary Buffers: \t\t" + std::string(numauxbufs)
        + "\n");

    int   vpsize_raw[2];
    glGetIntegerv(GL_MAX_VIEWPORT_DIMS, vpsize_raw);
    char  vpsize[kWorkBufferSize];
    sprintf(vpsize, "%d x %d", vpsize_raw[0], vpsize_raw[1]);
    transcript.write("Max Viewport Size: \t\t" + std::string(vpsize) + "\n");
    
    transcript.flushln( );
}




/* attempt to compile the GLSL vertex shader program whose source code
   is pacakged in 'src'; if the shader program compiles successfully,
   then return a handle to a shader object encapsulating the newly
   compiled progra; if the shader program fails to compile successfully,
   then return 0; if ENABLE_DIAGNOSTICS is #defined, shader log
   output and status information are written to the running Application's
   transcript file */
GLuint   GPUInterface::uCompileVertexProgram(const std::string& src)
{
    #ifdef ENABLE_DIAGNOSTICS
        TranscriptionServer&  trans =
            Application::instance( ).transcriptionServer( );
    #endif

    GLuint  pgmhandle = glCreateShader(GL_VERTEX_SHADER);

    if (!pgmhandle)
        throw std::runtime_error("GPUInterface: uCompileVertexProgram( ): "
            "could not allocate a shader program handle.");

    const char* srcbuffer = src.c_str( );
    glShaderSource(pgmhandle, 1, &srcbuffer, NULL);

    glCompileShader(pgmhandle);
    GLint  compile_ok = GL_FALSE;
    glGetShaderiv(pgmhandle, GL_COMPILE_STATUS, &compile_ok);

    #ifdef ENABLE_DIAGNOSTICS
    trans.write("\n");
    trans.write(">>>>> BEGIN VERTEX COMPILER INFO LOG >>>>>>>>>>>>>>>>>>>"
        ">>>\n\n");
    std::string  infolog = uGetShaderInfoLog(pgmhandle);
    trans.write(infolog);
    trans.write("\n");
    trans.write("<<<<< END VERTEX COMPILER INFO LOG <<<<<<<<<<<<<<<<<<<<<"
        "<<<");
    trans.flushln( );
    #endif

    if (!compile_ok) {

        // deallocate the shader object we allocated above because we're
        // about to throw an exception
        glDeleteShader(pgmhandle);

        throw std::runtime_error("GPUInterface: uCompileVertexProgram( ): "
            "program source failed to compile.");
    }

    return pgmhandle;
}





/* attempt to compile the GLSL fragment shader program whose source code
   is pacakged in 'src'; if the shader program compiles successfully,
   then return a handle to a shader object encapsulating the newly
   compiled progra; if the shader program fails to compile successfully,
   then return 0; if ENABLE_DIAGNOSTICS is #defined, shader log
   output and status information are written to the running Application's
   transcript file */
GLuint   GPUInterface::uCompileFragmentProgram(const std::string& src)
{
    #ifdef ENABLE_DIAGNOSTICS
        TranscriptionServer&  trans =
            Application::instance( ).transcriptionServer( );
    #endif

    GLuint  pgmhandle = glCreateShader(GL_FRAGMENT_SHADER);

    if (!pgmhandle)
        throw std::runtime_error("GPUInterface: uCompileFragmentProgram( ): "
            "could not allocate a shader program handle.");

    const char* srcbuffer = src.c_str( );
    glShaderSource(pgmhandle, 1, &srcbuffer, NULL);


    glCompileShader(pgmhandle);
    GLint  compile_ok = GL_FALSE;
    glGetShaderiv(pgmhandle, GL_COMPILE_STATUS, &compile_ok);

    #ifdef ENABLE_DIAGNOSTICS
    trans.write("\n");
    trans.write(">>>>> BEGIN FRAGMENT COMPILER INFO LOG >>>>>>>>>>>>>>>>>>>"
        ">>>\n\n");
    std::string  infolog = uGetShaderInfoLog(pgmhandle);
    trans.write(infolog);
    trans.write("\n");
    trans.write("<<<<< END FRAGMENT COMPILER INFO LOG <<<<<<<<<<<<<<<<<<<<<"
        "<<<");
    trans.flushln( );
    #endif

    if (!compile_ok) {

        // deallocate the shader object we allocated above because we're
        // about to throw an exception
        glDeleteShader(pgmhandle);

        throw std::runtime_error("GPUInterface: uCompileFragmentProgram( ): "
            "program source failed to compile.");
    }

    return pgmhandle;
}




std::string  GPUInterface::uGetShaderInfoLog(GLuint shader)
{
    GLint  len = 0;
    glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &len);   // length includes
                                                       // the '\0'
    if (len == 0)
        return std::string("");

    GLchar*  logbuffer = new GLchar[len];
    glGetShaderInfoLog(shader, len, NULL, logbuffer);
    std::string  result = logbuffer;
    delete [ ] logbuffer;
    
    return result;
}




GLint  GPUInterface::uGetUniformLocation(GLuint program,
    const std::string& name)
{
    static  const GLint  kErrorOccured = -1;
    
    const GLchar*  glname = name.c_str( );
    GLint  varloc = glGetUniformLocation(program, glname);

    if ((varloc == kErrorOccured) && (fCallFailureMode == kExplicitMode))
        throw std::runtime_error("GPUInterface: can't get TRAM location of "
            "uniform variable '" + name + "'");
    else
        return varloc;
}




GLuint  GPUInterface::prepareVertexProgram(const std::string& filename)
{
    std::string srccode = uReadProgramSource(filename);
    return uCompileVertexProgram(srccode);
}




GLuint  GPUInterface::prepareFragmentProgram(const std::string& filename)
{
    std::string srccode = uReadProgramSource(filename);
    return uCompileFragmentProgram(srccode);
}




std::string  GPUInterface::uGetProgramInfoLog(GLuint program)
{
    GLint  len = 0;
    glGetProgramiv(program, GL_INFO_LOG_LENGTH, &len);  // length including
                                                        // terminal '\0'
    if (len == 0)
        return std::string("");

    GLchar*  logbuffer = new GLchar[len];
    glGetProgramInfoLog(program, len, NULL, logbuffer);
    std::string  result = logbuffer;
    delete [ ] logbuffer;
    
    return result;
}




void  GPUInterface::setUniform1i(GLuint progset, const std::string& name,
    int val)
{
    GLint  varloc = uGetUniformLocation(progset, name);
    glUniform1i(varloc, val);
}




void  GPUInterface::setUniform1f(GLuint progset, const std::string& name,
        float val)
{
    GLint  varloc = uGetUniformLocation(progset, name);
    glUniform1f(varloc, val);
}




void  GPUInterface::setUniform2f(GLuint progset, const::std::string& name,
    const Vector2Df& val)
{
    GLint  varloc = uGetUniformLocation(progset, name);
    glUniform2f(varloc, val.x, val.y);
}




void  GPUInterface::setSampler2D(GLuint progset, const std::string& name,
    int val)
{
    setUniform1i(progset, name, val);
}




std::string  GPUInterface::uReadProgramSource(const std::string& filename)
{
    #ifdef ENABLE_DIAGNOSTICS
    TranscriptionServer&  trans =
        Application::instance( ).transcriptionServer( );
    #endif

    std::string  fullpath = fImagingProgDir + "\\" + filename;

    std::ifstream  srcstream(fullpath.c_str( ), std::ios::in);

    if (!srcstream)
        throw std::runtime_error("GPUInterface: unable to open "
            "shader program source file.");

    std::string  linebuffer;
    unsigned     lineticker = 0;
    std::string  code;
    while (srcstream) {

        linebuffer = "";
        std::getline(srcstream, linebuffer);
        code += (linebuffer + "\n");
        lineticker++;
    }

    #ifdef ENABLE_DIAGNOSTICS
    char  linecount[255];
    std::sprintf(linecount, "%d", lineticker);
    trans.writeln("read " + std::string(linecount) + " lines from shader "
        "source file '" + filename + "'.");
    #endif    

    srcstream.close( );

    return code;
}




GLuint  GPUInterface::downloadTexture(const RGBAImage& texdata)
{
    GLuint newtex;
    glGenTextures(1, &newtex);
    glBindTexture(GL_TEXTURE_2D, newtex);

    glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_FALSE);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);

    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, texdata.width( ),
        texdata.height( ), 0, GL_RGBA, GL_UNSIGNED_INT_8_8_8_8_REV,
        texdata.imageBuffer( ));

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    return newtex;
}




void  GPUInterface::updateTexture(GLuint texid, const RGBAImage& texdata)
{
    glBindTexture(GL_TEXTURE_2D, texid);

    glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_FALSE);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);

    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, texdata.width( ),
        texdata.height( ), 0, GL_RGBA, GL_UNSIGNED_INT_8_8_8_8_REV,
        texdata.imageBuffer( ));
}




GPUInterface::CallFailureMode  GPUInterface::callFailureMode( ) const
{
    return static_cast<CallFailureMode>(fCallFailureMode);
}




void  GPUInterface::setCallFailureMode(GPUInterface::CallFailureMode mode)
{
    if (callFailureMode( ) != mode) {

        fCallFailureMode = mode;

        #ifdef ENABLE_DIAGNOSTICS
        TranscriptionServer& trans =
            Application::instance( ).transcriptionServer( );
        if (mode == kExplicitMode)
            trans.writeln("GPUInterface: setting OpenGL API call failure "
                "mode to explicit");
        else if (mode == kSilentMode)
            trans.writeln("GPUInterface: setting OpenGL API call failure "
                "mode to silent");
        else
            trans.writeln("GPUInterface: setting unkown OpenGL API call "
                "failure mode");
        #endif
    }
}



const  std::string&  GPUInterface::imagingProgramsDirectory( ) const
{
    return fImagingProgDir;
}



void  GPUInterface::setImagingProgramsDirectory(const std::string& dir)
{
    fImagingProgDir = dir;
}
/////////////////////////////////////////////////////////////////////////////
// END  GPUInterface                                                       //
/////////////////////////////////////////////////////////////////////////////









/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  TextureBuffer                                                    //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
bool  TextureBuffer::sWasExtensionCheckDone = false;

TextureBuffer::TextureBuffer(int width, int height)
    : fWidth(width), fHeight(height)
{
    if ((width < kMinDimension) || (height < kMinDimension)) {

        std::ostringstream msgstream;
        msgstream << "TextureBuffer: both the width and height of a texture "
            "buffer must be at least " << kMinDimension << " pixels";

        throw std::logic_error(msgstream.str( ));
    } /* if either width or height less than min permissible dimension */

    if (!sWasExtensionCheckDone) {

    const char* extnlist =
        reinterpret_cast<const char*>(glGetString(GL_EXTENSIONS));

        if (! std::strstr(extnlist, "EXT_framebuffer_object"))
            throw std::runtime_error("TextureBuffer: the OpenGL "
                "framebuffer object extension is not available on this "
                "system");

        sWasExtensionCheckDone = true;
    } /* if extension check hasn't been done */

    /* create the OpenGL framebuffer object and the texture that we'll
       soon attach to it */
    glGenFramebuffersEXT(1, &fHandle);
    glGenTextures(1, &fTexHandle);

    /* bind the newly created FBO into the OpenGL machine state */
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fHandle);

    /* bind the newly created texture object into the OpenGL machine
       state, allocate storage for it, and set up its parameters */
    glBindTexture(GL_TEXTURE_2D, fTexHandle);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGBA,
        GL_UNSIGNED_BYTE, 0);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    /* attach the texture to the FBO's color buffer */
    glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT,
        GL_TEXTURE_2D, fTexHandle, 0);

    /* check to see whether the FBO is consistent and complete -- it
       should be -- but if it's not, then we'll throw an exception,
       but first let's delete the OpenGL objects we've allocated */
    GLenum statcode = glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
    if (statcode != GL_FRAMEBUFFER_COMPLETE_EXT) {

        uDeleteObjects( );
        throw std::runtime_error("TextureBuffer: expected OpenGL "
            "framebuffer object to be complete, but it isn't");
    }

    /* unbind the FBO from the OpenGL machine state */
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
}



TextureBuffer::~TextureBuffer( )
{
    uDeleteObjects( );
}



void  TextureBuffer::uDeleteObjects( )
{
    glDeleteFramebuffersEXT(1, &fHandle);
    glDeleteTextures(1, &fTexHandle);
}



int  TextureBuffer::width( ) const
{
    return fWidth;
}



int  TextureBuffer::height( ) const
{
    return fHeight;
}



GLuint  TextureBuffer::textureHandle( )
{
    return fTexHandle;
}



GLuint  TextureBuffer::handle( )
{
    return fHandle;
}
/////////////////////////////////////////////////////////////////////////////
// END    TextureBuffer                                                    //
/////////////////////////////////////////////////////////////////////////////
