//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ExoticThingies.cpp                                                       //
//                                                                          //
// Implements methods in class SliderThingy only                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  <unversioned module>                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  Copyright (c) 2008 Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "ExoticThingies.h"
#include "Application.h"
#include "GPUInterface.h"

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  SliderController                                                  //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class  NullSliderController :
    public virtual SliderController {

public:

    NullSliderController( ) { }

    ~NullSliderController( ) { }

    void  thingyValueChanged(SliderThingy& sender) { }
};


SliderController*  SliderController::nullController( )
{
    return new NullSliderController( );
}
//////////////////////////////////////////////////////////////////////////////
// END  SliderController                                                    //
//////////////////////////////////////////////////////////////////////////////









//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  SliderThingy                                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
SliderThingy::SliderThingy(CompositeThingy& composite)
{
    fController = SliderController::nullController( );

    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );
    HWND w = Win32Tools::widgetFactory( ).makeHorizontalTrackbar(
        parent.primitive( ), nextChildID( ));
    fAdapter.reset(new ThingyAdapter(*this, w));
}





LRESULT  SliderThingy::processParentMessage(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    switch (msgcode) {

        case WM_HSCROLL:
            fController->thingyValueChanged(*this);
            return 0;
        break;

        default:
            return DefWindowProc(target, msgcode, wparam, lparam);
        break;
    }
}




ThingyAdapter&  SliderThingy::adapter( )
{
    return *fAdapter;
}




const ThingyAdapter&  SliderThingy::adapter( ) const
{
    return *fAdapter;
}




void  SliderThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  SliderThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  SliderThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  SliderThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  SliderThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  SliderThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  SliderThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  SliderThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}




unsigned short  SliderThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  SliderThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  SliderThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}




void  SliderThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}




bool  SliderThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}




SliderController&  SliderThingy::controller( )
{
    return *fController;
}




const SliderController&  SliderThingy::controller( ) const
{
    return *fController;
}




SliderController&  SliderThingy::installController(SliderController& newctlr)
{
    SliderController* oldctlr = fController;
    fController = & newctlr;
    return *oldctlr;
}




void  SliderThingy::setRangeMin(int rmin)
{
    SendMessage(fAdapter->primitive( ), TBM_SETRANGEMIN, (WPARAM) TRUE,
        (LPARAM) rmin);
}




void  SliderThingy::setRangeMax(int rmax)
{
    SendMessage(fAdapter->primitive( ), TBM_SETRANGEMAX, (WPARAM) TRUE,
        (LPARAM) rmax);
}




void  SliderThingy::setRange(int rmin, int rmax)
{
    setRangeMin(rmin);
    setRangeMax(rmax);
}




int   SliderThingy::rangeMin( ) const
{
    return SendMessage(fAdapter->primitive( ), TBM_GETRANGEMIN, 0, 0); 
}




int   SliderThingy::rangeMax( ) const
{
    return SendMessage(fAdapter->primitive( ), TBM_GETRANGEMAX, 0, 0); 
}




int   SliderThingy::value( ) const
{
    return SendMessage(fAdapter->primitive( ), TBM_GETPOS, 0, 0);   
}



void  SliderThingy::setValue(int val)
{
    SendMessage(fAdapter->primitive( ), TBM_SETPOS, (WPARAM) TRUE,
        (LPARAM) val);   
}
//////////////////////////////////////////////////////////////////////////////
// END  SliderThingy                                                        //
//////////////////////////////////////////////////////////////////////////////










//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  HorizontalSeparatorThingy                                         //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
HorizontalSeparatorThingy::HorizontalSeparatorThingy(CompositeThingy&
    composite)
{
    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );
    HWND w = Win32Tools::widgetFactory( ).makeHorizontalSeparator(
        parent.primitive( ), nextChildID( ));
    fAdapter.reset(new ThingyAdapter(*this, w));
}




ThingyAdapter&  HorizontalSeparatorThingy::adapter( )
{
    return *fAdapter;
}




const ThingyAdapter&  HorizontalSeparatorThingy::adapter( ) const
{
    return *fAdapter;
}




void  HorizontalSeparatorThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  HorizontalSeparatorThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  HorizontalSeparatorThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  HorizontalSeparatorThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  HorizontalSeparatorThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  HorizontalSeparatorThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  HorizontalSeparatorThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  HorizontalSeparatorThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}




unsigned short  HorizontalSeparatorThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  HorizontalSeparatorThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  HorizontalSeparatorThingy::enable( )
{
}




void  HorizontalSeparatorThingy::disable( )
{
}




bool  HorizontalSeparatorThingy::isEnabled( ) const
{
    return false;
}
//////////////////////////////////////////////////////////////////////////////
// END  HorizontalSeparatorThingy                                           //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ProgressBarThingy                                                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
ProgressBarThingy::ProgressBarThingy(CompositeThingy& composite)
{
    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );
    HWND w = Win32Tools::widgetFactory( ).makeProgressBar(parent.primitive( ),
        nextChildID( ));
    fAdapter.reset(new ThingyAdapter(*this, w));
}




ThingyAdapter&  ProgressBarThingy::adapter( )
{
    return *fAdapter;
}




const ThingyAdapter&  ProgressBarThingy::adapter( ) const
{
    return *fAdapter;
}




void  ProgressBarThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  ProgressBarThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  ProgressBarThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  ProgressBarThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  ProgressBarThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  ProgressBarThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  ProgressBarThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  ProgressBarThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}




unsigned short  ProgressBarThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  ProgressBarThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  ProgressBarThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}




void  ProgressBarThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}




bool  ProgressBarThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}




int   ProgressBarThingy::value( ) const
{
    return SendMessage(fAdapter->primitive( ), PBM_GETPOS, 0, 0);  
}




void  ProgressBarThingy::setValue(int val)
{
    SendMessage(fAdapter->primitive( ), PBM_SETPOS, (WPARAM) val, 0);
}
//////////////////////////////////////////////////////////////////////////////
// END  ProgressBarThingy                                                   //
//////////////////////////////////////////////////////////////////////////////









//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  GLCanvasThingy                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const std::string  GLCanvasThingy::kPrimitiveClassName = "GLCanvasThingy";
bool               GLCanvasThingy::sIsClassRegistered = false;


GLCanvasThingy::GLCanvasThingy(CompositeThingy& composite)
    : fRenderingContext(0), fRenderingContextDC(0),
      fController(GLCanvasController::nullController( )),
      fIsTrackInProgress(false)
{
    if (!sIsClassRegistered) {

        WNDCLASS wdc;
        wdc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
        wdc.lpfnWndProc = ThingyAdapter::sUniversalProc;
        wdc.cbClsExtra = 0;
        wdc.cbWndExtra = 0;
        wdc.hInstance = Application::instance( ).systemModule( );
        wdc.hIcon = 0;
        wdc.hCursor = LoadCursor(NULL, IDC_ARROW);
        wdc.hbrBackground = (HBRUSH)(GetStockObject(WHITE_BRUSH));
        wdc.lpszMenuName = 0;
        wdc.lpszClassName = kPrimitiveClassName.c_str( );

        ATOM regresult = RegisterClass(&wdc);
        if (!regresult)
            throw std::runtime_error("GLCanvasThingy: unable to register "
                "window class");

        sIsClassRegistered = true;
    }

    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );
    HWND w = CreateWindowEx(0, kPrimitiveClassName.c_str( ), NULL,
        WS_CHILD, 0, 0, 64, 64, parent.primitive( ), (HMENU)(nextChildID( )),
        NULL, 0);
    fAdapter.reset(new ThingyAdapter(*this, w));

    uInitGLContext( );
    fDeviceInterface.reset(new GPUInterface(fRenderingContextDC,
        fRenderingContext));
}




ThingyAdapter&  GLCanvasThingy::adapter( )
{
    return *fAdapter;
}




const ThingyAdapter&  GLCanvasThingy::adapter( ) const
{
    return *fAdapter;
}




void  GLCanvasThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  GLCanvasThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  GLCanvasThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  GLCanvasThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  GLCanvasThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  GLCanvasThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  GLCanvasThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  GLCanvasThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}




unsigned short  GLCanvasThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  GLCanvasThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  GLCanvasThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}




void  GLCanvasThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}




bool  GLCanvasThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}



void  GLCanvasThingy::uInitGLContext( )
{
    PIXELFORMATDESCRIPTOR canvasPFD;
    
    canvasPFD.nSize = sizeof(PIXELFORMATDESCRIPTOR);
    canvasPFD.nVersion = 1;
    canvasPFD.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL |
        PFD_DOUBLEBUFFER;
    canvasPFD.iPixelType = PFD_TYPE_RGBA;
    canvasPFD.cColorBits = 24;
    canvasPFD.cRedBits = 0;
    canvasPFD.cRedShift = 0;
    canvasPFD.cGreenBits = 0;
    canvasPFD.cGreenShift = 0;
    canvasPFD.cBlueBits= 0;
    canvasPFD.cBlueShift = 0;
    canvasPFD.cAlphaBits = 0;
    canvasPFD.cAlphaShift = 0;
    canvasPFD.cAccumBits = 0;
    canvasPFD.cAccumRedBits = 0;
    canvasPFD.cAccumGreenBits = 0;
    canvasPFD.cAccumBlueBits = 0;
    canvasPFD.cAccumAlphaBits = 0;
    canvasPFD.cDepthBits = 32;
    canvasPFD.cStencilBits = 0;
    canvasPFD.cAuxBuffers = 4;
    canvasPFD.iLayerType = PFD_MAIN_PLANE;
    canvasPFD.bReserved = 0;
    canvasPFD.dwLayerMask = 0;
    canvasPFD.dwVisibleMask = 0;
    canvasPFD.dwDamageMask = 0;

    fRenderingContextDC = GetDC(fAdapter->primitive( ));

    int  canvasPixelFormatID = ChoosePixelFormat(fRenderingContextDC,
        &canvasPFD);
    SetPixelFormat(fRenderingContextDC, canvasPixelFormatID, &canvasPFD);
    fRenderingContext = wglCreateContext(fRenderingContextDC);
    if (!fRenderingContext)
        throw std::runtime_error("GLCanvasThingy: couldn't create an "
            "OpenGL rendering context");
    wglMakeCurrent(fRenderingContextDC, fRenderingContext);
    Win32Tools::initializeGLClientState( );
}



LRESULT  GLCanvasThingy::processMessage(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    HDC           canvasDC;
    PAINTSTRUCT   eventPS;
    int           canvas_h = 0;
    short         mouse_x = 0;
    short         mouse_y = 0;

    switch (msgcode) {

        case WM_PAINT:
            canvasDC = BeginPaint(target, &eventPS);

            wglMakeCurrent(fRenderingContextDC, fRenderingContext);
            fController->imageThingy(*this);
            SwapBuffers(fRenderingContextDC);
            EndPaint(target, &eventPS);

            return 0;
        break;

        case WM_LBUTTONDOWN:
            wglMakeCurrent(fRenderingContextDC, fRenderingContext);

            canvas_h = Win32Tools::getClientAreaHeight(target);
            mouse_x = LOWORD(lparam);
            mouse_y = HIWORD(lparam);

            fController->thingyMousePressed(*this,
                MouseInteractionToken(Vector2Di(mouse_x, canvas_h -
                mouse_y), MouseInteractionToken::kLeftButton));;

            SetCapture(target);
            fIsTrackInProgress = true;

            return 0;
        break;

        case WM_LBUTTONUP:
             /* avoid extraneous mouse ups generated by the standard
                open & save dialogs */
            if (! fIsTrackInProgress)
                return DefWindowProc(target, msgcode, wparam, lparam);

            wglMakeCurrent(fRenderingContextDC, fRenderingContext);

            canvas_h = Win32Tools::getClientAreaHeight(target);
            mouse_x = LOWORD(lparam);
            mouse_y = HIWORD(lparam);

            fController->thingyMouseReleased(*this,
                MouseInteractionToken(Vector2Di(mouse_x, canvas_h -
                mouse_y), MouseInteractionToken::kLeftButton));

            ReleaseCapture( );
            fIsTrackInProgress = false;

            return 0;
        break;


        case WM_RBUTTONDOWN:
            wglMakeCurrent(fRenderingContextDC, fRenderingContext);

            canvas_h = Win32Tools::getClientAreaHeight(target);
            mouse_x = LOWORD(lparam);
            mouse_y = HIWORD(lparam);

            fController->thingyMousePressed(*this,
                MouseInteractionToken(Vector2Di(mouse_x, canvas_h -
                mouse_y), MouseInteractionToken::kRightButton));

            return 0;
        break;

        case WM_RBUTTONUP:
            wglMakeCurrent(fRenderingContextDC, fRenderingContext);

            canvas_h = Win32Tools::getClientAreaHeight(target);
            mouse_x = LOWORD(lparam);
            mouse_y = HIWORD(lparam);

            fController->thingyMouseReleased(*this,
                MouseInteractionToken(Vector2Di(mouse_x, canvas_h -
                mouse_y), MouseInteractionToken::kRightButton));

            return 0;
        break;

        case WM_MOUSEMOVE:
            wglMakeCurrent(fRenderingContextDC, fRenderingContext);

            canvas_h = Win32Tools::getClientAreaHeight(target);
            mouse_x = LOWORD(lparam);
            mouse_y = HIWORD(lparam);

            if (fIsTrackInProgress)
                fController->thingyMouseTracked(*this,
                    MouseInteractionToken(Vector2Di(mouse_x, canvas_h -
                    mouse_y), MouseInteractionToken::kLeftButton));

            return 0;
        break;

        case WM_SIZE:
            fController->thingySized(*this);
            return 0;
        break;

        default:
            return DefWindowProc(target, msgcode, wparam, lparam);
        break;
    }
}



GLCanvasController&  GLCanvasThingy::controller( )
{
    return *fController;
}



const GLCanvasController&  GLCanvasThingy::controller( ) const
{
    return *fController;
}



GLCanvasController&  GLCanvasThingy::installController(
    GLCanvasController& newctlr)
{
    GLCanvasController* oldctlr = fController;

    fController = & newctlr;

    newctlr.initializeThingy(*this);
    newctlr.thingySized(*this);

    return *oldctlr;
}



void  GLCanvasThingy::makeRenderingTarget( )
{
    wglMakeCurrent(fRenderingContextDC, fRenderingContext);
}



void  GLCanvasThingy::reimage( )
{
    wglMakeCurrent(fRenderingContextDC, fRenderingContext);
    fController->imageThingy(*this);
    SwapBuffers(fRenderingContextDC);
}


GPUInterface&  GLCanvasThingy::gpuInterface( )
{
    return *fDeviceInterface;
}
//////////////////////////////////////////////////////////////////////////////
// END  GLCanvasThingy                                                      //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  GLCanvasController                                                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class  NullGLCanvasController :
    public virtual GLCanvasController {

public:

    NullGLCanvasController( ) { }

    ~NullGLCanvasController( ) { }

    void  imageThingy(GLCanvasThingy&) { }
    void  thingySized(GLCanvasThingy&) { }
    void  initializeThingy(GLCanvasThingy&) { }
    void  thingyMousePressed(GLCanvasThingy&, const MouseInteractionToken&)
        { }
    void  thingyMouseReleased(GLCanvasThingy&, const MouseInteractionToken&)
        { }
    void  thingyMouseTracked(GLCanvasThingy&, const MouseInteractionToken&)
        { }
};


GLCanvasController*  GLCanvasController::nullController( )
{
    return new NullGLCanvasController( );
}
//////////////////////////////////////////////////////////////////////////////
// END  GLCanvasController                                                  //
//////////////////////////////////////////////////////////////////////////////
