//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CanvasWindow.cpp                                                         //
//                                                                          //
// Implements methods in class CanvasWindowView only                        //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// version 2.5.2 of Tue 20 May 2008 @ 2:16pm EDT                            //
//                                                                          //
//    added ShowWindow( ) calls immediately following calls to widget       //
//    factory creation routines; this was necessitated by changes in the    //
//    Win32Support module that caused the widget factory to create          //
//    initially hidden widgets                                              //
//                                                                          //
// version 2.5.1 of Fri 3 May 2008 @ 7:33pm EDT                             //
//                                                                          //
//    changed all references of class template Vertex2D to Vector2D; see    //
//    the AffineGeometry module's header for more info                      //
//                                                                          //
// ver. 2.5.0 of Fri 15-Feb-2008 @ 11:27pm EST                              //
//                                                                          //
//    Added progress bar support, child management support, enable-disable  //
//    functionality, and wait state support.                                //
//                                                                          //
// ver. 2.0.0 of Thu 27-Sep-2007 @ 4:59pm EDT                               //
//                                                                          //
//    This file has been around for months and has seen several evolutions  //
//    through CVS commit cycles of projects GPUISupport and PyramidBuilder, //
//    however this current version 2.0.0 is the first formal "version"      //
//    of the file and is considered stable.                                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008 Lucas Stephen Beeler. All Rights Reserved.       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "CanvasWindow.h"
#include "Application.h"

bool  CanvasWindowView::sAreWindowClassesRegistered = false;
const std::string  CanvasWindowView::kWindowClassName =
    "CanvasWindowView_window";
const std::string  CanvasWindowView::kCanvasClassName =
    "CanvasWindowView_canvas";
const Vector2Di    CanvasWindowView::kDefaultPosition = Vector2Di(100, 100);


CanvasWindowView::CanvasWindowView(const std::string& title) :
    fMainWindowPrimitive(0), fCanvasWindowPrimitive(0),
    fIsWindowVisible(false), fLeftMargin(kDefaultLeftMargin),
    fRightMargin(kDefaultRightMargin), fTopMargin(kDefaultTopMargin),
    fBottomMargin(kDefaultBottomMargin), fResizeGripWindowPrimitive(0),
    fMinHorizTrackSize(kDefaultMinHorizTrackSize),
    fMinVerticalTrackSize(kDefaultMinVerticalTrackSize),
    fInfoAreaPrimitive(0), fIsWindowEnabled(true), fProgressBarPrimitive(0),
    fIsWindowWaiting(false), fStatusRegionHeight(0),
    fStatusRgnDivPrimitive(0)
{
    if (!sAreWindowClassesRegistered) {

        sRegisterWindowClasses( );
        sAreWindowClassesRegistered = true;
    }

    /* create the main window and check to make sure it was created
       successully; if it wasn't, throw a runtime error; if it was, configure
       it */
    fMainWindowPrimitive = CreateWindowEx(0,
        kWindowClassName.c_str( ), title.c_str( ), WS_OVERLAPPEDWINDOW,
        kDefaultPosition.x, kDefaultPosition.y, kDefaultWidth,
        kDefaultHeight, NULL, NULL, NULL, 0);
    if (!fMainWindowPrimitive)
        throw std::runtime_error("CanvasWindowView: CanvasWindowView( ) failed"
            " to create main window");
    SetWindowLongPtr(fMainWindowPrimitive, GWLP_USERDATA, (LONG_PTR) this);

    /* create the inset canvas child window and check to make sure it was
       created successully; if it wasn't, throw a runtime error; if it was,
       configure it */
    fCanvasWindowPrimitive = CreateWindowEx(WS_EX_CLIENTEDGE,
        kCanvasClassName.c_str( ), TEXT(""), WS_CHILD | WS_VISIBLE, 0, 0, 10,
        10, fMainWindowPrimitive, (HMENU)(kCanvasWindowChildID), NULL, 0);
    if (!fCanvasWindowPrimitive)
        throw std::runtime_error("CanvasWindowView: CanvasWindowView( ) failed"
            " to create inset canvas window");
    SetWindowLongPtr(fCanvasWindowPrimitive, GWLP_USERDATA, LONG_PTR(this));

    /* create the resize grip control child window and check to make sure that
       it was created successully; if it wasn't, throw a runtime error; if it
       was, leave it as-is -- there's nothing to configure for it */
    fResizeGripWindowPrimitive = CreateWindow(TEXT("scrollbar"), TEXT(""),
        WS_CHILD | SBS_SIZEGRIP | WS_VISIBLE | SBS_SIZEBOXBOTTOMRIGHTALIGN,
        1, 1, 11, 11, fMainWindowPrimitive, (HMENU)(kResizeGripWindowChildID),
        NULL, 0);
    if (!fResizeGripWindowPrimitive)
        throw std::runtime_error("CanvasWindowView: CanvasWindowView( ) failed"
            " to create inset canvas window");
}




CanvasWindowView::~CanvasWindowView( )
{
    /* hide the top-level  window */
    this->hide( );

    /* destroy the canvas window first (remember, its a child of the
       main window) */
    Win32Tools::destroyPrimitive(fCanvasWindowPrimitive);

    /* disable the info area and progress bar -- this destroys their
       primitives */
    disableInfoArea( );
    disableProgressBar( );

    /* now, we can destroy the main window */
    Win32Tools::destroyPrimitive(fMainWindowPrimitive);
}





std::string  CanvasWindowView::title( ) const
{
    return Win32Tools::getWindowTextString(fMainWindowPrimitive);
}




void  CanvasWindowView::setTitle(const std::string& s)
{
    Win32Tools::setWindowTextString(fMainWindowPrimitive, s);
}




unsigned short  CanvasWindowView::clientWidth( ) const
{
    return Win32Tools::getClientAreaWidth(fMainWindowPrimitive);
}




unsigned short  CanvasWindowView::clientHeight( ) const
{
    return Win32Tools::getClientAreaHeight(fMainWindowPrimitive);
}




unsigned short  CanvasWindowView::width( ) const
{
    return Win32Tools::getWindowWidth(fMainWindowPrimitive);
}




void  CanvasWindowView::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fMainWindowPrimitive, w);
}




unsigned short  CanvasWindowView::height( ) const
{
    return Win32Tools::getWindowHeight(fMainWindowPrimitive);
}




void  CanvasWindowView::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fMainWindowPrimitive, h);
}




Vector2Di  CanvasWindowView::position( ) const
{
    RECT  mainWindowBoundsRect;
    GetWindowRect(fMainWindowPrimitive, &mainWindowBoundsRect);

    return Vector2Di(mainWindowBoundsRect.left, mainWindowBoundsRect.top);
}




void  CanvasWindowView::setPosition(const Vector2Di& pos)
{
    unsigned short w = this->width( );
    unsigned short h = this->height( );

    MoveWindow(fMainWindowPrimitive, pos.x, pos.y, w, h, TRUE);
}




HWND  CanvasWindowView::peerHandle( )
{
    return fMainWindowPrimitive;
}




unsigned short  CanvasWindowView::leftMargin( ) const
{
    return fLeftMargin;
}




void  CanvasWindowView::setLeftMargin(unsigned short lm)
{
    fLeftMargin = lm;

    uLayoutChildren( );
    invalidate( );
}




unsigned short  CanvasWindowView::rightMargin( ) const
{
    return fRightMargin;
}




void  CanvasWindowView::setRightMargin(unsigned short rm)
{
    fRightMargin = rm;
    
    uLayoutChildren( );
    invalidate( );
}




unsigned short  CanvasWindowView::topMargin( ) const
{
    return fTopMargin;
}




void  CanvasWindowView::setTopMargin(unsigned short tm)
{
    fTopMargin = tm;

    uLayoutChildren( );
    invalidate( );
}




unsigned short  CanvasWindowView::bottomMargin( ) const
{
    return fBottomMargin;
}




void  CanvasWindowView::setBottomMargin(unsigned short bm)
{
    if (isInfoAreaEnabled( ))
        if (bm < (kStatusRegionHeight))
            disableInfoArea( );

    fBottomMargin = bm;
}




void  CanvasWindowView::invalidate( )
{
    RECT  mainWindowClientRect;
    GetClientRect(fMainWindowPrimitive, &mainWindowClientRect);
    
    InvalidateRect(fMainWindowPrimitive, &mainWindowClientRect, FALSE);
    UpdateWindow(fMainWindowPrimitive);
}




LRESULT WINAPI CanvasWindowView::sWindowProc(HWND targetWindow,
    UINT messageCode, WPARAM wParam, LPARAM lParam)
{
    CanvasWindowView*  obj = 0;

    switch (messageCode) {
    
        case WM_SIZE:
        case WM_GETMINMAXINFO:
        case WM_CLOSE:
        case WM_COMMAND:
        case WM_SETCURSOR:
            obj = sGetObject(targetWindow);
            if (obj)
                return obj->uProcessWindowMessage(targetWindow, messageCode,
                    wParam, lParam);
            else
                return DefWindowProc(targetWindow, messageCode, wParam,
                    lParam);
        break;

        default:
            return DefWindowProc(targetWindow, messageCode, wParam, lParam);
        break;
    }
}




LRESULT  CanvasWindowView::uProcessWindowMessage(HWND targetWindow,
    UINT messageCode, WPARAM wParam, LPARAM lParam)
{
    MINMAXINFO*                   constraints = 0;

    switch (messageCode) {
    
        case WM_SIZE:
            uLayoutChildren( );
            return 0;
        break;

        case WM_GETMINMAXINFO:
            constraints = (MINMAXINFO*) lParam;
            constraints->ptMinTrackSize.x = fMinHorizTrackSize;
            constraints->ptMinTrackSize.y = fMinVerticalTrackSize;
            return 0;
        break;

        case WM_SETCURSOR:
            if (isWaiting( )) {

                HCURSOR hourglass = LoadCursor(0, IDC_WAIT);

                SetCursor(hourglass);

                return TRUE;
            }
            else {

                return DefWindowProc(targetWindow, messageCode, wParam,
                    lParam);
            }
        break;

        default:
            return 0;
        break;
    }
}




LRESULT WINAPI CanvasWindowView::sCanvasProc(HWND targetWindow,
    UINT messageCode, WPARAM wParam, LPARAM lParam)
{
    CanvasWindowView* obj = 0;

    switch (messageCode) {
    
        case WM_SIZE:
        case WM_PAINT:
        case WM_LBUTTONDOWN:
        case WM_RBUTTONDOWN:
        case WM_MOUSEMOVE:
        case WM_LBUTTONUP:
        case WM_RBUTTONUP:
            obj = sGetObject(targetWindow);
            if (obj)
                return obj->uProcessCanvasMessage(targetWindow, messageCode,
                    wParam, lParam);
            else
                return 0;
        break;
    
        default:
            return DefWindowProc(targetWindow, messageCode, wParam, lParam);
        break;
    }
}



LRESULT  CanvasWindowView::uProcessCanvasMessage(HWND targetWindow,
    UINT messageCode, WPARAM wParam, LPARAM lParam)
{
    HDC          canvasDC;
    PAINTSTRUCT  eventPS;

    switch (messageCode) {
    
        case WM_SIZE:
            return 0;
        break;

        case WM_PAINT:
            canvasDC = BeginPaint(targetWindow, &eventPS);
            EndPaint(targetWindow, &eventPS);
            return 0;
        break;
        
        default:
            return 0;
        break;
    }
}




void  CanvasWindowView::sRegisterWindowClasses( )
{
    WNDCLASSEX  windowClass;
    WNDCLASS    canvasClass;

    windowClass.cbSize = sizeof(WNDCLASSEX);
    windowClass.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
    windowClass.lpfnWndProc = CanvasWindowView::sWindowProc;
    windowClass.cbClsExtra = 0;
    windowClass.cbWndExtra = sizeof(CanvasWindowView*);
    windowClass.hInstance = NULL;
    windowClass.hIcon = NULL;
    windowClass.hCursor = LoadCursor(NULL, IDC_ARROW);
    windowClass.hbrBackground = (HBRUSH)(COLOR_BTNFACE + 1);
    windowClass.lpszMenuName = NULL;
    windowClass.lpszClassName = kWindowClassName.c_str( );
    windowClass.hIconSm = NULL;

    canvasClass.style = CS_OWNDC;
    canvasClass.lpfnWndProc = CanvasWindowView::sCanvasProc;
    canvasClass.cbClsExtra = 0;
    canvasClass.cbWndExtra = sizeof(CanvasWindowView*);
    canvasClass.hInstance = NULL;
    canvasClass.hIcon = NULL;
    canvasClass.hCursor = LoadCursor(NULL, IDC_ARROW);
    canvasClass.hbrBackground = NULL;
    canvasClass.lpszMenuName = NULL;
    canvasClass.lpszClassName = kCanvasClassName.c_str( );

    ATOM  windowResult = RegisterClassEx(&windowClass);
    if (!windowResult)
        throw std::runtime_error("CanvasWindowView: sRegisterWindowClasses( )"
            " failed to register main window class");

    ATOM  canvasResult = RegisterClass(&canvasClass);
    if (!canvasResult)
        throw std::runtime_error("CanvasWindowView: sRegisterWindowClasses( )"
            " failed to register canvas window class");
}




CanvasWindowView*  CanvasWindowView::sGetObject(HWND w)
{
    LONG_PTR  rawResult = GetWindowLongPtr(w, GWLP_USERDATA);

    return  (CanvasWindowView*)(rawResult);
}




void  CanvasWindowView::show( )
{
    ShowWindow(fMainWindowPrimitive, SW_SHOWNORMAL);
    fIsWindowVisible = true;
    invalidate( );
}




void  CanvasWindowView::hide( )
{
    ShowWindow(fMainWindowPrimitive, SW_HIDE);
    fIsWindowVisible = false;
}




bool  CanvasWindowView::isVisible( ) const
{
    return fIsWindowVisible;
}




unsigned short  CanvasWindowView::minHorizTrackSize( ) const
{
    return fMinHorizTrackSize;
}




void  CanvasWindowView::setMinHorizTrackSize(unsigned short mhs)
{
    fMinHorizTrackSize = mhs;
}




unsigned short  CanvasWindowView::minVerticalTrackSize( ) const
{
    return fMinVerticalTrackSize;
}




void  CanvasWindowView::setMinVerticalTrackSize(unsigned short mvs)
{
    fMinVerticalTrackSize = mvs;
}




void  CanvasWindowView::uLayoutChildren( )
{
    RECT  mainWindowRect;

    /* get the main window size information */
    GetClientRect(fMainWindowPrimitive, &mainWindowRect);

    /* reposition the canvas child window appropriately, given the
       size information */
    MoveWindow(fCanvasWindowPrimitive, fLeftMargin, fTopMargin,
        mainWindowRect.right - fRightMargin - fLeftMargin,
        mainWindowRect.bottom - fBottomMargin - fTopMargin, TRUE);

    /* reposition the size grip child window appropriately, given the
       size information */
    MoveWindow(fResizeGripWindowPrimitive, mainWindowRect.right -
        kSizeGripPixelSize, mainWindowRect.bottom -
        kSizeGripPixelSize, kSizeGripPixelSize, kSizeGripPixelSize,
        TRUE);

    /* if the progress bar is enabled, then set its horizontal frame size
       to its width plus some spacing pixels; if it isn't enabled, then set
       its horizontal frame size to zero */
    int pb_hframe;
    if (isProgressBarEnabled( ))
        pb_hframe = Win32Tools::getWindowWidth(fProgressBarPrimitive) + 4;
    else
        pb_hframe = 0;

    /* if the progress bar is enabled, then compute its metrics and position
       it */
    if (isProgressBarEnabled( )) {

        const int pbw = Win32Tools::getWindowWidth(fProgressBarPrimitive);
        const int pbh = Win32Tools::getWindowHeight(fProgressBarPrimitive);

        Win32Tools::setWindowOrigin(fProgressBarPrimitive,
            mainWindowRect.right - 16 - pbw,
            mainWindowRect.bottom -  (fStatusRegionHeight / 2) - (pbh / 2));
    }

    /* if the info area is enabled, then reposition it too */
    if (isInfoAreaEnabled( )) {

        int infarea_ht = Win32Tools::getWindowHeight(fInfoAreaPrimitive);

        MoveWindow(fInfoAreaPrimitive, 16, mainWindowRect.bottom -
            (fStatusRegionHeight / 2) - (infarea_ht / 2),
            mainWindowRect.right - 16 - 16 - pb_hframe,
            infarea_ht, TRUE);
    }

    /* if either the progress bar or info area are enabled, then place the
       status region divider */
    if (isInfoAreaEnabled( ) || isProgressBarEnabled( )) {

        Win32Tools::setWindowOrigin(fStatusRgnDivPrimitive, 0,
            mainWindowRect.bottom - fStatusRegionHeight);
        Win32Tools::setWindowWidth(fStatusRgnDivPrimitive,
            mainWindowRect.right + 2);
    }

}




unsigned short  CanvasWindowView::canvasClientWidth( ) const
{
    return Win32Tools::getClientAreaWidth(fCanvasWindowPrimitive);
}




unsigned short  CanvasWindowView::canvasClientHeight( ) const
{
    return Win32Tools::getClientAreaHeight(fCanvasWindowPrimitive);
}




unsigned short  CanvasWindowView::canvasWidth( ) const
{
    return Win32Tools::getWindowWidth(fCanvasWindowPrimitive);
}




unsigned short  CanvasWindowView::canvasHeight( ) const
{
    return Win32Tools::getWindowHeight(fCanvasWindowPrimitive);
}




HWND  CanvasWindowView::canvasPeerHandle( )
{
    return fCanvasWindowPrimitive;
}




void  CanvasWindowView::enableInfoArea( )
{
    // if the info area is already enabled, do nothing
    if (isInfoAreaEnabled( ))
        return;

    if (bottomMargin( ) >= kStatusRegionHeight) {

        fInfoAreaPrimitive = Win32Tools::widgetFactory( ).makeLabel(
            fMainWindowPrimitive, kInfoAreaChildID);
        ShowWindow(fInfoAreaPrimitive, SW_SHOW);

        if (!isProgressBarEnabled( ))
            uEnableStatusRegion( );

        uLayoutChildren( );
    }
    else {

        throw std::logic_error("CanvasWindowView: enableInfoArea( ) bottom"
            " margin is too small to create info area child window");
    }
}




void  CanvasWindowView::disableInfoArea( )
{
    // if the info area isn't enabled, do nothing
    if (!isInfoAreaEnabled( ))
        return;

    Win32Tools::destroyPrimitive(fInfoAreaPrimitive);

    if (!isProgressBarEnabled( ))
        uDisableStatusRegion( );
}




bool  CanvasWindowView::isInfoAreaEnabled( ) const
{
    if (fInfoAreaPrimitive)
        return true;
    else
        return false;
}




std::string  CanvasWindowView::infoAreaText( ) const
{
    if (!isInfoAreaEnabled( ))
        throw std::logic_error("CanvasWindowView: infoAreaText( ) called,"
            " but info area isn't enabled");

    return  Win32Tools::getWindowTextString(fInfoAreaPrimitive);
}




void  CanvasWindowView::setInfoAreaText(const std::string& s)
{
    if (!isInfoAreaEnabled( ))
        throw std::logic_error("CanvasWindowView: infoAreaText( ) called,"
            " but info area isn't enabled");

    Win32Tools::setWindowTextString(fInfoAreaPrimitive, s);
}




void  CanvasWindowView::enableProgressBar( )
{
    /* if the progress bar is already enabled, then do nothing */
    if (isProgressBarEnabled( ))
        return;

    /* check to see if the user has left enough margin space for us to
       create the progress bar -- if he has, then go ahead and create it;
       if he hasn't, then throw a logic_error */
    if (bottomMargin( ) >= kStatusRegionHeight) {

        fProgressBarPrimitive = Win32Tools::widgetFactory( ).makeProgressBar(
            fMainWindowPrimitive, kProgressBarChildID);
        ShowWindow(fProgressBarPrimitive, SW_SHOW);

        Win32Tools::applyControlFaceProperties(fProgressBarPrimitive);

        if (!isInfoAreaEnabled( ))
            uEnableStatusRegion( );

        uLayoutChildren( );
    }
    else {

        throw std::logic_error("CanvasWindowView: enableProgressBar( ): "
            "bottom margin is too small to create child window");
    }
}




void  CanvasWindowView::disableProgressBar( )
{
    if (!isProgressBarEnabled( ))
        return;

    Win32Tools::destroyPrimitive(fProgressBarPrimitive);

    if (!isInfoAreaEnabled( ))
        uDisableStatusRegion( );
}




bool  CanvasWindowView::isProgressBarEnabled( ) const
{
    if (fProgressBarPrimitive)
        return true;
    else
        return false;
}




void  CanvasWindowView::setPercentProgress(int pctprog)
{
    if (!isProgressBarEnabled( ))
        return;

    SendMessage(fProgressBarPrimitive, PBM_SETPOS, pctprog, 0);
}




void  CanvasWindowView::enable( )
{
    std::set<HWND>::iterator it;
    for (it = fAutoEnableSet.begin( ); it != fAutoEnableSet.end( ); it++)
        EnableWindow(*it, TRUE);
    fAutoEnableSet.clear( );

    fIsWindowEnabled = true;
}




void  CanvasWindowView::disable( )
{
    fIsWindowEnabled = false;

    fAutoEnableSet.clear( );
    std::set<HWND>::iterator it;
    for (it = fChildSet.begin( ); it != fChildSet.end( ); it++) {

        if (IsWindowEnabled(*it)) {

            fAutoEnableSet.insert(*it);
            EnableWindow(*it, FALSE);
        }
    } /* for */
}




bool  CanvasWindowView::isEnabled( ) const
{
    return fIsWindowEnabled;
}




void  CanvasWindowView::manageChild(HWND c)
{
    fChildSet.insert(c);
}




void  CanvasWindowView::unmanageChild(HWND c)
{
    if (isChildManaged(c))
        fChildSet.erase(c);
}




bool  CanvasWindowView::isChildManaged(HWND c) const
{
    if (fChildSet.find(c) != fChildSet.end( ))
        return true;
    else
        return false;
}




void  CanvasWindowView::wait( )
{
    SetCursor(LoadCursor(0, IDC_WAIT));

    fAutoEnableSet.clear( );
    std::set<HWND>::iterator it;
    for (it = fChildSet.begin( ); it != fChildSet.end( ); it++) {

        if (IsWindowEnabled(*it)) {

            fAutoEnableSet.insert(*it);
            EnableWindow(*it, FALSE);
        }
    } /* for */

    fIsWindowWaiting = true;
}




void  CanvasWindowView::unwait( )
{
    SetCursor(LoadCursor(0, IDC_ARROW));

    std::set<HWND>::iterator it;
    for (it = fAutoEnableSet.begin( ); it != fAutoEnableSet.end( ); it++)
        EnableWindow(*it, TRUE);
    fAutoEnableSet.clear( );

    fIsWindowWaiting = false;
}




bool  CanvasWindowView::isWaiting( ) const
{
    return fIsWindowWaiting;
}



unsigned short  CanvasWindowView::statusRegionHeight( ) const
{
    return fStatusRegionHeight;
}




void  CanvasWindowView::uEnableStatusRegion( )
{
    fStatusRegionHeight = kStatusRegionHeight;

    fStatusRgnDivPrimitive =
        Win32Tools::widgetFactory( ).makeHorizontalSeparator(
        fMainWindowPrimitive, kStatusRegionDividerChildID);
    ShowWindow(fStatusRgnDivPrimitive, SW_SHOW);
}




void  CanvasWindowView::uDisableStatusRegion( )
{
    fStatusRegionHeight = 0;

    Win32Tools::destroyPrimitive(fStatusRgnDivPrimitive);
}
