//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Application.cpp                                                          //
//                                                                          //
// Implements methods in class Application and in the ApplicationController //
// interface. Defines the NullApplicationController class and implements    //
// methods in it.                                                           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 1.3.0 of Fri 07-Nov-2008 @ 6:39pm EDT                               //
//                                                                          //
//     Added accelerator key focus window management                        //
//                                                                          //
// ver. 1.2.0 of Fri 24-Oct-2008 @ 6:12pm EDT                               //
//                                                                          //
//     Application objects now maintain the path of the directory in which  //
//     they were launched, and provide a method for clients to access this  //
//     information.                                                         //
//                                                                          //
// ver. 1.1.1 of Mon 12-May-2008 @ 7:19pm EDT                               //
//                                                                          //
//     Changed const qualifiers on intersitial action related methods       //
//     in accordance with the interface changes in the UserInteraction      //
//     module                                                               //
//                                                                          //
// older change history elided; check out an older CVS rev to get it        //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "Application.h"
#include "TranscriptionServer.h"
#include "EssentialThingies.h"
#include <cstdlib>
#include <stdexcept>


Application*  Application::sSingletonInstance = 0;


//
// CLASS  NullApplicationController
//

class NullApplicationController : public virtual ApplicationController {

public:

    NullApplicationController( )
    {
    }

    ~NullApplicationController( )
    {
    }

    void  installed(Application& sender)
    {
    }

    void  deinstalled(Application& sender)
    {
    }

    void  applicationStarted(Application& sender)
    {
    }

    void  applicationStopped(Application& sender)
    {
    }

    bool  applicationStopRequested(Application& sender)
    {
        return true;
    }
};








//
// CLASS Application
//

Application::Application(const std::string& appname, HINSTANCE sysmod)
    : fIsRunning(false), fController(new NullApplicationController( )),
      fApplicationName(appname),
      fTransServer(new TranscriptionServer(appname)), fSysModule(sysmod),
      fKeyFocus(0)
{
    // if an Application instance already exists, throw a logic_error: any
    // given session of program execution can have at most one Application
    // instance that models that execution session
    if (sSingletonInstance)
        throw std::logic_error("Application: attempt to instantiate more than"
            "one Application object");

    // setup support for the common controls library
    INITCOMMONCONTROLSEX  controlUsageDesc;
    controlUsageDesc.dwSize = sizeof(INITCOMMONCONTROLSEX);
    controlUsageDesc.dwICC = ICC_WIN95_CLASSES;
    InitCommonControlsEx(&controlUsageDesc);

    fStartupDir = Win32Tools::filesystem( ).cwd( );

    sSingletonInstance = this;
}




Application::~Application( )
{
    delete fTransServer;

    if (typeid(*fController) == typeid(NullApplicationController))
        delete fController;
}




/** places the application into the running state and begins its main event
    loop. The event loop will continue to run (and the application will
    continue to execute) until terminate( ) is called. */
void  Application::start( )
{
    fIsRunning = true;

    fController->applicationStarted(*this);

    while (fIsRunning) {
    
        MSG  currentMessage;
        
        GetMessage(&currentMessage, NULL, 0, 0);

        if (isKeyFocusSet( )) {

            const ApplicationWindowThingy& focuswin = keyFocus( );

            if (focuswin.areKeysInstalled( )) {

                HACCEL  keys = focuswin.installedKeys( );
                
                ApplicationWindowThingy& focuswin_mutable =
                    const_cast<ApplicationWindowThingy&>(focuswin);

                HWND  focuswin_primitive =
                    focuswin_mutable.adapter( ).primitive( );

                if (TranslateAccelerator(focuswin_primitive, keys,
                    &currentMessage))
                        continue;
            } /* if the focus window has keys installed */
        } /* if the accelerator key focus is set to some window */
        
        TranslateMessage(&currentMessage);
        DispatchMessage(&currentMessage);
    }
}



/** terminates program execution immediately, does not notify the installed
    application controller and does not release any acquired system resources;
    upon termination, returns "resultCode" to the host operating system. */
void  Application::terminate(int resultCode)
{
    std::exit(resultCode);
}




/** gracefully stops program execution by first requesting permission from
    the currently installed controller object to stop execution (this
    gives the controller the opportunity to ask the user if he's sure he
    wants to quit, etc.). If the controller object denies the stop request,
    this method returns immediately without any effect. If, however, the
    controller object confirms the request, system resources are freed,
    the controller is notified that the application has stopped,
    and terminate(0) is called */
void  Application::stop( )
{
    /* if the currently installed controller object denies the stop request,
       then just return immediately without any effect */
    if (!fController->applicationStopRequested(*this))
        return;

    fIsRunning = false;

    fController->applicationStopped(*this);

    this->terminate(0);
}



/** returns a reference to the singleton instance to the caller */
Application& Application::instance( )
{
    if (!sSingletonInstance)
        throw std::logic_error("Application: instance( ) invoked but no "
            "application instance has yet been created");
    
    return *sSingletonInstance;
}




/** returns true if the application has been started, false otherwise */
bool  Application::isRunning( )
{
    return fIsRunning;
}




/** returns a reference to the currently installed controller */
ApplicationController&  Application::controller( )
{
    return *fController;
}




/** installs 'ctlr' to serve as the new applictation controller for this
    application, then returns a reference to whatever object was
    previously installed */
ApplicationController&  Application::installController(
    ApplicationController& ctlr)
{
    ApplicationController*  prevController = fController;
    prevController->deinstalled(*this);

    fController = & ctlr;
    fController->installed(*this);

    return *prevController;
}



/** returns a reference to the TranscriptionServer instance used by this
    application to transcribe error and diagnostic information at runtime */
TranscriptionServer&  Application::transcriptionServer( )
{
    return *fTransServer;
}




/** returns a handle to this Win32 executable module in which this
    application is running */
HINSTANCE  Application::systemModule( )
{
    return fSysModule;
}




/** runs the application's event loop in response to an interstitial action
    event */
void  Application::interstitialActionEvent(InterstitialActionSender& sender,
    const InterstitialActionToken& evt)
{    
    MSG  currentMessage;

    while (PeekMessage(&currentMessage, NULL, 0, 0, PM_REMOVE)) {
    
        TranslateMessage(&currentMessage);
        DispatchMessage(&currentMessage);
    }
}




/** returns the name of the application */
const std::string&  Application::name( ) const
{
    return fApplicationName;
}



/** returns the complete path to the directory where the application was
    launched */
const std::string&  Application::startupDirectory( ) const
{
    return fStartupDir;
}



/** make 'target' the window that will be notified when an accelerator key
    combination (ex: Ctrl+X for Exit Application) is pressed */
void  Application::setKeyFocus(const ApplicationWindowThingy& target)
{
    fKeyFocus = &target;
}



/** clears the current key focus window if one has been previously set via
    a call to setKeyFocus; if the current key focus is clear, no notification
    is sent when an accelerator key combination is pressed */
void  Application::clearKeyFocus( )
{
    fKeyFocus = 0;
}



/** returns true if the accelerator key focus is currently set to some
    window, and false if the key focus is currently clear */
bool  Application::isKeyFocusSet( ) const
{
    if (fKeyFocus)
        return true;
    else
        return false;
}



/** returns a reference to the window that currently has the accelerator
    key focus, or throws a logic_error exception if no key focus window
    is currently set */
const ApplicationWindowThingy&  Application::keyFocus( ) const
{
    if (! isKeyFocusSet( ))
        throw std::logic_error("Application: can't get accelerator key "
            "focus window: focus window is currently set");

    return *fKeyFocus;
}










//
// CLASS ApplicationController
//

ApplicationController::ApplicationController( )
{
}




ApplicationController::~ApplicationController( )
{
}




ApplicationController*  ApplicationController::nullController( )
{
    return new NullApplicationController( );
}
