//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// AffineGeometry.h                                                         //
//                                                                          //
// Packages the Vector2D LineSegment2D, and Rectangle2D class templates     //
// and defines named types for int and float instanatations of them. Also   //
// included in this file are constant definitions for numerical             //
// constants important in graphics (e.g. Pi) as well as definitions for     //
// inline functions and function templates useful for graphics programming  //
// (i.e. radian-degree interconversion functions and a function template to //
// perform linear interpolation over any arbitrary type supporting the      //
// addition and scalar-multiply operators).                                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 1.6.4 of Fri 8-Aug-2008 @ 1:17pm EDT                                //
//                                                                          //
//     new:    none                                                         //
//     fixed:  fixed operator!= in class template Vector2D                  //
//                                                                          //
// ver. 1.6.3 of Wed 6-Aug-2008 @ 3:26pm EDT                                //
//                                                                          //
//     new:    added operator*= to class template Vector2D                  //
//     fixed:  none                                                         //
//                                                                          //
// ver. 1.6.2 of Wed 9-July-2008 @ 11:27pm EDT                              //
//                                                                          //
//     new:    added component-wise ceil( ) function template for vectors   //
//     fixed:  none                                                         //
//                                                                          //
// ver. 1.6.1 of Wed 18-June-2008 @ 12:42pm EDT                             //
//                                                                          //
//     new:    added component-wise floor( ) function template for vectors  //
//     fixed:  none                                                         //
//                                                                          //
// change history for older versions elided; check out an older cvs rev     //
// to get it                                                                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef AFFINE_GEOMETRY__HXX
#define AFFINE_GEOMETRY__HXX

#include <stdexcept>
#include <cmath>
#include <climits>

//-------------------------------------------------------------------------//
// NUMERICAL CONSTANTS                                                     //
//-------------------------------------------------------------------------//
const float kPi = 3.14159265f;
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// INCOMPLETE TYPE SPECIFICATIONS                                          //
//-------------------------------------------------------------------------//
template<typename TComponent>  class Vector2D;
template<typename TComponent>  class LineSegment2D;
template<typename TComponent>  class Rectangle2D;
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// BASIC GRAPHICS & MATH FUNCTIONS                                         //
//-------------------------------------------------------------------------//
template<typename TObj, typename TMult>
inline  TObj  interpolate(const TObj& o1, const TObj& o2, const TMult& w2)
{
    TMult  w1 = static_cast<TMult>(1.0) - w2;

    return (o1 * w1) + (o2 * w2);
}




inline  float  rad2deg(float rad)
{
    return (rad / kPi) * 180.0f;
}




inline  float  deg2rad(float deg)
{
    return (deg / 180.0f) * kPi;
}



inline  int   clamp(int val, int valmin, int valmax)
{
    if (val < valmin)
        val = valmin;
    else if (val > valmax)
        val = valmax;

    return val;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS TEMPLATE  Vector2D                                                //
//-------------------------------------------------------------------------//
template<typename TComponent>
class  Vector2D {

public:

    TComponent x;
    TComponent y;

    Vector2D( )
        : x(static_cast<TComponent>(0)), y(static_cast<TComponent>(0))
    {
    }

    Vector2D(TComponent userX, TComponent userY)
        : x(userX), y(userY)
    {
    }

    const TComponent&  operator[ ](unsigned i) const
    {
        if (i == 0)
            return x;
        else if (i == 1)
            return y;
        else
            throw std::logic_error("Vector2D: operator[ ] index out of "
                "range");
    }

    TComponent&  operator[ ](unsigned i)
    {
        if (i == 0)
            return x;
        else if (i == 1)
            return y;
        else
            throw std::logic_error("Vector2D: operator[ ] index out of "
                "range");
    }

    Vector2D<TComponent>  operator-( ) const
    {
        return Vector2D<TComponent>(-x, -y);
    }

    Vector2D<TComponent>  operator/(const TComponent& divisor) const
    {
        return Vector2D<TComponent>(x / divisor, y / divisor);
    }

    bool  operator==(const Vector2D<TComponent>& rhs) const
    {
        if (x == rhs.x)
            if (y == rhs.y)
                return true;
        return false;
    }

    bool  operator!=(const Vector2D<TComponent>& rhs) const
    {
        if (! (*this == rhs))
            return true;
        else
            return false;
    }

    template<typename TRHS>
    bool  operator==(const Vector2D<TRHS>& rhs) const
    {
        if (x == rhs.x)
            if (y == rhs.y)
                return true;
        return false;
    }

    template<typename TRHS>
    bool  operator!=(const Vector2D<TRHS>& rhs) const
    {
        if !(*this == rhs)
            return true;
        else
            return false;
    }

    TComponent  length( ) const
    {
        return static_cast<TComponent>(std::sqrt((x * x) + (y * y)));
    }

    Vector2D<TComponent>  normalized( ) const
    {
        TComponent  len = length( );

        return Vector2D<TComponent>(x / len, y / len);
    }

    Vector2D<TComponent>  perp( ) const
    {
        return Vector2D<TComponent>(-y, x);
    }

    const Vector2D<TComponent>&  operator*=(const TComponent& scalar)
    {
        x = scalar * x;
        y = scalar * y;

        return *this;
    }
};
//-------------------------------------------------------------------------//





//-------------------------------------------------------------------------//
// ARITHMETIC OPERATIONS ON VECTORS                                        //
//-------------------------------------------------------------------------//
/* add two vectors to get a vector */
template<typename TComponent>
Vector2D<TComponent>  operator+(const Vector2D<TComponent>& v1,
    const Vector2D<TComponent>& v2)
{
    return Vector2D<TComponent>(v1.x + v2.x, v1.y + v2.y);
}




/* subtract one vector from another vector to get a vector */
template<typename TComponent>
Vector2D<TComponent>  operator-(const Vector2D<TComponent>& v1,
    const Vector2D<TComponent>& v2)
{
    return Vector2D<TComponent>(v1.x - v2.x, v1.y - v2.y);
}




/* vector-scalar pre-multiply */
template<typename TComponent, typename TScalar>
Vector2D<TComponent>  operator*(const TScalar& s,
    const Vector2D<TComponent>& v)
{
    return Vector2D<TComponent>(s * v.x, s * v.y);
}




/* vector-scalar post-multiply */
template<typename TComponent, typename TScalar>
Vector2D<TComponent>  operator*(const Vector2D<TComponent>& v,
    const TScalar& s)
{
    return Vector2D<TComponent>(s * v.x, s * v.y);
}




/* vector divide-by-scalar */ 
template<typename TComponent, typename TScalar>
Vector2D<TComponent>  operator/(const Vector2D<TComponent>& v,
    const TScalar& s)
{
    static  const TComponent  one = static_cast<TComponent>(1);
    TComponent  inv = one / static_cast<TComponent>(s);

    return Vector2D<TComponent>(inv * v.x, inv * v.y);
}




/* vector dot product */
template<typename TComponent>
TComponent  dot(const Vector2D<TComponent>& vec1,
    const Vector2D<TComponent>& vec2)
{
    return (vec1.x * vec2.x) + (vec1.y * vec2.y);
}




/* angle between two vectors (in radians) */
template<typename TComponent>
TComponent  angle(const Vector2D<TComponent>& vec1,
    const Vector2D<TComponent>& vec2)
{
    static const TComponent result_min = static_cast<TComponent>(0.0);
    static const TComponent result_max =
        static_cast<TComponent>(3.141592653589793);

    TComponent  result =
        std::acos(dot(vec1, vec2) / (vec1.length( ) * vec2.length( )));

    if (result < result_min)
        result = result_min;
    else if (result > result_max)
        result = result_max;

    return result;
}



/* average of two vectors */
template<typename TComponent>
Vector2D<TComponent>  average(const Vector2D<TComponent>& vec1,
    const Vector2D<TComponent>& vec2)
{
    return Vector2D<TComponent>((vec1.x + vec2.x) * 0.5,
        (vec1.y + vec2.y) * 0.5);
}


/* component-wise floor function for vectors */
template<typename TComponent>
Vector2D<TComponent>  floor(const Vector2D<TComponent>& invec)
{
    Vector2D<TComponent> result;

    result.x = static_cast<TComponent>(std::floor(invec.x));
    result.y = static_cast<TComponent>(std::floor(invec.y));

    return result;
}




/* component-wise ceiling function for vectors */
template<typename TComponent>
Vector2D<TComponent>  ceil(const Vector2D<TComponent>& invec)
{
    Vector2D<TComponent> result;

    result.x = static_cast<TComponent>(std::ceil(invec.x));
    result.y = static_cast<TComponent>(std::ceil(invec.y));

    return result;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS TEMPLATE  LineSegment2D                                           //
//-------------------------------------------------------------------------//
template<typename TComponent>
class  LineSegment2D {

public:

    static  const  Vector2D<TComponent>  kNoUniqueIntersection;

    Vector2D<TComponent>  v0;
    Vector2D<TComponent>  v1;

    LineSegment2D( )
        : v0(static_cast<TComponent>(0), static_cast<TComponent>(0)),
          v1(static_cast<TComponent>(1), static_cast<TComponent>(1))
    {
    }

    LineSegment2D(const Vector2D<TComponent>& userV0,
        const Vector2D<TComponent>& userV1)
        : v0(userV0), v1(userV1)
    {
    }

    LineSegment2D(const TComponent& v0x, const TComponent& v0y,
        const TComponent& v1x, const TComponent& v1y)
        : v0(v0x, v0y), v1(v1x, v1y)
    {
    }


    const Vector2D<TComponent>&  operator[ ](unsigned i) const
    {
        if (i == 0)
            return v0;
        else if (i == 1)
            return v1;
        else
            throw std::logic_error("LineSegment2D: operator[ ] index out "
                "of range");
    }

    Vector2D<TComponent>&  operator[ ](unsigned i)
    {
        if (i == 0)
            return v0;
        else if (i == 1)
            return v1;
        else
            throw std::logic_error("LineSegment2D: operator[ ] index out "
                "of range");
    }

    bool  operator==(const LineSegment2D<TComponent>& rhs) const
    {
        if (v0 == rhs.v0)
            if (v1 == rhs.v1)
                return true;

        return false;
    }

    bool  operator!=(const LineSegment2D<TComponent>& rhs) const
    {
        return !(*this == rhs);
    }

    Vector2D<TComponent>  displacement( ) const
    {
        return v1 - v0;
    }

    TComponent  length( ) const
    {
        return displacement( ).length( );
    }

    LineSegment2D<TComponent>  reversed( ) const
    {
        return LineSegment2D<TComponent>(this->v1, this->v0);
    }

    LineSegment2D<TComponent>  average(const LineSegment2D<TComponent>&
        other) const
    {
        return LineSegment2D<TComponent>((this->v0 + other.v0) * 0.5f,
            (this->v1 + other.v1) * 0.5f);
    }

    Vector2D<TComponent>  intersect(const LineSegment2D<TComponent>& other)
    {
        /* determine if this and other are parallel */
        Vector2D<TComponent>  thisdisp = this->displacement( );
        Vector2D<TComponent>  otherdisp = other.displacement( );
        Vector2D<TComponent>  od_perp = otherdisp.perp( );
        Vector2D<TComponent>  thisd_perp = thisdisp.perp( );

        if (dot(thisdisp, od_perp) == static_cast<TComponent>(0))
            return kNoUniqueIntersection;

        /* if they aren't parallel, then execution has reached this point,
           so determine if they intersect, and if so, where */
        Vector2D<TComponent>  nav =  this->v0 - other.v0;
        Vector2D<TComponent>  nav_rev = other.v0 - this->v0;
        TComponent  sparam = dot(-od_perp, nav) / dot(od_perp, thisdisp);
        TComponent  tparam = dot(-thisd_perp, nav_rev) / dot(thisd_perp,
            otherdisp);
        
        if ((sparam < static_cast<TComponent>(0.0f)) ||
            (sparam > static_cast<TComponent>(1.0f)) ||
            (tparam < static_cast<TComponent>(0.0f)) ||
            (tparam > static_cast<TComponent>(1.0f)))
                return kNoUniqueIntersection;
        else
                return interpolate(v0, v1, sparam);
    }

    LineSegment2D<TComponent>  operator-( ) const
    {
        return LineSegment2D<TComponent>(this->v1, this->v0);
    }
};

template<typename TComponent>
const  Vector2D<TComponent> 
    LineSegment2D<typename TComponent>::kNoUniqueIntersection =
        Vector2D<TComponent>(static_cast<TComponent>(INT_MAX),
        static_cast<TComponent>(INT_MAX));
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS TEMPLATE  Rectangle2D                                             //
//-------------------------------------------------------------------------//
template<typename TComponent>
class  Rectangle2D {

public:

    Vector2D<TComponent>  origin;
    TComponent            width;
    TComponent            height;

    Rectangle2D( )
        : origin(Vector2D<TComponent>(0, 0)), width(0), height(0)
    {
    }

    Rectangle2D(const Vector2D<TComponent>& client_ogn,
        const TComponent& client_wd, const TComponent& client_ht)
            : origin(client_ogn), width(client_wd), height(client_ht)
    {
    }

    Rectangle2D(const TComponent& client_x, const TComponent& client_y,
        const TComponent& client_wd, const TComponent& client_ht)
            : origin(Vector2D<TComponent>(client_x, client_y)),
              width(client_wd), height(client_ht)
    {
    }

    Rectangle2D<TComponent>  scale(int hfactor, int vfactor) const
    {
        return Rectangle2D<TComponent>(origin,
            static_cast<TComponent>(static_cast<int>(width) * hfactor),
            static_cast<TComponent>(static_cast<int>(height) * vfactor));
    }

    Rectangle2D<TComponent>  scale(float hfactor, float vfactor) const
    {
        return Rectangle2D<TComponent>(origin,
            static_cast<TComponent>(static_cast<float>(width) * hfactor),
            static_cast<TComponent>(static_cast<float>(height) * vfactor));
    }

    Rectangle2D<TComponent>  scale(double hfactor, double vfactor) const
    {
        return Rectangle2D<TComponent>(origin,
            static_cast<TComponent>(static_cast<double>(width) * hfactor),
            static_cast<TComponent>(static_cast<double>(height) * vfactor));
    }
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CONVENIENCE TYPEDEFS                                                    //
//-------------------------------------------------------------------------//
typedef Vector2D<int> Vector2Di;
typedef Vector2D<float> Vector2Df;
typedef LineSegment2D<int> LineSegment2Di;
typedef LineSegment2D<float> LineSegment2Df;
typedef Rectangle2D<int> Rectangle2Di;
typedef Rectangle2D<float> Rectangle2Df;
//-------------------------------------------------------------------------//

#endif
