/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// S580MainWindow.cpp                                                      //
//                                                                         //
// Implements methods in class S580MainWinController only                  //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  <unversioned module>                                                   //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Copyright (c) 2008 Lucas Stephen Beeler. All rights reserved.           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "S580MainWindow.h"
#include "S580Application.h"
#include "Brushes.h"
#include "Erasers.h"
#include "resource.h"
#include <sstream>

const RGBColor  S580MainWinController::kCanvasGutterColor =
    RGBColor(0.8f, 0.8f, 0.8f);

S580MainWinController::S580MainWinController(S580AppController& parent)
    : fParent(& parent), fIsScrollTrackInProgress(false),
      fIsPixWriteRequired(false), fIsEraseOpRequired(false),
      fIsBrushDrawRequired(false)
{
    /* create the managed view */
    fView.reset(new ApplicationWindowThingy( ));

    /* set the view's title to the name of the application */
    view( ).setTitle("Studio580");

    /* install the menu specific to Studio580 into the view */
    view( ).installMenu(kS580MainWinMenuID);

    /* install the accelerator table */
    view( ).installKeys(kS580MainWinAccelTableID);
    Application::instance( ).setKeyFocus(view( ));

    /* set up the toolbar */
    view( ).enableToolbar( );
    uConfigureToolbar( );

    /* synchronize the menu state so that the items that the user can
       choose in the menu bar reflects the current state of the
       application */
    uSyncMenuState( );

    /* divide the status bar into the brush information and eraser information
       panes and sync the text to reflect the current state */
    fBrushStatPane = view( ).appendStatusPane(384);
    fEraserStatPane = view( ).appendStatusPane(384);
    uSyncStatusBarText( );

    /* install *this as our managed view's controller */
    delete & view( ).installController(*this);

    /* set the scrollbar minor track increments to 32 pixels each */
    view( ).setScrollbarMinorIncrement(kHorizontalScrollbar, 32);
    view( ).setScrollbarMinorIncrement(kVerticalScrollbar, 32);

    /* create the OpenGL rendering canvas as a child of the view, then
       configure it */
    fRenderingCanvas.reset(new GLCanvasThingy(view( )));
    delete & fRenderingCanvas->installController(*this);
    fRenderingCanvas->setOrigin(0, 0);
    fRenderingCanvas->setWidth(view( ).composableWidth( ));
    fRenderingCanvas->setHeight(view( ).composableHeight( ));

    /* set the minimum size to which the user can track the view to
       794 x 500 pixels */
    view( ).setMinimumTrackSize(794, 500);
    view( ).setWidth(800);
    view( ).setHeight(600);
}



S580MainWinController::~S580MainWinController( )
{
}



void  S580MainWinController::thingyClosed(ApplicationWindowThingy& sender)
{
    Application::instance( ).stop( );
}



void  S580MainWinController::thingyMaximized(ApplicationWindowThingy& sender)
{
}



void  S580MainWinController::thingyRestored(ApplicationWindowThingy& sender)
{
}



void  S580MainWinController::thingyMenuItemChosen(
    ApplicationWindowThingy& sender, int itemID)
{
    if (itemID == kExitMenuItemID) {

        Application::instance( ).stop( );
    }
    else if (itemID == kNewMenuItemID) {

        parent( ).newPainting( );
    }
    else if (itemID == kCloseMenuItemID) {

        parent( ).closePainting( );
    }
    else if (itemID == kSaveAsMenuItemID) {

        uRunSaveInteraction( );
        parent( ).notifyDocumentStateChanged( );
    }
    else if (itemID == kOpenMenuItemID) {

        parent( ).openPainting( );
    }
    else if (itemID == kEraserOptsMenuItemID) {

        parent( ).changeEraserOptions( );
    }
    else if (itemID == kBrushOptsMenuItemID) {

        parent( ).changeBrushOptions( );
    }
    else if (itemID == kUndoMenuItemID) {

        parent( ).currentPainting( ).undo( );
        parent( ).notifyDocumentStateChanged( );
        fRenderingCanvas->reimage( );
    }
    else if (itemID == kRedoMenuItemID) {

        parent( ).currentPainting( ).redo( );
        parent( ).notifyDocumentStateChanged( );
        fRenderingCanvas->reimage( );
    }
    else if (itemID == kSaveMenuItemID) {

        if (parent( ).currentPainting( ).isMapped( ))
            parent( ).currentPainting( ).save( );
        else
            uRunSaveInteraction( );

        parent( ).notifyDocumentStateChanged( );
    }
}



void  S580MainWinController::thingyToolbarButtonPushed(
    ApplicationWindowThingy& sender, ToolbarButtonToken buttonID)
{
    if (buttonID == fEraserOptsButton)
        parent( ).changeEraserOptions( );
    else if (buttonID == fBrushOptsButton)
        parent( ).changeBrushOptions( );
}



void  S580MainWinController::thingyToolbarButtonSelected(
    ApplicationWindowThingy& sender, ToolbarButtonToken buttonID)
{
    if ((buttonID == fBrushToolButton) &&
        (parent( ).currentTool( ) == kEraserTool)) {

            parent( ).setCurrentTool(kBrushTool);
    }
    if ((buttonID == fEraserToolButton) &&
        (parent( ).currentTool( ) == kBrushTool)) {

            parent( ).setCurrentTool(kEraserTool);
    }
}



void  S580MainWinController::thingyToolbarButtonDeselected(
    ApplicationWindowThingy& sender, ToolbarButtonToken buttonID)
{
    if ((buttonID == fBrushToolButton) &&
        (parent( ).currentTool( ) == kBrushTool)) {

            sender.selectToolbarButton(fBrushToolButton);
    }

    if ((buttonID == fEraserToolButton) &&
        (parent( ).currentTool( ) == kEraserTool)) {

            sender.selectToolbarButton(fEraserToolButton);
    }
}



void  S580MainWinController::thingySized(ApplicationWindowThingy& sender)
{
    fRenderingCanvas->setWidth(sender.composableWidth( ));
    fRenderingCanvas->setHeight(sender.composableHeight( ));

    uSyncScrollbars( );
}



void  S580MainWinController::layoutSubthingies(CompositeThingy& sender)
{
}



void  S580MainWinController::enableSubthingies(CompositeThingy& sender)
{
}



void  S580MainWinController::disableSubthingies(CompositeThingy& sender)
{
}



void  S580MainWinController::showView( )
{
    view( ).show( );
}




void  S580MainWinController::uConfigureToolbar( )
{
    HBITMAP brushbmp =
        (HBITMAP) LoadImage(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(kS580BrushToolBitmapID), IMAGE_BITMAP,
        ApplicationWindowThingy::kToolbarIconDimension,
        ApplicationWindowThingy::kToolbarIconDimension,
        LR_SHARED | LR_LOADMAP3DCOLORS);
    fBrushToolButton = view( ).appendToolbarButton(kSelectButtonKind,
        brushbmp, "Brush Tool   ");

    HBITMAP brushoptsbmp =
        (HBITMAP) LoadImage(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(kS580BrushOptsBitmapID), IMAGE_BITMAP,
        ApplicationWindowThingy::kToolbarIconDimension,
        ApplicationWindowThingy::kToolbarIconDimension,
        LR_SHARED | LR_LOADMAP3DCOLORS);
    fBrushOptsButton = view( ).appendToolbarButton(kPushButtonKind,
        brushoptsbmp, "Options...  ");

    HBITMAP eraserbmp =
        (HBITMAP) LoadImage(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(kS580EraserToolBitmapID), IMAGE_BITMAP,
        ApplicationWindowThingy::kToolbarIconDimension,
        ApplicationWindowThingy::kToolbarIconDimension,
        LR_SHARED | LR_LOADMAP3DCOLORS);
    view( ).appendToolbarSeparator(3);
    view( ).appendToolbarSeparator(3);
    fEraserToolButton = view( ).appendToolbarButton(kSelectButtonKind,
        eraserbmp, "Eraser Tool   ");

    HBITMAP eraseropts =
        (HBITMAP) LoadImage(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(kS580EraserOptsBitmapID), IMAGE_BITMAP,
        ApplicationWindowThingy::kToolbarIconDimension,
        ApplicationWindowThingy::kToolbarIconDimension,
        LR_SHARED | LR_LOADMAP3DCOLORS);
    fEraserOptsButton = view( ).appendToolbarButton(kPushButtonKind,
        eraseropts, "Options...  ");
    view( ).appendToolbarSeparator(3);
}



void  S580MainWinController::uSyncMenuState( )
{
    if (! parent( ).isPaintingOpen( )) {

        view( ).disableMenuItem(kSaveMenuItemID);
        view( ).disableMenuItem(kSaveAsMenuItemID);
        view( ).disableMenuItem(kCloseMenuItemID);
        view( ).disableMenuItem(kUndoMenuItemID);
        view( ).disableMenuItem(kRedoMenuItemID);
    }
    else {

        view( ).enableMenuItem(kSaveAsMenuItemID);
        view( ).enableMenuItem(kCloseMenuItemID);

        if (parent( ).currentPainting( ).isUndoAvailable( ))
            view( ).enableMenuItem(kUndoMenuItemID);
        else
            view( ).disableMenuItem(kUndoMenuItemID);

        if (parent( ).currentPainting( ).isRedoAvailable( ))
            view( ).enableMenuItem(kRedoMenuItemID);
        else
            view( ).disableMenuItem(kRedoMenuItemID);

        if (parent( ).currentPainting( ).isMutated( ))
            view( ).enableMenuItem(kSaveMenuItemID);
        else
            view( ).disableMenuItem(kSaveMenuItemID);

    } /* else if a painting document is open */
}



S580AppController&  S580MainWinController::parent( )
{
    return *fParent;
}




void  S580MainWinController::uSyncStatusBarText( )
{
    const std::string  kBrushPanePrefix = "Current Brush:  ";
    const std::string  kEraserPanePrefix = "Current Eraser:  ";

    std::string  brushstatus = kBrushPanePrefix +
        parent( ).brushManager( ).activeBrush( ).name( );
    std::string  eraserstatus = kEraserPanePrefix +
        parent( ).eraser( ).description( );

    view( ).setStatusPaneText(fBrushStatPane, brushstatus);
    view( ).setStatusPaneText(fEraserStatPane, eraserstatus);
}



void  S580MainWinController::setSelectedTool(const ToolSpecifier& t)
{
    if (t == kBrushTool) {

        view( ).deselectToolbarButton(fEraserToolButton);
        view( ).selectToolbarButton(fBrushToolButton);
    }
    else if (t == kEraserTool) {

        view( ).deselectToolbarButton(fBrushToolButton);
        view( ).selectToolbarButton(fEraserToolButton);
    }
}




ApplicationWindowThingy&  S580MainWinController::view( )
{
    return *fView;
}



void  S580MainWinController::notifyDocumentChanged( )
{
    if (parent( ).isPaintingOpen( )) {

        fRenderingCanvas->show( );
    }

    if (! parent( ).isPaintingOpen( )) {

        fRenderingCanvas->hide( );
    }

    uSyncScrollbars( );
    uSyncMenuState( );
    uSyncTitleBarText( );

    /* the innocent hack of resizing the view window is used here to force
       it to redraw, such that the "no man's land" rectangle between the
       horizontal and vertical scroll bars is redrawn in the background
       color of the canvas gutter */
    int view_wd = view( ).width( );
    view( ).setWidth(view_wd - 1);
    view( ).setWidth(view_wd);
}



void  S580MainWinController::uSyncScrollbars( )
{
    if (parent( ).isPaintingOpen( )) {

        Vector2Di  canvasuniv_ext =
            parent( ).currentPainting( ).dimension( ) +
            Vector2Di(2 * kCanvasGutterWidth, 2 * kCanvasGutterWidth);

        Vector2Di  window_ext;
        window_ext.x = fRenderingCanvas->width( );
        window_ext.y = fRenderingCanvas->height( );

        if (canvasuniv_ext.x > window_ext.x) {

            view( ).enableScrollbar(kHorizontalScrollbar);
        }
        else {

            view( ).disableScrollbar(kHorizontalScrollbar);
        }

        if (canvasuniv_ext.y > window_ext.y) {

            view( ).enableScrollbar(kVerticalScrollbar);
        }
        else {

            view( ).disableScrollbar(kVerticalScrollbar);
        }

        fRenderingCanvas->setWidth(view( ).composableWidth( ));
        fRenderingCanvas->setHeight(view( ).composableHeight( ));

        Vector2Di  window_ext_new;
        window_ext_new.x = fRenderingCanvas->width( );
        window_ext_new.y = fRenderingCanvas->height( );

        view( ).setScrollbarRange(kHorizontalScrollbar, 0, canvasuniv_ext.x);
        view( ).setScrollbarRange(kVerticalScrollbar, 0, canvasuniv_ext.y);

        view( ).setScrollbarPageSize(kHorizontalScrollbar, window_ext_new.x);
        view( ).setScrollbarPageSize(kVerticalScrollbar, window_ext_new.y);
    }
    else {

        view( ).disableScrollbar(kHorizontalScrollbar);
        view( ).disableScrollbar(kVerticalScrollbar);
    }
}



void  S580MainWinController::imageThingy(GLCanvasThingy& sender)
{
    glClear(GL_COLOR_BUFFER_BIT);

    Vector2Di  window_ext = Vector2Di(sender.width( ), sender.height( ));
    Vector2Di  window_origin = uGetViewfinderOrigin( );

    glViewport(0, 0, window_ext.x, window_ext.y);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity( );
    glOrtho(window_origin.x, window_origin.x + window_ext.x, window_origin.y,
        window_origin.y + window_ext.y, 1, -1);

    uDrawPixelsFromDocument( );

    uSetContentConstraints( );
    uDrawSampleContent( );
    uUnsetContentConstraints( );

    if (fIsPixWriteRequired) {

        uWritePixelsToDocument( );
        fIsPixWriteRequired = false;
    }

    uDrawCanvasFrame( );

    glFlush( );
}



void  S580MainWinController::thingySized(GLCanvasThingy& sender)
{
}



void  S580MainWinController::initializeThingy(GLCanvasThingy&)
{
    glClearColor(kCanvasGutterColor.r, kCanvasGutterColor.g, 
        kCanvasGutterColor.b, 1.0f);

    /* configure pixel format, transfer, packing and unpacking to
       enable pixel transfer into & out of the current painting
       document's image buffer */
    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
    glPixelStorei(GL_PACK_ALIGNMENT, 4);
    glPixelStorei(GL_PACK_ROW_LENGTH, 0);
    glPixelStorei(GL_PACK_SKIP_ROWS, 0);
    glPixelStorei(GL_PACK_SKIP_PIXELS, 0);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity( );
}



void  S580MainWinController::uDrawCanvasFrame( )
{
    Vector2Df  ext = Vector2Df(
        static_cast<float>(parent( ).currentPainting( ).dimension( ).x),
        static_cast<float>(parent( ).currentPainting( ).dimension( ).y));

    glColor4f(0.0f, 0.0f, 0.0f, 1.0f);

    glBegin(GL_LINE_LOOP);
        glVertex2f(-0.5f, -0.5f);
        glVertex2f(ext.x + 0.5f, -0.5f);
        glVertex2f(ext.x + 0.5f, ext.y + 0.5f);
        glVertex2f(-0.5f, ext.y + 0.5f);
    glEnd( );

}



void  S580MainWinController::thingyScrollbarPositioned(
    ApplicationWindowThingy& sender, const ScrollbarSelector& bar)
{
    if (fIsScrollTrackInProgress)  fIsScrollTrackInProgress = false;

    int botval = uGetBottomRelScrollPos( );

    TranscriptionServer&  tranny =
        Application::instance( ).transcriptionServer( );
    std::ostringstream valuestream;
    valuestream << "\nbottom rel vscroll position (place) = " << botval;
    tranny.writeln(valuestream.str( ));


    fRenderingCanvas->reimage( );
}



void  S580MainWinController::thingyScrollbarTracked(
    ApplicationWindowThingy& sender, const ScrollbarSelector& bar,
    int trackval)
{
    if (! fIsScrollTrackInProgress)  fIsScrollTrackInProgress = true;

    fTrackingScroll = bar;
    fScrollTrackValue = trackval;

    int botval = uGetBottomRelScrollPos( );

    TranscriptionServer&  tranny =
        Application::instance( ).transcriptionServer( );
    std::ostringstream valuestream;
    valuestream << "\nbottom rel vscroll position (track) = " << botval;
    tranny.writeln(valuestream.str( ));

    fRenderingCanvas->reimage( );
}



int  S580MainWinController::uGetBottomRelScrollPos( ) const
{
    int vscroll_value;
    if ((fIsScrollTrackInProgress) && (fTrackingScroll == kVerticalScrollbar))
        vscroll_value = fScrollTrackValue;
    else
        vscroll_value = view( ).scrollbarPosition(kVerticalScrollbar);

    return view( ).scrollbarRangeMax(kVerticalScrollbar) - vscroll_value -
        view( ).scrollbarPageSize(kVerticalScrollbar) + 1;
}



const ApplicationWindowThingy&  S580MainWinController::view( ) const
{
    return *fView;
}



void  S580MainWinController::uDrawSampleContent( )
{
    if (parent( ).currentTool( ) == kBrushTool) {

        Brush& currbrush = parent( ).universalBrush( );

        if (currbrush.isStrokeInProgress( )) {

            currbrush.refreshStroke( );
        }
        else {

            if (fIsBrushDrawRequired) {

                currbrush.drawStroke( );
                fIsBrushDrawRequired = false;
                fIsPixWriteRequired = true;
            } /* if a stroke draw is required */

        } /* else if no stroke is in progress */
    }
    else if (parent( ).currentTool( ) == kEraserTool) {

        if (fIsEraseOpRequired) {

            parent( ).eraser( ).erase(fErasePoint);
            fIsEraseOpRequired = false;
            fIsPixWriteRequired = true;
        }
    } /* if the current tool is the eraser */
}



void  S580MainWinController::uSetContentConstraints( )
{
    Rectangle2Di  const_viewport = uGetConstrainedViewport( );
    Rectangle2Di  vis_subcanvas = uGetVisibleSubcanvas( );

    glPushAttrib(GL_VIEWPORT_BIT);
    glViewport(const_viewport.origin.x, const_viewport.origin.y,
        const_viewport.width, const_viewport.height);

    glMatrixMode(GL_PROJECTION);
    glPushMatrix( );
    glLoadIdentity( );
    glOrtho(vis_subcanvas.origin.x, vis_subcanvas.origin.x +
        vis_subcanvas.width, vis_subcanvas.origin.y,
        vis_subcanvas.origin.y + vis_subcanvas.height, 1, -1);
}



void  S580MainWinController::uUnsetContentConstraints( )
{
    glPopAttrib( );
    glMatrixMode(GL_PROJECTION);
    glPopMatrix( );
}



void  S580MainWinController::thingyMousePressed(GLCanvasThingy& sender,
    const MouseInteractionToken& evt)
{
    Vector2Di canvaspt = evt.location( ) + uGetViewfinderOrigin( );

    Rectangle2Di  viscanvas = uGetVisibleSubcanvas( );
    parent( ).currentPainting( ).setUndoPoint(viscanvas);
    parent( ).notifyDocumentStateChanged( );

    if (parent( ).currentTool( ) == kBrushTool) {

        parent( ).universalBrush( ).startStroke( );
    }
    else if (parent( ).currentTool( ) == kEraserTool) {

        fIsEraseOpRequired = true;
        fErasePoint = canvaspt;
        sender.reimage( );
    }
}



void  S580MainWinController::thingyMouseReleased(GLCanvasThingy& sender,
    const MouseInteractionToken& evt)
{
    Vector2Di canvaspt = evt.location( ) + uGetViewfinderOrigin( );

    if (parent( ).currentTool( ) == kBrushTool) {

        parent( ).universalBrush( ).stopStroke( );
        fIsBrushDrawRequired = true;
        sender.reimage( );
    }
    else if (parent( ).currentTool( ) == kEraserTool) {
    }

        parent( ).currentPainting( ).setMutated( );
        parent( ).notifyDocumentStateChanged( );
}



void  S580MainWinController::thingyMouseTracked(GLCanvasThingy& sender,
    const MouseInteractionToken& evt)
{
    Vector2Di canvaspt = evt.location( ) + uGetViewfinderOrigin( );

    if (parent( ).currentTool( ) == kBrushTool) {

        parent( ).universalBrush( ).appendStroke(static_cast<float>(
            canvaspt.x), static_cast<float>(canvaspt.y));

        sender.reimage( );
    }
    else if (parent( ).currentTool( ) == kEraserTool) {

        fIsEraseOpRequired = true;
        fErasePoint = canvaspt;
        sender.reimage( );
    }
}



const S580AppController&  S580MainWinController::parent( ) const
{
    return *fParent;
}



void  S580MainWinController::uWritePixelsToDocument( )
{
    Rectangle2Di  subcanvas = uGetVisibleSubcanvas( );
    int           imgwidth = parent( ).currentPainting( ).width( );
    int           imgheight = parent( ).currentPainting( ).height( );

    glPixelStorei(GL_PACK_ALIGNMENT, 4);
    glPixelStorei(GL_PACK_ROW_LENGTH, imgwidth);
    glPixelStorei(GL_PACK_SKIP_ROWS, subcanvas.origin.y);
    glPixelStorei(GL_PACK_SKIP_PIXELS, subcanvas.origin.x);

    Rectangle2Di  viewport = uGetConstrainedViewport( );

    glReadBuffer(GL_BACK);
    glReadPixels(viewport.origin.x, viewport.origin.y, viewport.width,
        viewport.height, GL_RGBA, GL_UNSIGNED_INT_8_8_8_8_REV,
        parent( ).currentPainting( ).image( ).imageBuffer( ));
}



void  S580MainWinController::uDrawPixelsFromDocument( )
{
    Rectangle2Di  subcanvas = uGetVisibleSubcanvas( );
    int           imgwidth = parent( ).currentPainting( ).width( );
    int           imgheight = parent( ).currentPainting( ).height( );

    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, imgwidth);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, subcanvas.origin.y);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, subcanvas.origin.x);

    Rectangle2Di  viewport = uGetConstrainedViewport( );

    glWindowPos2i(viewport.origin.x, viewport.origin.y);

    glDrawPixels(subcanvas.width, subcanvas.height, GL_RGBA,
        GL_UNSIGNED_INT_8_8_8_8_REV,
        parent( ).currentPainting( ).image( ).imageBuffer( ));
}



void  S580MainWinController::uRunSaveInteraction( )
{
    SaveFileChooser sfc;
    sfc.setTitle("Save Painting As...");
    sfc.addFiletypeMask("Truevision TARGA (*.tga)", "*.tga");

    std::string  savefn;

    bool  isfilevalid = true;
    do {

        savefn = sfc.runFileChoiceInteraction(view( ));
        if (savefn == "")
            return;

        /* check to see if the filename entered by the user includes the
          .tga extension. If it doesn't, then add it */
        int extstart = savefn.length( ) - 4;

        if (extstart < 1) {

            savefn += ".tga";
        }
        else {

            std::string  extstring = savefn.substr(extstart, 4);
            if (extstring != ".tga")
                savefn += ".tga";
        }

        /* now that we've got the filename, check to see if file already
           exists; if it does, ask the user if he/she wants overwrite it
           or pick another file */
        if (Win32Tools::filesystem( ).doesFileExist(savefn)) {

            std::string  simplefn = Win32Tools::simplifyFilename(savefn);

            std::string  msgstr = "The file '" + simplefn + "' already "
                "exists.\nClick 'Yes' to overwrite the existing file, or "
                "'No' to choose another file.";

            int userchoice = MessageBox(view( ).adapter( ).primitive( ),
                msgstr.c_str( ), Application::instance( ).name( ).c_str( ),
                MB_YESNO);

            if (userchoice == IDYES)
                isfilevalid = true;
            else
                isfilevalid = false;
        }

    } while (!isfilevalid);

    parent( ).currentPainting( ).saveAs(savefn);
}



Vector2Di     S580MainWinController::uGetViewfinderOrigin( ) const
{
    /* extent of the canvas universe, including the gutter */
    Vector2Di  canvasuniv_ext =
        this->parent( ).currentPainting( ).dimension( ) +
        Vector2Di(2 * kCanvasGutterWidth, 2 * kCanvasGutterWidth);

    /* extent of painting itself */
    Vector2Di  painting_ext = this->parent( ).currentPainting( ).dimension( );

    /* extent of the window */
    Vector2Di  window_ext;
    window_ext.x = Win32Tools::getClientAreaWidth(
        fRenderingCanvas->adapter( ).primitive( ));
    window_ext.y = Win32Tools::getClientAreaHeight(
        fRenderingCanvas->adapter( ).primitive( ));

    /* horizontal scrollbar value */
    int  hscroll_value;
    if ((fIsScrollTrackInProgress) && (fTrackingScroll == kHorizontalScrollbar))
        hscroll_value = fScrollTrackValue;
    else
        hscroll_value = view( ).scrollbarPosition(kHorizontalScrollbar);

    /* vertical scrollbar value (relative to the bottom of the scroll
       gutter -- note that this is different from the standard PTCI & Win32
       way of measuring scroll values, which is to measure relative to the
       top of the scroll gutter) */
    int  vscroll_value = uGetBottomRelScrollPos( );

    Vector2Di  win_offset;

    if (window_ext.x >= canvasuniv_ext.x)
        win_offset.x = (painting_ext.x - window_ext.x) / 2;
    else
        win_offset.x = hscroll_value - kCanvasGutterWidth;

    if (window_ext.y >= canvasuniv_ext.y)
        win_offset.y = (painting_ext.y - window_ext.y) / 2;
    else
        win_offset.y = vscroll_value - kCanvasGutterWidth;

    return win_offset;
}



Vector2Di  S580MainWinController::uGetViewfinderExtent( ) const
{
    Vector2Di  window_ext;

    window_ext.x = Win32Tools::getClientAreaWidth(
        fRenderingCanvas->adapter( ).primitive( ));
    window_ext.y = Win32Tools::getClientAreaHeight(
        fRenderingCanvas->adapter( ).primitive( ));

    return window_ext;
}



Rectangle2Di  S580MainWinController::uGetConstrainedViewport( ) const
{
    Vector2Di  viewf_ogn = uGetViewfinderOrigin( );

    /* if one of the viewfinder origin's coordinates is negative, then
       some of the gutter is visible along the corresponding axis, so
       we need to constraint the viewport */
    Rectangle2Di  constr_vp;
    constr_vp.origin.x = (viewf_ogn.x >= 0) ? 0 : -viewf_ogn.x;
    constr_vp.origin.y = (viewf_ogn.y >= 0) ? 0 : -viewf_ogn.y;

    Vector2Di  painting_ext = parent( ).currentPainting( ).dimension( );
    Vector2Di  viewf_ext = uGetViewfinderExtent( );

    int  rt_gut_pix = viewf_ogn.x + viewf_ext.x - painting_ext.x;
    rt_gut_pix = clamp(rt_gut_pix, 0, INT_MAX);

    int  top_gut_pix = viewf_ogn.y + viewf_ext.y - painting_ext.y;
    top_gut_pix = clamp(top_gut_pix, 0, INT_MAX);

    constr_vp.width = viewf_ext.x - rt_gut_pix - constr_vp.origin.x;

    constr_vp.height = viewf_ext.y - top_gut_pix - constr_vp.origin.y;

    return constr_vp;
}



Rectangle2Di  S580MainWinController::uGetVisibleSubcanvas( ) const
{
    Rectangle2Di  result;

    Vector2Di     viewf_ogn = uGetViewfinderOrigin( );

    result.origin.x = clamp(viewf_ogn.x, 0, INT_MAX);
    result.origin.y = clamp(viewf_ogn.y, 0, INT_MAX);

    Rectangle2Di  constr_vp = uGetConstrainedViewport( );

    result.width = constr_vp.width;
    result.height = constr_vp.height;

    return result;
}



void  S580MainWinController::notifyEraserChanged( )
{
    uSyncStatusBarText( );
}




GPUInterface&  S580MainWinController::gpuInterface( )
{
    return fRenderingCanvas->gpuInterface( );
}



void  S580MainWinController::notifyBrushChanged( )
{
    uSyncStatusBarText( );
}



void  S580MainWinController::notifyDocumentStateChanged( )
{
    uSyncTitleBarText( );
    uSyncMenuState( );
}



void  S580MainWinController::uSyncTitleBarText( )
{
    std::string  titletext = "Studio580";

    if (parent( ).isPaintingOpen( )) {

        std::string  appendage = "  " +
            parent( ).currentPainting( ).name( );

        if (parent( ).currentPainting( ).isMutated( ))
            appendage += " [changed]";

        titletext += appendage;
    }

    view( ).setTitle(titletext);
}




void  S580MainWinController::runSaveInteraction( )
{
    uRunSaveInteraction( );
}
