/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Painting.cpp                                                            //
//                                                                         //
// Implements methods in classes PaintingDocument and UndoPoint            //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  <unversioned module>                                                   //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Copyright (c) 2008 Lucas Stephen Beeler. All rights reserved.           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "Painting.h"

int         UndoPoint::sNextSequenceNumber = 0;
UndoPoint*  UndoPoint::sNullPoint = 0;

const UndoPoint&  UndoPoint::nullPoint( )
{
    if (! sNullPoint) {

        sNullPoint = new UndoPoint(0, 0, 0, 0);
        sNextSequenceNumber--;
        sNullPoint->fSequenceNumber = -1;
    }

    return *sNullPoint;
}



UndoPoint::UndoPoint(int x, int y, int w, int h)
    : fSequenceNumber(sNextSequenceNumber++), fEffectsRegion(x, y, w, h)
{
}



UndoPoint::UndoPoint(const Vector2Di& origin, int w, int h)
    : fSequenceNumber(sNextSequenceNumber++), fEffectsRegion(origin, w, h)
{
}



UndoPoint::UndoPoint(const Rectangle2Di fxreg)
    : fSequenceNumber(sNextSequenceNumber++), fEffectsRegion(fxreg)
{
}



UndoPoint::UndoPoint(const UndoPoint& src)
    : fSequenceNumber(src.fSequenceNumber),
      fEffectsRegion(src.fEffectsRegion)
{
}



UndoPoint::~UndoPoint( )
{
}



const UndoPoint&  UndoPoint::operator=(const UndoPoint& rhs)
{
    fSequenceNumber = rhs.fSequenceNumber;
    fEffectsRegion = rhs.fEffectsRegion;

    return *this;
}



bool  UndoPoint::operator==(const UndoPoint& rhs) const
{
    if (this->fSequenceNumber == rhs.fSequenceNumber)
        return true;
    else
        return false;
}



bool  UndoPoint::operator!=(const UndoPoint& rhs) const
{
    return (! (*this == rhs));
}



const Rectangle2Di&  UndoPoint::effectsRegion( ) const
{
    return fEffectsRegion;
}











PaintingDocument::PaintingDocument(const std::string& filepath)
    : fUndoPoint(UndoPoint::nullPoint( )),
      fRedoPoint(UndoPoint::nullPoint( )),
      fIsMutated(false)
{
    fImageBuffer.reset(RGBAImage::createMappedImage(
        RGBAImage::kTargaImageFormat, filepath));
    fUndoBuffer.reset(new RGBAImage(fImageBuffer->width( ),
        fImageBuffer->height( )));
    fName = Win32Tools::simplifyFilename(filepath);
}



PaintingDocument::PaintingDocument(int width, int height)
    : fImageBuffer(new RGBAImage(width, height)),
      fUndoPoint(UndoPoint::nullPoint( )),
      fRedoPoint(UndoPoint::nullPoint( )),
      fUndoBuffer(new RGBAImage(width, height)),
      fIsMutated(false)
{
    fName = "untitled";
}



PaintingDocument::PaintingDocument(const Vector2Di& dim)
    : fImageBuffer(new RGBAImage(dim.x, dim.y)),
      fUndoPoint(UndoPoint::nullPoint( )),
      fRedoPoint(UndoPoint::nullPoint( )),
      fUndoBuffer(new RGBAImage(dim.x, dim.y)),
      fIsMutated(false)
{
    fName = "untitled";
}



PaintingDocument::~PaintingDocument( )
{
}



int  PaintingDocument::width( ) const
{
    return image( ).width( );
}



int  PaintingDocument::height( ) const
{
    return image( ).height( );
}



Vector2Di  PaintingDocument::dimension( ) const
{
    return Vector2Di(width( ), height( ));
}



RGBAImage&  PaintingDocument::image( )
{
    return *fImageBuffer;
}



const RGBAImage&  PaintingDocument::image( ) const
{
    return *fImageBuffer;
}



void  PaintingDocument::saveAs(const std::string& path)
{
    image( ).installMapping(RGBAImage::kTargaImageFormat, path);
    image( ).write( );

    fIsMutated = false;
    fName = Win32Tools::simplifyFilename(path);
}



bool  PaintingDocument::isUndoAvailable( ) const
{
    if (fUndoPoint != UndoPoint::nullPoint( ))
        return true;
    else
        return false;
}



void  PaintingDocument::setUndoPoint(const Rectangle2Di& fxreg)
{
    fRedoPoint = UndoPoint::nullPoint( );

    fUndoPoint = UndoPoint(fxreg);

    RGBAImage::transferPixels(*fImageBuffer, *fUndoBuffer, fxreg.origin.x,
        fxreg.origin.y, fxreg.origin.x, fxreg.origin.y, fxreg.width,
        fxreg.height);
}



void  PaintingDocument::undo( )
{
    if (! isUndoAvailable( ))
        throw std::logic_error("PaintingDocument: can't undo: undo isn't "
            "available");

    Rectangle2Di  fxreg = fUndoPoint.effectsRegion( );

    std::auto_ptr<RGBAImage>  redoimg(new RGBAImage(fxreg.width,
        fxreg.height));

    RGBAImage::transferPixels(*fImageBuffer, *redoimg, fxreg.origin.x,
        fxreg.origin.y, 0, 0, fxreg.width, fxreg.height);

    RGBAImage::transferPixels(*fUndoBuffer, *fImageBuffer, fxreg.origin.x,
        fxreg.origin.y, fxreg.origin.x, fxreg.origin.y, fxreg.width,
        fxreg.height);

    RGBAImage::transferPixels(*redoimg, *fUndoBuffer, 0, 0, fxreg.origin.x,
        fxreg.origin.y, fxreg.width, fxreg.height);

    RGBAImage*  redoimg_ptr = redoimg.release( );
    delete redoimg_ptr;
    redoimg.reset(0);

    fUndoPoint = UndoPoint::nullPoint( );
    fRedoPoint = UndoPoint(fxreg);
    fIsMutated = true;
}



const std::string&  PaintingDocument::name( ) const
{
    return fName;
}




bool  PaintingDocument::isRedoAvailable( ) const
{
    if (fRedoPoint != UndoPoint::nullPoint( ))
        return true;
    else
        return false;
}



void  PaintingDocument::redo( )
{
    if (! isRedoAvailable( ))
        throw std::logic_error("PaintingDocument: can't redo: redo isn't "
            "available");

    Rectangle2Di  fxreg = fRedoPoint.effectsRegion( );

    std::auto_ptr<RGBAImage>  undoimg(new RGBAImage(fxreg.width,
        fxreg.height));

    RGBAImage::transferPixels(*fImageBuffer, *undoimg, fxreg.origin.x,
        fxreg.origin.y, 0, 0, fxreg.width, fxreg.height);

    RGBAImage::transferPixels(*fUndoBuffer, *fImageBuffer, fxreg.origin.x,
        fxreg.origin.y, fxreg.origin.x, fxreg.origin.y, fxreg.width,
        fxreg.height);

    RGBAImage::transferPixels(*undoimg, *fUndoBuffer, 0, 0, fxreg.origin.x,
        fxreg.origin.y, fxreg.width, fxreg.height);

    RGBAImage*  undoimg_ptr = undoimg.release( );
    delete undoimg_ptr;
    undoimg.reset(0);

    fRedoPoint = UndoPoint::nullPoint( );
    fUndoPoint = UndoPoint(fxreg);
    fIsMutated = true;
}



bool  PaintingDocument::isMutated( ) const
{
    return fIsMutated;
}



void  PaintingDocument::setMutated( )
{
    fIsMutated = true;
}



bool  PaintingDocument::isMapped( ) const
{
    if (fImageBuffer->imageKind( ) == RGBAImage::kFileMappedImage)
        return true;
    else
        return false;
}



void  PaintingDocument::save( )
{
    if (! isMapped( ))
        throw std::logic_error("PaintingDocument: can't save document: "
            "document isn't mapped to a file");

    fImageBuffer->write( );

    fIsMutated = false;
}
