/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Brushes.h                                                               //
//                                                                         //
// Packages the interface definitions of classes Brush and BrushManager    //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  ver 1.0.1 of Thu Feb-12-2009 @ 10:16pm EST                             //
//                                                                         //
//     Modified the save( ) method of class BrushManager to correct a bug  //
//     where floating-point values that happened to be whole numbers       //
//     written out to the brush database file without a decimal point.     //
//     This was problematic, because when the brush database was read      //
//     back in, the StructuredReader through an exception because it       //
//     always expects floating point values to have a decimal point.       //
//                                                                         //
//     this is the version used to compile to the 1.0pr4 distributable     //
//                                                                         //
//  ver 1.0.0 of Fri Jan-09-2009 @ 5:14am EST                              //
//                                                                         //
//     Differences from development versions: when the brush manager       //
//     writes out the brush database file in method save( ), it now        //
//     replaces the user and library paths with the macros "$(USER)" and   //
//     "$(LIBRARY)," respectively. Whenever brush paths are dealt with in  //
//     the brush manager (in its constructor and the save( ) method) the   //
//     initial drive letters in fully-qualified path names are made        //
//     converted to lowercase before any comparison operations are         //
//     performed on them. This is because all PTCI string-comparison       //
//     operations are case-sensitive, and Windows API routines can hand    //
//     back paths with either capitalized or non-capitalized drive letters //
//     (this is not surprising, as windows is case-insensitive).           //
//                                                                         //
//     note that this is the version used to compile to the 1.0 pr1 - pr3  //
//     distributables.                                                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Copyright (c) 2008-2009 Lucas Stephen Beeler. All rights reserved.      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#ifndef BRUSHES__HXX
#define BRUSHES__HXX

#include <ColorImaging.h>
#include <string>
#include <vector>
#include <map>
#include "Studio580Types.h"

//-------------------------------------------------------------------------//
/* A Brush is a Studio580, texture synthesis-based based paintbrush. Many
   paintbrush objects can be instantiated within a running instance of the
   Studio580 application, but only one can be current at any given time.
   The current brush is used to create new strokes. */
class Brush :
    public virtual DialogController {

private:

    struct  DecalRun {

        int         userStartVertex;
        int         userNumVertices;
        int         extStartVertex;
        int         extNumVertices;
        int         synHorizOffset;
        int         synLength;
        bool        isOpen;
        Vector2Di   anchor;
    };

    enum    PerpMode {
        
        kRightHandedMode,
        kLeftHandedMode
    };

    static  const int               kSlotsPerFrameVertex = 2;
    static  const int               kSlotsPerSample = 2;
    static  const int               kNumInitVerticesAllocd = 8192;
    static  const int               kNumInitSamplesAllocd = 2049;
    static  const Vector2Df         kNullNode;
    static  const Vector2Df         kNullPerp;
    static  const Vector2Df         kNullDisplacement;
    static  const float             kSepThresholdCosine;
    static  const std::string       kDefaultBrushSource;
    static  const std::string       kDefaultBrushSimset;

    static  DialogThingy*           sBrushLinkingDialog;

    std::string                            fSourceImagePath;
    std::string                            fSimMapPath;
    std::auto_ptr<SynthesisAgent>          fSynAgent;
    std::auto_ptr<PyramidalSimilarityMap>  fSimMap;
    std::string                            fName;
    float                                  fHalfWidth;
    float                                  fNumBlendPixels;
    GLfloat*                               fFrameVtxCoordStore;
    GLfloat*                               fFeatherTexCoordStore;
    GLfloat*                               fDecalTexCoordStore;
    GLfloat*                               fSamplePointStore;
    float                                  fBlendBoundaries[2];
    float                                  fAppendThreshold;
    int                                    fNumUserFrameVerts;
    int                                    fNumExtFrameVerts;
    int                                    fNumSamples;
    int                                    fNumAllocdFrameSlots;
    bool                                   fIsEdgeNoiseEnabled;
    bool                                   fIsStrokeAssemblyInProgress;
    float                                  fJumpParameter;
    GLuint                                 fUnifVertexProgID;
    GLuint                                 fProxyFragmentProgID;
    GLuint                                 fDecalFragmentProgID;
    GLuint                                 fProxyProgSetID;
    GLuint                                 fDecalProgSetID;
    GLuint                                 fEdgeNoiseTexID;
    GPUInterface*                          fGPU;
    PerpMode                               fPerpMode;
    Vector2Df                              fPreNode;
    Vector2Df                              fPrePerp;
    Vector2Df                              fPreDisp;
    float                                  fPreArcLength;
    std::vector<int>                       fNeedsSeparationCap;
    std::vector<DecalRun>                  fDecalRuns;

    void       uGrowStores( );
    void       uComputeBlendBounds( );
    void       uComputeAppendThreshold( );
    void       uInterchangePerpMode( );
    void       uInitTextures( );
    void       uInitPrograms( );
    void       uSetSamplePos(int, const Vector2Df&);
    int        uGetSampleSlot(int);
    void       uUseProxyShaders( ) const;
    int        uGetFrameSlot(int) const;
    Vector2Df  uGetFramePos(int);
    void       uSetFramePos(int, const Vector2Df&);
    void       uSetFrameFeatherTexS(int, float);
    float      uGetFrameFeatherTexT(int) const;
    void       uSetFrameFeatherTexT(int, float);
    float      uGetFrameDecalTexS(int) const;
    void       uSetFrameDecalTexS(int, float);
    float      uGetFrameDecalTexT(int) const;
    void       uSetFrameDecalTexT(int, float);
    Vector2Df  uGetSamplePos(int);
    Vector2Df  uGetAssociatedSamplePos(int);
    void       uMakeFadeInCap(int, int);
    void       uMakeFadeOutCap(int, int);
    void       uMakeSeparationCap(int, int);
    void       uPostProcess( );
    void       uUseDecalShaders( ) const;
    bool       uIsDecalRunOpen( ) const;
    void       uOpenDecalRun(int, int);
    void       uCloseDecalRun(int, int);
    void       uGenerateExtGeometry( );
    void       uComputeDecalRuns( );
    void       uNormalizeDecalCoords( );
    void       uDrawDecalRun(const DecalRun& dr) const;
    void       uCreateSynAgent( );

    Vector2Df  uRightPerp(const Vector2Df& v)
    { 
        return Vector2Df(-v.y, v.x);
    }

    Vector2Df  uLeftPerp(const Vector2Df& v)
    {
        return Vector2Df(v.y, -v.x);
    }

    Vector2Df  uGeneratePerp(const Vector2Df& v)
    {
        if (fPerpMode == kRightHandedMode)
            return  uRightPerp(v).normalized( );
        else
            return  uLeftPerp(v).normalized( );
    }

public:

    static  const unsigned short  kDefaultWidth = 64;
    static  const unsigned short  kDefaultBlendPix = 16;

    Brush(const std::string name, const std::string& srcpath,
        const std::string& simpath);
    Brush(const std::string& name, const std::string& srcpath,
        const std::string& simpath, unsigned short width,
        unsigned short blendpix, float jumparam,
        bool isnoiseenabled);
    Brush(const Brush&);

    virtual  ~Brush( );

    const Brush&  operator=(const Brush&);
    bool  operator==(const Brush&) const;

    virtual  unsigned short      width( ) const;
    virtual  void                setWidth(unsigned short);
    virtual  unsigned short      blendPixels( ) const;
    virtual  void                setBlendPixels(unsigned short);
    virtual  bool                isEdgeNoiseEnabled( ) const;
    virtual  void                enableEdgeNoise( );
    virtual  void                disableEdgeNoise( );
    virtual  const std::string&  sourceImagePath( ) const;
    virtual  void                setSourceImagePath(const std::string&);
    virtual  const std::string&  similarityMapPath( ) const;
    virtual  void                setSimilarityMapPath(const std::string&);
    virtual  const std::string&  name( ) const;
    virtual  void                setName(const std::string&);
    virtual  float               jumpParameter( ) const;
    virtual  void                setJumpParameter(float);
    virtual  bool                isLinked( ) const;
    virtual  GPUInterface&       linkedGPU( ) const;
    virtual  void                link(GPUInterface&);
    virtual  void                unlink( );
    virtual  void                startStroke( );
    virtual  void                stopStroke( );
    virtual  void                refreshStroke( ) const;
    virtual  bool                isStrokeInProgress( ) const;
    virtual  void                appendStroke(float x, float y);
    virtual  void                drawStroke( );
    virtual  void                thingyItemActivated(DialogThingy&, int);
    virtual  void                thingyClosed(DialogThingy&);
    virtual  void                thingyInteractionStarted(DialogThingy&);
    virtual  void                thingyInteractionStopped(DialogThingy&);
    virtual  void                layoutSubthingies(CompositeThingy& sender);
    virtual  void                enableSubthingies(CompositeThingy& sender);
    virtual  void                disableSubthingies(CompositeThingy& sender);
    virtual  void                cloneAttributesFrom(const Brush&);
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
/* A BrushManager manages the collection of Brushes instantiated within a
   running instance of the Studio580 application. A BrushManager allows
   clients to get and set the current brush, as well as to read brush
   descriptions from files on disk */
class BrushManager {

private:

    static  Brush*  sSystemBrush;

    std::map<std::string, Brush*>  fBrushStore;
    std::string                    fActiveBrushName;
    S580AppController*             fParent;

    Brush*       uReadBrush(StructuredReader& reader);
    void         uReadStore(StructuredReader& reader);
    std::string  uGenerateNewBrushName( ) const;
    std::string  uGenerateCloneBrushName(const std::string&);

public:

    BrushManager(S580AppController& parent, std::string& dbpath);
    virtual ~BrushManager( );

    virtual  std::vector<std::string>  manifest( ) const;
    virtual  Brush&                    find(const std::string&);
    virtual  const Brush&              find(const std::string&) const;
    virtual  bool                      isInstalled(const std::string&) const;
    virtual  void                      install(Brush*);
    virtual  void                      remove(const std::string&);
    virtual  void                      save(const std::string&) const;
    virtual  Brush&                    activeBrush( );
    virtual  const Brush&              activeBrush( ) const;
    virtual  void                      setActiveBrush(const std::string&);
    virtual  const Brush&              systemBrush( ) const;
    virtual  Brush&                    newBrush( );
    virtual  int                       numBrushes( ) const;
    virtual  Brush&                    cloneBrush(const std::string&);
    virtual  S580AppController&        parent( );
    virtual  const S580AppController&  parent( ) const;
};
//-------------------------------------------------------------------------//

#endif
