/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Brushes.cpp                                                             //
//                                                                         //
// Implements methods in classes Brush and BrushManager                    //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  ver 1.0.1 of Thu Feb-12-2009 @ 10:16pm EST                             //
//                                                                         //
//     Modified the save( ) method of class BrushManager to correct a bug  //
//     where floating-point values that happened to be whole numbers       //
//     written out to the brush database file without a decimal point.     //
//     This was problematic, because when the brush database was read      //
//     back in, the StructuredReader through an exception because it       //
//     always expects floating point values to have a decimal point.       //
//                                                                         //
//     this is the version used to compile to the 1.0pr4 distributable     //
//                                                                         //
//  ver 1.0.0 of Fri Jan-09-2009 @ 5:14am EST                              //
//                                                                         //
//     Differences from development versions: when the brush manager       //
//     writes out the brush database file in method save( ), it now        //
//     replaces the user and library paths with the macros "$(USER)" and   //
//     "$(LIBRARY)," respectively. Whenever brush paths are dealt with in  //
//     the brush manager (in its constructor and the save( ) method) the   //
//     initial drive letters in fully-qualified path names are made        //
//     converted to lowercase before any comparison operations are         //
//     performed on them. This is because all PTCI string-comparison       //
//     operations are case-sensitive, and Windows API routines can hand    //
//     back paths with either capitalized or non-capitalized drive letters //
//     (this is not surprising, as windows is case-insensitive).           //
//                                                                         //
//     note that this is the version used to compile to the 1.0 pr1 - pr3  //
//     distributables.                                                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Copyright (c) 2008-2009 Lucas Stephen Beeler. All rights reserved.      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "Brushes.h"
#include "S580Application.h"
#include "resource.h"
#include <fstream>
#include <sstream>
#include <cctype>
#include <iomanip>

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  Brush                                                            //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
const Vector2Df    Brush::kNullNode = Vector2Df(-65537.0f, -65537.0f);
const Vector2Df    Brush::kNullPerp = Vector2Df(0.0f, 0.0f);
const Vector2Df    Brush::kNullDisplacement = Vector2Df(0.0f, 0.0f);
const float        Brush::kSepThresholdCosine = 0.01f;
DialogThingy*      Brush::sBrushLinkingDialog = 0;
const std::string  Brush::kDefaultBrushSource = "Van Gogh Default.tga";
const std::string  Brush::kDefaultBrushSimset = "Van Gogh Default.simset";

Brush::Brush(const std::string name, const std::string& srcpath,
    const std::string& simpath)
        : fName(name), fSourceImagePath(srcpath), fSimMapPath(simpath),
          fIsEdgeNoiseEnabled(true), fJumpParameter(1.1f), fGPU(0),
          fFrameVtxCoordStore(0), fFeatherTexCoordStore(0),
          fDecalTexCoordStore(0), fSamplePointStore(0),
          fIsStrokeAssemblyInProgress(false), fPerpMode(kRightHandedMode),
          fPreArcLength(0.0f)
{
    if (fSourceImagePath == "::default::")
        fSourceImagePath = Application::instance( ).startupDirectory( ) + 
        "\\Library\\" + kDefaultBrushSource;

    if (fSimMapPath == "::default::")
        fSimMapPath = Application::instance( ).startupDirectory( ) + 
        "\\Library\\" + kDefaultBrushSimset;

    fHalfWidth = static_cast<float>(kDefaultWidth) / 2.0f;
    fNumBlendPixels = static_cast<float>(kDefaultBlendPix);

    uComputeBlendBounds( );
    uComputeAppendThreshold( );
}



Brush::Brush(const std::string& name, const std::string& srcpath,
    const std::string& simpath, unsigned short width,
    unsigned short blendpix, float jumparam, bool isnoiseenabled)
        : fName(name), fSourceImagePath(srcpath), fSimMapPath(simpath),
          fIsEdgeNoiseEnabled(isnoiseenabled), fJumpParameter(jumparam),
          fGPU(0), fFrameVtxCoordStore(0), fFeatherTexCoordStore(0),
          fDecalTexCoordStore(0), fSamplePointStore(0),
          fIsStrokeAssemblyInProgress(false), fPerpMode(kRightHandedMode),
          fPreArcLength(0.0f)
{
    if (fSourceImagePath == "::default::")
        fSourceImagePath = Application::instance( ).startupDirectory( ) + 
        "\\Library\\" + kDefaultBrushSource;

    if (fSimMapPath == "::default::")
        fSimMapPath = Application::instance( ).startupDirectory( ) + 
        "\\Library\\" + kDefaultBrushSimset;

    if ((width > 128) || (width < 4))
        throw std::logic_error("Brush: can't construct new Brush: width "
            "value out of range");
    if ((width % 2) != 0)
        throw std::logic_error("Brush: can't construct new Brush: width "
            "value is odd");
    fHalfWidth = static_cast<float>(width) / 2.0f;

    if ((blendpix > (width / 4)) || (blendpix < 1))
        throw std::logic_error("Brush: can't construct new Brush: blending "
            "region pixel size value out of range");
    fNumBlendPixels = static_cast<float>(blendpix);

    uComputeBlendBounds( );
    uComputeAppendThreshold( );
}



Brush::Brush(const Brush& src)
    : fSourceImagePath(src.fSourceImagePath), fSimMapPath(src.fSimMapPath),
      fName(src.fName), fHalfWidth(src.fHalfWidth),
      fNumBlendPixels(src.fNumBlendPixels),
      fIsEdgeNoiseEnabled(src.fIsEdgeNoiseEnabled),
      fJumpParameter(src.fJumpParameter), fGPU(0),
      fIsStrokeAssemblyInProgress(false), fPerpMode(kRightHandedMode),
      fPreArcLength(0.0f)
{
    uComputeBlendBounds( );
    uComputeAppendThreshold( );
}



Brush::~Brush( )
{
    if (isLinked( ))
        unlink( );
}



const Brush&  Brush::operator=(const Brush& src)
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't assign to brush: brush "
            "is linked");

    fSourceImagePath = src.fSourceImagePath;
    fSimMapPath = src.fSimMapPath;
    fName = src.fName;
    fHalfWidth = src.fHalfWidth;
    fNumBlendPixels = src.fNumBlendPixels;
    fIsEdgeNoiseEnabled = src.fIsEdgeNoiseEnabled;
    fJumpParameter = src.fJumpParameter;
    fGPU = 0;

    uComputeBlendBounds( );
    uComputeAppendThreshold( );

    return *this;
}



bool  Brush::operator==(const Brush& rhs) const
{
    if (fSourceImagePath != rhs.fSourceImagePath)
        return false;

    if (fSimMapPath != rhs.fSimMapPath)
        return false;

    if (fName != rhs.fName)
        return false;

    if (fHalfWidth != rhs.fHalfWidth)
        return false;

    if (fNumBlendPixels != rhs.fNumBlendPixels)
        return false;

    if (fIsEdgeNoiseEnabled != rhs.fIsEdgeNoiseEnabled)
        return false;

    if (fJumpParameter != rhs.fJumpParameter)
        return false;

    return true;
}



unsigned short  Brush::width( ) const
{
    return static_cast<unsigned short>(fHalfWidth * 2.0f);
}



void  Brush::setWidth(unsigned short width)
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't set width: brush is "
            "linked");

    if ((width > 128) || (width < 4))
        throw std::logic_error("Brush: can't set width: value out of "
            "range");
    if ((width % 2) != 0)
        throw std::logic_error("Brush: can't set width: value is odd");
    fHalfWidth = static_cast<float>(width) / 2.0f;

    uComputeBlendBounds( );
    uComputeAppendThreshold( );
}



unsigned short  Brush::blendPixels( ) const
{
    return static_cast<unsigned short>(fNumBlendPixels);
}



void  Brush::setBlendPixels(unsigned short blendpix)
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't set blend pixels: brush is "
            "linked");

    if ((blendpix > (width( ) / 4)) || (blendpix < 1))
        throw std::logic_error("Brush: can't set blending region pixel size: "
            "value out of range");
    fNumBlendPixels = static_cast<float>(blendpix);

    uComputeBlendBounds( );
}



bool  Brush::isEdgeNoiseEnabled( ) const
{
    return fIsEdgeNoiseEnabled;
}



void  Brush::enableEdgeNoise( )
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't enable edge noise: brush is "
            "linked");

    fIsEdgeNoiseEnabled = true;
}



void  Brush::disableEdgeNoise( )
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't disable edge noise: brush is "
            "linked");

    fIsEdgeNoiseEnabled = false;
}



const std::string&  Brush::sourceImagePath( ) const
{
    return fSourceImagePath;
}



const std::string&  Brush::similarityMapPath( ) const
{
    return fSimMapPath;
}



const std::string&  Brush::name( ) const
{
    return fName;
}



void  Brush::setSourceImagePath(const std::string& path)
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't set source image path: brush "
            "is linked");

    fSourceImagePath = path;
}



void  Brush::setSimilarityMapPath(const std::string& path)
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't set similarity map path: "
            "brush is linked");

    fSimMapPath = path;
}



void  Brush::setName(const std::string& name)
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't set brush name: brush is "
            "linked");

    fName = name;
}



float  Brush::jumpParameter( ) const
{
    return fJumpParameter;
}



void   Brush::setJumpParameter(float param)
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't set jump parameter: brush "
            "is linked");

    fJumpParameter = param;
}



bool  Brush::isLinked( ) const
{
    if (fGPU)
        return true;
    else
        return false;
}



GPUInterface&  Brush::linkedGPU( ) const
{
    if (! isLinked( ))
        throw std::logic_error("Brush: can't get linked GPU interface: this "
            "brush isn't linked");

    return *fGPU;
}



void  Brush::link(GPUInterface& gpu)
{
    if (isLinked( ))
        throw std::logic_error("Brush: can't link brush: brush is already "
            " linked");

    fGPU = (& gpu);

    /* upon linkage, allocate the geometrty stores */
    const int  kNumInitFrameSlotsAllocd = kSlotsPerFrameVertex *
        kNumInitVerticesAllocd;
    const int  kNumInitSampleSlotsAllocd = kSlotsPerSample *
        kNumInitSamplesAllocd;

    fFrameVtxCoordStore = new GLfloat [kNumInitFrameSlotsAllocd];
    fFeatherTexCoordStore = new GLfloat [kNumInitFrameSlotsAllocd];
    fDecalTexCoordStore = new GLfloat [kNumInitFrameSlotsAllocd];
    fSamplePointStore = new GLfloat [kNumInitSampleSlotsAllocd];
    fNumUserFrameVerts = 0;
    fNumSamples = 0;
    fNumAllocdFrameSlots = kNumInitFrameSlotsAllocd;

    /* upon linkage, load texture images from disk and download them into
       TRAM */
    uInitTextures( );

    /* upon linkage, prepare the vertex and fragment programs that are used
       to draw the decal and proxy strokes */
    uInitPrograms( );

    /* upon linkage, create the synthesis agent */
    uCreateSynAgent( );

    /* set the SynAgent's jump parameter */
    fSynAgent->setJumpParameter(fJumpParameter);

    /* once the synthesis agent is created and configured, "prime" it
       by having it perform any arbitrary synthesis operation. This strange
       hack is necessary because on several 2006-2007 generation ATI GPUs,
       the first synthesis operation executed by a synthesis agent is
       extremely slow, but all subsequent synthesis operations are quite
       fast. Since brush linking is already an expensive operation that
       is performed rarely, we chose to perform the first synthesis
       operation here to concentrate the most obscenely time consuming
       tasks into brush linking */
    fSynAgent->synthesize(32, 32, 32, 32);
}



void  Brush::unlink( )
{
    if (! isLinked( ))
        throw std::logic_error("Brush: can't unlink brush: brush isn't "
            " linked");

    if (isStrokeInProgress( ))
        throw std::logic_error("Brush: can't unlink brush: stroke assembly "
            "is in progress");

    /* since we allocate the geometrty stores on linkage, we deallocate
       them when we unlink */
    delete [ ] fFrameVtxCoordStore;
    fFrameVtxCoordStore = 0;
    
    delete [ ] fFeatherTexCoordStore;
    fFeatherTexCoordStore = 0;

    delete [ ] fDecalTexCoordStore;
    fDecalTexCoordStore = 0;

    delete [ ] fSamplePointStore;
    fSamplePointStore = 0;

    /* delete textures on unlink */
    glDeleteTextures(1, &fEdgeNoiseTexID);

    /* delete imaging programs on unlink */
    glDeleteShader(fUnifVertexProgID);
    glDeleteShader(fProxyFragmentProgID);
    glDeleteShader(fDecalFragmentProgID);
    glDeleteProgram(fProxyProgSetID);
    glDeleteProgram(fDecalProgSetID);

    /* delete the synthesis agent and similarity map */
    SynthesisAgent*  synagent = fSynAgent.release( );
    delete synagent;
    fSynAgent.reset(0);
    PyramidalSimilarityMap*  simmap = fSimMap.release( );
    delete simmap;
    fSimMap.reset(0);

    fGPU = 0;
}



void  Brush::uGrowStores( )
{
    int  num_framevts_new = 2 * fNumUserFrameVerts;
    int  num_frameslots_new = kSlotsPerFrameVertex * num_framevts_new;
    int  num_samples_new = 2 * fNumSamples;
    int  num_sampleslots_new = kSlotsPerSample * num_samples_new;

    /* grow the vertex coord store */
    GLfloat*  new_vtxstore = new GLfloat [num_frameslots_new];
    std::memcpy(new_vtxstore, fFrameVtxCoordStore, sizeof(GLfloat) *
        fNumAllocdFrameSlots);
    delete [ ] fFrameVtxCoordStore;
    fFrameVtxCoordStore = new_vtxstore;

    /* grow the feather tex coord store */
    GLfloat*  new_featherstore = new GLfloat [num_frameslots_new];
    std::memcpy(new_featherstore, fFeatherTexCoordStore, sizeof(GLfloat) *
        fNumAllocdFrameSlots);
    delete [ ] fFeatherTexCoordStore;
    fFeatherTexCoordStore = new_featherstore;

    /* grow the decal tex coord store */
    GLfloat*  new_decalstore = new GLfloat [num_frameslots_new];
    std::memcpy(new_decalstore, fDecalTexCoordStore, sizeof(GLfloat) *
        fNumAllocdFrameSlots);
    delete [ ] fDecalTexCoordStore;
    fDecalTexCoordStore = new_decalstore;

    /* grow the user sample point store */
    GLfloat*  new_samplestore = new GLfloat [num_sampleslots_new];
    std::memcpy(new_samplestore, fSamplePointStore, sizeof(GLfloat) *
        kSlotsPerSample * fNumSamples);
    delete [ ] fSamplePointStore;
    fSamplePointStore = new_samplestore;

    fNumAllocdFrameSlots = num_frameslots_new;
}



void  Brush::uComputeBlendBounds( )
{
    fBlendBoundaries[0] = (0.5f / fHalfWidth) * fNumBlendPixels;
    fBlendBoundaries[1] = 1.0f - ((0.5f / fHalfWidth) * fNumBlendPixels);
}



void  Brush::uComputeAppendThreshold( )
{
    static  const float  kCos45Deg = 0.7071067812f;
    fAppendThreshold = 1.4f * fHalfWidth * kCos45Deg;
}



void  Brush::startStroke( )
{
    if (! isLinked( ))
        throw std::logic_error("Brush: can't start stroke: brush "
            "isn't linked");

    if (isStrokeInProgress( ))
        throw std::logic_error("Brush: can't start stroke: stroke "
            "already in progress");

    fPerpMode = kRightHandedMode;
    fPreNode = kNullNode;
    fPrePerp = kNullPerp;
    fPreDisp = kNullDisplacement;
    fPreArcLength = 0.0f;
    fNumUserFrameVerts = 0;
    fNumSamples = 0;
    fNumExtFrameVerts = 0;
    fNeedsSeparationCap.clear( );
    fDecalRuns.clear( );

    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(kSlotsPerFrameVertex, GL_FLOAT, 0, fFrameVtxCoordStore);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glTexCoordPointer(kSlotsPerFrameVertex, GL_FLOAT, 0,
        fFeatherTexCoordStore);

    fIsStrokeAssemblyInProgress = true;
}



void  Brush::stopStroke( )
{
    if (! isStrokeInProgress( ))
        throw std::logic_error("Brush: can't stop stroke: no stroke "
            "is in progress");

    glUseProgram(0);
    glDisable(GL_BLEND);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);

    uPostProcess( );

    fIsStrokeAssemblyInProgress = false;
}



void  Brush::refreshStroke( ) const
{
    /* it's an error to visually refresh the stroke if no stroke is in
       progress */
    if (! isStrokeInProgress( ))
        throw std::logic_error("BrushStroke: can't refresh proxy: assembly "
            "isn't in progress");

    /* draw the translucent proxy stroke frame -- we use a programmable shader
       to do this and take our coordinates from the frame vertex coordinate
       store */
    uUseProxyShaders( );
    glVertexPointer(kSlotsPerFrameVertex, GL_FLOAT, 0, fFrameVtxCoordStore);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glDrawArrays(GL_QUADS, 0, fNumUserFrameVerts);
    glDisable(GL_BLEND);

    /* draw the stroke central axis -- we use fixed functionality to do this
       and take our coordinates from the sample point store */
    glUseProgram(0);
    glColor4f(0.5f, 0.7f, 1.0f, 1.0f);
    glVertexPointer(kSlotsPerSample, GL_FLOAT, 0, fSamplePointStore);
    glDrawArrays(GL_LINE_STRIP, 0, fNumSamples);

    glFlush( );
}



bool  Brush::isStrokeInProgress( ) const
{
    return fIsStrokeAssemblyInProgress;
}



void  Brush::uInterchangePerpMode( )
{
    if (fPerpMode == kRightHandedMode)
        fPerpMode = kLeftHandedMode;
    else
        fPerpMode = kRightHandedMode;
}



void  Brush::uInitTextures( )
{
    /* this method now has a null implementation, because we no longer use
       pre-computed noise texture; instead, we compute noise values in
       the shader */
    ;
}



void  Brush::uInitPrograms( )
{
    GPUInterface& gpu = * fGPU;

    const std::string  kLibPath =
        Application::instance( ).startupDirectory( ) + "\\Library\\";

    gpu.setImagingProgramsDirectory(kLibPath);

    fUnifVertexProgID = gpu.prepareVertexProgram("StrokeV.glsl");
    fProxyFragmentProgID = gpu.prepareFragmentProgram("ProxyStrokeF.glsl");
    fDecalFragmentProgID = gpu.prepareFragmentProgram("DecalStrokeF.glsl");

    GLuint  proxyprogs[2];
    proxyprogs[0] = fUnifVertexProgID;
    proxyprogs[1] = fProxyFragmentProgID;
    fProxyProgSetID = gpu.prepareProgramSet(proxyprogs, 2);

    GLuint  decalprogs[2];
    decalprogs[0] = fUnifVertexProgID;
    decalprogs[1] = fDecalFragmentProgID;
    fDecalProgSetID = gpu.prepareProgramSet(decalprogs, 2);
}



void  Brush::appendStroke(float x, float y)
{
    /* append can only be called when stroke assembly is in progress */
    if (! isStrokeInProgress( ))
        throw std::logic_error("Brush: can't append to stroke: no stroke "
            "is in progress");

    /* if we've used all the available array slots, then grow the array
       before we add any more data to it */
    if ((fNumUserFrameVerts * kSlotsPerFrameVertex) == fNumAllocdFrameSlots)
        uGrowStores( );

    Vector2Df  thisnode = Vector2Df(x, y);

    /* if the prenode is null, then this is the first node in the
       stroke, but since we need at least two nodes to build a quad,
       just set up the prenode value for the next invocation of
       append( ) and return */
    if (fPreNode == kNullNode) {

        fPreNode = thisnode;

        uSetSamplePos(fNumSamples, thisnode);
        fNumSamples++;

        return;
    } /* if the prenode is null */

    /* if the length of displacement vector from the prenode to this node
       is less than the threshold value, then just return control to the
       caller since the two nodes aren't sufficiently separated to build
       a quad from */
    Vector2Df  thisdisp = thisnode - fPreNode;
    if (thisdisp.length( ) < fAppendThreshold)
        return;

    /* add the current sample to the sample point store */
    uSetSamplePos(fNumSamples, thisnode);
    fNumSamples++;

    /* compute the value of the arc length parameter, that is, the distance
       along the path of the brush stroke from the point the user first
       depressed the mouse to the current node -- we use this value for
       generating texture coordinates */
    float  thisarclength = fPreArcLength + thisdisp.length( );

    /* generate a vector perpendicular to the displacement vector, whose
       length is one half the cross-sectional width of the brush. Note
       that in the plane, there are two such valid perpendicular vectors;
       they lie on the same line, but point in opposite directions. We
       call them the right-handed and left-handed perpediculars, and
       we need to compute the one approrpriate to the mode currently
       in effect. */
    Vector2Df  thisperp = uGeneratePerp(thisdisp);

    /* detect separation */
    bool  isseparated = false;
    if (fPreDisp != kNullDisplacement) {
        if (dot(thisperp, fPrePerp.normalized( )) < kSepThresholdCosine) {

            uInterchangePerpMode( );
            thisperp = uGeneratePerp(thisdisp);

            isseparated = true;
        }
    }

    thisperp *= fHalfWidth;

    /* if the previously captured node has no perpendicular, then clone
       this one into it (this, for example, is case for a brush stroke
       made up of only two nodes -- a start node and stop node -- from
       which we can generate a single quad). But if the previously
       captured node does have a perpendicular, then average that
       perpendicular with the current one and clone the averaged
       perpendicular back into the pre-node (this averaging smooths
       stroke edges) */
    if (fPrePerp == kNullPerp) {

        fPrePerp = thisperp;
    }
    else {

        fPrePerp = ((thisperp + fPrePerp) * 0.5f).normalized( ) * fHalfWidth;
    }

    /* compute the four vertices of the quad formed between the prenode
       and the current node */
    Vector2Df  quadvert0 = fPreNode + fPrePerp;
    Vector2Df  quadvert1 = fPreNode - fPrePerp;
    Vector2Df  quadvert2 = thisnode - thisperp;
    Vector2Df  quadvert3 = thisnode + thisperp;

    /* write the texture and vertex coordinate values for all four points
       making up the quad into the appropriate slots in the packed geometry &
       texture data array */
    int  startslot = fNumUserFrameVerts * kSlotsPerFrameVertex;
    int  currslot = startslot;

    /* feather tex s coord for v0 */
    fFeatherTexCoordStore[currslot] = 1.0f;

    /* decal tex s coord for v0 */
    fDecalTexCoordStore[currslot] = fPreArcLength;

    /* x coord for v0 */
    fFrameVtxCoordStore[currslot] = quadvert0.x;

    currslot++;

    /* feather tex t coord for v0 */
    fFeatherTexCoordStore[currslot] = 1.0f;

    /* decal tex t coord for v0 */
    fDecalTexCoordStore[currslot] = 2.0f * fHalfWidth;

    /* y coord for v0 */
    fFrameVtxCoordStore[currslot] = quadvert0.y;

    currslot++;

    /* feather tex s coord for v1 */
    fFeatherTexCoordStore[currslot] = 1.0f;

    /* decal tex s coord for v1 */
    fDecalTexCoordStore[currslot] = fPreArcLength;

    /* x coord for v1 */
    fFrameVtxCoordStore[currslot] = quadvert1.x;

    currslot++;

    /* feather tex t coord for v1 */
    fFeatherTexCoordStore[currslot] = 0.0f;

    /* decal tex t coord for v1 */
    fDecalTexCoordStore[currslot] = 0.0f;

    /* y coord for v1 */
    fFrameVtxCoordStore[currslot] = quadvert1.y;

    currslot++;

    /* feather tex s coord for v2 */
    fFeatherTexCoordStore[currslot] = 1.0f;

    /* decal tex s coord for v2 */
    fDecalTexCoordStore[currslot] = thisarclength;

    /* x coord for v2 */
    fFrameVtxCoordStore[currslot] = quadvert2.x;

    currslot++;

    /* feather tex t coord for v2 */
    fFeatherTexCoordStore[currslot] = 0.0f;

    /* decal tex t coord for v2 */
    fDecalTexCoordStore[currslot] = 0.0f;

    /* y coord for v2 */
    fFrameVtxCoordStore[currslot] = quadvert2.y;

    currslot++;

    /* feather tex s coord for v3 */
    fFeatherTexCoordStore[currslot] = 1.0f;

    /* decal tex s coord for v3 */
    fDecalTexCoordStore[currslot] = thisarclength;

    /* x coord for v3 */
    fFrameVtxCoordStore[currslot] = quadvert3.x;

    currslot++;

    /* feather tex t coord for v3 */
    fFeatherTexCoordStore[currslot] = 1.0f;

    /* decal tex t coord for v3 */
    fDecalTexCoordStore[currslot] = 2.0f * fHalfWidth;

    /* y coord for v3 */
    fFrameVtxCoordStore[currslot] = quadvert3.y;

    currslot++;

    /* if we've already drawn at least one complete quad (which is to say
       we're in the process of drawing at least the second quad now), then
       we need to back-correct the two forward vertices of the quad before
       this one so that two forward vertices of the previous quad align
       with the two rear vertices of the current quad */
    if (fNumUserFrameVerts > 3) {

        /* get slot number where (x, y) coordinates of vertex 2 of the
           previous quad begin */
        int modslot = ((fNumUserFrameVerts - 2) * kSlotsPerFrameVertex);

        fFrameVtxCoordStore[modslot++] = quadvert1.x;

        fFrameVtxCoordStore[modslot++] = quadvert1.y;

        fFrameVtxCoordStore[modslot++] = quadvert0.x;

        fFrameVtxCoordStore[modslot] = quadvert0.y;
    }

    fPreNode = thisnode;
    fPrePerp = thisperp;
    fPreDisp = thisdisp.normalized( );
    fPreArcLength = thisarclength;
    fNumUserFrameVerts += 4;

    if (isseparated)
        fNeedsSeparationCap.push_back(fNumUserFrameVerts - 8);
}



void  Brush::uSetSamplePos(int samplenum, const Vector2Df& pos)
{
    int slot = uGetSampleSlot(samplenum);

    fSamplePointStore[slot] = pos.x;
    fSamplePointStore[slot + 1] = pos.y;
}



int  Brush::uGetSampleSlot(int samplenum)
{
    return samplenum * kSlotsPerSample;
}



void  Brush::uUseProxyShaders( ) const
{
    glUseProgram(fProxyProgSetID);
}



int  Brush::uGetFrameSlot(int vtxnum) const
{
    return vtxnum * kSlotsPerFrameVertex;
}



Vector2Df  Brush::uGetFramePos(int vtxnum)
{
    int slot = uGetFrameSlot(vtxnum);

    return Vector2Df(fFrameVtxCoordStore[slot],
        fFrameVtxCoordStore[slot + 1]);
}



void  Brush::uSetFramePos(int vtxnum, const Vector2Df& pos)
{
    int slot = uGetFrameSlot(vtxnum);

    fFrameVtxCoordStore[slot] = pos.x;
    fFrameVtxCoordStore[slot + 1] = pos.y;
}



void  Brush::uSetFrameFeatherTexS(int vtxnum, float s)
{
    int slot = uGetFrameSlot(vtxnum);

    fFeatherTexCoordStore[slot] = s;
}



void  Brush::uSetFrameFeatherTexT(int vtxnum, float t)
{
    int slot = uGetFrameSlot(vtxnum) + 1;

    fFeatherTexCoordStore[slot] = t;
}



float  Brush::uGetFrameFeatherTexT(int vtxnum) const
{
    int slot = uGetFrameSlot(vtxnum) + 1;

    return fFeatherTexCoordStore[slot];
}



float  Brush::uGetFrameDecalTexS(int vtxnum) const
{
    int slot = uGetFrameSlot(vtxnum);

    return fDecalTexCoordStore[slot];
}



void   Brush::uSetFrameDecalTexS(int vtxnum, float s)
{
    int slot = uGetFrameSlot(vtxnum);

    fDecalTexCoordStore[slot] = s;
}



float  Brush::uGetFrameDecalTexT(int vtxnum) const
{
    int slot = uGetFrameSlot(vtxnum) + 1;

    return fDecalTexCoordStore[slot];
}



void   Brush::uSetFrameDecalTexT(int vtxnum, float t)
{
    int slot = uGetFrameSlot(vtxnum) + 1;

    fDecalTexCoordStore[slot] = t;
}



Vector2Df  Brush::uGetSamplePos(int samplenum)
{
    int slot = uGetSampleSlot(samplenum);

    return Vector2Df(fSamplePointStore[slot], fSamplePointStore[slot + 1]);
}



Vector2Df  Brush::uGetAssociatedSamplePos(int vtxnum)
{
    int samplenum = (vtxnum / 4) + ((vtxnum % 4) / 2);

    return uGetSamplePos(samplenum);
}



void  Brush::uMakeFadeInCap(int src_vnum, int dest_vnum)
{
    Vector2Df  v0 = uGetFramePos(src_vnum);
    Vector2Df  v1 = uGetFramePos(src_vnum + 1);
    Vector2Df  v2 = uGetFramePos(src_vnum + 2);
    Vector2Df  v3 = uGetFramePos(src_vnum + 3);

    Vector2Df  va = v1 + ((v1 - v2).normalized( ) * fNumBlendPixels);
    Vector2Df  vb = v0 + ((v0 - v3).normalized( ) * fNumBlendPixels);

    float  feat_t03 = uGetFrameFeatherTexT(src_vnum);
    float  dec_t03 = uGetFrameDecalTexT(src_vnum);
    src_vnum++;

    float  dec_t12 = uGetFrameDecalTexT(src_vnum);
    float  feat_t12 = uGetFrameFeatherTexT(src_vnum);

    uSetFramePos(dest_vnum, va);
    uSetFrameFeatherTexS(dest_vnum, 0.0f);
    uSetFrameFeatherTexT(dest_vnum, feat_t12);
    uSetFrameDecalTexS(dest_vnum, 0.0f);
    uSetFrameDecalTexT(dest_vnum, dec_t12);
    dest_vnum++;

    uSetFramePos(dest_vnum, vb);
    uSetFrameFeatherTexS(dest_vnum, 0.0f);
    uSetFrameFeatherTexT(dest_vnum, feat_t03);
    uSetFrameDecalTexS(dest_vnum, 0.0f);
    uSetFrameDecalTexT(dest_vnum, dec_t03);
    dest_vnum++;

    uSetFramePos(dest_vnum, v0);
    uSetFrameFeatherTexS(dest_vnum, 1.0f);
    uSetFrameFeatherTexT(dest_vnum, feat_t03);
    uSetFrameDecalTexS(dest_vnum, fNumBlendPixels);
    uSetFrameDecalTexT(dest_vnum, dec_t03);
    dest_vnum++;

    uSetFramePos(dest_vnum, v1);
    uSetFrameFeatherTexS(dest_vnum, 1.0f);
    uSetFrameFeatherTexT(dest_vnum, feat_t12);
    uSetFrameDecalTexS(dest_vnum, fNumBlendPixels);
    uSetFrameDecalTexT(dest_vnum, dec_t12);
}



void  Brush::uMakeFadeOutCap(int src_vnum, int dest_vnum)
{
    Vector2Df  v0 = uGetFramePos(src_vnum);
    Vector2Df  v1 = uGetFramePos(src_vnum + 1);
    Vector2Df  v2 = uGetFramePos(src_vnum + 2);
    Vector2Df  v3 = uGetFramePos(src_vnum + 3);

    Vector2Df  va = v2 + ((v2 - v1).normalized( ) * fNumBlendPixels);
    Vector2Df  vb = v3 + ((v3 - v0).normalized( ) * fNumBlendPixels);

    float  feat_t03 = uGetFrameFeatherTexT(src_vnum);
    float  dec_t03 = uGetFrameDecalTexT(src_vnum);
    src_vnum++;

    float  dec_t12 = uGetFrameDecalTexT(src_vnum);
    float  feat_t12 = uGetFrameFeatherTexT(src_vnum);

    uSetFramePos(dest_vnum, va);
    uSetFrameFeatherTexS(dest_vnum, 0.0f);
    uSetFrameFeatherTexT(dest_vnum, feat_t12);
    uSetFrameDecalTexT(dest_vnum, dec_t12);
    dest_vnum++;

    uSetFramePos(dest_vnum, vb);
    uSetFrameFeatherTexS(dest_vnum, 0.0f);
    uSetFrameFeatherTexT(dest_vnum, feat_t03);
    uSetFrameDecalTexT(dest_vnum, dec_t03);
    dest_vnum++;

    uSetFramePos(dest_vnum, v3);
    uSetFrameFeatherTexS(dest_vnum, 1.0f);
    uSetFrameFeatherTexT(dest_vnum, feat_t03);
    uSetFrameDecalTexT(dest_vnum, dec_t03);
    dest_vnum++;

    uSetFramePos(dest_vnum, v2);
    uSetFrameFeatherTexS(dest_vnum, 1.0f);
    uSetFrameFeatherTexT(dest_vnum, feat_t12);
    uSetFrameDecalTexT(dest_vnum, dec_t12);
}



void  Brush::uMakeSeparationCap(int src_v0num, int destnum)
{
    /* see diagram in design memo ??? for a visual representation of
       these points */
    Vector2Df  va = uGetAssociatedSamplePos(src_v0num + 6);
    Vector2Df  vb = uGetAssociatedSamplePos(src_v0num);
    Vector2Df  inb_v0 = uGetFramePos(src_v0num);
    Vector2Df  inb_v1 = uGetFramePos(src_v0num + 1);
    Vector2Df  inb_v2 = uGetFramePos(src_v0num + 2);
    Vector2Df  inb_v3 = uGetFramePos(src_v0num + 3);
    Vector2Df  outb_v0 = uGetFramePos(src_v0num + 4);
    Vector2Df  outb_v1 = uGetFramePos(src_v0num + 5);
    Vector2Df  outb_v2 = uGetFramePos(src_v0num + 6);
    Vector2Df  outb_v3 = uGetFramePos(src_v0num + 7);

    float  a_to_3 = (inb_v3 - va).length( );
    float  a_to_2 = (inb_v2 - va).length( );
    float  b_to_1 = (outb_v1 - vb).length( );
    float  b_to_0 = (outb_v0 - vb).length( );

    Vector2Df inb_remote;
    Vector2Df inb_link;
    if (a_to_3 > a_to_2) {

        inb_remote = inb_v3 + ((inb_v3 - inb_v0).normalized( ) *
            fNumBlendPixels);
        inb_link = inb_v3;
    }
    else {

        inb_remote = inb_v2 + ((inb_v2 - inb_v1).normalized( ) *
            fNumBlendPixels);
        inb_link = inb_v2;
    }

    Vector2Df outb_remote;
    Vector2Df outb_link;
    float     indep_feat_t;
    float     indep_dec_t;
    if (b_to_1 > b_to_0) {

        outb_remote = outb_v1 + ((outb_v1 - outb_v2).normalized( ) *
            fNumBlendPixels);
        outb_link = outb_v1;
        indep_feat_t = uGetFrameFeatherTexT(src_v0num + 5);
        indep_dec_t = uGetFrameDecalTexT(src_v0num + 5);
    }
    else {

        outb_remote = outb_v0 + ((outb_v0 - outb_v3).normalized( ) *
            fNumBlendPixels);
        outb_link = outb_v0;
        indep_feat_t = uGetFrameFeatherTexT(src_v0num + 4);
        indep_dec_t = uGetFrameDecalTexT(src_v0num + 4);
    }
    float  dep_feat_t = (indep_feat_t == 0.0f) ? 1.0f : 0.0f;
    float  dep_dec_t = (indep_dec_t == 0.0f) ? (2.0f * fHalfWidth) : 0.0f;

    uSetFramePos(destnum, inb_remote);
    uSetFrameFeatherTexS(destnum, 0.0f);
    uSetFrameFeatherTexT(destnum, dep_feat_t);
    uSetFrameDecalTexT(destnum, dep_dec_t);
    destnum++;
    
    uSetFramePos(destnum, outb_remote);
    uSetFrameFeatherTexS(destnum, 0.0f);
    uSetFrameFeatherTexT(destnum, indep_feat_t);
    uSetFrameDecalTexT(destnum, indep_dec_t);
    destnum++;
    
    uSetFramePos(destnum, outb_link);
    uSetFrameFeatherTexS(destnum, 1.0f);
    uSetFrameFeatherTexT(destnum, indep_feat_t);
    uSetFrameDecalTexT(destnum, indep_dec_t);
    destnum++;

    uSetFramePos(destnum, inb_link);
    uSetFrameFeatherTexS(destnum, 1.0f);
    uSetFrameFeatherTexT(destnum, dep_feat_t);
    uSetFrameDecalTexT(destnum, dep_dec_t);
}



void  Brush::uPostProcess( )
{
    uGenerateExtGeometry( );
    uComputeDecalRuns( );
    uNormalizeDecalCoords( );
}



void  Brush::uUseDecalShaders( ) const
{
    glUseProgram(fDecalProgSetID);

    GPUInterface& gpu = (* fGPU);
    gpu.setUniform1fv(fDecalProgSetID, "BlendBoundaries",
        & fBlendBoundaries[0], & fBlendBoundaries[2]);
    gpu.setUniform1i(fDecalProgSetID, "DecalTex", 0);
    if (fIsEdgeNoiseEnabled)
        gpu.setUniform1i(fDecalProgSetID, "IsNoiseEnabled", 1);
    else
        gpu.setUniform1i(fDecalProgSetID, "IsNoiseEnabled", 0);
}



bool  Brush::uIsDecalRunOpen( ) const
{
    int  lastrunidx = static_cast<int>(fDecalRuns.size( )) - 1;

    /* if the decal run collection is empty, no run is open */
    if (lastrunidx == -1)
        return false;
    
    return  fDecalRuns[lastrunidx].isOpen;
}




void  Brush::uOpenDecalRun(int userstartvtx, int extstartvtx)
{
    if (uIsDecalRunOpen( ))
        throw std::logic_error("uOpenDecalRun: can't open a new run. An "
            "existing run is already open");


    int        offset;
    Vector2Di  anchor;
    if (fDecalRuns.empty( )) {

        offset = 0;
        anchor.x = (std::rand( ) % 64) * fSynAgent->precursorQuantization( );
        anchor.y = (std::rand( ) % 64) * fSynAgent->precursorQuantization( );
    }
    else {

        offset = fDecalRuns[fDecalRuns.size( ) - 1].synHorizOffset +
            fDecalRuns[fDecalRuns.size( ) - 1].synLength;
        anchor = fDecalRuns[fDecalRuns.size( ) - 1].anchor;
    }

    DecalRun  newrun;
    
    newrun.userStartVertex = userstartvtx;
    newrun.extStartVertex = extstartvtx;
    newrun.userNumVertices = 0;
    newrun.extNumVertices = 0;
    newrun.synHorizOffset = offset;
    newrun.isOpen = true;
    newrun.anchor = anchor;
    
    fDecalRuns.push_back(newrun);
}



void  Brush::uCloseDecalRun(int userstopvtx, int extstopvtx)
{
    if (! uIsDecalRunOpen( ))
        throw std::logic_error("uCloseDecalRun: can't close run. No run is "
            "open");

    int  runidx = static_cast<int>(fDecalRuns.size( )) - 1;

    int  usernumverts = userstopvtx - fDecalRuns[runidx].userStartVertex + 1;
    int  extnumverts = extstopvtx - fDecalRuns[runidx].extStartVertex + 1;

    int  user_lobound = (usernumverts > 0) ? static_cast<int>(
        uGetFrameDecalTexS(fDecalRuns[runidx].userStartVertex)) : INT_MAX;
    int  ext_lobound = (extnumverts > 0) ? static_cast<int>(
        uGetFrameDecalTexS(fDecalRuns[runidx].extStartVertex)) : INT_MAX;
    int  lobound = min(user_lobound, ext_lobound);

    int  ext_hibound = -1;
    if (extnumverts > 0) {
        
        if (extstopvtx == (fNumUserFrameVerts + fNumExtFrameVerts - 1))
            ext_hibound = static_cast<int>(uGetFrameDecalTexS(extstopvtx - 2));
        else
            ext_hibound = static_cast<int>(uGetFrameDecalTexS(extstopvtx));
    }
    int  user_hibound = (usernumverts > 0) ? static_cast<int>(
        uGetFrameDecalTexS(userstopvtx)) : -1;
    int  hibound = max(user_hibound, ext_hibound);

    int  synlength = (hibound - lobound) + 2;

    fDecalRuns[runidx].isOpen = false;
    fDecalRuns[runidx].userNumVertices = usernumverts;
    fDecalRuns[runidx].extNumVertices = extnumverts;
    fDecalRuns[runidx].synLength = synlength;
}



void  Brush::uGenerateExtGeometry( )
{    
    int  vtxticker = fNumUserFrameVerts;
    
    /* check to see if the existing stores are full and grow them if
       necessary */
    if ((vtxticker * kSlotsPerFrameVertex) == fNumAllocdFrameSlots)
        uGrowStores( );
    
    /* generate the fade-in cap at the beginning of the stroke and advance
       the vertex ticker */
    uMakeFadeInCap(0, vtxticker);
    vtxticker += 4;
    
    /* generate the separation point caps for all separation points along
       the stroke that need caps, advancing the vertex ticker each time and
       growing the stores if necessary */
    std::vector<int>::const_iterator  needscap_it =
        fNeedsSeparationCap.begin( );
    for ( ; needscap_it != fNeedsSeparationCap.end( ); needscap_it++) {
    
        if ((vtxticker * kSlotsPerFrameVertex) == fNumAllocdFrameSlots)
            uGrowStores( ); 
        uMakeSeparationCap(*needscap_it, vtxticker);
        vtxticker += 4;
        
    } /* for all separation points needing cap generation */
    
    /* grow the stores if necessary, then generate the fade-out cap at the
       end of the stroke and advance the vertex ticker */
    if ((vtxticker * kSlotsPerFrameVertex) == fNumAllocdFrameSlots)
        uGrowStores( ); 
    uMakeFadeOutCap(fNumUserFrameVerts - 4, vtxticker);
    vtxticker += 4;
    
    /* use the vertex ticker to compute the number of extended geometry
       vertices we've added to the stores */
    fNumExtFrameVerts = vtxticker - fNumUserFrameVerts;   
}



void  Brush::uComputeDecalRuns( )
{
    const float  caplen = fNumBlendPixels;     // length of fade caps (px)

    float        pd_arclen = caplen;           // per-decal run arclength

    int          septicker = 0;                // the index number of the
                                               // next separation cap in the
                                               // ordered sequence of caps

    const float  breakincr =                   // max arc length increment
        static_cast<float>(                    // between decal run breaks
        SynthesisAgent::kWorkTexExtent -
        fSynAgent->precursorQuantization( ));
      
    /* before starting to examine quads open the first run */
    uOpenDecalRun(0, fNumUserFrameVerts);
    
    /* examine all the quads in the stroke, generating decal run breaks where
       necessary and computing decal texture s-coords; we examine all quads
       (both user quads and extended quads), but since each extended quad is
       associated with a user quad, we only loop over the user quads, getting
       at the associated extended quads where necessary */
    for (int vnum = 0; vnum < fNumUserFrameVerts; vnum += 4) {
    
        float  raw_arclen_01 = uGetFrameDecalTexS(vnum);
        float  raw_arclen_23 = uGetFrameDecalTexS(vnum + 2);
        float  delta_arclen = (raw_arclen_23 - raw_arclen_01);
        
        /* processing this user quad could trigger a decal run break */
        if ((pd_arclen + delta_arclen) >= breakincr) {
        
            int  userstopvtx = vnum - 1;
            int  extstopvtx = fNumUserFrameVerts + (4 * septicker) + 3;
        
            uCloseDecalRun(userstopvtx, extstopvtx);
            uOpenDecalRun(userstopvtx + 1, extstopvtx + 1);
            
            pd_arclen = 0.0f;
        } /* if a decal run break was triggered */

        uSetFrameDecalTexS(vnum, pd_arclen );
        uSetFrameDecalTexS(vnum + 1, pd_arclen);
        uSetFrameDecalTexS(vnum + 2, pd_arclen + delta_arclen);
        uSetFrameDecalTexS(vnum + 3, pd_arclen + delta_arclen);

        pd_arclen += delta_arclen;

        int  nextsep_vnum = -1;
        if (septicker < static_cast<int>(fNeedsSeparationCap.size( )))
            nextsep_vnum = fNeedsSeparationCap[septicker];
        
        if (vnum == nextsep_vnum) {
        
            /* 4 is added because we have to skip over the 4 verts of fade-in
               quad */
            int  capv0num = fNumUserFrameVerts + (4 * septicker) + 4;

            /* processing this separation cap could trigger a run break */
            if ((pd_arclen + caplen) >= breakincr) {
            
                int  userstopvtx = vnum + 3;
                int  extstopvtx = capv0num - 1;
            
                uCloseDecalRun(userstopvtx, extstopvtx);
                uOpenDecalRun(userstopvtx + 1, extstopvtx + 1);

                pd_arclen = 0.0f;
            } /* if a decal run break was triggered */
                        
            uSetFrameDecalTexS(capv0num, pd_arclen);
            uSetFrameDecalTexS(capv0num + 1, pd_arclen);
            uSetFrameDecalTexS(capv0num + 2, pd_arclen + caplen);
            uSetFrameDecalTexS(capv0num + 3, pd_arclen + caplen);
            
            pd_arclen += caplen;
            
            septicker++;
        } /* if vtx with seq number 'vnum' anchors the next separation */
    } /* for the number 0 vertex of all user quads */
    
    /* process decal texture s-coordinates for the fade-out endcap quad;
       note that processing these could trigger a decal run break */
    int    fadeout_vnum = (4 * septicker) + fNumUserFrameVerts + 4;
    float  endstart_arclen = uGetFrameDecalTexS(fNumUserFrameVerts - 1);
    if ((pd_arclen + caplen) >= breakincr) {
    
        int  userstopvtx = fNumUserFrameVerts - 1;
        int  extstopvtx = fNumUserFrameVerts + (4 * septicker) + 3;
    
        uCloseDecalRun(userstopvtx, extstopvtx);
        uOpenDecalRun(fNumUserFrameVerts, extstopvtx + 1);
    
        pd_arclen = 0.0f;
    } /* if a decal run break was triggered */
    
    uSetFrameDecalTexS(fadeout_vnum, endstart_arclen + caplen);
    uSetFrameDecalTexS(fadeout_vnum + 1, endstart_arclen + caplen);
    uSetFrameDecalTexS(fadeout_vnum + 2, endstart_arclen);
    uSetFrameDecalTexS(fadeout_vnum + 3, endstart_arclen);

    pd_arclen += caplen;
    
    /* close the last open decal run */
    uCloseDecalRun(fNumUserFrameVerts - 1, fNumUserFrameVerts +
        fNumExtFrameVerts - 1);    
}



void  Brush::uNormalizeDecalCoords( )
{
    int    storesize = (fNumUserFrameVerts + fNumExtFrameVerts) *
        kSlotsPerFrameVertex;
    for (int i = 0; i < storesize; i += kSlotsPerFrameVertex) {

        fDecalTexCoordStore[i] *= SynthesisAgent::kOutTexPixelFraction;
        fDecalTexCoordStore[i + 1] *= SynthesisAgent::kOutTexPixelFraction;
    }
}



void  Brush::uCreateSynAgent( )
{
    RGBAImage*  preimg = RGBAImage::createMappedImage(
        RGBAImage::kTargaImageFormat, fSourceImagePath);

    fSimMap.reset(new PyramidalSimilarityMap(fSimMapPath));

    fSynAgent.reset(new SynthesisAgent(*preimg, *fGPU, *fSimMap));

    delete preimg;
}



void  Brush::uDrawDecalRun(const DecalRun& dr) const
{
    int  syn_x = dr.synHorizOffset + dr.anchor.x;
    int  syn_y = dr.anchor.y;
    int  syn_width = dr.synLength + 8;
    int  syn_height = static_cast<int>(2.0f * fHalfWidth);

    GLuint syn_texid = fSynAgent->synthesize(syn_x, syn_y, syn_width,
        syn_height);

    /* engage the decal shaders */
    uUseDecalShaders( );

    /* bind the brush surface texture into texture unit 0 and configure it
       for smoother filtering -- this is necessary since the SynthesisAgent
       sets the filtering mode to GL_NEAREST for pixel-exactness */
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, syn_texid);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    /* enable blending and vertex arrays */
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(kSlotsPerFrameVertex, GL_FLOAT, 0, fFrameVtxCoordStore);

    /* enable & setup texcoord arrays on texture unit 1 */
    glClientActiveTexture(GL_TEXTURE1);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glTexCoordPointer(kSlotsPerFrameVertex, GL_FLOAT, 0,
        fFeatherTexCoordStore);

    /* enable & setup texcoord arrays on texture unit 0 */
    glClientActiveTexture(GL_TEXTURE0);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glTexCoordPointer(kSlotsPerFrameVertex, GL_FLOAT, 0,
        fDecalTexCoordStore);

    /* draw the user geometry */
    if (dr.userNumVertices > 0)
        glDrawArrays(GL_QUADS, dr.userStartVertex, dr.userNumVertices);

    /* draw the extended geometry */
    if (dr.extNumVertices > 0)
        glDrawArrays(GL_QUADS, dr.extStartVertex, dr.extNumVertices);

    /* disable texcoord arrays on texture unit 1 */
    glClientActiveTexture(GL_TEXTURE1);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);

    /* disable texcoord arrays on texture unit 0 */
    glClientActiveTexture(GL_TEXTURE0);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);

    /* disable vertex arrays & blending */
    glDisableClientState(GL_VERTEX_ARRAY);
    glDisable(GL_BLEND);

    /* disengage the decal shaders */
    glUseProgram(0);
}



void  Brush::drawStroke( )
{
    /* it's an error to invoke draw on an unlinked brush */
    if (! isLinked( ))
        throw std::logic_error("Brush: can't draw stroke: brush isn't "
            "linked");

    /* draw is a no-op if called when stroke assembly is in progress */
    if (isStrokeInProgress( ))
        return;

    /* if there's nothing to draw then return immediately */
    if (fPrePerp == kNullPerp)
        return;

    std::vector<DecalRun>::const_iterator  it = fDecalRuns.begin( );
    for ( ; it != fDecalRuns.end( ); it++)
        uDrawDecalRun(*it);
}



void  Brush::thingyItemActivated(DialogThingy&, int)
{
}



void  Brush::thingyClosed(DialogThingy&)
{
}



void  Brush::thingyInteractionStarted(DialogThingy& sender)
{
    sender.themeItem(kBrushLinkWaitTextResID);
    sender.show( );
    ShowWindow(GetDlgItem(sender.adapter( ).primitive( ), kBrushLinkWaitTextResID),
        SW_SHOW);
    UpdateWindow(sender.adapter( ).primitive( ));

    /* upon linkage, allocate the geometrty stores */
    const int  kNumInitFrameSlotsAllocd = kSlotsPerFrameVertex *
        kNumInitVerticesAllocd;
    const int  kNumInitSampleSlotsAllocd = kSlotsPerSample *
        kNumInitSamplesAllocd;

    fFrameVtxCoordStore = new GLfloat [kNumInitFrameSlotsAllocd];
    fFeatherTexCoordStore = new GLfloat [kNumInitFrameSlotsAllocd];
    fDecalTexCoordStore = new GLfloat [kNumInitFrameSlotsAllocd];
    fSamplePointStore = new GLfloat [kNumInitSampleSlotsAllocd];
    fNumUserFrameVerts = 0;
    fNumSamples = 0;
    fNumAllocdFrameSlots = kNumInitFrameSlotsAllocd;

    /* upon linkage, load texture images from disk and download them into
       TRAM */
    uInitTextures( );

    /* upon linkage, prepare the vertex and fragment programs that are used
       to draw the decal and proxy strokes */
    uInitPrograms( );

    /* upon linkage, create the synthesis agent */
    uCreateSynAgent( );

    /* set the SynAgent's jump parameter */
    fSynAgent->setJumpParameter(fJumpParameter);

    sender.stopInteraction( );
}



void  Brush::thingyInteractionStopped(DialogThingy&)
{
}



void  Brush::layoutSubthingies(CompositeThingy& sender)
{
}



void  Brush::enableSubthingies(CompositeThingy& sender)
{
}



void  Brush::disableSubthingies(CompositeThingy& sender)
{
}



void  Brush::cloneAttributesFrom(const Brush& src)
{
    this->fHalfWidth = src.fHalfWidth;
    this->fNumBlendPixels = src.fNumBlendPixels;
    this->fIsEdgeNoiseEnabled = src.fIsEdgeNoiseEnabled;
    this->fSourceImagePath = src.fSourceImagePath;
    this->fSimMapPath = src.fSimMapPath;
    this->fJumpParameter = src.fJumpParameter;

    uComputeBlendBounds( );

    if (isLinked( )) {

        fSynAgent->setJumpParameter(src.fJumpParameter);

        /* load the new similarity map from disk */
        PyramidalSimilarityMap*  newmap = new PyramidalSimilarityMap(
            src.fSimMapPath);

        /* save a pointer to the old similarity map */
        PyramidalSimilarityMap*  oldmap = fSimMap.release( );

        /* set the new map as the map in use */
        fSimMap.reset(newmap);

        /* deallocate the old map */
        delete oldmap;

        /* tell the synthesis agent use the new map */
        fSynAgent->setSimilarityMap(*fSimMap);

        /* load the new source image from disk */
        RGBAImage*  newsrcimg = RGBAImage::createMappedImage(
            RGBAImage::kTargaImageFormat, fSourceImagePath);

        /* tell the synthesis agent to use the new source image */
        fSynAgent->setSourceImage(*newsrcimg);

        /* the synthesis agent copies the new source image into its
           state, so we can delete the loaded image now */
        delete newsrcimg;
    }
}
/////////////////////////////////////////////////////////////////////////////
// END  Brush                                                              //
/////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  BrushManager                                                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
Brush*  BrushManager::sSystemBrush = 0;

BrushManager::BrushManager(S580AppController& parent, std::string& dbpath)
    : fParent(&parent)
{
    /* map the brush database file into memory */
    std::string  dbcontents =
        Win32Tools::filesystem( ).mapTextFile(dbpath);

    /* create a structured reader to parse the brush database file */
    StructuredReader  dbparser = StructuredReader(dbcontents);

    /* install the library path macro on the structured reader */
    std::string  libpath = Application::instance( ).startupDirectory( ) +
        "\\Library";
    libpath[0] = std::tolower(libpath[0]);
    dbparser.installMacro("LIBPATH", libpath);

    /* install the user path macro on the structured reader */
    std::string  userpath = Application::instance( ).startupDirectory( ) +
        "\\User";
    userpath[0] = std::tolower(userpath[0]);
    dbparser.installMacro("USERPATH", userpath);

    /* parse the brush database file */
    uReadStore(dbparser);
}



BrushManager::~BrushManager( )
{
}



Brush*  BrushManager::uReadBrush(StructuredReader& reader)
{
    std::string  brushname = reader.readQuotedString( );

    reader.readGroupOpen( );

    std::string  simpath;
    bool         gotsimpath = false;
    std::string  srcpath;
    bool         gotsrcpath = false;
    int          width;
    bool         gotwidth = false;
    int          blendsize;
    bool         gotblendsize = false;
    bool         isnoisy;
    bool         gotisnoisy = false;
    float        jumpparam;
    bool         gotjumpparam = false;

    while ((!gotsimpath) || (!gotsrcpath) || (!gotwidth) ||
           (!gotblendsize) || (!gotisnoisy) || (!gotjumpparam)) {

        std::string  decl = reader.readIdentifier( );

        if (decl == "sourceImagePath") {

            reader.readAssignmentOperator( );
            srcpath = reader.readQuotedString( );
            gotsrcpath = true;
        }
        else if (decl == "similarityMapPath") {

            reader.readAssignmentOperator( );
            simpath = reader.readQuotedString( );
            gotsimpath = true;
        }
        else if (decl == "crossSecWidth") {

            reader.readAssignmentOperator( );
            width = reader.readInteger( );
            gotwidth = true;
        }
        else if (decl == "blendRegionSize") {

            reader.readAssignmentOperator( );
            blendsize = reader.readInteger( );
            gotblendsize = true;
        }
        else if (decl == "isNoiseEnabled") {

            reader.readAssignmentOperator( );
            isnoisy = reader.readBoolean( );
            gotisnoisy = true;
        }
        else if (decl == "jumpParameter") {

            reader.readAssignmentOperator( );
            jumpparam = reader.readFloat( );
            gotjumpparam = true;
        }
        else {

            throw std::runtime_error("BrushManager: malformed brush database "
                "file: brush declares an unrecognized property");
        }
    } /* while we haven't got all required brush properties */

    reader.readGroupClose( );

    return new Brush(brushname, srcpath, simpath, width, blendsize,
        jumpparam, isnoisy);
}



void  BrushManager::uReadStore(StructuredReader& reader)
{
    std::string  storetype = reader.readFiletypeID( );

    /* read and verify the filetype */
    if (storetype != "BRUSHDBASE")
        throw std::logic_error("BrushManager: brush database file is of the "
            "wrong type");

    /* read the file header segment */
    std::string  headertag = reader.readIdentifier( );
    if (headertag != "brushDatabase")
        throw std::logic_error("BrushManager: brush database file has a "
            "malformed header");

    reader.readGroupOpen( );

    bool         gotnumrecs = false;
    int          numrecs;
    bool         gotactivename = false;
    std::string  activename;
    while ((!gotnumrecs) || (!gotactivename)) {

        std::string  currtag = reader.readIdentifier( );

        if (currtag == "numRecords") {

            reader.readAssignmentOperator( );
            numrecs = reader.readInteger( );
            gotnumrecs = true;
        }
        else if (currtag == "activeBrush") {

            reader.readAssignmentOperator( );
            activename = reader.readQuotedString( );
            gotactivename = true;
        }
        else {

            throw std::logic_error("BrushManager: unrecognized property "
                "declaration in brush database header");
        }
    } /* while we haven't got all needed info */

    reader.readGroupClose( );

    for (int i = 0; i < numrecs; i++) {

        Brush* newbrush = uReadBrush(reader);

        install(newbrush);

    } /* for all brush records in the database */

    fActiveBrushName = activename;
}



std::vector<std::string>  BrushManager::manifest( ) const
{
    std::vector<std::string>  result;

    std::map<std::string, Brush*>::const_iterator  it;
    for (it = fBrushStore.begin( ); it != fBrushStore.end( ); it++) {

        result.push_back(it->second->name( ));
    } /* for all the brushes in the store */

    return result;
}



Brush&  BrushManager::find(const std::string& name)
{
    if (! isInstalled(name))
        throw std::logic_error("BrushManager: can't find brush: "
            "brush isn't installed");

    return (* (fBrushStore.find(name)->second));
}



const Brush&  BrushManager::find(const std::string& name) const
{
    if (! isInstalled(name))
        throw std::logic_error("BrushManager: can't find brush: "
            "brush isn't installed");

    return (* (fBrushStore.find(name)->second));
}



bool  BrushManager::isInstalled(const std::string& name) const
{
    if (fBrushStore.find(name) != fBrushStore.end( ))
        return true;
    else
        return false;
}



void  BrushManager::install(Brush* newbrush)
{
    if (isInstalled(newbrush->name( )))
        throw std::logic_error("BrushManager: can't install brush: a brush "
            "with the same name is already installed");

    fBrushStore.insert(std::make_pair(newbrush->name( ), newbrush));
}



void  BrushManager::remove(const std::string& name)
{
    if (! isInstalled(name))
        throw std::logic_error("BrushManager: can't remove brush: brush "
            "isn't installed");

    Brush*  nameval = & find(name);
    fBrushStore.erase(name);
    delete nameval;
}



void  BrushManager::save(const std::string& outpath) const
{
    std::ofstream  outstream(outpath.c_str( ), std::ios::out);

    if (!outstream)
        throw std::runtime_error("BrushManager: couldn't save brush "
            "database: output file couldn't be opened for writing");

    /* write the filetype identifier */
    outstream << "(# BRUSHDBASE)\n\n";

    /* write the file header */
    outstream << "brushDatabase {\n\n";
    outstream << "    numRecords  := " << fBrushStore.size( ) << "\n";
    outstream << "    activeBrush := \"" << activeBrush( ).name( ) << "\"\n";
    outstream << "}\n\n";

    /* step through each brush in the store and write its properties */
    std::map<std::string, Brush*>::const_iterator  it;
    for (it = fBrushStore.begin( ); it != fBrushStore.end( ); it++) {

        Brush&  currbrush = * it->second;

        outstream << "\"" << currbrush.name( ) << "\" {\n\n";

        /* if either of the properties whose values are filesystem path
           names (i.e. the "sourceImagePath" and "similarityMapPath"
           properties) have values that contain the library path, then
           don't write the literal library path when writing the
           values of these properties; instead write a macro (i.e.
           "$(LIBPATH)") that will expand into the full library path
           when the brush database is read back in */
        std::string  libpath =
            Application::instance( ).startupDirectory( ) + "\\Library";
        libpath[0] = std::tolower(libpath[0]);
        std::string  userpath =
            Application::instance( ).startupDirectory( ) + "\\User";
        userpath[0] = std::tolower(userpath[0]);

        std::string srcpath = currbrush.sourceImagePath( );
        srcpath[0] = std::tolower(srcpath[0]);
        if (srcpath.find(libpath) != std::string::npos) {

            size_t startpt = srcpath.find(libpath);
            srcpath.replace(startpt, libpath.length( ), "$(LIBPATH)");
        }
        if (srcpath.find(userpath) != std::string::npos) {

            size_t startpt = srcpath.find(userpath);
            srcpath.replace(startpt, userpath.length( ), "$(USERPATH)");
        }
        outstream << "    sourceImagePath   := \"" << srcpath << "\"\n";

        std::string simpath = currbrush.similarityMapPath( );
        simpath[0] = std::tolower(simpath[0]);
        if (simpath.find(libpath) != std::string::npos) {

            size_t startpt = simpath.find(libpath);
            simpath.replace(startpt, libpath.length( ), "$(LIBPATH)");
        }
        if (simpath.find(userpath) != std::string::npos) {

            size_t startpt = simpath.find(userpath);
            simpath.replace(startpt, userpath.length( ), "$(USERPATH)");
        }
        outstream << "    similarityMapPath := \"" << simpath << "\"\n";

        outstream << "    crossSecWidth     := " << currbrush.width( ) <<
            "\n";
        outstream << "    blendRegionSize   := " << currbrush.blendPixels( )
            << "\n";
		outstream.setf(std::ios::fixed);
		outstream << "    jumpParameter     := " << std::setprecision(4) <<
            currbrush.jumpParameter( ) << "\n";
        outstream << "    isNoiseEnabled    := ";
        if (currbrush.isEdgeNoiseEnabled( ))
            outstream << "true\n";
        else
            outstream << "false\n";
        outstream << "}\n\n";

    } /* for all brushes in the store */

    /* close the filestream */
    outstream.close( );
}



Brush&  BrushManager::activeBrush( )
{
    return find(fActiveBrushName);
}



const Brush&  BrushManager::activeBrush( ) const
{
    return find(fActiveBrushName);
}



void  BrushManager::setActiveBrush(const std::string& brushname)
{
    if (! isInstalled(brushname))
        throw std::logic_error("BrushManager: can't make brush '" +
            brushname + "' active: brush isn't installed");

    fActiveBrushName = brushname;

    parent( ).notifyBrushChanged( );
}



const Brush&  BrushManager::systemBrush( ) const
{
    if (! sSystemBrush) {

        const std::string  srcpath =
            Application::instance( ).startupDirectory( ) +
            "\\Library\\Van Gogh Default.tga";

        const std::string  simpath =
            Application::instance( ).startupDirectory( ) +
            "\\Library\\Van Gogh Default.simset";

        sSystemBrush = new Brush("System Brush", srcpath, simpath, 64, 16,
            1.1f, true);
    }

    return (* sSystemBrush);
}



std::string  BrushManager::uGenerateNewBrushName( ) const
{
    std::ostringstream  candidstream;

    int numticker = 1;
    while (true) {

        candidstream.str("");

        candidstream << "New Brush " << numticker++;

        if (! isInstalled(candidstream.str( )))
            return candidstream.str( );
    }
}



Brush&  BrushManager::newBrush( )
{
    std::string name = uGenerateNewBrushName( );

    Brush*  newbrush = new Brush(systemBrush( ));

    newbrush->setName(name);

    install(newbrush);

    return *newbrush;
}



int  BrushManager::numBrushes( ) const
{
    return ((int) fBrushStore.size( ));
}



Brush&  BrushManager::cloneBrush(const std::string& srcname)
{
    if (! isInstalled(srcname))
        throw std::logic_error("BrushManager: can't clone brush: specified "
            "clone source brush isn't installed");

    std::string  name = uGenerateCloneBrushName(srcname);

    Brush*  newbrush = new Brush(find(srcname));

    newbrush->setName(name);

    install(newbrush);

    return *newbrush;
}



std::string  BrushManager::uGenerateCloneBrushName(const std::string& name)
{
    std::ostringstream  candidstream;

    int numticker = 0;
    while (true) {

        candidstream.str("");

        candidstream << name;
        if (numticker == 0)
            candidstream << " Clone";
        else
            candidstream << " Clone " << numticker;
        numticker++;

        if (! isInstalled(candidstream.str( )))
            return candidstream.str( );
    }
}



S580AppController&  BrushManager::parent( )
{
    return *fParent;
}



const S580AppController&  BrushManager::parent( ) const
{
    return *fParent;
}
/////////////////////////////////////////////////////////////////////////////
// END  BrushManager                                                       //
/////////////////////////////////////////////////////////////////////////////
