//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// UIThingies.h                                                             //
//                                                                          //
// Packages interface definitions for classes Thingy, ThingyController,     //
// CompositeThingy, CompositeThingyController, ThingyAdapter and            //
// ThingyAdapterClient                                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  <unversioned module>                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  Copyright (c) 2008 Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef UI_THINGIES__HXX
#define UI_THINGIES__HXX

#include <Win32Support.h>
#include <PTCITypes.h>
#include <memory>
#include <string>
#include <map>

//-----------------------------------------------------------------------------
/* The cosmic base class for Thingies, it defines a basic set of methods
   to support standard user interface component management tasks,
   such as showing, hiding, sizing, positioning, enabling and disabling
   UI components. All Thingies implement this interface.*/
class  Thingy {

private:

    static int sChildIDTicker;

    /* Thingies don't have value semantics */
    Thingy(const Thingy&) { }
    const Thingy& operator=(const Thingy&) { return *this; }

protected:

    static int nextChildID( );

    Thingy( ) { }

public:

    virtual ~Thingy( ) { }

    /* equality for Thingies is reference equality */
    virtual  bool  operator==(const Thingy& rhs)
        { if (this == &rhs) return true; else return false; }

    virtual  void  show( ) = 0;
    virtual  void  hide( ) = 0;
    virtual  bool  isVisible( ) const = 0;
    virtual  Vector2Di  origin( ) const = 0;
    virtual  void  setOrigin(const Vector2Di& ogn) = 0;
    virtual  void  setOrigin(unsigned short x, unsigned short y) = 0;
    virtual  unsigned short  width( ) const = 0;
    virtual  void  setWidth(unsigned short w) = 0;
    virtual  unsigned short  height( ) const = 0;
    virtual  void  setHeight(unsigned short h) = 0;
    virtual  void  enable( ) = 0;
    virtual  void  disable( ) = 0;
    virtual  bool  isEnabled( ) const = 0;
};
//-----------------------------------------------------------------------------




//-----------------------------------------------------------------------------
/* The cosmic base class for ThingyControllers. All Thingy controllers
   implement this interface, but generally do so indirectly, through one of
   its more specialized subclasses. For example, objects that control
   PushButtonThingies implement the PushButtonController interface,
   which is a subclass of ThingyController */
class  ThingyController {

protected:

    ThingyController( ) { }

public:

    virtual  ~ThingyController( ) { }
};
//-----------------------------------------------------------------------------




//-----------------------------------------------------------------------------
/* Defines an interface that must extended by "Composite Thingies."
   Composite Thingies comprise the subset of thingies that can manage the
   appearance, placement and lifecycle of other thingies. For example,
   imagine a UI where a push button appears in a window. The push button is
   logically "owned" or "managed" by the window that encloses it, and it is
   placed relative to the coordinate system of the window. Here, the
   window acts a composite thingy managing a simple thingy (the button). The
   ensemble of thingies managed by a composite thingy are called that thingy's
   "subthingies" */
class  CompositeThingy
    : public virtual Thingy {

private:

    /* CompositeThingies don't have value semantics */
    CompositeThingy(const CompositeThingy&) { }
    const CompositeThingy& operator=(const CompositeThingy&) { return *this; }

protected:

    CompositeThingy( ) { }

public:

    virtual  ~CompositeThingy( ) { }

    virtual  unsigned short  composableWidth( ) const = 0;
    virtual  unsigned short  composableHeight( ) const = 0;
};
//-----------------------------------------------------------------------------




//-----------------------------------------------------------------------------
/* Defines a common interface that must be implemented by any object that
   wishes to control a CompositeThingy. Since the controllers of
   CompositeThingies must manage the layout and enabling/disabling of
   subthingies, this interface mandates methods to do so. */
class  CompositeThingyController
    : public virtual ThingyController {

protected:

    CompositeThingyController( ) { }

public:

    static  CompositeThingyController*  nullController( );

    virtual ~CompositeThingyController( ) { }

    virtual void  layoutSubthingies(CompositeThingy& sender) = 0;
    virtual void  enableSubthingies(CompositeThingy& sender) = 0;
    virtual void  disableSubthingies(CompositeThingy& sender) = 0;
};
//-----------------------------------------------------------------------------





//-----------------------------------------------------------------------------
/* A ThingyAdapter manages a Win32 HWND primitive on behalf of a Thingy. The
   Thingy for which the ThingyAdapter manages the HWND is that ThingyAdapter's
   "client." */
class  ThingyAdapter {

private:

    static  std::map<HWND, ThingyAdapter*>  sObjectMap;

    HWND                  fPrimitive;
    ThingyAdapterClient*  fClient;

    /* ThingyAdapters don't have value semantics */
    ThingyAdapter(const ThingyAdapter&) { }
    const ThingyAdapter& operator=(const ThingyAdapter&) { return *this; }

public:

    static  LRESULT  CALLBACK  sUniversalProc(HWND, UINT, WPARAM, LPARAM);
    static  ThingyAdapter&  getObject(HWND primitive);
    static  void  associateObject(HWND primitive, ThingyAdapter& obj);
    static  void  deleteAssociation(HWND primitive);
    static  bool  isPrimitiveAssociated(HWND primitive);

    ThingyAdapter(ThingyAdapterClient& client, HWND managed);
    virtual  ~ThingyAdapter( );

    ThingyAdapter&  parent( );
    const ThingyAdapter&  parent( ) const;
    bool  hasParent( ) const;
    HWND  primitive( );
    ThingyAdapterClient&  client( );
    const ThingyAdapterClient&  client( ) const;
    void  release( );
    
    LRESULT  propogateChildMessage(HWND, UINT, WPARAM, LPARAM);
};

typedef std::auto_ptr<ThingyAdapter>  SafeThingyAdapter;
//-----------------------------------------------------------------------------




//-----------------------------------------------------------------------------
/* Defines the interface through which a ThingyAdapter asks its client to
   process Win32 HWND primitive messages */
class  ThingyAdapterClient {

protected:

    ThingyAdapterClient( ) { }

public:

    virtual  ~ThingyAdapterClient( ) { }

    virtual  LRESULT  processMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam)
        { return DefWindowProc(target, message, wparam, lparam); }
    virtual  LRESULT  processParentMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam)
        { return DefWindowProc(target, message, wparam, lparam); }
    virtual  ThingyAdapter&  adapter( ) = 0;
    virtual  const ThingyAdapter&  adapter( ) const = 0;
};
//-----------------------------------------------------------------------------





//-----------------------------------------------------------------------------
/* Defines the special interface required of Thingy adapter clients who are
   capable of composing other Thingies. */
class  CompositeThingyAdapterClient :
    public virtual ThingyAdapterClient {

protected:

    CompositeThingyAdapterClient( ) { }

public:

    virtual  ~CompositeThingyAdapterClient( ) { }

    virtual  ThingyAdapter&  composerAdapter( ) = 0;
    virtual  const ThingyAdapter&  composerAdapter( ) const = 0;
};
//-----------------------------------------------------------------------------

#endif
