//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ThingyScrapbook.cpp                                                      //
//                                                                          //
// Implements methods in classes SourceImageChooserThingy and               //
// SourceImageChooserController                                             //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// <unversioned module>                                                     //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008, Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "ImageRepresentation.h"
#include "OpenFileChooser.h"
#include "Application.h"
#include "ThingyScrapbook.h"
#include <sstream>

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  SourceImageChooserController                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// CLASS  SourceImageChooserController                                      //
//////////////////////////////////////////////////////////////////////////////





//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  SourceImageChooserThingy                                          //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const float  SourceImageChooserThingy::kNoAspectRatioConstraint = -1.0f;


SourceImageChooserThingy::SourceImageChooserThingy(CompositeThingy& parthingy)
    : fMasterControl(parthingy), fSetSourceButton(fMasterControl),
      fFileSourceLabel(fMasterControl),
      fWidthMinConst(kDefaultDimMinConstraint),
      fWidthMaxConst(kDefaultDimMaxConstraint),
      fHeightMinConst(kDefaultDimMinConstraint),
      fHeightMaxConst(kDefaultDimMaxConstraint),
      fAspectRatioConst(kNoAspectRatioConstraint),
      fIsPowerOf2ConstEnabled(true), fSrcImgData(0)
{
    delete & fMasterControl.installController(*this);
    fMasterControl.setTitle("Brush Source Image:");
    fMasterControl.setHeight(56);

    delete & fSetSourceButton.installController(*this);
    fSetSourceButton.setTitle("Set...");
    fSetSourceButton.show( );

    fFileSourceLabel.setTitle("none set.");
    fFileSourceLabel.show( );
}




SourceImageChooserThingy::~SourceImageChooserThingy( )
{
}




void  SourceImageChooserThingy::show( )
{
    fMasterControl.show( );
}




void  SourceImageChooserThingy::hide( )
{
    fMasterControl.hide( );
}




bool  SourceImageChooserThingy::isVisible( )
{
    return fMasterControl.isVisible( );
}




Vector2Di  SourceImageChooserThingy::origin( ) const
{
    return fMasterControl.origin( );
}




void  SourceImageChooserThingy::setOrigin(const Vector2Di& ogn)
{
    fMasterControl.setOrigin(ogn);
}




void  SourceImageChooserThingy::setOrigin(unsigned short x, unsigned short y)
{
    fMasterControl.setOrigin(x, y);
}




unsigned short  SourceImageChooserThingy::width( ) const
{
    return fMasterControl.width( );
}




void  SourceImageChooserThingy::setWidth(unsigned short w)
{
    fMasterControl.setWidth(w);
}




unsigned short  SourceImageChooserThingy::height( ) const
{
    return fMasterControl.height( );
}




void  SourceImageChooserThingy::setHeight(unsigned short h)
{
    fMasterControl.setHeight(h);
}




void  SourceImageChooserThingy::enable( )
{
    fMasterControl.enable( );
}




void  SourceImageChooserThingy::disable( )
{
    fMasterControl.disable( );
}




bool  SourceImageChooserThingy::isEnabled( ) const
{
    return fMasterControl.isEnabled( );
}




void  SourceImageChooserThingy::layoutSubthingies(CompositeThingy& sender)
{
    static const int  kButtonLabelSpacing = 8;
    static const int  kUnifHorizInset = 4;

    int surface_wd = fMasterControl.composableWidth( );
    int surface_ht = fMasterControl.composableHeight( );
    int setbtn_wd = fSetSourceButton.width( );
    int setbtn_ht = fSetSourceButton.height( );
    int label_ht = fFileSourceLabel.height( );

    fSetSourceButton.setOrigin(surface_wd - setbtn_wd - kUnifHorizInset,
        (surface_ht - setbtn_ht) / 2);

    fFileSourceLabel.setOrigin(kUnifHorizInset, (surface_ht - label_ht) / 2);
    fFileSourceLabel.setWidth(surface_wd - setbtn_wd - kButtonLabelSpacing -
        (2 * kUnifHorizInset));
}




void  SourceImageChooserThingy::enableSubthingies(CompositeThingy& sender)
{
}




void  SourceImageChooserThingy::disableSubthingies(CompositeThingy& sender)
{
}




void  SourceImageChooserThingy::thingyClicked(PushButtonThingy& sender)
{
    //BCMainWindow& hostwin =
    //    dynamic_cast<BCMainWindow&>(**fReceiverSet.begin( ));

    OpenFileChooser ofc;
    ofc.setTitle("Choose a Source Image");
    ofc.addFiletypeMask("Truevision TARGA (*.tga; *.tpic)", "*.tga");
    ofc.addFiletypeMask("Truevision TARGA (*.tga; *.tpic)", "*.tpic");

    std::string  imgfn = ofc.runFileChoiceInteraction(
        sender.adapter( ).primitive( ));

    /* if the image filename 'imgfn' returned is the null string, then the
       user clicked 'Cancel' in the 'Open...' dialog box before choosing an
       image file to open, so just return */
    if (imgfn == "")
        return;

    /* try to load image data from the the file the user selected */
    //hostwin.setWaitingVisual( );
    RGBAImage*  userimg = 0;
    try {

        userimg = RGBAImage::createMappedImage(RGBAImage::kTargaImageFormat,
            imgfn, fReceiverSet, *this);
    }
    catch (std::runtime_error&) {

        const std::string&  appname = Application::instance( ).name( );
        MessageBox(NULL, TEXT("The file selected does not contain "
            "valid TARGA image data"), appname.c_str( ), MB_ICONERROR);

        //hostwin.setInteractionVisual( );
        delete userimg;
        return;
    }

    /* if execution reaches this point, then we were able read image data
       from the file chosen by the user sucessfully. But we're still not in
       the clear, since the image setting interaction can still fail if the
       image chosen by the user doesn't meet the constraints specified for
       *this, so let's check it against these constraints now */

    if ((userimg->width( ) < fWidthMinConst) ||
        (userimg->width( ) > fWidthMaxConst)) {

        const std::string&  appname = Application::instance( ).name( );
        std::ostringstream  msgstream;
        msgstream << "The image you've chosen is too wide.\n" <<
            "Valid source images have widths between " << fWidthMinConst <<
            " pixels and " << fWidthMaxConst << " pixels, inclusive.";

        MessageBox(fMasterControl.adapter( ).primitive( ),
            msgstream.str( ).c_str( ), appname.c_str( ), MB_ICONERROR);

        delete userimg;
        //hostwin.setInteractionVisual( );
        return;
    }

    if ((userimg->height( ) < fHeightMinConst) ||
        (userimg->height( ) > fHeightMaxConst)) {

        const std::string&  appname = Application::instance( ).name( );
        std::ostringstream  msgstream;
        msgstream << "The image you've chosen is too tall.\n" <<
            "Valid source images have heights between " << fHeightMinConst <<
            " pixels and " << fHeightMaxConst << " pixels, inclusive.";

            MessageBox(fMasterControl.adapter( ).primitive( ),
                msgstream.str( ).c_str( ), appname.c_str( ), MB_ICONERROR);

        delete userimg;
        //hostwin.setInteractionVisual( );
        return;
    }

    if (fAspectRatioConst != kNoAspectRatioConstraint) {

        float imgw = static_cast<float>(userimg->width( ));
        float imgh = static_cast<float>(userimg->height( ));
        float rat = (imgw / imgh);

        if (rat != fAspectRatioConst) {

            const std::string&  appname = Application::instance( ).name( );
            std::ostringstream  msgstream;
            msgstream << "The image you've chosen has the wrong aspect " <<
                "ratio.\nValid source images must have an aspect ratio (" <<
                "width / height) of " << fAspectRatioConst << ".";

            MessageBox(fMasterControl.adapter( ).primitive( ),
                msgstream.str( ).c_str( ), appname.c_str( ), MB_ICONERROR);

            delete userimg;
            //hostwin.setInteractionVisual( );
            return;
        }
    }

    if (fIsPowerOf2ConstEnabled) {

        double lgwidth = std::log(static_cast<double>(userimg->width( ))) /
            std::log(2.0);
        double lgheight = std::log(static_cast<double>(userimg->height( ))) /
            std::log(2.0);

        double fracwidth;
        std::modf(lgwidth, &fracwidth);
        fracwidth = lgwidth - fracwidth;

        double fracheight;
        std::modf(lgheight, &fracheight);
        fracheight = lgheight - fracheight;

        if (fracwidth != 0.0) {

            const std::string&  appname = Application::instance( ).name( );
            std::ostringstream  msgstream;
            msgstream << "The image you've chosen has a width that is not "
                "a power of two.\nValid source images must have widths and "
                "heights that are powers of two.";

            MessageBox(fMasterControl.adapter( ).primitive( ),
                msgstream.str( ).c_str( ), appname.c_str( ), MB_ICONERROR);

            delete userimg;
            //hostwin.setInteractionVisual( );
            return;
        }

        if (fracheight != 0.0) {

            const std::string&  appname = Application::instance( ).name( );
            std::ostringstream  msgstream;
            msgstream << "The image you've chosen has a height that is not "
                "a power of two.\nValid source images must have widths and "
                "heights that are powers of two.";

            MessageBox(fMasterControl.adapter( ).primitive( ),
                msgstream.str( ).c_str( ), appname.c_str( ), MB_ICONERROR);

            delete userimg;
            //hostwin.setInteractionVisual( );
            return;
        }
    }

    /* if execution reaches this point, then none of the constraint tests
       failed, so lets go ahead and set the chosen filename readout and
       notify our parent that the user has chosen a new source image */
    fSrcFilename = imgfn;
    int  lastPathDelimLoc = (int) imgfn.find_last_of("\\/");

    std::string  simpleFilename;
    if (lastPathDelimLoc != std::string::npos)
        simpleFilename = imgfn.substr(lastPathDelimLoc + 1,
            imgfn.length( ) - lastPathDelimLoc - 1);
    else
        simpleFilename = imgfn;

    fFileSourceLabel.setTitle(simpleFilename);

    RGBAImage* oldsrc = fSrcImgData;
    fSrcImgData = userimg;
    if (oldsrc)
        delete oldsrc;

    //hostwin.setInteractionVisual( );
    //hostwin.notifySourceReset( );
}




void  SourceImageChooserThingy::registerReceiver(
    InterstitialActionReceiver& recv)
{
    fReceiverSet.insert(&recv);
}




void  SourceImageChooserThingy::deregisterReceiver(
    InterstitialActionReceiver& recv)
{
    fReceiverSet.erase(&recv);
}




void  SourceImageChooserThingy::cancelAction(
    InterstitialActionReceiver& recv)
{
}




const RGBAImage&  SourceImageChooserThingy::sourceImage( ) const
{
    return *fSrcImgData;
}




const std::string&  SourceImageChooserThingy::sourceFile( ) const
{
    return fSrcFilename;
}




bool  SourceImageChooserThingy::isSourceImageSet( ) const
{
    if (fSrcImgData)
        return true;
    else
        return false;
}
//////////////////////////////////////////////////////////////////////////////
// CLASS  SourceImageChooserThingy                                          //
//////////////////////////////////////////////////////////////////////////////
