//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// SynthesisAgent.cpp                                                       //
//                                                                          //
// Defines the static constants of and implements methods in class          //
// SynthesisAgent only                                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 2.0.0 of Fri 09-Jan-2009 @ 5:37am EST                               //
//                                                                          //
//     SynthesisAgent's now allow their clients to change their source      //
//     images and similarity maps dynamically, via public member functions. //
//     In the past, these were set once, in a SynthesisAgent's constructor, //
//     and remained fixed throughout an object's lifetime.                  //
//                                                                          //
// ver. 1.5.5 of Mon 1-Sep-2008 @ 10:55am EDT                               //
//                                                                          //
//     Made the synthesis shader's jump penalization parameter client       //
//     configurable                                                         //
//                                                                          //
// ver. 1.5.0 of Sun 31-Aug-2008 @ 10:52pm EDT                              //
//                                                                          //
//     Added a publicly accessible debugging method that draws the contents //
//     of the active work texture buffer. Texture-filtering options are now //
//     re-configured to be pixel-exact (GL_NEAREST) before the synthesis    //
//     passes are run. This aids interoperability with client code that     //
//     may change the filtering options arbitrarily.                        //
//                                                                          //
// prior change history elided; check out an older CVS rev to get it        //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008-2009, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include <SynthesisAgent.h>
#include <GPUInterface.h>
#include <ImageRepresentation.h>
#include <TextureSynthesis.h>
#include <stdexcept>
#include <algorithm>
#include <map>
#include <climits>
#include <sstream>
#include <ctime>

const float  SynthesisAgent::kOutTexPixelFraction = 1.0f /
    static_cast<float>(SynthesisAgent::kWorkTexExtent);

const float  SynthesisAgent::kDefaultJumpParameter = 1.3f;


const Vector2Df  SynthesisAgent::kUpsampleOutsets[ ] = {

    Vector2Df(1.0f, 1.0f), /* level 0 */
    Vector2Df(2.0f, 2.0f), /* level 1 */
    Vector2Df(4.0f, 4.0f), /* level 2 */
    Vector2Df(8.0f, 8.0f), /* level 3 */
    Vector2Df(8.0f, 8.0f), /* level 4 */
    Vector2Df(8.0f, 8.0f), /* level 5 */
    Vector2Df(8.0f, 8.0f), /* level 6 */
    Vector2Df(8.0f, 8.0f), /* level 7 */
};


const Vector2Df  SynthesisAgent::kCorr1Outsets[ ] = {

    Vector2Df(6.0f, 6.0f), /* level 3 */
    Vector2Df(6.0f, 6.0f), /* level 4 */
    Vector2Df(6.0f, 6.0f), /* level 5 */
    Vector2Df(6.0f, 6.0f), /* level 6 */
    Vector2Df(6.0f, 6.0f), /* level 7 */
};


const Vector2Df  SynthesisAgent::kCorr2Outsets[ ] = {

    Vector2Df(4.0f, 4.0f), /* level 3 */
    Vector2Df(4.0f, 4.0f), /* level 4 */
    Vector2Df(4.0f, 4.0f), /* level 5 */
    Vector2Df(4.0f, 4.0f), /* level 6 */
    Vector2Df(4.0f, 4.0f), /* level 7 */
};




SynthesisAgent::SynthesisAgent(const RGBAImage& preimg, GPUInterface& gpui,
    const PyramidalSimilarityMap& simmap)
        : fGPU(gpui), fSourceMosaicTex(0), fInvarUnifState(0),
          fSimMap(&simmap), fSourceMosaicTexID(0), fSimMapMosaicTexID(0),
          fVertexProgID(0), fFragmentProgID(0), fProgramSetID(0),
          fActiveWorkTexBuffer(0), fAltWorkTexBuffer(0),
          fPerInvocUnifState(0), fStateCacheTexID(0)
{
    uConstructorHelper(preimg);
}




SynthesisAgent::SynthesisAgent(const RGBAImage& preimg, GPUInterface& gpui,
     const PyramidalSimilarityMap& simmap, InterstitialActionReceiver& iar)
        : fGPU(gpui), fSourceMosaicTex(0), fInvarUnifState(0),
          fSimMap(&simmap), fSourceMosaicTexID(0), fSimMapMosaicTexID(0),
          fVertexProgID(0), fFragmentProgID(0), fProgramSetID(0),
          fActiveWorkTexBuffer(0), fAltWorkTexBuffer(0),
          fPerInvocUnifState(0), fStateCacheTexID(0)
{
    registerReceiver(iar);
    uConstructorHelper(preimg);
}




SynthesisAgent::~SynthesisAgent( )
{
    delete fInvarUnifState;
    fInvarUnifState = 0;

    delete fPerInvocUnifState;
    fPerInvocUnifState = 0;

    delete fSourceMosaicTex;
    fSourceMosaicTex = 0;

    delete fActiveWorkTexBuffer;
    fActiveWorkTexBuffer = 0;

    delete fAltWorkTexBuffer;
    fAltWorkTexBuffer = 0;

    glDeleteTextures(1, &fSourceMosaicTexID);
    glDeleteTextures(1, &fSimMapMosaicTexID);
    glDeleteTextures(1, &fStateCacheTexID);

    glDeleteShader(fVertexProgID);
    glDeleteShader(fFragmentProgID);

    glDeleteProgram(fProgramSetID);
}




void  SynthesisAgent::uConstructorHelper(const RGBAImage& srcimg)
{
    fPrecursorQuantization = srcimg.width( );

    /* create the invariant and per-invocation uniform state caches */
    fInvarUnifState = new InvariantUniformState;
    fPerInvocUnifState = new PerInvocationUniformState( );

    /* set the jump penalization parameter to its default value */
    setJumpParameter(kDefaultJumpParameter);

    /* build a Gaussian image pyramid out of the client-provided precursor
       image, and get the pyramid's mosaic image -- the mosaic image allows
       us to download the entire image pyramid into TRAM on the GPU in a
       relatively compact fashion */
    ImagePyramid* pyr;
    if (!fReceiverSet.empty( ))
        pyr = new ImagePyramid(srcimg, **fReceiverSet.begin( ));
    else
        pyr = new ImagePyramid(srcimg);
    fSourceMosaicTex = new RGBAImage(pyr->mosaicImage( ));

    /* cache the extent of the source mosaic texture */
    fInvarUnifState->sourceTexExtent[0] =
        static_cast<float>(fSourceMosaicTex->width( ));
    fInvarUnifState->sourceTexExtent[1] =
        static_cast<float>(fSourceMosaicTex->height( ));

    /* cache the locations & sizes of the individual pyramid level tile images
       within the overall mosaic image */
    int idxticker = 0;
    std::vector<Rectangle2Di>::const_iterator it =
        pyr->mosaicTiles( ).begin( );
    for ( ; it != pyr->mosaicTiles( ).end( ); it++) {

        fInvarUnifState->sourceTileExtents[idxticker] =
            static_cast<float>(it->width);

        /* origin coordinate data has to be inserted in (y, x) order,
           instead of in customary (x, y) order, because the entire
           array is reversed after it's assembled -- why? because that's
           the way the shader program wants it */
        fInvarUnifState->sourceTileOrigins[2 * idxticker] =
            static_cast<float>(it->origin.y);
        fInvarUnifState->sourceTileOrigins[(2 * idxticker) + 1] =
            static_cast<float>(it->origin.x);

        fInvarUnifState->sourceTileExtents[idxticker] =
            static_cast<float>(it->width);

        idxticker++;
    }
    std::reverse(& fInvarUnifState->sourceTileOrigins[0],
        & fInvarUnifState->sourceTileOrigins[2 * idxticker]);
    std::reverse(& fInvarUnifState->sourceTileExtents[0],
        & fInvarUnifState->sourceTileExtents[idxticker]);

    /* cache the number of refinement levels */
    fInvarUnifState->numRefinementLevels = static_cast<int>(pyr->numLevels( ));

    /* after we've got the mosaic image and the tiles, we don't need the
       pyramid itself any more */
    delete pyr;

    /* cache the extent of the work texture */
    fInvarUnifState->workTexExtent[0] = static_cast<float>(kWorkTexExtent);
    fInvarUnifState->workTexExtent[1] = static_cast<float>(kWorkTexExtent);

    /* cache the extent of the similarity map mosaic texture */
    fInvarUnifState->simTexExtent[0] =
        static_cast<float>(fSimMap->mosaicImage( ).width( ));
    fInvarUnifState->simTexExtent[1] =
        static_cast<float>(fSimMap->mosaicImage( ).height( ));

    /* cache the number of similarity map entries per pixel */
    fInvarUnifState->simNumEntries =
        static_cast<float>(fSimMap->entriesPerPixel( ));

    /* cache the texture unit numbers used to access the work texture,
       source texture, similarity map texture, and encoded state texture */
    fInvarUnifState->workTex = 0;
    fInvarUnifState->sourceTex = 1;
    fInvarUnifState->simTex = 2;
    fInvarUnifState->stateTex = 3;

    /* compute the locations of the similarity map mosiac tiles, then download
       them into TRAM as an encoded 1D texture -- note that once nVidia
       corrects their 8xxx series driver issues, we can just
       store this in a uniform instead of downloading it as a texture */
    std::vector<Vector2Df>  simTileOrigins;
    for (size_t i = 0; i < fSimMap->mosaicTiles( ).size( ); i++) {

        const std::vector<Rectangle2Di>& simtiles =
            fSimMap->mosaicTiles( );
        Vector2Df simogn = Vector2Df(
            static_cast<float>(simtiles[i].origin.x),
            static_cast<float>(simtiles[i].origin.y));
        simTileOrigins.push_back(simogn);
    }
    glActiveTexture(GL_TEXTURE3);
    fStateCacheTexID = uDownloadTexture1D(simTileOrigins);
    glActiveTexture(GL_TEXTURE0);

    /* create the two work texture buffers */
    fActiveWorkTexBuffer = new TextureBuffer(kWorkTexExtent, kWorkTexExtent);
    fAltWorkTexBuffer = new TextureBuffer(kWorkTexExtent, kWorkTexExtent);

    /* load the shader programs from disk */
    uLoadDiskResources( );

    /* download the texture image data to TRAM */
    uConfigureTextures( );

    /* briefly enable then disable the synthesis shader program to cache the
       locations of the uniform variables in it */
    glUseProgram(fProgramSetID);
    uConfigureInvariantLocs(*fInvarUnifState);
    uConfigurePerInvocationLocs(*fPerInvocUnifState);
    glUseProgram(0);
}




GLuint  SynthesisAgent::synthesize(int x, int y, int width, int height)
{
    int maxref = fInvarUnifState->numRefinementLevels - 1;

    /* set up an orthographic 2D coordinate system that corresponds to the
       pixels of the work texture */
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix( );
    glLoadIdentity( );
    glMatrixMode(GL_PROJECTION);
    glPushMatrix( );
    glLoadIdentity( );
    glOrtho(0.0, kWorkTexExtent, 0.0, kWorkTexExtent, 1.0, -1.0);
    glPushAttrib(GL_VIEWPORT_BIT);
    glViewport(0, 0, kWorkTexExtent, kWorkTexExtent);

    /* engage the synthesis shaders, write the cached values of invariant
       uniforms back onto the GPU and bind the invariant textures into
       their texture units */
    glUseProgram(fProgramSetID);
    uWriteInvariantUnifState(*fInvarUnifState);
    uRebindInvariantTextures( );

    /* Compute the geometry of the various work image effects regions at each
       level of refinement and download them onto the GPU */
    uComputeEffectsRegions(*fPerInvocUnifState, x, y, width, height);
    uWritePerInvocationUnifState(*fPerInvocUnifState);

    /* make texture unit 0 active and bind the alternate work texture buffer
       into it */
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, fAltWorkTexBuffer->textureHandle( ));

    /* make the active work texture buffer the destination for rendering
       operations */
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fActiveWorkTexBuffer->handle( ));

    /* run the main rendering loop */
    for (int i = 0; i < numRefinementLevels( ); i++) {

        fGPU.setUniform1i(fProgramSetID, "RefinementLevel", i);

        /* perform the upsample & jitter pass */
        uRunUpsampleJitterPass(i);

        /* if the levels are large enough, run the correction passes */
        if (i > 2)
            uRunCorrectionPasses(i);

    } /* end main rendering loop */

    uRunMaxOutputPass(x, y, width, height);

    /* disengage the synthesis shaders */
    glUseProgram(0);

    /* return the viewport and coordinate systems to their previous states */
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix( );
    glMatrixMode(GL_PROJECTION);
    glPopMatrix( );
    glPopAttrib( );

    /* make the display the target for rendering operations once more */
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

    return fActiveWorkTexBuffer->textureHandle( );
}




void  SynthesisAgent::registerReceiver(InterstitialActionReceiver& recv)
{
    fReceiverSet.insert(&recv);
}




void  SynthesisAgent::deregisterReceiver(InterstitialActionReceiver& recv)
{
    fReceiverSet.erase(&recv);
}




void  SynthesisAgent::cancelAction(InterstitialActionReceiver& recv)
{
}




void  SynthesisAgent::uRenderEffectsRegion(float x, float y, float w, float h)
{
    glBegin(GL_QUADS);
        glMultiTexCoord2i(GL_TEXTURE0, 0, 0);
        glVertex2f(x, y);

        glMultiTexCoord2i(GL_TEXTURE0, 1, 0);
        glVertex2f(x + w, y);

        glMultiTexCoord2i(GL_TEXTURE0, 1, 1);
        glVertex2f(x + w, y + h);

        glMultiTexCoord2i(GL_TEXTURE0, 0, 1);
        glVertex2f(x, y + h);
    glEnd( );

    glFlush( );
}




void  SynthesisAgent::uLoadDiskResources( )
{
    /* load the vertex and fragment shaders source files from disk, compile
       them, link them into a program, and install the program on the GPU */
    fVertexProgID = fGPU.prepareVertexProgram("SynAgentV.glsl");
    fFragmentProgID = fGPU.prepareFragmentProgram("SynAgentF.glsl");
    GLuint progset [ ] = { fVertexProgID, fFragmentProgID };
    fProgramSetID = fGPU.prepareProgramSet(progset, 2);
}




void  SynthesisAgent::uConfigureTextures( )
{
    /* source mosaic is downloaded to texture unit 1 */
    glActiveTexture(GL_TEXTURE1);
    fSourceMosaicTexID = fGPU.downloadTexture(*fSourceMosaicTex);

    /* similarity map mosaic is downloaded to texture unit 2 */
    glActiveTexture(GL_TEXTURE2);
    fSimMapMosaicTexID = fGPU.downloadTexture(fSimMap->mosaicImage( ));

    glActiveTexture(GL_TEXTURE0);
}




void  SynthesisAgent::uWriteInvariantUnifState(InvariantUniformState& state)
{
    glUniform2f(state.workTexExtentLoc,
        state.workTexExtent[0], state.workTexExtent[1]);
    glUniform2f(state.sourceTexExtentLoc,
        state.sourceTexExtent[0], state.sourceTexExtent[1]);
    glUniform2fv(state.sourceTileOriginsLoc, 16, state.sourceTileOrigins);
    glUniform1fv(state.sourceTileExtentsLoc, 8, state.sourceTileExtents);
    glUniform2f(state.simTexExtentLoc, state.simTexExtent[0],
        state.simTexExtent[1]);
    glUniform1f(state.simNumEntriesLoc, state.simNumEntries);
    glUniform1i(state.numRefinementLevelsLoc, state.numRefinementLevels);
    glUniform1i(state.workTexLoc, state.workTex);
    glUniform1i(state.sourceTexLoc, state.sourceTex);
    glUniform1i(state.simTexLoc, state.simTex);
    glUniform1i(state.stateTexLoc, state.stateTex);
    glUniform1fv(state.pixelWeightsLoc, 25, PixelNeighborhood::kNHWeights2);
    glUniform1f(state.jumpPenalizationParamLoc, state.jumpPenalizationParam);
}




void  SynthesisAgent::uRunUpsampleJitterPass(int reflev)
{
    fGPU.setUniform1i(fProgramSetID, "OperationMode",
        kUpsampleJitterMode);

    float currext_x =
        fPerInvocUnifState->upsampleEffectsGeom[(4 * reflev) + 2];
    float currext_y =
        fPerInvocUnifState->upsampleEffectsGeom[(4 * reflev) + 3];

    uRenderEffectsRegion(0.0f, 0.0f, currext_x, currext_y);

    /* interchange the work texture buffers */
    uSwapWorkTexBuffers( );
}




void  SynthesisAgent::uComputeEffectsRegions(
    PerInvocationUniformState& state, int userx, int usery, int userwd,
    int userht)
{
    int    numlevs = numRefinementLevels( );
    float  srcdim = fInvarUnifState->sourceTileExtents[numlevs - 1];

    /* set up the output effects geometry at the maximum level of
       refinement: note that this is just a matter of pushing the user
       origin and extent information into the vector -- but remember to
       add 1 source unit dimension to the origins because of the way
       regions are shifted to prevent negative values from ever
       entering computations */
    state.maxOutputEffectsGeom[0] = static_cast<float>(userx) + srcdim;
    state.maxOutputEffectsGeom[1] = static_cast<float>(usery) + srcdim;
    state.maxOutputEffectsGeom[2] = static_cast<float>(userwd);
    state.maxOutputEffectsGeom[3] = static_cast<float>(userht);

    /* compute the logical origin of the output effects region at the coarsest
       level of refinement */
    Vector2Df  outogn0 = floor(Vector2Df(static_cast<float>(userx),
        static_cast<float>(usery)) / srcdim) + Vector2Df(1.0f, 1.0f);

    /* compute the extent of the output effects region at the coarsest level
       of refinement */
    Vector2Df outext0 = ceil((Vector2Df(static_cast<float>(userx),
        static_cast<float>(usery)) + Vector2Df(static_cast<float>(userwd),
        static_cast<float>(userht))) / srcdim) - outogn0 +
        Vector2Df(1.0f, 1.0f);

    /* use the output effects region geometry to compute geometry for the
       upsample/jitter effects region at the coarsest level of refinement */
    Vector2Df upsampogn0 = outogn0 - kUpsampleOutsets[0];
    Vector2Df upsampext0 = outext0 + (2.0f * kUpsampleOutsets[0]);

    /* insert the initial geometry information computed about into the
       appropriate collection */
    state.upsampleEffectsGeom[0] = upsampogn0.x;
    state.upsampleEffectsGeom[1] = upsampogn0.y;
    state.upsampleEffectsGeom[2] = upsampext0.x;
    state.upsampleEffectsGeom[3] = upsampext0.y;

    /* compute geometry information for all the remaining levels */
    Vector2Df  outogn = outogn0;
    Vector2Df  outext = outext0;
    for (int i = 1; i < numlevs; i++) {

        outogn = 2.0f * outogn;
        outext = 2.0f * outext;

        int  upsamp_baseidx = i * 4;

        Vector2Df upsampogn = outogn - kUpsampleOutsets[i];
        Vector2Df upsampext = outext + (2.0f * kUpsampleOutsets[i]);
        state.upsampleEffectsGeom[upsamp_baseidx] = upsampogn.x;
        state.upsampleEffectsGeom[upsamp_baseidx + 1] = upsampogn.y;
        state.upsampleEffectsGeom[upsamp_baseidx + 2] = upsampext.x;
        state.upsampleEffectsGeom[upsamp_baseidx + 3] = upsampext.y;

        if (i > 2) {

            int  corr_baseidx = (i - 3) * 4;

            Vector2Df corr1ogn = outogn - kCorr1Outsets[i - 3];
            Vector2Df corr1ext = outext + (2.0f * kCorr1Outsets[i - 3]);
            Vector2Df corr2ogn = outogn - kCorr2Outsets[i - 3];
            Vector2Df corr2ext = outext + (2.0f * kCorr2Outsets[i - 3]);

            state.corr1EffectsGeom[corr_baseidx] = corr1ogn.x;
            state.corr1EffectsGeom[corr_baseidx + 1] = corr1ogn.y;
            state.corr1EffectsGeom[corr_baseidx + 2] = corr1ext.x;
            state.corr1EffectsGeom[corr_baseidx + 3] = corr1ext.y;

            state.corr2EffectsGeom[corr_baseidx] = corr2ogn.x;
            state.corr2EffectsGeom[corr_baseidx + 1] = corr2ogn.y;
            state.corr2EffectsGeom[corr_baseidx + 2] = corr2ext.x;
            state.corr2EffectsGeom[corr_baseidx + 3] = corr2ext.y;

        } /* if correction has kicked in (note: it starts at level 3) */
    } /* for all remaining levels of refinement */
}




void  SynthesisAgent::uRunCorrectionPasses(int reflev)
{     
    fGPU.setUniform1i(fProgramSetID, "OperationMode", kCorrectionMode);

    /* run the first correction pass */
    float  currext_x =
        fPerInvocUnifState->corr1EffectsGeom[4 * (reflev - 3) + 2];
    float  currext_y =
        fPerInvocUnifState->corr1EffectsGeom[4 * (reflev - 3) + 3];

    fGPU.setUniform1i(fProgramSetID, "CorrectionPassNumber", 1);
    uRenderEffectsRegion(0.0f, 0.0f, currext_x, currext_y);

    /* interchange the work texture buffers */
    uSwapWorkTexBuffers( );

    /* run the second correction pass */
    currext_x =
        fPerInvocUnifState->corr2EffectsGeom[4 * (reflev - 3) + 2];
    currext_y =
        fPerInvocUnifState->corr2EffectsGeom[4 * (reflev - 3) + 3];

    fGPU.setUniform1i(fProgramSetID, "CorrectionPassNumber", 2);
    uRenderEffectsRegion(0.0f, 0.0f, currext_x, currext_y);

    /* interchange the work texture buffers */
    uSwapWorkTexBuffers( );
}




void  SynthesisAgent::uRunMaxOutputPass(int x, int y, int w, int h)
{
    fGPU.setUniform1i(fProgramSetID, "OperationMode", kOutputMode);

    uRenderEffectsRegion(0.0f, 0.0f, static_cast<float>(w),
        static_cast<float>(h));
}




GLuint  SynthesisAgent::uDownloadTexture1D(
    const std::vector<Vector2Df> texdata)
{
    static  const float    kTexture1DEncodeMult = 1.0f / 4095.0f;
    static  const GLsizei  kTexture1DBufferSize = 1024;

    std::vector<Vector2Df>::const_iterator it;    

    int sizeticker = 0;
    for (it = texdata.begin( ); it != texdata.end( ); it++)
        sizeticker++;

    float* transferbuf = new float[kTexture1DBufferSize];

    it = texdata.begin( );
    for (int i = 0; i < (2 * sizeticker); i += 2, it++) {

        transferbuf[i] = it->x * kTexture1DEncodeMult;
        transferbuf[i + 1] = it->y * kTexture1DEncodeMult;
    }

    GLuint newtex;
    glGenTextures(1, &newtex);
    glBindTexture(GL_TEXTURE_1D, newtex);

    glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_FALSE);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);

    glTexImage1D(GL_TEXTURE_1D, 0, GL_LUMINANCE16, kTexture1DBufferSize, 0,
        GL_LUMINANCE, GL_FLOAT, transferbuf);

    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    delete [ ] transferbuf;

    return newtex;
}




void  SynthesisAgent::uUpdateTexture1D(GLuint texid,
    std::vector<Vector2Df> texdata)
{
    static  const float    kTexture1DEncodeMult = 1.0f / 4095.0f;
    static  const GLsizei  kTexture1DBufferSize = 1024;

    std::vector<Vector2Df>::const_iterator it;    

    int sizeticker = 0;
    for (it = texdata.begin( ); it != texdata.end( ); it++)
        sizeticker++;

    float* transferbuf = new float[kTexture1DBufferSize];

    it = texdata.begin( );
    for (int i = 0; i < (2 * sizeticker); i += 2, it++) {

        transferbuf[i] = it->x * kTexture1DEncodeMult;
        transferbuf[i + 1] = it->y * kTexture1DEncodeMult;
    }

    glBindTexture(GL_TEXTURE_1D, texid);

    glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_FALSE);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
    glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);

    glTexImage1D(GL_TEXTURE_1D, 0, GL_LUMINANCE16, kTexture1DBufferSize, 0,
        GL_LUMINANCE, GL_FLOAT, transferbuf);

    delete [ ] transferbuf;
}




void    SynthesisAgent::uSwapWorkTexBuffers( )
{
    /* make the alternate work texture buffer the destination for rendering
       operations */
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fAltWorkTexBuffer->handle( ));

    /* bind the active buffer into texture unit 0 to be used as a source for
       rendering operations */
    glBindTexture(GL_TEXTURE_2D, fActiveWorkTexBuffer->textureHandle( ));

    /* swap the alternate & active pointers */
    TextureBuffer* temp = fAltWorkTexBuffer;
    fAltWorkTexBuffer = fActiveWorkTexBuffer;
    fActiveWorkTexBuffer = temp;
}




void   SynthesisAgent::uConfigureInvariantLocs(
    SynthesisAgent::InvariantUniformState& state)
{
    state.workTexExtentLoc = glGetUniformLocation(fProgramSetID,
        "WorkTexExtent");
    state.sourceTexExtentLoc = glGetUniformLocation(fProgramSetID,
        "SourceTexExtent");
    state.sourceTileOriginsLoc = glGetUniformLocation(fProgramSetID,
        "SourceTileOrigins");
    state.sourceTileExtentsLoc = glGetUniformLocation(fProgramSetID,
        "SourceTileExtents");
    state.simTexExtentLoc = glGetUniformLocation(fProgramSetID,
        "SimTexExtent");
    state.simNumEntriesLoc = glGetUniformLocation(fProgramSetID,
        "SimNumEntries");
    state.numRefinementLevelsLoc = glGetUniformLocation(fProgramSetID,
        "NumRefinementLevels");
    state.workTexLoc = glGetUniformLocation(fProgramSetID, "WorkTex");
    state.sourceTexLoc = glGetUniformLocation(fProgramSetID, "SourceTex");
    state.simTexLoc = glGetUniformLocation(fProgramSetID, "SimTex");
    state.stateTexLoc = glGetUniformLocation(fProgramSetID, "StateTex");
    state.pixelWeightsLoc = glGetUniformLocation(fProgramSetID,
        "PixelWeights");
    state.jumpPenalizationParamLoc = glGetUniformLocation(fProgramSetID,
        "JumpPenalizationParam");
}




void  SynthesisAgent::uConfigurePerInvocationLocs(
    SynthesisAgent::PerInvocationUniformState& state)
{
    state.upsampleEffectsGeomLoc = glGetUniformLocation(fProgramSetID,
        "UpsampleEffectsGeometry");
    state.corr1EffectsGeomLoc = glGetUniformLocation(fProgramSetID,
        "Corr1EffectsGeometry");
    state.corr2EffectsGeomLoc = glGetUniformLocation(fProgramSetID,
        "Corr2EffectsGeometry");
    state.maxOutputEffectsGeomLoc = glGetUniformLocation(fProgramSetID,
        "MaxOutputEffectsGeometry");
}




void  SynthesisAgent::uWritePerInvocationUnifState(
    SynthesisAgent::PerInvocationUniformState& state)
{
    glUniform2fv(state.upsampleEffectsGeomLoc, 32,
        state.upsampleEffectsGeom);
    glUniform2fv(state.corr1EffectsGeomLoc, 20, state.corr1EffectsGeom);
    glUniform2fv(state.corr2EffectsGeomLoc, 20, state.corr2EffectsGeom);
    glUniform2fv(state.maxOutputEffectsGeomLoc, 4,
        state.maxOutputEffectsGeom);
}



void  SynthesisAgent::uRebindInvariantTextures( )
{
    /* bind the source mosaic texture to texture unit 1, and configure
       the filtering and wrapping parameters for texture unit 1 for
       pixel-exactness */
    glActiveTexture(GL_TEXTURE1);
    glBindTexture(GL_TEXTURE_2D, fSourceMosaicTexID);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);


    /* bind the similarity map mosaic texture to texture unit 2, and
       configure the filtering and wrapping parameters for texture unit 2
       for pixel-exactness */
    glActiveTexture(GL_TEXTURE2);
    glBindTexture(GL_TEXTURE_2D, fSimMapMosaicTexID);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    /* bind the persistent state texture to texture unit 3, and configure
       the filtering and wrapping parameters for texture unit 3 for pixel-
       exactness */
    glActiveTexture(GL_TEXTURE3);
    glBindTexture(GL_TEXTURE_1D, fStateCacheTexID);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    glActiveTexture(GL_TEXTURE0);
}




void  SynthesisAgent::dbgDrawActiveBuffer( ) const
{
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, fActiveWorkTexBuffer->textureHandle( ));
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

    glBegin(GL_QUADS);
        glTexCoord2i(0, 0);
        glVertex2i(0, 0);

        glTexCoord2i(1, 0);
        glVertex2i(kWorkTexExtent, 0);

        glTexCoord2i(1, 1);
        glVertex2i(kWorkTexExtent, kWorkTexExtent);

        glTexCoord2i(0, 1);
        glVertex2i(0, kWorkTexExtent);
    glEnd( );
    glDisable(GL_TEXTURE_2D);
}




float  SynthesisAgent::jumpParameter( ) const
{
    return fInvarUnifState->jumpPenalizationParam;
}




void   SynthesisAgent::setJumpParameter(float j)
{
    if ((j < 0.8f) || (j > 2.6f))
        throw std::logic_error("SynthesisAgent: jump penalization parameter"
            " out of range; valid range is 0.8 - 2.6, inclusive");

    fInvarUnifState->jumpPenalizationParam = j;
}




void  SynthesisAgent::setSimilarityMap(const PyramidalSimilarityMap& simmap)
{
    fSimMap = (& simmap);

    /* cache the extent of the similarity map mosaic texture */
    fInvarUnifState->simTexExtent[0] =
        static_cast<float>(fSimMap->mosaicImage( ).width( ));
    fInvarUnifState->simTexExtent[1] =
        static_cast<float>(fSimMap->mosaicImage( ).height( ));

    /* cache the number of similarity map entries per pixel */
    fInvarUnifState->simNumEntries =
        static_cast<float>(fSimMap->entriesPerPixel( ));

    /* compute the locations of the similarity map mosiac tiles, then download
       them into TRAM as an encoded 1D texture -- note that once nVidia
       corrects their 8xxx series driver issues, we can just
       store this in a uniform instead of downloading it as a texture */
    std::vector<Vector2Df>  simTileOrigins;
    for (size_t i = 0; i < fSimMap->mosaicTiles( ).size( ); i++) {

        const std::vector<Rectangle2Di>& simtiles =
            fSimMap->mosaicTiles( );
        Vector2Df simogn = Vector2Df(
            static_cast<float>(simtiles[i].origin.x),
            static_cast<float>(simtiles[i].origin.y));
        simTileOrigins.push_back(simogn);
    }
    glActiveTexture(GL_TEXTURE3);
    uUpdateTexture1D(fStateCacheTexID, simTileOrigins);
    glActiveTexture(GL_TEXTURE0);

    /* update similarity map mosaic image in texture unit 2 */
    glActiveTexture(GL_TEXTURE2);
    fGPU.updateTexture(fSimMapMosaicTexID, fSimMap->mosaicImage( ));
    glActiveTexture(GL_TEXTURE0);
}




void  SynthesisAgent::setSourceImage(const RGBAImage& srcimg)
{
    fPrecursorQuantization = srcimg.width( );

    /* build a Gaussian image pyramid out of the client-provided precursor
       image, and get the pyramid's mosaic image -- the mosaic image allows
       us to download the entire image pyramid into TRAM on the GPU in a
       relatively compact fashion */
    ImagePyramid* pyr;
    pyr = new ImagePyramid(srcimg);
    RGBAImage* oldmosaic = fSourceMosaicTex;
    fSourceMosaicTex = new RGBAImage(pyr->mosaicImage( ));
    delete oldmosaic;

    /* cache the extent of the source mosaic texture */
    fInvarUnifState->sourceTexExtent[0] =
        static_cast<float>(fSourceMosaicTex->width( ));
    fInvarUnifState->sourceTexExtent[1] =
        static_cast<float>(fSourceMosaicTex->height( ));

    /* cache the locations & sizes of the individual pyramid level tile images
       within the overall mosaic image */
    int idxticker = 0;
    std::vector<Rectangle2Di>::const_iterator it =
        pyr->mosaicTiles( ).begin( );
    for ( ; it != pyr->mosaicTiles( ).end( ); it++) {

        fInvarUnifState->sourceTileExtents[idxticker] =
            static_cast<float>(it->width);

        /* origin coordinate data has to be inserted in (y, x) order,
           instead of in customary (x, y) order, because the entire
           array is reversed after it's assembled -- why? because that's
           the way the shader program wants it */
        fInvarUnifState->sourceTileOrigins[2 * idxticker] =
            static_cast<float>(it->origin.y);
        fInvarUnifState->sourceTileOrigins[(2 * idxticker) + 1] =
            static_cast<float>(it->origin.x);

        fInvarUnifState->sourceTileExtents[idxticker] =
            static_cast<float>(it->width);

        idxticker++;
    }
    std::reverse(& fInvarUnifState->sourceTileOrigins[0],
        & fInvarUnifState->sourceTileOrigins[2 * idxticker]);
    std::reverse(& fInvarUnifState->sourceTileExtents[0],
        & fInvarUnifState->sourceTileExtents[idxticker]);

    /* cache the number of refinement levels */
    fInvarUnifState->numRefinementLevels = static_cast<int>(pyr->numLevels( ));

    /* after we've got the mosaic image and the tiles, we don't need the
       pyramid itself any more */
    delete pyr;

    /* update the source mosaic image in texture unit 1 */
    glActiveTexture(GL_TEXTURE1);
    fGPU.updateTexture(fSourceMosaicTexID, *fSourceMosaicTex);
}
