//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// MapPicker.cpp                                                            //
//                                                                          //
// Implements methods in class MapPicker only                               //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// <unversioned module>                                                     //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008, Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MapPicker.h"
#include "PTCIResources.h"
#include "StructuredIO.h"
#include <sstream>

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  MapPicker                                                         //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const std::string  MapPicker::kNoMapsFoundMessage =
    "MapPicker: no corresponding similarity maps found";


MapPicker::MapPicker(HWND overlaypeer)
    : fManagedView(overlaypeer, kMapPickerDialogResID),
      fOverlayPeer(overlaypeer), fBrushPath(""), fDidUserCancel(true)
{
    delete & fManagedView.installController(*this);
}




MapPicker::~MapPicker( )
{
}




std::string  MapPicker::runInteraction( )
{
    fDidUserCancel = true;
    fManagedView.startInteraction( );

    RECT  invalrect;
    GetClientRect(fOverlayPeer, &invalrect);
    InvalidateRect(fOverlayPeer, &invalrect, TRUE);
    UpdateWindow(fOverlayPeer);

    if (fDidUserCancel)
        return "";
    else
        return fUserMapPath;
}




void  MapPicker::setBrushPath(const std::string& brushpath)
{
    std::string dir = Win32Tools::filesystem( ).extractDirectory(brushpath);
    std::string fname = Win32Tools::filesystem( ).extractFilename(brushpath);

    std::string searchpat = dir + "\\" + fname + "*.simset";

    WIN32_FIND_DATA resultinfo;
    HANDLE searchresult = FindFirstFile(searchpat.c_str( ), &resultinfo);
    if (searchresult == INVALID_HANDLE_VALUE)
        throw std::runtime_error(kNoMapsFoundMessage);

    /* if execution reaches this point, at least one corresponding
       similarity map was found */
    fBrushPath = dir;
    fMatchingMaps.clear( );
    fMatchingMaps.push_back(resultinfo.cFileName);

    BOOL anotherfound;
    while (anotherfound = FindNextFile(searchresult, &resultinfo))
        fMatchingMaps.push_back(resultinfo.cFileName);
    
    BOOL closeok = FindClose(searchresult);
    if (!closeok)
        throw std::runtime_error("MapPicker: couldn't close file handle");
}




bool  MapPicker::isBrushPathSet( ) const
{
    if (fBrushPath == "")
        return false;
    else
        return true;
}




void  MapPicker::installed(ModalDialogView& sender)
{
}




void  MapPicker::deinstalled(ModalDialogView& sender)
{
}




bool  MapPicker::isInstalled( ) const
{
    return true;
}




void  MapPicker::layout(ModalDialogView& sender)
{
    if (sender.isInteractionRunning( )) {

        int conts_left = Win32Tools::getWindowOrigin(fMapListBox).x;
        int conts_right =
            Win32Tools::getWindowOrigin(fMapPropertiesGroup).x +
            Win32Tools::getWindowWidth(fMapPropertiesGroup);

        int separator_y =
            (Win32Tools::getWindowOrigin(fMapPropertiesGroup).y +
            Win32Tools::getWindowHeight(fMapPropertiesGroup) +
            Win32Tools::getWindowOrigin(fSelectButton).y) / 2;

        Win32Tools::setWindowOrigin(fHorizSeparator, conts_left, separator_y);
        Win32Tools::setWindowWidth(fHorizSeparator, conts_right -
            conts_left);
    }
}




void  MapPicker::image(ModalDialogView& sender)
{
}




void  MapPicker::commandPropagated(ModalDialogView& sender,
    HWND commandSource, unsigned short commandID)
{
    if (commandSource == fCancelButton) {

        sender.finishInteraction( );
    }
    else if (commandSource == fMapListBox) {

        if (commandID == LBN_SELCHANGE) {

            int newsel = static_cast<int>(SendMessage(fMapListBox,
                LB_GETCURSEL, 0, 0));

            if (newsel != LB_ERR)
                uLoadPropsFields(newsel);

        } /* if selection changed */
    } /* if sender was map list box */
    else if (commandSource == fSelectButton) {

        fDidUserCancel = false;

        int brushnum = SendMessage(fMapListBox, LB_GETCURSEL, 0, 0);
        fUserMapPath = fBrushPath + "\\" + fMatchingMaps[brushnum];

        sender.finishInteraction( );
    }
}




void  MapPicker::interactionStarted(ModalDialogView& sender)
{
    uAssociateResDefines( );
    uStylizeResDefines( );
    uPopulateMapListBox( );

    SendMessage(fMapListBox, LB_SETCURSEL, (WPARAM) 0, (LPARAM) 0);
    uLoadPropsFields(0);

    fHorizSeparator = Win32Tools::widgetFactory( ).makeHorizontalSeparator(
        sender.peerHandle( ), sender.kChildID0);
    ShowWindow(fHorizSeparator, SW_SHOW);

    
}




void  MapPicker::interactionFinished(ModalDialogView& sender)
{
    fMapListBox = 0;
    fMapPropertiesGroup = 0;
    fBaseDimLabel = 0;
    fBaseDimReadout = 0;
    fEncodedLevsLabel = 0;
    fEncodedLevsReadout = 0;
    fEntriesLabel = 0;
    fEntriesReadout = 0;
    fVarBoostLabel = 0;
    fVarBoostReadout = 0;
    fBoostFracLabel = 0;
    fBoostFracReadout = 0;
    fCancelButton = 0;
    fSelectButton = 0;

    Win32Tools::destroyPrimitive(fHorizSeparator);
}




void  MapPicker::dialogClosed(ModalDialogView& sender)
{
    sender.finishInteraction( );
}




void  MapPicker::viewEnabled(UIComponentView& sender)
{
}




void  MapPicker::viewDisabled(UIComponentView& sender)
{
}




void  MapPicker::uAssociateResDefines( )
{
    fMapListBox = fManagedView.itemPeer(kMapPickerListElemID);
    fMapPropertiesGroup = fManagedView.itemPeer(kMapPickerGroupElemID);
    fBaseDimLabel = fManagedView.itemPeer(kMapPickerBaseDimLabelElemID);
    fBaseDimReadout = fManagedView.itemPeer(kMapPickerBaseDimReadoutElemID);
    fEncodedLevsLabel = fManagedView.itemPeer(kMapPickerEncLevsLabelElemID);
    fEncodedLevsReadout =
        fManagedView.itemPeer(kMapPickerEncLevsReadoutElemID);
    fEntriesLabel = fManagedView.itemPeer(kMapPickerEntsLabelElemID);
    fEntriesReadout = fManagedView.itemPeer(kMapPickerEntsReadoutElemID);
    fVarBoostLabel = fManagedView.itemPeer(kMapPickerIsBoostLabelElemID);
    fVarBoostReadout = fManagedView.itemPeer(kMapPickerIsBoostReadoutElemID);
    fBoostFracLabel = fManagedView.itemPeer(kMapPickerBoostFracLabelElemID);
    fBoostFracReadout =
        fManagedView.itemPeer(kMapPickerBoostFracReadoutElemID);
    fCancelButton = fManagedView.itemPeer(kMapPickerCancelBtnElemID);
    fSelectButton = fManagedView.itemPeer(kMapPickerSelectBtnElemID);
}




void  MapPicker::uStylizeResDefines( )
{
    Win32Tools::applyControlFaceProperties(fMapListBox);
    Win32Tools::applyControlFaceProperties(fMapPropertiesGroup);
    Win32Tools::applyControlFaceProperties(fBaseDimLabel);
    Win32Tools::applyControlFaceProperties(fBaseDimReadout);
    Win32Tools::applyControlFaceProperties(fEncodedLevsLabel);
    Win32Tools::applyControlFaceProperties(fEncodedLevsReadout);
    Win32Tools::applyControlFaceProperties(fEntriesLabel);
    Win32Tools::applyControlFaceProperties(fEntriesReadout);
    Win32Tools::applyControlFaceProperties(fVarBoostLabel);
    Win32Tools::applyControlFaceProperties(fVarBoostReadout);
    Win32Tools::applyControlFaceProperties(fBoostFracLabel);
    Win32Tools::applyControlFaceProperties(fBoostFracReadout);
    Win32Tools::applyControlFaceProperties(fCancelButton);
    Win32Tools::applyControlFaceProperties(fSelectButton);
}




void  MapPicker::uPopulateMapListBox( )
{
    for (int i = 0; i < fMatchingMaps.size( ); i++) {

        std::string fname_noext =
            Win32Tools::filesystem( ).extractFilename(fMatchingMaps[i]);

        SendMessage(fMapListBox, LB_ADDSTRING, 0,
            (LPARAM)fname_noext.c_str( ));
    }
}




void  MapPicker::uReadMapHeader(int mapnum,
    MapPicker::MapHeaderRecord& output)
{
    std::string  mappath = fBrushPath + "\\" + fMatchingMaps[mapnum];

    std::string  headerlines =
        Win32Tools::filesystem( ).mapTextFile(mappath, 20);

    StructuredReader reader(headerlines);

    std::string mapfilemagic = reader.readFiletypeID( );
    if (mapfilemagic != "PYRSIM")
        throw std::runtime_error("MapPicker: bad similarity map file magic "
            "number");

    std::string headerid = reader.readIdentifier( );
    if (headerid != "pyramidalSimset")
        throw std::runtime_error("MapPicker: bad similarity map file header "
            "group declaration");

    reader.readGroupOpen( );
    bool gotBWid = false, gotBHt = false, gotEnts = false, gotLevs = false,
        gotIsBoosted = false, gotBoostFrac = false;
    for (int i = 0; i < 6; i++) {

        std::string curtok = reader.readIdentifier( );

        if (curtok == "baseHorizPixels") {
    
            reader.readAssignmentOperator( );
            output.basedim = reader.readInteger( );
            gotBWid = true;
        }
        else if (curtok == "baseVertPixels") {
    
            reader.readAssignmentOperator( );
            reader.readInteger( );
            gotBHt = true;
        }
        else if (curtok == "numLevels") {
    
            reader.readAssignmentOperator( );
            output.numlevs = reader.readInteger( );
            gotLevs = true;
        }
        else if (curtok == "entriesPerPix") {
    
            reader.readAssignmentOperator( );
            output.entries = reader.readInteger( );
            gotEnts = true;
        }
        else if (curtok == "isVarBoosted") {
    
            reader.readAssignmentOperator( );
            output.isboosted = reader.readBoolean( );
            gotIsBoosted = true;
        }
        else if (curtok == "boostSepFrac") {
    
            reader.readAssignmentOperator( );
            output.boostfrac = reader.readFloat( );
            gotBoostFrac = true;
        }
        else {
    
            throw std::runtime_error("MapPicker: bad token in map file "
                "header");
        }
    } /* for all 6 required header fields */
    if (! (gotBWid && gotBHt && gotEnts && gotLevs && gotIsBoosted &&
           gotBoostFrac)) {

            throw std::runtime_error("MapPicker: one or more required "
                "header fields not present");
    }
}




void  MapPicker::uLoadPropsFields(int mapnum)
{
    MapHeaderRecord mapinfo;

    uReadMapHeader(mapnum, mapinfo);

    std::ostringstream  formatstream;

    Win32Tools::setWindowTextIntegral(fBaseDimReadout, mapinfo.basedim);

    Win32Tools::setWindowTextIntegral(fEncodedLevsReadout, mapinfo.numlevs);

    Win32Tools::setWindowTextIntegral(fEntriesReadout, mapinfo.entries);

    std::string enableBoostVal;
    if (mapinfo.isboosted)
        enableBoostVal = "Enabled";
    else
        enableBoostVal = "Disabled";
    Win32Tools::setWindowTextString(fVarBoostReadout, enableBoostVal);

    std::string boostFracVal;
    if (mapinfo.isboosted) {

        if ((mapinfo.boostfrac > 0.032f) && (mapinfo.boostfrac < 0.034f))
            boostFracVal = "1 / 30";
        else if ((mapinfo.boostfrac > 0.049f) && (mapinfo.boostfrac < 0.051f))
            boostFracVal = "1 / 20";
        else if ((mapinfo.boostfrac > 0.065f) && (mapinfo.boostfrac < 0.067f))
            boostFracVal = "1 / 15";
        else if ((mapinfo.boostfrac > 0.09f) && (mapinfo.boostfrac < 0.11f))
            boostFracVal = "1 / 10";
        else if ((mapinfo.boostfrac > 0.19f) && (mapinfo.boostfrac < 0.21f))
            boostFracVal = "1 / 5";
        else if ((mapinfo.boostfrac > 0.24f) && (mapinfo.boostfrac < 0.26f))
            boostFracVal = "1 / 4";
        else if ((mapinfo.boostfrac > 0.32f) && (mapinfo.boostfrac < 0.34f))
            boostFracVal = "1 / 3";
        else if ((mapinfo.boostfrac > 0.49f) && (mapinfo.boostfrac < 0.51f))
            boostFracVal = "1 / 2";
        else
            boostFracVal = "BAD";
    }
    else {

        boostFracVal = "n / a";
    }
    Win32Tools::setWindowTextString(fBoostFracReadout, boostFracVal);
}
//////////////////////////////////////////////////////////////////////////////
// END  MapPicker                                                           //
//////////////////////////////////////////////////////////////////////////////
