//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ImageRepresentation.h                                                    //
//                                                                          //
// Defines the lightweight types RGBColor and PixelConfiguration. Defines   //
// the RGBAImageClass and its nested classes fragment, const_fragment,      //
// subspace, and const_subspace. Defines the ImageTools procedure wrapper.  //
// Defines the PTCI image convolution classes Convolver, ConvolverFactory,  //
// ContinuousConvolver, and GaussianContinuousConvolver.                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver 2.2.0 of Sat 18 Oct 2008 @ 10:15pm EDT                               //
//                                                                          //
//     added methods installMapping( ) and write( ) to class RGBAImage;     //
//     added method saveTARGA to class ImageTools                           //
//                                                                          //
// ver 2.1.5 of Mon 12 May 2008 @ 3:24pm EDT                                //
//                                                                          //
//     added method cancelAction( ) to class RGBAImage to make the class    //
//     compliant with the new InterstitialActionSender interface            //
//                                                                          //
// ver 2.1.1 of Fri 3 May 2008 @ 7:38pm EDT                                 //
//                                                                          //
//     changed all references of class template Vertex2D to Vector2D; see   //
//     the AffineGeometry module's header for more info                     //
//                                                                          //
// previous change history elided; check out an older CVS rev to get it     //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008 Lucas Stephen Beeler. All Rights Reserved.       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef  IMAGE_REPRESENTATION__HXX
#define  IMAGE_REPRESENTATION__HXX

#include <PTCITypes.h>
#include <libtarga.h>
#include <AffineGeometry.h>
#include <UserInteraction.h>
#include <string>
#include <set>
#include <typeinfo>
#include <stdexcept>
#include <vector>

//-------------------------------------------------------------------------//
// TYPE  RGBColor                                                          //
//-------------------------------------------------------------------------//
struct  RGBColor {

private:

    void  uClamp( );
    void  uClampCheck( ) const;

public:

    float r;
    float g;
    float b;

    RGBColor( );
    RGBColor(float ur, float ug, float ub);

    bool       operator==(const RGBColor& rhs) const;
    bool       operator!=(const RGBColor& rhs) const;
    bool       operator<(const RGBColor& rhs) const;
    bool       operator>(const RGBColor& rhs) const;
    bool       operator<=(const RGBColor& rhs) const;
    bool       operator>=(const RGBColor& rhs) const;
    RGBColor   operator*(float x) const;
    RGBColor   operator/(float x) const;
    RGBColor   operator+(const RGBColor& rhs) const;
    RGBColor   operator-(const RGBColor& rhs) const;
    RGBColor   operator+=(const RGBColor& rhs);
    float      distance(const RGBColor& rhs) const;
};

RGBColor  operator*(float x, RGBColor c);
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// TYPE  PixelConfiguration                                                //
//-------------------------------------------------------------------------//
enum  PixelConfiguration { kRGBAConfiguration, kBGRConfiguration };
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS  RGBAImage                                                        //
//-------------------------------------------------------------------------//
class RGBAImage :
    public virtual InterstitialActionSender {

    public : enum  ImageKind  { kFileMappedImage, kInMemoryImage };
    public : enum  ImageFileFormat { kTargaImageFormat };

private:

    static const unsigned short  sBitsPerChannel = 8;
    static const unsigned short  sChannelsPerPixel = 4;
    static const unsigned short  sBitsPerPixel =
        sBitsPerChannel * sChannelsPerPixel;
    static const PixelConfiguration  sPixelConfiguration = kRGBAConfiguration;

    unsigned char*                         fImageBuffer;
    unsigned                               fImageWidth;
    unsigned                               fImageHeight;
    unsigned                               fImageBufferSize;
    ImageKind                              fImageKind;
    std::string                            fMapSourcePath;
    ImageFileFormat                        fMapSourceFormat;
    std::set<InterstitialActionReceiver*>  fReceiverSet;

public:

    class  fragment;
    class  const_fragment;
    class  subspace;
    class  const_subspace;

    friend  class  fragment;
    friend  class  const_fragment;
    friend  class  subspace;
    friend  class  const_subspace;

    //---------------------------------------------------------------------//
    // NESTED CLASS  RGBAImage::fragment                                   //
    //---------------------------------------------------------------------//
    class  fragment  {

        friend class RGBAImage;
        friend class RGBAImage::const_fragment;
        friend class RGBAImage::subspace;

    private:

        subspace*           fHostSubspace;
        mutable RGBAImage*  fHostImage;
        unsigned            fX;
        unsigned            fY;

        fragment(RGBAImage* img, unsigned x, unsigned y);
        fragment(subspace* sub, unsigned x, unsigned y);

        void  uCheckSameHosts(const fragment& other) const;
        void  uCheckSameHosts(const const_fragment& other) const;
        void  uAdvance( );
        unsigned char*  uAddrRed( ) const;
        unsigned char*  uAddrGreen( ) const;
        unsigned char*  uAddrBlue( ) const;

    public:

        fragment(const fragment& frag);

        const fragment&  operator=(const fragment& rhs);
        bool             operator==(const fragment& rhs) const;
        bool             operator==(const const_fragment& rhs) const;
        bool             operator!=(const fragment& rhs) const;
        bool             operator!=(const const_fragment& rhs) const;
        bool             operator<(const fragment& rhs) const;
        bool             operator<(const const_fragment& rhs) const;
        bool             operator>(const fragment& rhs) const;
        bool             operator>(const const_fragment& rhs) const;
        bool             operator<=(const fragment& rhs) const;
        bool             operator<=(const const_fragment& rhs) const;
        bool             operator>=(const fragment& rhs) const;
        bool             operator>=(const const_fragment& rhs) const;
        fragment&        operator++( );
        fragment         operator++(int);
        RGBColor         color( ) const;
        void             setColor(const RGBColor&);
    };


    //---------------------------------------------------------------------//

    //---------------------------------------------------------------------//
    // NESTED CLASS  RGBAImage::const_fragment                             //
    //---------------------------------------------------------------------//
    class  const_fragment  {

        friend class RGBAImage;
        friend class RGBAImage::fragment;
        friend class RGBAImage::subspace;
        friend class RGBAImage::const_subspace;

    private:

        const RGBAImage*       fHostImage;
        const subspace*        fHostSubspace;
        const const_subspace*  fHostConstSubspace;
        unsigned               fX;
        unsigned               fY;

        const_fragment(const RGBAImage*, unsigned x, unsigned y);
        const_fragment(const subspace* sub, unsigned x, unsigned y);
        const_fragment(const const_subspace* csub, unsigned x, unsigned y);

        void  uCheckSameHosts(const fragment& other) const;
        void  uCheckSameHosts(const const_fragment& other) const;
        void  uAdvance( );
        const unsigned char*  uAddrRed( ) const;
        const unsigned char*  uAddrGreen( ) const;
        const unsigned char*  uAddrBlue( ) const;

    public:

        const_fragment(const const_fragment& frag);
        const_fragment(const fragment& frag);

        const const_fragment&  operator=(const fragment& rhs);
        const const_fragment&  operator=(const const_fragment& rhs);
        bool                   operator==(const fragment& rhs) const;
        bool                   operator==(const const_fragment& rhs) const;
        bool                   operator!=(const fragment& rhs) const;
        bool                   operator!=(const const_fragment& rhs) const;
        bool                   operator<(const fragment& rhs) const;
        bool                   operator<(const const_fragment& rhs) const;
        bool                   operator>(const fragment& rhs) const;
        bool                   operator>(const const_fragment& rhs) const;
        bool                   operator<=(const fragment& rhs) const;
        bool                   operator<=(const const_fragment& rhs) const;
        bool                   operator>=(const fragment& rhs) const;
        bool                   operator>=(const const_fragment& rhs) const;
        const_fragment&        operator++( );
        const_fragment         operator++(int);
        RGBColor               color( ) const;
    };
    //---------------------------------------------------------------------//

    //---------------------------------------------------------------------//
    // NESTED CLASS  RGBAImage::subspace                                   //
    //---------------------------------------------------------------------//
    class subspace {

        friend class RGBAImage;
        friend class RGBAImage::fragment;
        friend class RGBAImage::const_fragment;
        friend class RGBAImage::const_subspace;

    private:

        RGBAImage*       fHostImage;
        Vector2Di        fOrigin;
        unsigned short   fWidth;
        unsigned short   fHeight;

        subspace(RGBAImage* img, const Vector2Di& origin, unsigned short w,
            unsigned short h);
    public:

        typedef  RGBAImage::fragment  fragment;
        typedef  RGBAImage::const_fragment  const_fragment;

        subspace(const subspace&);

        const subspace&        operator=(const subspace&);
        bool                   operator==(const subspace& rhs) const;
        bool                   operator!=(const subspace& rhs) const;
        bool                   operator==(const const_subspace& rhs) const;
        bool                   operator!=(const const_subspace& rhs) const;

        fragment  firstFragment( );
        const_fragment  firstFragment( ) const;
        fragment  lastFragment( );
        const_fragment  lastFragment( ) const;
        fragment  fragmentAt(unsigned short x, unsigned short y);
        const_fragment  fragmentAt(unsigned short x, unsigned short y) const;
        unsigned short  width( ) const;
        unsigned short  height( ) const;
        const Vector2Di&  origin( ) const;

        RGBAImage  makeImage( ) const;
        RGBAImage&  hostImage( );
        const RGBAImage&  hostImage( ) const;
        RGBAImage::subspace  subspaceAt(unsigned short x, unsigned short y,
            unsigned short w, unsigned short h);
        RGBAImage::const_subspace  subspaceAt(unsigned short x,
            unsigned short y, unsigned short w, unsigned short h) const;

    };
    //---------------------------------------------------------------------//

    //---------------------------------------------------------------------//
    // NESTED CLASS  RGBAImage::const_subspace                             //
    //---------------------------------------------------------------------//
    class const_subspace {

        friend class RGBAImage;
        friend class RGBAImage::const_fragment;
        friend class RGBAImage::const_subspace;

    private:

        const RGBAImage*  fHostImage;
        Vector2Di         fOrigin;
        unsigned short    fWidth;
        unsigned short    fHeight;

        const_subspace(const RGBAImage* img, const Vector2Di& origin,
            unsigned short w, unsigned short h);
    public:

        typedef  RGBAImage::fragment  fragment;
        typedef  RGBAImage::const_fragment  const_fragment;

        const_subspace(const const_subspace& sub);
        const_subspace(const subspace& sub);

        const const_subspace&  operator=(const const_subspace& sub);
        const const_subspace&  operator=(const subspace& sub);
        bool                   operator==(const const_subspace& rhs) const;
        bool                   operator!=(const const_subspace& rhs) const;
        bool                   operator==(const subspace& rhs) const;
        bool                   operator!=(const subspace& rhs) const;

        const_fragment  firstFragment( ) const;
        const_fragment  lastFragment( ) const;
        const_fragment  fragmentAt(unsigned short x, unsigned short y) const;
        unsigned short  width( ) const;
        unsigned short  height( ) const;
        const Vector2Di&  origin( ) const;

        RGBAImage  makeImage( ) const;
        const RGBAImage&  hostImage( ) const;
        RGBAImage::const_subspace  subspaceAt(unsigned short x,
            unsigned short y, unsigned short w, unsigned short h) const;
        void  draw( ) const;
    };

    //---------------------------------------------------------------------//

    static  void  transferPixels(const RGBAImage& src, RGBAImage& dest,
        unsigned short srcx, unsigned short srcy, unsigned short destx,
        unsigned short desty, unsigned short width, unsigned short height);

    static  RGBAImage*  createMappedImage(ImageFileFormat format,
        const std::string& path);
    static  RGBAImage*  createMappedImage(ImageFileFormat format,
        const std::string& path, InterstitialActionReceiverSet& notifyset,
        InterstitialActionSender& proxyhost);

    RGBAImage(unsigned short width, unsigned short height);
    RGBAImage(const RGBAImage& source);

    virtual ~RGBAImage( );

    const RGBAImage&  operator=(const RGBAImage& source);

    inline  unsigned short  width( ) const
    {
        return fImageWidth;
    }

    inline  unsigned short  height( ) const
    {
        return fImageHeight;
    }

    inline  unsigned char*  imageBuffer( )
    {
        return fImageBuffer;
    }

    inline  const unsigned char*  imageBuffer( ) const
    {
        return fImageBuffer;
    }

    unsigned        imageBufferSize( ) const;
    unsigned short  bitsPerChannel( ) const;
    unsigned short  bitsPerPixel( ) const;
    unsigned short  channelsPerPixel( ) const;
    PixelConfiguration  pixelConfiguration( ) const;
    fragment  fragmentAt(unsigned short x, unsigned short y);
    const_fragment  fragmentAt(unsigned short x, unsigned short y) const;
    fragment  firstFragment( );
    const_fragment  firstFragment( ) const;
    fragment  lastFragment( );
    const_fragment  lastFragment( ) const;
    subspace  subspaceAt(unsigned short x, unsigned short y, unsigned short w,
        unsigned short h);
    const_subspace  subspaceAt(unsigned short x, unsigned short y,
        unsigned short w, unsigned short h) const;
    void  convolve(const Convolver&);
    void  convolve(const ContinuousConvolver&);
    void  resample(unsigned short outw, unsigned short outh,
        const ContinuousConvolver& filter);
    void  draw( ) const;
    ImageKind  imageKind( ) const;
    const std::string&  mappingSource( ) const;
    ImageFileFormat  sourceFormat( ) const;
    void  installMapping(ImageFileFormat format, const std::string& path);
    void  write( ) const;
    void  registerReceiver(InterstitialActionReceiver& recv);
    void  deregisterReceiver(InterstitialActionReceiver& recv);
    void  cancelAction(InterstitialActionReceiver& recv);
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS  ImageTools                                                       //
//-------------------------------------------------------------------------//
class  ImageTools {

private:

    ImageTools( );

public:

    virtual  ~ImageTools( );

    static  RGBAImage*  loadTARGA(const std::string& filename);
    static  RGBAImage*  loadTARGA(const std::string& filename,
        InterstitialActionReceiverSet& notifyset,
        InterstitialActionSender& proxyhost);
    static  void        saveTARGA(const RGBAImage& img,
        const std::string& filename);

    template<typename TImageData1, typename TImageData2>
    static  float  perceptualDistance(const TImageData1& idat1,
        const TImageData2& idat2)
    {
        if ((idat1.width( ) != idat2.width( )) ||
            (idat1.height( ) != idat2.height( )))
            throw  std::logic_error("ImageTools: perceptualDistance( ) "
                "cannot compare two image entities of differing dimensions");

        TImageData1::const_fragment  fragment1 = idat1.firstFragment( );
        TImageData2::const_fragment  fragment2 = idat2.firstFragment( );

        float  accum = 0.0f;

        while (fragment1 <= idat1.lastFragment( )) {

            RGBColor c1 = fragment1.color( );
            RGBColor c2 = fragment2.color( );

            accum += c2.distance(c1);

            fragment1++;
            fragment2++;
        }

        return accum;
    }

    template<typename TImageData1, typename TImageData2>
    static  void  transferPixels(const TImageData1& src,
        TImageData2& dest, float srcFrac)
    {
        if ((src.width( ) != dest.width( )) &&
            (src.height( ) != dest.height( )))
            throw  std::logic_error("ImageTools: transferPixels( ) received "
                "source and destination pixel spaces of different sizes");

        if (srcFrac > 1.0f)
            srcFrac = 1.0f;
        else if (srcFrac < 0.0f)
            srcFrac = 0.0f;
        else
            ;

        float  destFrac = 1.0f - srcFrac;

        TImageData1::const_fragment  srcIter = src.firstFragment( );
        TImageData2::fragment        destIter = dest.firstFragment( );

        while (srcIter <= src.lastFragment( )) {

            RGBColor  srcColor = srcIter.color( );
            RGBColor  destColor = destIter.color( );

            RGBColor  resultColor = (srcColor * srcFrac) +
                (destColor * destFrac);

            destIter.setColor(resultColor);

            srcIter++;
            destIter++;
        }
    }
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS  Convolver                                                        //
//-------------------------------------------------------------------------//
class  Convolver {

private:

    float*           fKernelData;
    unsigned short   fSideDimension;

public:

    Convolver(unsigned short radius = 1);

    Convolver(const Convolver&);

    ~Convolver( )
    {
        delete [ ] fKernelData;
    }

    const Convolver&  operator=(const Convolver&);

    unsigned short  radius( ) const
    {
        return (fSideDimension - 1) / 2;
    }

    unsigned short  sideDimension( ) const
    {
        return fSideDimension;
    }

    const float&  element(unsigned short x, unsigned short y) const
    {
        if ((x >= fSideDimension) || (y >= fSideDimension))
            throw std::logic_error("Convolver::element( ): index "
                "out of range");

        return fKernelData[(y * fSideDimension) + x];
    }

    float&  element(unsigned short x, unsigned short y)
    {
        if ((x >= fSideDimension) || (y >= fSideDimension))
            throw std::logic_error("Convolver::element( ): index "
                "out of range");

        return fKernelData[(y * fSideDimension) + x];
    }

    float  subspaceSum(unsigned short left, unsigned short bottom,
        unsigned short right, unsigned short top) const
    {
        float accum = 0.0f;

        for (int j = bottom; j <= top; j++)
            for (int i = left; i <= right; i++)
                accum += this->element(i, j);

        return accum;
    }

    void  normalize( )
    {
        float scale = subspaceSum(0, 0, sideDimension( ) - 1,
            sideDimension( ) - 1);

        scale = 1.0f / scale;

        for (int j = 0; j < sideDimension( ); j++)
            for (int i = 0; i < sideDimension( ); i++)
                element(i, j) = element(i, j) * scale;
    }
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS  ConvolverFactory                                                 //
//-------------------------------------------------------------------------//
class  ConvolverFactory {

private:

    static  ConvolverFactory*  sSingletonInstance;

protected:

    ConvolverFactory( );

public:

    static  ConvolverFactory&  instance( );

    virtual  ~ConvolverFactory( );

    virtual  Convolver*  createGaussian(unsigned short);
    virtual  Convolver*  createBox(unsigned short);
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS  ContinuousConvolver                                              //
//-------------------------------------------------------------------------//
class  ContinuousConvolver {

protected:

    float  fRadius;

    ContinuousConvolver(float radius = 1.0f);

public:

    virtual  ~ContinuousConvolver( );

    virtual  float  evaluate(float, float) const = 0;

    virtual  float  radius( ) const
    {
        return fRadius;
    }
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// CLASS  GaussianContinuousConvolver                                      //
//-------------------------------------------------------------------------//
class  GaussianContinuousConvolver : public virtual ContinuousConvolver {

protected:

    float  fVariance;

public:

    GaussianContinuousConvolver(float radius = 2.0f);
    GaussianContinuousConvolver(float radius, float variance);

    virtual  ~GaussianContinuousConvolver( );

    float  evaluate(float, float) const;
};
//-------------------------------------------------------------------------//

#endif
