//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ImageInspector.cpp                                                       //
//                                                                          //
// Implements methods in the ImageInspector class. Defines the              //
// ImageInspector::DialogController and                                     //
// ImageInspector::ImageStatsPaneController nested classes and implements   //
// methods in them.                                                         //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 1.1.2 of Tue 20 May 2008 @ 2:32pm EDT                               //
//                                                                          //
//    fixed hidden widget problem by adding ShowWindow( ) calls following   //
//    calls to widget factory creation routines                             //
//                                                                          //
// ver. 1.1.1 of Fri 3 May 2008 @ 11:33pm EDT                               //
//                                                                          //
//     changed all references of class template Vertex2D to Vector2D; see   //
//     the AffineGeometry module's header for more info                     //
//                                                                          //
// previous change history elided; check out an older CVS rev to get it     //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008, Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "ImageInspector.h"
#include "PTCIResources.h"
#include "WorkPane.h"
#include "Application.h"
#include <typeinfo>
#include <sstream>

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ImageInspector :: ImageStatsPaneController                        //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class ImageInspector::ImageStatsPaneController
    : public virtual WorkPaneController {

private:

    WorkPaneView*     fManagedView;
    HWND              fHostClassLabel;
    HWND              fHostClassReadout;
    HWND              fHostAddressLabel;
    HWND              fHostAddressReadout;
    HWND              fImageKindLabel;
    HWND              fImageKindReadout;
    HWND              fMapSourceLabel;
    HWND              fMapSourceReadout;
    HWND              fSourceFormatLabel;
    HWND              fSourceFormatReadout;
    HWND              fDimensionLabel;
    HWND              fDimensionReadout;
    const RGBAImage*  fTargetImage;

    void  uSynchronizeReadouts(const RGBAImage& img);
    void  uClearReadouts( );

public:

    static  const unsigned short  kPaneHeight = 84;
    static  const unsigned short  kTitleColumnWidth = 108;
    static  const unsigned short  kLeftMargin = 12;
    static  const unsigned short  kRightMargin = 0;
    static  const unsigned short  kHorizontalPadding = 16;
    static  const unsigned short  kVerticalPadding = 6;
    static  const unsigned short  kReadoutColumnWidth = 132;

    ImageStatsPaneController( );
    ImageStatsPaneController(const RGBAImage& img);
    virtual  ~ImageStatsPaneController( );

    void  installed(WorkPaneView& sender);
    void  deinstalled(WorkPaneView& sender);
    void  layout(WorkPaneView& sender);
    void  commandPropagated(WorkPaneView& sender,
        HWND commandSource, unsigned short commandID);
    bool  isInstalled( ) const;
    void  setTargetImage(const RGBAImage& img);
    void  viewEnabled(UIComponentView& sender);
    void  viewDisabled(UIComponentView& sender);

};




ImageInspector::ImageStatsPaneController::ImageStatsPaneController( )
    : fManagedView(0), fTargetImage(0)
{
}




ImageInspector::ImageStatsPaneController::ImageStatsPaneController(
    const RGBAImage& img)
        : fManagedView(0), fTargetImage(& img)
{
}




ImageInspector::ImageStatsPaneController::~ImageStatsPaneController( )
{
    if (isInstalled( ))
        fManagedView->installController(
            * WorkPaneController::nullController( ));
}




void  ImageInspector::ImageStatsPaneController::installed(
    WorkPaneView& sender)
{
    sender.setHeight(kPaneHeight);

    int idticker = sender.kChildID0;

    fHostClassLabel = Win32Tools::widgetFactory( ).makeBoldLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fHostClassLabel, "Object Type:");
    Win32Tools::setWindowWidth(fHostClassLabel, kTitleColumnWidth);
    ShowWindow(fHostClassLabel, SW_SHOW);

    fHostClassReadout = Win32Tools::widgetFactory( ).makeLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fHostClassReadout, "RGBAImage");
    Win32Tools::setWindowWidth(fHostClassReadout, kReadoutColumnWidth);
    ShowWindow(fHostClassReadout, SW_SHOW);

    fHostAddressLabel = Win32Tools::widgetFactory( ).makeBoldLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fHostAddressLabel, "Object Address:");
    Win32Tools::setWindowWidth(fHostAddressLabel, kTitleColumnWidth);
    ShowWindow(fHostAddressLabel, SW_SHOW);

    fHostAddressReadout = Win32Tools::widgetFactory( ).makeLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fHostAddressReadout, "0x7FA8E2");
    Win32Tools::setWindowWidth(fHostAddressReadout, kReadoutColumnWidth);
    ShowWindow(fHostAddressReadout, SW_SHOW);

    fImageKindLabel = Win32Tools::widgetFactory( ).makeBoldLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fImageKindLabel, "Image Kind:");
    Win32Tools::setWindowWidth(fImageKindLabel, kTitleColumnWidth);
    ShowWindow(fImageKindLabel, SW_SHOW);

    fImageKindReadout = Win32Tools::widgetFactory( ).makeLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fImageKindReadout, "file-mapped");
    Win32Tools::setWindowWidth(fImageKindReadout, kReadoutColumnWidth);
    ShowWindow(fImageKindReadout, SW_SHOW);

    fMapSourceLabel = Win32Tools::widgetFactory( ).makeBoldLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fMapSourceLabel, "Mapping Source:");
    Win32Tools::setWindowWidth(fMapSourceLabel, kTitleColumnWidth);
    ShowWindow(fMapSourceLabel, SW_SHOW);

    fMapSourceReadout = Win32Tools::widgetFactory( ).makeLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fMapSourceReadout,
        "ConteCrayon_large_800.tga");
    Win32Tools::setWindowWidth(fMapSourceReadout, kReadoutColumnWidth);
    ShowWindow(fMapSourceReadout, SW_SHOW);

    fSourceFormatLabel = Win32Tools::widgetFactory( ).makeBoldLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fSourceFormatLabel, "Source Format:");
    Win32Tools::setWindowWidth(fSourceFormatLabel, kTitleColumnWidth);
    ShowWindow(fSourceFormatLabel, SW_SHOW);

    fSourceFormatReadout = Win32Tools::widgetFactory( ).makeLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fSourceFormatReadout,
        "TrueVision TARGA");
    Win32Tools::setWindowWidth(fSourceFormatReadout, kReadoutColumnWidth);
    ShowWindow(fSourceFormatReadout, SW_SHOW);

    fDimensionLabel = Win32Tools::widgetFactory( ).makeBoldLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fDimensionLabel, "Dimension:");
    Win32Tools::setWindowWidth(fDimensionLabel, kTitleColumnWidth);
    ShowWindow(fDimensionLabel, SW_SHOW);

    fDimensionReadout = Win32Tools::widgetFactory( ).makeLabel(
        sender.peerHandle( ), idticker++);
    Win32Tools::setWindowTextString(fDimensionReadout,
        "1400  1050 pixels");
    Win32Tools::setWindowWidth(fDimensionReadout, kReadoutColumnWidth);
    ShowWindow(fDimensionReadout, SW_SHOW);

    if (fTargetImage)
        uSynchronizeReadouts(*fTargetImage);
    else
        uClearReadouts( );

    fManagedView = & sender;
}




void  ImageInspector::ImageStatsPaneController::deinstalled(
    WorkPaneView& sender)
{
    fManagedView = 0;

    Win32Tools::destroyPrimitive(fHostClassLabel);
    Win32Tools::destroyPrimitive(fHostClassReadout);

    Win32Tools::destroyPrimitive(fHostAddressLabel);
    Win32Tools::destroyPrimitive(fHostAddressReadout);

    Win32Tools::destroyPrimitive(fImageKindLabel);
    Win32Tools::destroyPrimitive(fImageKindReadout);

    Win32Tools::destroyPrimitive(fMapSourceLabel);
    Win32Tools::destroyPrimitive(fMapSourceReadout);

    Win32Tools::destroyPrimitive(fSourceFormatLabel);
    Win32Tools::destroyPrimitive(fSourceFormatReadout);

    Win32Tools::destroyPrimitive(fDimensionLabel);
    Win32Tools::destroyPrimitive(fDimensionReadout);
}




void  ImageInspector::ImageStatsPaneController::layout(
    WorkPaneView& sender)
{
    if (isInstalled( )) {

        const int  pane_ht = sender.height( );
        const int  pane_wd = sender.width( );
        const int  unif_label_ht =
            Win32Tools::getWindowHeight(fHostClassLabel);
        const int  content_ht = (3 * unif_label_ht) + (2 * kVerticalPadding);
        const int  content_y = (pane_ht - content_ht) / 2;
        const int  c1_title_x = kLeftMargin;
        const int  c1_readout_x = c1_title_x + kTitleColumnWidth;
        const int  c2_title_x = c1_readout_x + kReadoutColumnWidth +
            kHorizontalPadding;
        const int  c2_readout_x = c2_title_x + kTitleColumnWidth;

        int running_y = content_y;

        Win32Tools::setWindowOrigin(fHostClassLabel, c1_title_x, running_y);
        Win32Tools::setWindowOrigin(fHostClassReadout, c1_readout_x,
            running_y);
        Win32Tools::setWindowOrigin(fMapSourceLabel, c2_title_x, running_y);
        Win32Tools::setWindowOrigin(fMapSourceReadout, c2_readout_x,
            running_y);

        running_y += unif_label_ht;
        running_y += kVerticalPadding;

        Win32Tools::setWindowOrigin(fHostAddressLabel, c1_title_x,
            running_y);
        Win32Tools::setWindowOrigin(fHostAddressReadout, c1_readout_x,
            running_y);
        Win32Tools::setWindowOrigin(fSourceFormatLabel, c2_title_x,
            running_y);
        Win32Tools::setWindowOrigin(fSourceFormatReadout, c2_readout_x,
            running_y);
        running_y += unif_label_ht;
        running_y += kVerticalPadding;

        Win32Tools::setWindowOrigin(fImageKindLabel, c1_title_x, running_y);
        Win32Tools::setWindowOrigin(fImageKindReadout, c1_readout_x,
            running_y);
        Win32Tools::setWindowOrigin(fDimensionLabel, c2_title_x,
            running_y);
        Win32Tools::setWindowOrigin(fDimensionReadout, c2_readout_x,
            running_y);
    }
}




void  ImageInspector::ImageStatsPaneController::commandPropagated(
    WorkPaneView& sender, HWND commandSource, unsigned short commandID)
{
}




bool  ImageInspector::ImageStatsPaneController::isInstalled( ) const
{
    if (fManagedView)
        return true;
    else
        return false;
}




void  ImageInspector::ImageStatsPaneController::uSynchronizeReadouts(
    const RGBAImage& img)
{
    const std::string kUnknown = "unknown";

    Win32Tools::setWindowTextString(fHostClassReadout, typeid(img).name( ));

    std::ostringstream  addr_stream;
    addr_stream << "0x" << std::hex << (& img);
    Win32Tools::setWindowTextString(fHostAddressReadout, addr_stream.str( ));

    std::string  kind_string;
    if (img.imageKind( ) == RGBAImage::kInMemoryImage)
        kind_string = "in-memory image";
    else if (img.imageKind( ) == RGBAImage::kFileMappedImage)
        kind_string = "file-mapped image";
    else
        kind_string = "unknown kind";
    Win32Tools::setWindowTextString(fImageKindReadout, kind_string);

    if (img.imageKind( ) == RGBAImage::kFileMappedImage) {

        Win32Tools::setWindowTextString(fMapSourceReadout,
            img.mappingSource( ));
        Win32Tools::setWindowTextString(fSourceFormatReadout,
            ImageInspector::sFormatNames[img.sourceFormat( )]);
    }
    else {

        Win32Tools::setWindowTextString(fMapSourceReadout, "not mapped");
        Win32Tools::setWindowTextString(fSourceFormatReadout, "not mapped");
    }

    std::ostringstream  dimension_stream;
    dimension_stream << img.width( ) << "  " << img.height( ) << " pixels";
    Win32Tools::setWindowTextString(fDimensionReadout,
        dimension_stream.str( ));
}




void  ImageInspector::ImageStatsPaneController::uClearReadouts( )
{
    const std::string  kClearMessage = "n/a (no image)";

    Win32Tools::setWindowTextString(fHostClassReadout, kClearMessage);
    Win32Tools::setWindowTextString(fHostAddressReadout, kClearMessage);
    Win32Tools::setWindowTextString(fImageKindReadout, kClearMessage);
    Win32Tools::setWindowTextString(fMapSourceReadout, kClearMessage);
    Win32Tools::setWindowTextString(fSourceFormatReadout, kClearMessage);
    Win32Tools::setWindowTextString(fDimensionReadout, kClearMessage);
}




void  ImageInspector::ImageStatsPaneController::setTargetImage(
    const RGBAImage& img)
{
    fTargetImage = & img;

    if (isInstalled( ))
        uSynchronizeReadouts(img);
}




void  ImageInspector::ImageStatsPaneController::viewEnabled(
    UIComponentView& sender)
{
}




void  ImageInspector::ImageStatsPaneController::viewDisabled(
    UIComponentView& sender)
{
}
//////////////////////////////////////////////////////////////////////////////
// END  ImageStatsPaneController                                            //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ImageInspector :: DialogController                                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class ImageInspector::DialogController
    : public virtual ModalDialogController {

private:

    static  bool                  sIsViewPaneClassRegistered;
    static  bool                  sAreCursorsLoaded;
    static  HCURSOR               sGripNormalCursor;
    static  HCURSOR               sGripEngagedCursor;
    static  const char*           kViewPaneClassName;
    static  const unsigned short  kResizeGripChildID =
        ModalDialogView::kChildID0;
    static  const unsigned short  kViewPaneChildID =
        ModalDialogView::kChildID0 + 1;
    static  const unsigned short  kStatsPaneChildID =
        ModalDialogView::kChildID0 + 2;
    static  const unsigned short  kBottomZoneHeight = 82;
    static  const unsigned short  kViewPaneGutterSize = 32;

    static  LRESULT  WINAPI  sViewPaneProc(HWND targetWindow,
        UINT messageCode, WPARAM wParam, LPARAM lParam);
    static  HWND  sMakeViewPane(ImageInspector::DialogController* parentobj);
    static  void  sLoadCursors( );
    static  ImageInspector::DialogController*  sGetObject(HWND w);


    ModalDialogView*           fManagedView;
    WorkPaneView*              fStatsPaneView;
    ImageStatsPaneController*  fStatsPaneCtlr;
    HWND                       fResizeGripPrimitive;
    HWND                       fDoneButton;
    HWND                       fViewingPane;
    HWND                       fDescriptiveText;
    unsigned short             fRightMargin;
    unsigned short             fLeftMargin;
    unsigned short             fTopMargin;
    unsigned short             fPadding;
    bool                       fIsTrackingInProgress;
    Vector2Di                  fPrevTrackPoint;
    const RGBAImage*           fTargetImage;
    HBITMAP                    fDrawableDDB;
    HBITMAP                    fCompositionDDB;
    bool                       fIsVertPanEnabled;
    bool                       fIsHorizPanEnabled;
    Vector2Di                  fImageDrawOrigin;
    RECT                       fIDOMotionBounds;

    LRESULT  uProcessViewPaneMessage(HWND targetWindow, UINT messageCode,
        WPARAM wParam, LPARAM lParam);
    void  uDoInitialImageLayout( );
    RECT  uViewPaneToImage(const RECT& vprect);
    void  uComputeIDOMotionBounds( );

public:

    DialogController( );
    DialogController(const RGBAImage&);
    virtual ~DialogController( );

    void  installed(ModalDialogView& sender);
    void  deinstalled(ModalDialogView& sender);
    bool  isInstalled( ) const;
    void  layout(ModalDialogView& sender);
    void  image(ModalDialogView& sender);
    void  commandPropagated(ModalDialogView& sender,
        HWND commandSource, unsigned short commandID);
    void  interactionStarted(ModalDialogView& sender);
    void  interactionFinished(ModalDialogView& sender);
    void  dialogClosed(ModalDialogView& sender);
    void  setTargetImage(const RGBAImage& img);
    void  viewEnabled(UIComponentView& sender);
    void  viewDisabled(UIComponentView& sender);

};

bool  ImageInspector::DialogController::sIsViewPaneClassRegistered = false;
const char*  ImageInspector::DialogController::kViewPaneClassName =
    "ImageInspector_DialogController_ViewPane";
HCURSOR  ImageInspector::DialogController::sGripNormalCursor = 0;
HCURSOR  ImageInspector::DialogController::sGripEngagedCursor = 0;
bool     ImageInspector::DialogController::sAreCursorsLoaded = false;


ImageInspector::DialogController::DialogController( )
    : fManagedView(0), fResizeGripPrimitive(0), fDoneButton(0),
      fRightMargin(0), fViewingPane(0), fLeftMargin(0),
      fPadding(0), fTopMargin(0), fStatsPaneView(0),
      fStatsPaneCtlr(0), fIsTrackingInProgress(false),
      fTargetImage(0), fDrawableDDB(0), fIsVertPanEnabled(false),
      fIsHorizPanEnabled(false), fCompositionDDB(0)
{
}



ImageInspector::DialogController::DialogController(const RGBAImage& img)
    : fManagedView(0), fResizeGripPrimitive(0), fDoneButton(0),
      fRightMargin(0), fViewingPane(0), fLeftMargin(0),
      fPadding(0), fTopMargin(0), fStatsPaneView(0),
      fStatsPaneCtlr(0), fIsTrackingInProgress(false),
      fTargetImage(0), fDrawableDDB(0), fIsVertPanEnabled(false),
      fIsHorizPanEnabled(false), fCompositionDDB(0)
{
    setTargetImage(img);
}





ImageInspector::DialogController::~DialogController( )
{
    if (isInstalled( ))
        fManagedView->installController(
            * ModalDialogController::nullController( ));

    if (fDrawableDDB)
        DeleteObject(fDrawableDDB);

    if (fCompositionDDB)
        DeleteObject(fCompositionDDB);
}




void  ImageInspector::DialogController::installed(ModalDialogView& sender)
{
    sender.setMinHorizTrackSize(564);
    sender.setMinVerticalTrackSize(384);

    fManagedView = & sender;
}




void  ImageInspector::DialogController::deinstalled(ModalDialogView& sender)
{
    fManagedView = 0;
}




bool  ImageInspector::DialogController::isInstalled( ) const
{
    if (fManagedView)
        return true;
    else
        return false;
}




void  ImageInspector::DialogController::layout(ModalDialogView& sender)
{
    if (sender.isInteractionRunning( )) {

        const int  dlog_wd = sender.clientWidth( );
        const int  dlog_ht = sender.clientHeight( );
        const int  donebut_wd = Win32Tools::getWindowWidth(fDoneButton);
        const int  donebut_y = Win32Tools::getWindowOrigin(fDoneButton).y;
        const int  viewpane_ht = dlog_ht - fTopMargin - kBottomZoneHeight;
        const int  viewpane_wd = dlog_wd - fRightMargin - fLeftMargin;
        const int  image_wd = fTargetImage->width( );
        const int  image_ht = fTargetImage->height( );

        Win32Tools::setWindowOrigin(fResizeGripPrimitive, dlog_wd - 16,
            dlog_ht - 16);

        Win32Tools::setWindowOrigin(fDoneButton, dlog_wd - fRightMargin -
            donebut_wd, donebut_y);

        Win32Tools::setWindowWidth(fViewingPane, dlog_wd - fRightMargin -
            fLeftMargin);
        Win32Tools::setWindowHeight(fViewingPane, viewpane_ht);

        fStatsPaneView->setPosition(fLeftMargin, fTopMargin + viewpane_ht);
        fStatsPaneView->setWidth(dlog_wd - fRightMargin - fLeftMargin);

        uComputeIDOMotionBounds( );

        HDC displayDC = CreateDC(TEXT("DISPLAY"), NULL, NULL,NULL);
        if (!displayDC)
            throw std::runtime_error("ImageInspector: couldn't create a "
                "device context for the display");
        if (fCompositionDDB)
            DeleteObject(fCompositionDDB);
        fCompositionDDB = CreateCompatibleBitmap(displayDC, viewpane_wd,
            viewpane_ht);
        if (!fCompositionDDB)
            throw std::runtime_error("ImageInspector: couldn't create a "
                "Win32 DDB for offscreen view composition");
        DeleteDC(displayDC);

        RECT vp_drawrect;
        GetClientRect(fViewingPane, &vp_drawrect);

        if (fIsHorizPanEnabled) {

            const int img_re = fImageDrawOrigin.x + image_wd;

            if (img_re < (vp_drawrect.right - kViewPaneGutterSize)) {

                const int  img_re_new = vp_drawrect.right -
                    kViewPaneGutterSize;

                if (kViewPaneGutterSize <= (img_re_new - image_wd)) {

                    fImageDrawOrigin.x = (vp_drawrect.right - image_wd) / 2;
                    fIsHorizPanEnabled = false;
                }
                else {

                    fImageDrawOrigin.x = img_re_new - image_wd;
                }
            } /* if whitespace increased on right */
        } /* if horizontal panning is enabled */
        else {

            const int img_le = fImageDrawOrigin.x;

            if (img_le < kViewPaneGutterSize) {

                fIsHorizPanEnabled = true;
            }
            else {

                fImageDrawOrigin.x = (vp_drawrect.right - image_wd) / 2;
            }
        } /* if horizontal panning is not enabled */

        if (fIsVertPanEnabled) {

            const int img_be = fImageDrawOrigin.y + image_ht;

            if (img_be < (vp_drawrect.bottom - kViewPaneGutterSize)) {

                const int img_be_new = vp_drawrect.bottom -
                    kViewPaneGutterSize;

                if (kViewPaneGutterSize <= (img_be_new - image_ht)) {

                    fImageDrawOrigin.y = (vp_drawrect.bottom - image_ht) / 2;
                    fIsVertPanEnabled = false;
                }
                else {

                    fImageDrawOrigin.y = img_be_new - image_ht;
                }
            } /* if whitespace increased on bottom */
        } /* if vertical panning is enabled */
        else {

            const int img_te = fImageDrawOrigin.y;

            if (img_te < kViewPaneGutterSize) {

                fIsVertPanEnabled = true;
            }
            else {

                fImageDrawOrigin.y = (vp_drawrect.bottom - image_ht) / 2;
            }
        } /* if vertical panning is not enabled */
    } /* if dialog interaction is running */
}




void  ImageInspector::DialogController::image(ModalDialogView& sender)
{
}




void  ImageInspector::DialogController::commandPropagated(
    ModalDialogView& sender, HWND commandSource, unsigned short commandID)
{
    if (commandSource == fDoneButton)
        sender.finishInteraction( );
}




void  ImageInspector::DialogController::interactionStarted(
    ModalDialogView& sender)
{
    fResizeGripPrimitive = CreateWindow(TEXT("scrollbar"), TEXT(""),
        WS_CHILD | SBS_SIZEGRIP | WS_VISIBLE | SBS_SIZEBOXBOTTOMRIGHTALIGN,
        1, 1, 11, 11, sender.peerHandle( ), (HMENU)(kResizeGripChildID),
        NULL, 0);
    if (!fResizeGripPrimitive)
        throw std::runtime_error("ImageInspector: dialog controller couldn't"
            " create resize grip control.");

    fDoneButton = sender.itemPeer(kDoneButtonChildID);
    Win32Tools::applyControlFaceProperties(fDoneButton);
    fDescriptiveText = sender.itemPeer(kUnifDescTextChildID);
    Win32Tools::applyControlFaceProperties(fDescriptiveText);

    const int  dlog_wd = sender.clientWidth( );
    const int  donebtn_wd = Win32Tools::getWindowWidth(fDoneButton);
    const Vector2Di  donebtn_pos = Win32Tools::getWindowOrigin(fDoneButton);

    const HWND  desctext = sender.itemPeer(kUnifDescTextChildID);

    fRightMargin = dlog_wd - donebtn_pos.x - donebtn_wd;
    fLeftMargin = Win32Tools::getWindowOrigin(desctext).x;
    fPadding = Win32Tools::getWindowOrigin(desctext).y;
    fTopMargin = Win32Tools::getWindowHeight(desctext) + (2 * fPadding);

    fViewingPane = sMakeViewPane(this);
    Win32Tools::setWindowOrigin(fViewingPane, fLeftMargin, fTopMargin);

    fStatsPaneView = new WorkPaneView(sender.peerHandle( ),
        kStatsPaneChildID);
    fStatsPaneCtlr = new ImageInspector::ImageStatsPaneController( );
    delete & fStatsPaneView->installController(*fStatsPaneCtlr);
    if (fTargetImage)
        fStatsPaneCtlr->setTargetImage(*fTargetImage);

    layout(sender);
    uDoInitialImageLayout( );
}




void  ImageInspector::DialogController::interactionFinished(
    ModalDialogView& sender)
{
    Win32Tools::destroyPrimitive(fResizeGripPrimitive);
    Win32Tools::destroyPrimitive(fViewingPane);

    delete fStatsPaneView;
    fStatsPaneView = 0;

    delete fStatsPaneCtlr;
    fStatsPaneCtlr = 0;

    fDoneButton = 0;
}




void  ImageInspector::DialogController::dialogClosed(ModalDialogView& sender)
{
    sender.finishInteraction( );
}




LRESULT  WINAPI  ImageInspector::DialogController::sViewPaneProc(
    HWND targetwin, UINT msgcode, WPARAM wparam, LPARAM lparam)
{
        switch (msgcode) {

        case  WM_LBUTTONDOWN:
        case  WM_LBUTTONUP:
        case  WM_SETCURSOR:
        case  WM_PAINT:
        case  WM_MOUSEMOVE:
            return sGetObject(targetwin)->uProcessViewPaneMessage(
                targetwin, msgcode, wparam, lparam);
        break;
        
        default:
            return DefWindowProc(targetwin, msgcode, wparam, lparam);
        break;
    }
}




void  ImageInspector::DialogController::viewEnabled(UIComponentView& sender)
{
}




void  ImageInspector::DialogController::viewDisabled(UIComponentView& sender)
{
}




HWND  ImageInspector::DialogController::sMakeViewPane(
    ImageInspector::DialogController* parentobj)
{
    if (! sAreCursorsLoaded)
        sLoadCursors( );

    if (!sIsViewPaneClassRegistered) {
    
        WNDCLASS    paneclass;

        paneclass.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
        paneclass.lpfnWndProc =
            ImageInspector::DialogController::sViewPaneProc;
        paneclass.cbClsExtra = 0;
        paneclass.cbWndExtra = sizeof(ImageInspector::DialogController*);
        paneclass.hInstance = NULL;
        paneclass.hIcon = NULL;
        paneclass.hCursor = sGripNormalCursor;
        paneclass.hbrBackground = NULL;
        paneclass.lpszMenuName = NULL;
        paneclass.lpszClassName = kViewPaneClassName;

        ATOM  regresult = RegisterClass(&paneclass);
        if (!regresult)
            throw std::runtime_error("ImageInspector::DialogController: "
                " couldn't register view pane window class");

        sIsViewPaneClassRegistered = true;
    }

    HWND  result =
        CreateWindowEx(WS_EX_CLIENTEDGE, kViewPaneClassName, TEXT(""),
        WS_CHILD | WS_VISIBLE, 0, 0, 10, 10,
        parentobj->fManagedView->peerHandle( ), (HMENU)(kViewPaneChildID),
        NULL, 0);

    if (!result)
        throw std::runtime_error("ImageInspector::DialogController: failed "
            "to create view pane window");

    SetWindowLongPtr(result, GWLP_USERDATA, LONG_PTR(parentobj));

    return result;
}



void  ImageInspector::DialogController::sLoadCursors( )
{
    sGripNormalCursor = LoadCursor(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(kGripNormalCursorResID));
    if (sGripNormalCursor == NULL)
        throw std::runtime_error("ImageInspector::DialogController: couldn't "
            "load normal grip cursor from application module resources.");

    sGripEngagedCursor = LoadCursor(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(kGripEngagedCursorResID));
    if (sGripEngagedCursor == NULL)
        throw std::runtime_error("ImageInspector::DialogController: couldn't "
            "load normal grip cursor from application module resources.");

    sAreCursorsLoaded = true;
}




ImageInspector::DialogController*
    ImageInspector::DialogController::sGetObject(HWND w)
{
    LONG_PTR  rawResult = GetWindowLongPtr(w, GWLP_USERDATA);

    return  (ImageInspector::DialogController*)(rawResult);
}



LRESULT  ImageInspector::DialogController::uProcessViewPaneMessage(
    HWND targetwin, UINT msgcode, WPARAM wparam, LPARAM lparam)
{
    static  const int  kErrorOccurred = 0;

    if (msgcode == WM_LBUTTONDOWN) {

        fIsTrackingInProgress = true;
        fPrevTrackPoint.x = LOWORD(lparam);
        fPrevTrackPoint.y = HIWORD(lparam);
        SetCapture(targetwin);
        SetCursor(sGripEngagedCursor);

        return 0;
    }
    else if (msgcode == WM_LBUTTONUP) {

        fIsTrackingInProgress = false;
        ReleaseCapture( );

        return 0;
    }
    else if (msgcode == WM_PAINT) {

        PAINTSTRUCT  paintinfo;
        HDC          paintdc;

        paintdc = BeginPaint(targetwin, &paintinfo);

        if (fDrawableDDB) {

            int resultCode;
        
            RECT  viewrect;
            GetClientRect(targetwin, &viewrect);

            RECT imagerect;
            imagerect.left = fImageDrawOrigin.x;
            imagerect.top = fImageDrawOrigin.y;
            imagerect.right = fImageDrawOrigin.x + fTargetImage->width( );
            imagerect.bottom = fImageDrawOrigin.y + fTargetImage->height( );

            RECT drawrect;
            IntersectRect(&drawrect, &viewrect, &imagerect);

            RECT drawrect_relimg = uViewPaneToImage(drawrect);

            HDC  imagebmap_dc = CreateCompatibleDC(paintdc);
            SelectObject(imagebmap_dc, fDrawableDDB);

            HDC  composerbmap_dc = CreateCompatibleDC(paintdc);
            SelectObject(composerbmap_dc, fCompositionDDB);

            resultCode = FillRect(composerbmap_dc, &viewrect,
                GetSysColorBrush(COLOR_APPWORKSPACE));
            if (resultCode == kErrorOccurred)
                throw std::runtime_error("ImageInspector: Win32 GDI error "
                    "occured when attempting to fill a rectangle in the "
                    "offscreen compositon buffer");

            RECT  framerect;
            CopyRect(&framerect, &imagerect);
            InflateRect(&framerect, 1, 1);
            resultCode = FrameRect(composerbmap_dc, &framerect,
                (HBRUSH)GetStockObject(BLACK_BRUSH));
            if (resultCode == kErrorOccurred)
                throw std::runtime_error("ImageInspector: Win32 GDI error "
                    "occured when attempting to draw a frame around the "
                    "target image in the offscreen compositon buffer");

            resultCode = BitBlt(composerbmap_dc, drawrect.left, drawrect.top,
                drawrect.right - drawrect.left, drawrect.bottom -
                drawrect.top, imagebmap_dc, drawrect_relimg.left,
                drawrect_relimg.top, SRCCOPY);
            if (resultCode == kErrorOccurred)
                throw std::runtime_error("ImageInspector: Win32 GDI error "
                    "occured when attempting to transfer image data pixels "
                    "into the offscreen composition buffer");

            resultCode = BitBlt(paintdc, 0, 0, viewrect.right - viewrect.left
                - 1, viewrect.bottom - viewrect.top - 1, composerbmap_dc, 0,
                0, SRCCOPY);
            if (resultCode == kErrorOccurred)
                throw std::runtime_error("ImageInspector: Win32 GDI error "
                    "occured when attempting to transfer pixels from the "
                    "offscreen composition buffer into the viewing pane");

            DeleteDC(imagebmap_dc);
            DeleteDC(composerbmap_dc);

        } /* if a drawable DDB exists */

        EndPaint(targetwin, &paintinfo);
    }
    else if ((msgcode == WM_MOUSEMOVE) && fIsTrackingInProgress) {

        const short curx = static_cast<short>(LOWORD(lparam));
        const short cury = static_cast<short>(HIWORD(lparam));

        if (fIsHorizPanEnabled) {

            const int  deltax = curx - fPrevTrackPoint.x;

            int newdrawx = fImageDrawOrigin.x + deltax;

            if (newdrawx < fIDOMotionBounds.left)
                newdrawx = fIDOMotionBounds.left;
            if (newdrawx > fIDOMotionBounds.right)
                newdrawx = fIDOMotionBounds.right;

            fImageDrawOrigin.x = newdrawx;
        }

        if (fIsVertPanEnabled) {

            const int  deltay = cury - fPrevTrackPoint.y;

            int newdrawy = fImageDrawOrigin.y + deltay;

            if (newdrawy < fIDOMotionBounds.top)
                newdrawy = fIDOMotionBounds.top;
            if (newdrawy > fIDOMotionBounds.bottom)
                newdrawy = fIDOMotionBounds.bottom;

            fImageDrawOrigin.y = newdrawy;
        }

        if (fIsHorizPanEnabled || fIsVertPanEnabled) {

            RECT  vprect;
            GetClientRect(fViewingPane, &vprect);

            InvalidateRect(fViewingPane, &vprect, FALSE);
        }

        fPrevTrackPoint = Vector2Di(curx, cury);

        return 0;
    }
    else {

        return DefWindowProc(targetwin, msgcode, wparam, lparam);
    }
}



void  ImageInspector::DialogController::setTargetImage(const RGBAImage& img)
{
    fTargetImage = & img;

    HBITMAP  oldddb;
    oldddb = fDrawableDDB;

    PBITMAPINFO  imgdib = Win32Tools::toNative(img);
    unsigned char*  dib_base_addr = reinterpret_cast<unsigned char*>(imgdib);
    unsigned char*  dib_pix_buffer = dib_base_addr + sizeof(BITMAPINFO);
    HDC displayDC = CreateDC(TEXT("DISPLAY"), NULL, NULL,NULL);
    if (!displayDC)
        throw std::runtime_error("ImageInspector: couldn't create a device "
            "context for the display");

    fDrawableDDB = CreateDIBitmap(displayDC, &(imgdib->bmiHeader), CBM_INIT,
        dib_pix_buffer, imgdib, DIB_RGB_COLORS);
    if (!fDrawableDDB)
        throw std::runtime_error("ImageInspector: couldn't create a "
            "screen-compatible Win32 DDB from target image pixel data");

    DeleteDC(displayDC);
    delete [ ] imgdib;

    if (!oldddb)
        DeleteObject(oldddb);

    if (isInstalled( ) && fManagedView->isInteractionRunning( )) {

        fStatsPaneCtlr->setTargetImage(img);

        uDoInitialImageLayout( );

        RECT  vp_rect;
        GetClientRect(fManagedView->peerHandle( ), &vp_rect);
        InvalidateRect(fManagedView->peerHandle( ), &vp_rect, FALSE);
    }
}




void  ImageInspector::DialogController::uDoInitialImageLayout( )
{
    RECT  vp_rect;
    GetClientRect(fViewingPane, &vp_rect);
    const int  vp_wd = vp_rect.right - vp_rect.left;
    const int  vp_ht = vp_rect.bottom - vp_rect.top;
    const int  img_wd = fTargetImage->width( );
    const int  img_ht = fTargetImage->height( );

    if (img_wd <= (vp_wd - (2 * kViewPaneGutterSize))) {

        /* in this case, we set up an image that is initially
           centered horizontally, and disable horizontal panning */

        fImageDrawOrigin.x = (vp_wd - img_wd) / 2;
        fIsHorizPanEnabled = false;
    }
    else {

        /* in this case, we set up an image that is initially
           left-aligned horizontally, and we enable horizontal
           panning */

        fImageDrawOrigin.x = kViewPaneGutterSize;
        fIsHorizPanEnabled = true;
    }

    if (img_ht <= (vp_ht - (2 * kViewPaneGutterSize))) {

        /* in this case, we set up an image that is initially
           centered vertically, and disable vertical panning */

        fImageDrawOrigin.y = (vp_ht - img_ht) / 2;
        fIsVertPanEnabled = false;
    }
    else {

        /* in this case, we set up an image that is initially
           top-aligned vertically, and we enable vertical
           panning */

        fImageDrawOrigin.y = kViewPaneGutterSize;
        fIsVertPanEnabled = true;
    }
}




RECT  ImageInspector::DialogController::uViewPaneToImage(const RECT& vprect)
{
    RECT  imgrect;

    imgrect.left = vprect.left - fImageDrawOrigin.x;
    imgrect.top = vprect.top - fImageDrawOrigin.y;
    imgrect.right = vprect.right - fImageDrawOrigin.x;
    imgrect.bottom = vprect.bottom - fImageDrawOrigin.y;

    return imgrect;
}




void  ImageInspector::DialogController::uComputeIDOMotionBounds( )
{
    RECT  vprect;
    GetClientRect(fViewingPane, &vprect);

    const int vp_wd = vprect.right - vprect.left;
    const int vp_ht = vprect.bottom - vprect.top;
    const int img_wd = fTargetImage->width( );
    const int img_ht = fTargetImage->height( );

    fIDOMotionBounds.left = vp_wd - kViewPaneGutterSize - img_wd;
    fIDOMotionBounds.top = vp_ht - kViewPaneGutterSize - img_ht;
    fIDOMotionBounds.right = kViewPaneGutterSize;
    fIDOMotionBounds.bottom = kViewPaneGutterSize;
}
//////////////////////////////////////////////////////////////////////////////
// END  DialogController                                                    //
//////////////////////////////////////////////////////////////////////////////










//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ImageInspector                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const char*  ImageInspector::sFormatNames[ ] = { "Truevision TARGA" };


ImageInspector::ImageInspector(HWND overlayPeer)
    : fInspectorDlgCtlr(0), fInspectorDlgView(0), fTargetImage(0)
{
    fInspectorDlgView = new ModalDialogView(overlayPeer,
        kImageInspectorDialogResID);
    fInspectorDlgCtlr = new DialogController( );

    delete & fInspectorDlgView->installController(*fInspectorDlgCtlr);
}




ImageInspector::ImageInspector(HWND overlayPeer, const RGBAImage& target)
    : fInspectorDlgCtlr(0), fInspectorDlgView(0), fTargetImage(&target)
{
    fInspectorDlgView = new ModalDialogView(overlayPeer,
        kImageInspectorDialogResID);
    fInspectorDlgCtlr = new DialogController( );

    delete & fInspectorDlgView->installController(*fInspectorDlgCtlr);

    fInspectorDlgCtlr->setTargetImage(target);
}




ImageInspector::~ImageInspector( )
{
    delete fInspectorDlgView;
    delete fInspectorDlgCtlr;
}



void  ImageInspector::runInspectionInteraction( )
{
    if (! fTargetImage)
        throw std::logic_error("ImageInspector: tried to run inspection "
            "interaction, but no target image has been set");

    fInspectorDlgView->startInteraction( );
}




void  ImageInspector::setTargetImage(const RGBAImage& img)
{
    fTargetImage = & img;

    fInspectorDlgCtlr->setTargetImage(img);
}
//////////////////////////////////////////////////////////////////////////////
// END  ImageInspector                                                      //
//////////////////////////////////////////////////////////////////////////////
