//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ExoticThingies.h                                                         //
//                                                                          //
// Packages interface definition for class SliderThingy only                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  <unversioned module>                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  Copyright (c) 2008 Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef EXOTIC_THINGIES__HXX
#define EXOTIC_THINGIES__HXX

#include <UIThingies.h>

//-------------------------------------------------------------------------//
/* A SliderController is a kind of ThingyController that defines an
   interface that must be implemented by objects that wish to control
   on-screen slider (sometimes called "trackbar") widgets */
class SliderController :
    public virtual ThingyController {

protected:

    SliderController( ) { }

public:

    static  SliderController*  nullController( );

    virtual ~SliderController( ) { }

    virtual  void  thingyValueChanged(SliderThingy& sender) = 0;
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
/* A SliderThingy is a kind of Thingy that implements an on-screen slider
   (sometimes called a "trackbar") control. SliderThingies are primitive
   thingies (they can't serve as parents for subthingies) and are not free
   thingies (they must know who their parent is when they are created). */
class SliderThingy :
    public virtual Thingy,
    public virtual ThingyAdapterClient {

private:

    SafeThingyAdapter      fAdapter;
    SliderController*      fController;

    /* SliderThingies don't have value semantics */
    SliderThingy(const SliderThingy&) { }
    const SliderThingy& operator=(const SliderThingy&)
        { return *this; }

public:

    SliderThingy(CompositeThingy& composite);

    LRESULT  processParentMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam);
    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    virtual  SliderController&  controller( );
    virtual  const SliderController&  controller( ) const;
    virtual  SliderController&  installController(
        SliderController&);
    virtual  void  setRangeMin(int rmin);
    virtual  void  setRangeMax(int rmax);
    virtual  void  setRange(int rmin, int rmax);
    virtual  int   rangeMin( ) const;
    virtual  int   rangeMax( ) const;
    virtual  int   value( ) const;
    virtual  void  setValue(int val);
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
/* A HorizontalSeparatorThingy is a kind of Thingy that implements an
   on-screen horizontal separator bar control. HorizontalSeparatorThingies
   are primitive thingies (they can't serve as parents for subthingies)
   and are not free thingies (they must know who their parent is when they
   are created). */
class HorizontalSeparatorThingy :
    public virtual Thingy,
    public virtual ThingyAdapterClient {

private:

    SafeThingyAdapter      fAdapter;

    /* SliderThingies don't have value semantics */
    HorizontalSeparatorThingy(const HorizontalSeparatorThingy&) { }
    const HorizontalSeparatorThingy& operator=(
        const HorizontalSeparatorThingy&)
            { return *this; }

public:

    HorizontalSeparatorThingy(CompositeThingy& composite);

    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
};
//-------------------------------------------------------------------------//





//-------------------------------------------------------------------------//
/* A ProgressBarThingy is a kind of Thingy that implements an on-screen
   progress bar control. ProgressBarThingies are primitive thingies (they
   can't serve as parents for subthingies) and are not free thingies (they
   must know who their parent is when they are created). */
class ProgressBarThingy :
    public virtual Thingy,
    public virtual ThingyAdapterClient {

private:

    SafeThingyAdapter      fAdapter;

    /* ProgressBarThingies don't have value semantics */
    ProgressBarThingy(const ProgressBarThingy&) { }
    const ProgressBarThingy& operator=(const ProgressBarThingy&)
        { return *this; }

public:

    ProgressBarThingy(CompositeThingy& composite);

    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    virtual  int   value( ) const;
    virtual  void  setValue(int val);
};
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
/* A GLCanvasThingy is a kind of Thingy that implements an on-screen
   window in which clients can perform arbitrary OpenGL rendering
   operations. GLCanvasThingies are primitive thingies (they can't serve
   as parents for subthingies) and are not free thingies (they must know
   who their parent is when they are created). */
class GLCanvasThingy :
    public virtual Thingy,
    public virtual ThingyAdapterClient {

private:

    static  const std::string  kPrimitiveClassName;
    static  bool               sIsClassRegistered;

    SafeThingyAdapter            fAdapter;
    GLCanvasController*          fController;
    HGLRC                        fRenderingContext;
    HDC                          fRenderingContextDC;
    std::auto_ptr<GPUInterface>  fDeviceInterface;
    bool                         fIsTrackInProgress;

    /* GLCanvasThingies don't have value semantics */
    GLCanvasThingy(const GLCanvasThingy&) { }
    const GLCanvasThingy& operator=(const GLCanvasThingy&)
        { return *this; }

    void  uInitGLContext( );

public:

    GLCanvasThingy(CompositeThingy& composite);

    LRESULT  processMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam);
    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    virtual  GLCanvasController&  controller( );
    virtual  const GLCanvasController&  controller( ) const;
    virtual  GLCanvasController&  installController(
        GLCanvasController&);
    virtual  void  makeRenderingTarget( );
    virtual  void  reimage( );
    virtual  GPUInterface&  gpuInterface( );
};
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
/* A GLCanvasController is a kind of ThingyController that defines an
   interface that must be implemented by objects that wish to control
   on-screen OpenGL rendering canvases */
class GLCanvasController :
    public virtual ThingyController {

protected:

    GLCanvasController( ) { }

public:

    static  GLCanvasController*  nullController( );

    virtual ~GLCanvasController( ) { }

    virtual  void  imageThingy(GLCanvasThingy&) = 0;
    virtual  void  thingySized(GLCanvasThingy&) = 0;
    virtual  void  initializeThingy(GLCanvasThingy&) = 0;
    virtual  void  thingyMousePressed(GLCanvasThingy&,
        const MouseInteractionToken&) = 0;
    virtual  void  thingyMouseReleased(GLCanvasThingy&,
        const MouseInteractionToken&) = 0;
    virtual  void  thingyMouseTracked(GLCanvasThingy&,
        const MouseInteractionToken&) = 0;
};
//-------------------------------------------------------------------------//
#endif
