//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// EssentialThingies.h                                                      //
//                                                                          //
// Packages interface definitions for classes ApplicationPaneThingy,        //
// ApplicationPaneController, PushButtonThingy, and PushButtonController    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  <unversioned module>                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  Copyright (c) 2008 Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef ESSENTIAL_THINGIES__HXX
#define ESSENTIAL_THINGIES__HXX

#include <UIThingies.h>
#include <map>

//---------------------------------------------------------------------------//
/* An ApplicationPaneThingy is a kind of Thingy that serves as a top-level
   main application window for single-window, "panel" type applications.
   ApplicationPaneThingies can't be resized or maximized by the user, but
   they can be minimized and closed. ApplicationPaneThingies are composite
   thingies that can manage any subthingies created inside them */
class  ApplicationPaneThingy :
    public virtual CompositeThingy,
    public virtual CompositeThingyAdapterClient {

private:

    static  const std::string  kWindowClassName;
    static  bool               sIsClassRegistered;

    SafeThingyAdapter           fAdapter;
    ApplicationPaneController*  fController;

    /* ApplicationPaneThingies don't have value semantics */
    ApplicationPaneThingy(const ApplicationPaneThingy&) { }
    const ApplicationPaneThingy& operator=(const ApplicationPaneThingy&)
        { return *this; }

public:

    ApplicationPaneThingy( );

    LRESULT  processMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam);
    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    ThingyAdapter&  composerAdapter( );
    const ThingyAdapter&  composerAdapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    unsigned short  composableWidth( ) const;
    unsigned short  composableHeight( ) const;
    virtual  ApplicationPaneController&  controller( );
    virtual  const ApplicationPaneController&  controller( ) const;
    virtual  ApplicationPaneController&  installController(
        ApplicationPaneController&);
    virtual  std::string  title( ) const;
    virtual  void  setTitle(const std::string&);
};
//---------------------------------------------------------------------------//




//---------------------------------------------------------------------------//
/* An ApplicationPaneController is a kind of ThingyController that
   defines an interface for all objects that wish to control
   ApplicationPaneThingies. Since ApplicationPaneThingies are composite
   thingies, their controllers must manage the layout and appearance of
   any subthingies created inside */
class  ApplicationPaneController :
    public virtual CompositeThingyController {

protected:

    ApplicationPaneController( ) { }

public:

    static  ApplicationPaneController*  nullController( );

    virtual  ~ApplicationPaneController( ) { }

    virtual  void  thingyClosed(ApplicationPaneThingy& sender) = 0;
};
//---------------------------------------------------------------------------//



//---------------------------------------------------------------------------//
/* An ApplicationWindowThingy is a kind of Thingy that serves as a top-level
   main application window for full-fledged applications. It supports a
   menu bar, a status bar, a toolbar and horizontal and vertical scrollbars.
   ApplicationWindowThingies are resizable, minimizable and maximizable by
   the user. ApplicationWindowThingies are composite thingies that can manage
   any subthingies created inside them */
class  ApplicationWindowThingy :
    public virtual CompositeThingy,
    public virtual CompositeThingyAdapterClient {

private:

    static  const std::string  kFrameWindowClassName;
    static  const std::string  kContentPaneWindowClassName;
    static  const std::string  kClientPaneWindowClassName;
    static  bool               sAreClassesRegistered;
    static  long               sNextTBToken;

    ApplicationWindowController*  fController;
    SafeThingyAdapter             fHorizScrollAdapter;
    SafeThingyAdapter             fVertScrollAdapter;
    SafeThingyAdapter             fClientPaneAdapter;;
    SafeThingyAdapter             fContentPaneAdapter;
    SafeThingyAdapter             fToolbarAdapter;
    SafeThingyAdapter             fStatusBarAdapter;
    SafeThingyAdapter             fFrameAdapter;
    bool                          fIsToolbarEnabled;
    bool                          fIsHorizScrollEnabled;
    bool                          fIsVerticalScrollEnabled;
    bool                          fIsStatBarEnabled;
    bool                          fIsAssemblyComplete;
    std::map<long, long>          fToolbarButtKindDict;
    int                           fHorizScrollMinorIncr;
    int                           fHorizScrollMajorIncr;
    int                           fVertScrollMinorIncr;
    int                           fVertScrollMajorIncr;
    Vector2Di                     fMinTrackSize;
    HACCEL                        fInstalledKeys;
    bool                          fIsWaiting;

    /* ApplicationWindowThingies don't have value semantics */
    ApplicationWindowThingy(const ApplicationWindowThingy&) { }
    const ApplicationWindowThingy& operator=(const ApplicationWindowThingy&)
        { return *this; }

    void  uLayoutAssembly( );
    bool  uIsToolbarButton(long);
    bool  uIsButtonSelected(long);
    void  uProcessToolbarButtonClick(long);
    void  uEnableHorizontalScrollbar( );
    void  uEnableVerticalScrollbar( );
    void  uDisableHorizontalScrollbar( );
    void  uDisableVerticalScrollbar( );
    bool  uIsHorizontalScrollbarEnabled( ) const;
    bool  uIsVerticalScrollbarEnabled( ) const;
    void  uInterpretScrollAction(const ScrollbarSelector&, int);
    int   uGetScrollTrackValue(const ScrollbarSelector&) const;

public:

    static  const unsigned short  kToolbarIconDimension = 32;

    ApplicationWindowThingy( );

    LRESULT  processMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam);
    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    ThingyAdapter&  composerAdapter( );
    const ThingyAdapter&  composerAdapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    unsigned short  composableWidth( ) const;
    unsigned short  composableHeight( ) const;
    virtual  ApplicationWindowController&  controller( );
    virtual  const ApplicationWindowController&  controller( ) const;
    virtual  ApplicationWindowController&  installController(
        ApplicationWindowController&);
    virtual  std::string  title( ) const;
    virtual  void  setTitle(const std::string&);
    virtual  void  installMenu(int);
    virtual  void  installKeys(int);
    virtual  HACCEL  installedKeys( ) const;
    virtual  bool  areKeysInstalled( ) const;
    virtual  void  enableMenuItem(int);
    virtual  void  disableMenuItem(int);
    virtual  void  enableToolbar( );
    virtual  void  disableToolbar( );
    virtual  bool  isToolbarEnabled( ) const;
    virtual  ToolbarButtonToken  appendToolbarButton(ToolbarButtonKind,
        HBITMAP, const std::string&);
    virtual  void  appendToolbarSeparator(int);
    virtual  void  selectToolbarButton(ToolbarButtonToken);
    virtual  void  deselectToolbarButton(ToolbarButtonToken);
    virtual  bool  isToolbarButtonSelected(ToolbarButtonToken) const;
    virtual  void  enableScrollbar(const ScrollbarSelector&);
    virtual  void  disableScrollbar(const ScrollbarSelector&);
    virtual  bool  isScrollbarEnabled(const ScrollbarSelector&) const;
    virtual  void  setScrollbarPageSize(const ScrollbarSelector&, int);
    virtual  int   scrollbarPageSize(const ScrollbarSelector&) const;
    virtual  void  setScrollbarPosition(const ScrollbarSelector&, int);
    virtual  void  setScrollbarMinorIncrement(const ScrollbarSelector&, int);
    virtual  int   scrollbarPosition(const ScrollbarSelector&) const;
    virtual  void  setScrollbarRange(const ScrollbarSelector&, int, int);
    virtual  int   scrollbarRangeMin(const ScrollbarSelector&) const;
    virtual  int   scrollbarRangeMax(const ScrollbarSelector&) const;
    virtual  StatusBarPaneToken  appendStatusPane(int);
    virtual  void  setStatusPaneText(StatusBarPaneToken, const std::string&);
    virtual  void  setMinimumTrackSize(int, int);
    virtual  void  setMinimumTrackSize(Vector2Di&);
    virtual  const Vector2Di&  minimumTrackSize( ) const;
    virtual  void  wait( );
    virtual  void  unwait( );
    virtual  bool  isWaiting( ) const;
};
//---------------------------------------------------------------------------//



//---------------------------------------------------------------------------//
/* An ApplicationWindowController is a kind of ThingyController that
   defines an interface for all objects that wish to control
   ApplicationWindowThingies. Since ApplicationWindowThingies are composite
   thingies, their controllers must manage the layout and appearance of
   any subthingies created inside */
class  ApplicationWindowController :
    public virtual CompositeThingyController {

protected:

    ApplicationWindowController( ) { }

public:

    static  ApplicationWindowController*  nullController( );

    virtual  ~ApplicationWindowController( ) { }

    virtual  void  thingyClosed(ApplicationWindowThingy&) = 0;
    virtual  void  thingyMaximized(ApplicationWindowThingy&) = 0;
    virtual  void  thingyRestored(ApplicationWindowThingy&) = 0;
    virtual  void  thingyMenuItemChosen(ApplicationWindowThingy&, int) = 0;
    virtual  void  thingyToolbarButtonPushed(ApplicationWindowThingy&,
        ToolbarButtonToken) = 0;
    virtual  void  thingyToolbarButtonSelected(ApplicationWindowThingy&,
        ToolbarButtonToken) = 0;
    virtual  void  thingyToolbarButtonDeselected(ApplicationWindowThingy&,
        ToolbarButtonToken) = 0;
    virtual  void  thingySized(ApplicationWindowThingy&) = 0;
    virtual  void  thingyScrollbarPositioned(ApplicationWindowThingy&,
        const ScrollbarSelector&) = 0;
    virtual  void  thingyScrollbarTracked(ApplicationWindowThingy&,
        const ScrollbarSelector&, int) = 0;
};
//---------------------------------------------------------------------------//



//---------------------------------------------------------------------------//
/* A PushButtonThingy is a kind of Thingy that implements an on-screen push
   button control, like the "Ok" and "Cancel" controls found in dialog
   boxes. PushButtonThingies are primitive thingies (they can't serve as 
   parents for subthingies) and are not free thingies (they must know who
   their parent is when they are created). */
class  PushButtonThingy :
    public virtual Thingy,
    public virtual ThingyAdapterClient {

private:

    SafeThingyAdapter      fAdapter;
    PushButtonController*  fController;

    /* PushButtonThingies don't have value semantics */
    PushButtonThingy(const PushButtonThingy&) { }
    const PushButtonThingy& operator=(const PushButtonThingy&)
        { return *this; }

public:

    PushButtonThingy(CompositeThingy& composite);

    LRESULT  processParentMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam);
    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    virtual  PushButtonController&  controller( );
    virtual  const PushButtonController&  controller( ) const;
    virtual  PushButtonController&  installController(
        PushButtonController&);
    virtual  std::string  title( ) const;
    virtual  void  setTitle(const std::string&);
};
//---------------------------------------------------------------------------//




//---------------------------------------------------------------------------//
/* A PushButtonController is a kind of ThingyController that defines
   an interface to be implemented by all objects that wish to control
   on-screen push button widgets */
class  PushButtonController :
    public virtual ThingyController {

protected:

    PushButtonController( ) { }

public:

    static  PushButtonController*  nullController( );

    virtual ~PushButtonController( ) { }

    virtual  void  thingyClicked(PushButtonThingy& sender) = 0;
};
//---------------------------------------------------------------------------//




//---------------------------------------------------------------------------//
/* A GroupBoxThingy is a bound, composite Thingy that implements an on-screen
   box used to group related controls (such as a radio button group). */
class  GroupBoxThingy :
    public virtual CompositeThingy,
    public virtual CompositeThingyAdapterClient {

private:

    static  bool  sIsClassRegistered;

    SafeThingyAdapter             fOutsetPaneAdapter;
    SafeThingyAdapter             fGroupFrameAdapter;
    SafeThingyAdapter             fInsetPaneAdapter;
    CompositeThingyController*    fController;

    /* GroupBoxThingies don't have value semantics */
    GroupBoxThingy(const GroupBoxThingy&) { }
    const GroupBoxThingy& operator=(const GroupBoxThingy&)
        { return *this; }

    void  uLayoutInnerPanes( );

public:

    GroupBoxThingy(CompositeThingy& parent);

    LRESULT  processMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam);
    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    ThingyAdapter&  composerAdapter( );
    const ThingyAdapter&  composerAdapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    unsigned short  composableWidth( ) const;
    unsigned short  composableHeight( ) const;
    virtual  CompositeThingyController&  controller( );
    virtual  const CompositeThingyController&  controller( ) const;
    virtual  CompositeThingyController&  installController(
        CompositeThingyController&);
    virtual  std::string  title( ) const;
    virtual  void  setTitle(const std::string&);
};
//---------------------------------------------------------------------------//




//---------------------------------------------------------------------------//
/* A TextLabelThingy is a kind of Thingy that implements an on-screen text
   label control. TextLabelThingies are primitive thingies (they can't
   serve as parents for subthingies) and are not free thingies (they must
   know who their parent is when they are created). TextLabelThingies don't
   have any associated user events, so they have no controllers. */
class  TextLabelThingy :
    public virtual Thingy,
    public virtual ThingyAdapterClient {

private:

    SafeThingyAdapter      fAdapter;

    /* PushButtonThingies don't have value semantics */
    TextLabelThingy(const TextLabelThingy&) { }
    const TextLabelThingy& operator=(const TextLabelThingy&)
        { return *this; }

public:

    enum  TextAlignmentMode { kAlignLeft, kAlignRight };

    TextLabelThingy(CompositeThingy& composite);

    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    virtual  std::string  title( ) const;
    virtual  void  setTitle(const std::string&);
    virtual  TextLabelThingy::TextAlignmentMode  alignment( ) const;
    virtual  void  setAlignment(TextLabelThingy::TextAlignmentMode mode);
};
//---------------------------------------------------------------------------//




//---------------------------------------------------------------------------//
/* A CheckBoxThingy is a kind of Thingy that implements an on-screen checkbox
   control. CheckBoxThingies are primitive thingies (they can't serve as
   parents for subthingies) and are not free thingies. */
class  CheckBoxThingy :
    public virtual Thingy,
    public virtual ThingyAdapterClient {

private:

    SafeThingyAdapter      fAdapter;
    CheckBoxController*    fController;

    /* CheckBoxThingies don't have value semantics */
    CheckBoxThingy(const CheckBoxThingy&) { }
    const CheckBoxThingy& operator=(const CheckBoxThingy&)
        { return *this; }

public:

    CheckBoxThingy(CompositeThingy& composite);

    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    LRESULT  processParentMessage(HWND target, UINT msgcode, WPARAM wparam,
        LPARAM lparam);
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    virtual  std::string  title( ) const;
    virtual  void  setTitle(const std::string&);
    virtual  bool  isChecked( ) const;
    virtual  void  setChecked( );
    virtual  void  setUnchecked( );
    virtual  CheckBoxController&  controller( );
    virtual  const CheckBoxController&  controller( ) const;
    virtual  CheckBoxController&  installController(CheckBoxController&);
};
//---------------------------------------------------------------------------//




//---------------------------------------------------------------------------//
/* A CheckBoxController is a kind of ThingyController that defines
   an interface to be implemented by all objects that wish to receive events
   from on-screen check box widgets */
class  CheckBoxController :
    public virtual ThingyController {

protected:

    CheckBoxController( ) { }

public:

    static  CheckBoxController*  nullController( );

    virtual ~CheckBoxController( ) { }

    virtual  void  thingyChecked(CheckBoxThingy& sender) = 0;
    virtual  void  thingyUnchecked(CheckBoxThingy& sender) = 0;
};
//---------------------------------------------------------------------------//




//---------------------------------------------------------------------------//
/* A TextEntryThingy is a kind of Thingy that implements an on-screen
   single-line text entry box control. TextEditThingies are primitive
   thingies (they can't serve as parents for subthingies) and are bound. */
class  TextEntryThingy :
    public virtual Thingy,
    public virtual ThingyAdapterClient {

private:

    SafeThingyAdapter      fAdapter;

    /* TextEditThingies don't have value semantics */
    TextEntryThingy(const TextEntryThingy&) { }
    const TextEntryThingy& operator=(const TextEntryThingy&)
        { return *this; }

public:

    TextEntryThingy(CompositeThingy& composite);

    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    virtual  std::string  contents( ) const;
    virtual  void  setContents(const std::string&);
};
//---------------------------------------------------------------------------//




//---------------------------------------------------------------------------//
/* A DialogThingy is a kind of Thingy that implements an on-screen, 
   application-modal dialog box window. DialogThingies can't be resized, 
   maximized, or minimized by the user, but they can be closed.
   DialogThingies are created from a description stored in a resource script.
   DialogThingies are composite thingies that can manage any subthingies
   created inside them and are bound */
class DialogThingy :
    public virtual CompositeThingy,
    public virtual CompositeThingyAdapterClient {

private:

    static  DialogThingy*  sObjAwaitingAssoc;

    SafeThingyAdapter           fAdapter;
    CompositeThingy*            fParentWindow;
    DialogController*           fController;
    int                         fDialogResID;
    bool                        fIsInteractionStopping;

    static  INT_PTR  CALLBACK  sDialogProc(HWND, UINT, WPARAM, LPARAM);
    static  DialogThingy&  sGetObject(HWND);

    /* DialogThingies don't have value semantics */
    DialogThingy(const DialogThingy&) { }
    const DialogThingy& operator=(const DialogThingy&) { return *this; }

public:

    DialogThingy(CompositeThingy&, int);
    DialogThingy(int);
    virtual  ~DialogThingy( );

    LRESULT  processMessage(HWND target, UINT message,
        WPARAM wparam, LPARAM lparam);
    ThingyAdapter&  adapter( );
    const ThingyAdapter&  adapter( ) const;
    ThingyAdapter&  composerAdapter( );
    const ThingyAdapter&  composerAdapter( ) const;
    void  show( );
    void  hide( );
    bool  isVisible( ) const;
    Vector2Di  origin( ) const;
    void  setOrigin(const Vector2Di& ogn);
    void  setOrigin(unsigned short x, unsigned short y);
    unsigned short  width( ) const;
    void  setWidth(unsigned short w);
    unsigned short  height( ) const;
    void  setHeight(unsigned short h);
    void  enable( );
    void  disable( );
    bool  isEnabled( ) const;
    unsigned short  composableWidth( ) const;
    unsigned short  composableHeight( ) const;
    virtual  DialogController&  controller( );
    virtual  const DialogController&  controller( ) const;
    virtual  DialogController&  installController(DialogController&);
    virtual  std::string  title( ) const;
    virtual  void  setTitle(const std::string&);
    virtual  void  startInteraction( );
    virtual  void  stopInteraction( );
    virtual  bool  isInteractionRunning( ) const;
    virtual  void  themeItem(int);
    virtual  std::string  itemText(int) const;
    virtual  void  setItemText(int, const std::string&);
    virtual  void  enableItem(int);
    virtual  void  disableItem(int);
    virtual  bool  isItemEnabled(int) const;
};
//---------------------------------------------------------------------------//





//---------------------------------------------------------------------------//
/* A DialogController is a kind of ThingyController that defines an interface
   to be implemented by all objects that wish to receive events from
   on-screen dialog box widgets */
class  DialogController :
    public virtual CompositeThingyController {

protected:

    DialogController( ) { }

public:

    static  DialogController*  nullController( );
    virtual  ~DialogController( ) { }

    virtual  void  thingyItemActivated(DialogThingy&, int) = 0;
    virtual  void  thingyClosed(DialogThingy&) = 0;
    virtual  void  thingyInteractionStarted(DialogThingy&) = 0;
    virtual  void  thingyInteractionStopped(DialogThingy&) = 0;
};
//---------------------------------------------------------------------------//

#endif
