//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// EssentialThingies.cpp                                                    //
//                                                                          //
// Implements methods in classes ApplicationPaneThingy,                     //
// ApplicationPaneController, PushButtonThingy, and PushButtonController.   //
// Pacakages interface definitions for and implements methods in classes    //
// NullApplicationPaneController and NullPushButtonController               //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  <unversioned module>                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  Copyright (c) 2008 Lucas Stephen Beeler. All Rights Reserved.           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "EssentialThingies.h"
#include "Application.h"
#include "PTCIResources.h"

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ApplicationPaneThingy                                             //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const std::string  ApplicationPaneThingy::kWindowClassName =
    "winclass_ApplicationPaneThingy";
bool               ApplicationPaneThingy::sIsClassRegistered = false;


ApplicationPaneThingy::ApplicationPaneThingy( )
    : fController(ApplicationPaneController::nullController( ))
{
    if (!sIsClassRegistered) {

        WNDCLASS wdc;
        wdc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
        wdc.lpfnWndProc = ThingyAdapter::sUniversalProc;
        wdc.cbClsExtra = 0;
        wdc.cbWndExtra = 0;
        wdc.hInstance = Application::instance( ).systemModule( );
        wdc.hIcon = 0;
        wdc.hCursor = LoadCursor(NULL, IDC_ARROW);
        wdc.hbrBackground = (HBRUSH) (COLOR_BTNFACE + 1);
        wdc.lpszMenuName = 0;
        wdc.lpszClassName = kWindowClassName.c_str( );

        ATOM regresult = RegisterClass(&wdc);
        if (!regresult)
            throw std::runtime_error("ApplicationPaneThingy: unable to "
                "register window class");

        sIsClassRegistered = true;
    }

    HWND primitive = CreateWindow(kWindowClassName.c_str( ),
        "ApplicationPaneThingy", WS_BORDER | WS_CAPTION | WS_MINIMIZEBOX |
        WS_SYSMENU, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
        CW_USEDEFAULT, NULL, NULL, NULL, 0);

    fAdapter.reset(new ThingyAdapter(*this, primitive));
}




LRESULT  ApplicationPaneThingy::processMessage(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    switch (msgcode) {

        case WM_CLOSE:
            fController->thingyClosed(*this);
            return 0;
        break;

        case WM_SIZE:
            fController->layoutSubthingies(*this);
            return 0;
        break;

        case WM_COMMAND:
            if (lparam) /* only control msgs have non-null lparams */
                return fAdapter->propogateChildMessage(target, msgcode,
                    wparam, lparam);
            else
                return DefWindowProc(target, msgcode, wparam, lparam);
        break;

        default:
            return DefWindowProc(target, msgcode, wparam, lparam);
        break;
    }
}



ThingyAdapter&  ApplicationPaneThingy::adapter( )
{
    return *fAdapter;
}




const ThingyAdapter&  ApplicationPaneThingy::adapter( ) const
{
    return *fAdapter;
}




ThingyAdapter&  ApplicationPaneThingy::composerAdapter( )
{
    return adapter( );
}





const ThingyAdapter&  ApplicationPaneThingy::composerAdapter( ) const
{
    return adapter( );
}

    
    
        
    
void  ApplicationPaneThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  ApplicationPaneThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  ApplicationPaneThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  ApplicationPaneThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  ApplicationPaneThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  ApplicationPaneThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  ApplicationPaneThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  ApplicationPaneThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}




unsigned short  ApplicationPaneThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  ApplicationPaneThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  ApplicationPaneThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}




void  ApplicationPaneThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}




bool  ApplicationPaneThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}




unsigned short  ApplicationPaneThingy::composableWidth( ) const
{
    return Win32Tools::getClientAreaWidth(fAdapter->primitive( ));
}




unsigned short  ApplicationPaneThingy::composableHeight( ) const
{
    return Win32Tools::getClientAreaHeight(fAdapter->primitive( ));
}




ApplicationPaneController&  ApplicationPaneThingy::controller( )
{
    return *fController;
}




const ApplicationPaneController&
    ApplicationPaneThingy::controller( ) const
{
    return *fController;
}




ApplicationPaneController&  ApplicationPaneThingy::installController(
        ApplicationPaneController& newctlr)
{
    ApplicationPaneController* oldctlr = fController;
    fController = & newctlr;
    return *oldctlr;
}




std::string  ApplicationPaneThingy::title( ) const
{
    return Win32Tools::getWindowTextString(fAdapter->primitive( ));
}




void  ApplicationPaneThingy::setTitle(const std::string& t)
{
    Win32Tools::setWindowTextString(fAdapter->primitive( ), t);
}
//////////////////////////////////////////////////////////////////////////////
// END  ApplicationPaneThingy                                               //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ApplicationPaneController                                         //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class  NullApplicationPaneController :
    public virtual ApplicationPaneController {

public:

    NullApplicationPaneController( ) { }
    virtual  ~NullApplicationPaneController( ) { }

    void  thingyClosed(ApplicationPaneThingy& sender) { }
    void  layoutSubthingies(CompositeThingy& sender) { }
    void  enableSubthingies(CompositeThingy& sender) { }
    void  disableSubthingies(CompositeThingy& sender) { }
};


ApplicationPaneController*
    ApplicationPaneController::nullController( )
{
    return new NullApplicationPaneController( );
}
//////////////////////////////////////////////////////////////////////////////
// END  ApplicationPaneController                                           //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ApplicationWindowThingy                                           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const std::string    ApplicationWindowThingy::kFrameWindowClassName =
    "ApplicationWindowThingy_frame";
const std::string    ApplicationWindowThingy::kContentPaneWindowClassName =
    "ApplicationWindowThingy_content";
const std::string    ApplicationWindowThingy::kClientPaneWindowClassName =
    "ApplicationWindowThingy_client";
bool                 ApplicationWindowThingy::sAreClassesRegistered = false;
long                 ApplicationWindowThingy::sNextTBToken = 1024;

ApplicationWindowThingy::ApplicationWindowThingy( )
    : fController(ApplicationWindowController::nullController( )),
      fIsToolbarEnabled(false), fIsAssemblyComplete(false),
      fIsHorizScrollEnabled(false), fIsVerticalScrollEnabled(false),
      fIsStatBarEnabled(true), fHorizScrollMinorIncr(1),
      fHorizScrollMajorIncr(2), fVertScrollMinorIncr(1),
      fVertScrollMajorIncr(2), fMinTrackSize(320, 240),
      fInstalledKeys(0), fIsWaiting(false)
{
    if (!sAreClassesRegistered) {

        WNDCLASS wdc;
        wdc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
        wdc.lpfnWndProc = ThingyAdapter::sUniversalProc;
        wdc.cbClsExtra = 0;
        wdc.cbWndExtra = 0;
        wdc.hInstance = Application::instance( ).systemModule( );
        wdc.hIcon = 0;
        wdc.hCursor = LoadCursor(NULL, IDC_ARROW);
        wdc.hbrBackground = (HBRUSH) (COLOR_BTNFACE + 1);
        wdc.lpszMenuName = 0;
        wdc.lpszClassName = kFrameWindowClassName.c_str( );

        ATOM regresult = RegisterClass(&wdc);
        if (!regresult)
            throw std::runtime_error("ApplicationWindowThingy: unable to "
                "register frame window class");

        wdc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
        wdc.lpfnWndProc = ThingyAdapter::sUniversalProc;
        wdc.cbClsExtra = 0;
        wdc.cbWndExtra = 0;
        wdc.hInstance = Application::instance( ).systemModule( );
        wdc.hIcon = 0;
        wdc.hCursor = LoadCursor(NULL, IDC_ARROW);
        wdc.hbrBackground = (HBRUSH) NULL;
        wdc.lpszMenuName = 0;
        wdc.lpszClassName = kContentPaneWindowClassName.c_str( );

        regresult = RegisterClass(&wdc);
        if (!regresult)
            throw std::runtime_error("ApplicationWindowThingy: unable to "
                "register content pane window class");

        wdc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
        wdc.lpfnWndProc = ThingyAdapter::sUniversalProc;
        wdc.cbClsExtra = 0;
        wdc.cbWndExtra = 0;
        wdc.hInstance = Application::instance( ).systemModule( );
        wdc.hIcon = 0;
        wdc.hCursor = LoadCursor(NULL, IDC_ARROW);
        wdc.hbrBackground = (HBRUSH) (COLOR_APPWORKSPACE + 1);
        wdc.lpszMenuName = 0;
        wdc.lpszClassName = kClientPaneWindowClassName.c_str( );

        regresult = RegisterClass(&wdc);
        if (!regresult)
            throw std::runtime_error("ApplicationWindowThingy: unable to "
                "register client pane window class");

        sAreClassesRegistered = true;
    }

    HWND frame_primitive = CreateWindowEx(0,
        kFrameWindowClassName.c_str( ), "ApplicationWindowThingy",
        WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
        CW_USEDEFAULT, NULL, NULL, NULL, 0);

    fFrameAdapter.reset(new ThingyAdapter(*this, frame_primitive));

    installMenu(kAppWinDefaultMenuResID);

    HWND tb_primitive = CreateWindow(TOOLBARCLASSNAME, (LPSTR)(NULL),
        WS_CHILD | CCS_TOP | TBSTYLE_FLAT | TBSTYLE_LIST, 0, 0, 256, 64,
        frame_primitive, (HMENU)(nextChildID( )),
        Application::instance( ).systemModule( ), NULL);
    fToolbarAdapter.reset(new ThingyAdapter(*this, tb_primitive));
    SendMessage(tb_primitive, TB_SETBUTTONSIZE, 0,
        (LPARAM) MAKELONG(64, 64));
    SendMessage(tb_primitive, TB_SETBITMAPSIZE, 0,
        (LPARAM) MAKELONG(32, 32));
    SendMessage(tb_primitive, TB_BUTTONSTRUCTSIZE,
        (WPARAM) sizeof(TBBUTTON), 0);
    SendMessage(tb_primitive, TB_AUTOSIZE, 0, 0); 

    TBBUTTON  sepdesc;
    sepdesc.iBitmap = 3;
    sepdesc.idCommand = 0;
    sepdesc.fsState = TBSTATE_ENABLED;
    sepdesc.fsStyle = BTNS_SEP;
    sepdesc.dwData = 0;
    sepdesc.iString = -1;
    SendMessage(tb_primitive, TB_ADDBUTTONS, (WPARAM) 1, 
        (LPARAM) (LPTBBUTTON) &sepdesc);

    sepdesc.iBitmap = 3;
    sepdesc.idCommand = 0;
    sepdesc.fsState = TBSTATE_ENABLED;
    sepdesc.fsStyle = BTNS_SEP;
    sepdesc.dwData = 0;
    sepdesc.iString = -1;
    SendMessage(tb_primitive, TB_ADDBUTTONS, (WPARAM) 1, 
        (LPARAM) (LPTBBUTTON) &sepdesc);

    HWND  contentpane_primitive = CreateWindowEx(WS_EX_CLIENTEDGE,
        kContentPaneWindowClassName.c_str( ), TEXT(""), WS_CHILD,
        0, 0, 1, 1, frame_primitive, (HMENU)(nextChildID( )),
        Application::instance( ).systemModule( ), NULL);
    fContentPaneAdapter.reset(new ThingyAdapter(*this,
        contentpane_primitive));

    HWND  horizscroll_primitive = CreateWindow("SCROLLBAR", (LPSTR) NULL,
        WS_CHILD | SBS_HORZ | SBS_BOTTOMALIGN,
        0, 0, 128, 128, contentpane_primitive, (HMENU)(nextChildID( )),
        Application::instance( ).systemModule( ), NULL);
    fHorizScrollAdapter.reset(new ThingyAdapter(*this,
        horizscroll_primitive));

    HWND  vertscroll_primitive = CreateWindow("SCROLLBAR", (LPSTR) NULL,
        WS_CHILD | SBS_VERT | SBS_RIGHTALIGN,
        0, 0, 128, 128, contentpane_primitive, (HMENU)(nextChildID( )),
        Application::instance( ).systemModule( ), NULL);
    fVertScrollAdapter.reset(new ThingyAdapter(*this, vertscroll_primitive));

    HWND  clientpane_primitive =
        CreateWindow(kClientPaneWindowClassName.c_str( ), (LPSTR) NULL,
        WS_CHILD | WS_VISIBLE, 0, 0, 128, 128,
        contentpane_primitive, (HMENU)(nextChildID( )),
        Application::instance( ).systemModule( ), NULL);
    fClientPaneAdapter.reset(new ThingyAdapter(*this, clientpane_primitive));

    HWND  statbar_primitive = CreateWindow(STATUSCLASSNAME, TEXT(""),
        WS_CHILD | WS_VISIBLE | SBARS_SIZEGRIP | CCS_BOTTOM, 0, 0, 128,
        128, frame_primitive, (HMENU)(nextChildID( )),
        Application::instance( ).systemModule( ), NULL);
    Win32Tools::applyControlFaceProperties(statbar_primitive);
    fStatusBarAdapter.reset(new ThingyAdapter(*this, statbar_primitive));

    fIsAssemblyComplete = true;
}



LRESULT  ApplicationWindowThingy::processMessage(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    if (! fIsAssemblyComplete)
        return DefWindowProc(target, msgcode, wparam, lparam);

    MINMAXINFO*  constraints = 0;

    /* Process messages from the top-level window frame */
    if (target == fFrameAdapter->primitive( )) {

        switch (msgcode) {

            case WM_CLOSE:
                fController->thingyClosed(*this);
                return 0;
            break;

            case WM_SIZE:
                uLayoutAssembly( );
                fController->thingySized(*this);
                return 0;
            break;

            case WM_SETCURSOR:
                if (isWaiting( )) {

                    HCURSOR hourglass = LoadCursor(0, IDC_WAIT);

                    SetCursor(hourglass);

                    return TRUE;
                }
                else {

                    return DefWindowProc(target, msgcode, wparam, lparam);
                }
            break;

            case WM_GETMINMAXINFO:
                constraints = (MINMAXINFO*) lparam;
                constraints->ptMinTrackSize.x = fMinTrackSize.x;
                constraints->ptMinTrackSize.y = fMinTrackSize.y;
                return 0;
            break;

            case WM_COMMAND:
                if (lparam) { /* only control msgs have non-null lparams */

                    int buttontoken = LOWORD(wparam);
                    if (uIsToolbarButton(buttontoken)) {

                        uProcessToolbarButtonClick(buttontoken);
                        return 0;
                    }
                    else {

                        return fFrameAdapter->propogateChildMessage(target,
                            msgcode, wparam, lparam);
                    } /* else if the command's not from a toolbar button */
                } /* if the message is from a control */
                else {

                    fController->thingyMenuItemChosen(*this, LOWORD(wparam));
                    return 0;
                }
            break;

            default:
                return DefWindowProc(target, msgcode, wparam, lparam);
            break;
        } /* switch */
    } /* if the message target is the top-level window frame */
    else if (target == fClientPaneAdapter->primitive( )) {

        switch (msgcode) {

            case WM_SIZE:
                fController->layoutSubthingies(*this);
                return 0;
            break;

            case WM_COMMAND:
                if (lparam) { /* only control msgs have non-null lparams */

                    return fFrameAdapter->propogateChildMessage(target,
                        msgcode, wparam, lparam);
                } /* if the message is from a control */
                else {

                    return DefWindowProc(target, msgcode, wparam, lparam);
                } /* else if the message isn't from a control */
            break;

            default:
                return DefWindowProc(target, msgcode, wparam, lparam);
            break;
        } /* switch */
    } /* else if the message target is the client pane */
    else if (target == fContentPaneAdapter->primitive( )) {

        switch (msgcode) {

            case WM_HSCROLL:
                uInterpretScrollAction(kHorizontalScrollbar, LOWORD(wparam));
                return 0;
            break;

            case WM_VSCROLL:
                uInterpretScrollAction(kVerticalScrollbar, LOWORD(wparam));
                return 0;
            break;

            default:
                return DefWindowProc(target, msgcode, wparam, lparam);
            break;
        } /* switch */
    } /* else if the message target is the content pane */
    else {

        return DefWindowProc(target, msgcode, wparam, lparam);

    } /* else the message target is neither the frame, nor the
         client pane, nor the content pane */
}



ThingyAdapter&  ApplicationWindowThingy::adapter( )
{
    return *fFrameAdapter;
}



const ThingyAdapter&  ApplicationWindowThingy::adapter( ) const
{
    return *fFrameAdapter;
}



ThingyAdapter&  ApplicationWindowThingy::composerAdapter( )
{
    return *fClientPaneAdapter;
}



const ThingyAdapter&  ApplicationWindowThingy::composerAdapter( ) const
{
    return *fClientPaneAdapter;
}



void  ApplicationWindowThingy::show( )
{
    ShowWindow(fContentPaneAdapter->primitive( ), SW_SHOW);
    ShowWindow(fFrameAdapter->primitive( ), SW_SHOW);
}



void  ApplicationWindowThingy::hide( )
{
    ShowWindow(fFrameAdapter->primitive( ), SW_HIDE);
}



bool  ApplicationWindowThingy::isVisible( ) const
{
    if (IsWindowVisible(fFrameAdapter->primitive( )))
        return true;
    else
        return false;
}



Vector2Di  ApplicationWindowThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fFrameAdapter->primitive( ));
}



void  ApplicationWindowThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fFrameAdapter->primitive( ), ogn.x, ogn.y);
}



void  ApplicationWindowThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}



unsigned short  ApplicationWindowThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fFrameAdapter->primitive( ));
}



void  ApplicationWindowThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fFrameAdapter->primitive( ), w);
}



unsigned short  ApplicationWindowThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fFrameAdapter->primitive( ));
}



void  ApplicationWindowThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fFrameAdapter->primitive( ), h);
}



void  ApplicationWindowThingy::enable( )
{
    EnableWindow(fFrameAdapter->primitive( ), TRUE);
}



void  ApplicationWindowThingy::disable( )
{
    EnableWindow(fFrameAdapter->primitive( ), FALSE);
}



bool  ApplicationWindowThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fFrameAdapter->primitive( )))
        return true;
    else
        return false;
}



unsigned short  ApplicationWindowThingy::composableWidth( ) const
{
    return Win32Tools::getClientAreaWidth(fClientPaneAdapter->primitive( ));
}



unsigned short  ApplicationWindowThingy::composableHeight( ) const
{
    return Win32Tools::getClientAreaHeight(fClientPaneAdapter->primitive( ));
}



ApplicationWindowController&  ApplicationWindowThingy::controller( )
{
    return *fController;
}



const ApplicationWindowController&
    ApplicationWindowThingy::controller( ) const
{
    return *fController;
}



ApplicationWindowController&  ApplicationWindowThingy::installController(
    ApplicationWindowController& newctlr)
{
    ApplicationWindowController* oldctlr = fController;

    fController = & newctlr;

    newctlr.layoutSubthingies(*this);

    return *oldctlr;
}



std::string  ApplicationWindowThingy::title( ) const
{
    return Win32Tools::getWindowTextString(fFrameAdapter->primitive( ));
}



void  ApplicationWindowThingy::setTitle(const std::string& t)
{
    Win32Tools::setWindowTextString(fFrameAdapter->primitive( ), t);
}



void  ApplicationWindowThingy::installMenu(int resid)
{
    HMENU  newmenu = LoadMenu(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(resid));
    if (newmenu == NULL)
        throw std::runtime_error("ApplicationWindowThingy: couldn't install "
            "menu: specified menu resource can't be loaded");

    HMENU  currentmenu = GetMenu(fFrameAdapter->primitive( ));

    SetMenu(fFrameAdapter->primitive( ), newmenu);

    if (currentmenu != NULL)
        DestroyMenu(currentmenu);
}




void  ApplicationWindowThingy::enableToolbar( )
{
    ShowWindow(fToolbarAdapter->primitive( ), SW_SHOW);
    fIsToolbarEnabled = true;
}



void  ApplicationWindowThingy::disableToolbar( )
{
    ShowWindow(fToolbarAdapter->primitive( ), SW_HIDE);
    fIsToolbarEnabled = false;
}



bool  ApplicationWindowThingy::isToolbarEnabled( ) const
{
    return fIsToolbarEnabled;
}



ToolbarButtonToken  ApplicationWindowThingy::appendToolbarButton(
    ToolbarButtonKind kind, HBITMAP userbmp, const std::string& usertitle)
{
    TBADDBITMAP  bitmapsdesc;
    bitmapsdesc.hInst = NULL;
    bitmapsdesc.nID = (UINT_PTR)(userbmp);

    int  userbmpidx = SendMessage(fToolbarAdapter->primitive( ),
        TB_ADDBITMAP, 1, (LPARAM)(&bitmapsdesc));

    TBBUTTON  buttondesc;
    buttondesc.iBitmap = userbmpidx;
    buttondesc.idCommand = sNextTBToken++;
    buttondesc.fsState = TBSTATE_ENABLED;
    if (kind == kPushButtonKind)  buttondesc.fsStyle = BTNS_BUTTON;
    else if (kind == kSelectButtonKind)  buttondesc.fsStyle = BTNS_CHECK;
    buttondesc.dwData = 0;
    buttondesc.iString = (INT_PTR)(usertitle.c_str( ));
    SendMessage(fToolbarAdapter->primitive( ), TB_ADDBUTTONS, (WPARAM) 1,
        (LPARAM) (LPTBBUTTON) &buttondesc);

    fToolbarButtKindDict[(sNextTBToken - 1)] = kind;

    return (sNextTBToken - 1);
}



void  ApplicationWindowThingy::appendToolbarSeparator(int width)
{
    TBBUTTON  sepdesc;
    sepdesc.iBitmap = width;
    sepdesc.idCommand = 0;
    sepdesc.fsState = TBSTATE_ENABLED;
    sepdesc.fsStyle = BTNS_SEP;
    sepdesc.dwData = 0;
    sepdesc.iString = -1;
    SendMessage(fToolbarAdapter->primitive( ), TB_ADDBUTTONS, (WPARAM) 1, 
        (LPARAM) (LPTBBUTTON) &sepdesc);
}



void  ApplicationWindowThingy::uLayoutAssembly( )
{
    if (isToolbarEnabled( ))
        SendMessage(fToolbarAdapter->primitive( ), TB_AUTOSIZE, 0, 0);

    int  tbheight = 0;
    if (isToolbarEnabled( ))
        tbheight = Win32Tools::getWindowHeight(fToolbarAdapter->primitive( ))
        - 3;

    int  frame_clientheight = Win32Tools::getClientAreaHeight(
        fFrameAdapter->primitive( ));
    int  frame_clientwidth = Win32Tools::getClientAreaWidth(
        fFrameAdapter->primitive( ));

    int  statbar_height =
        Win32Tools::getWindowHeight(fStatusBarAdapter->primitive( ));

    Win32Tools::setWindowOrigin(fContentPaneAdapter->primitive( ),
        0, tbheight);
    Win32Tools::setWindowWidth(fContentPaneAdapter->primitive( ),
        frame_clientwidth);
    Win32Tools::setWindowHeight(fContentPaneAdapter->primitive( ),
        frame_clientheight - tbheight - statbar_height);

    Win32Tools::setWindowOrigin(fStatusBarAdapter->primitive( ),
        0, frame_clientheight - statbar_height);
    Win32Tools::setWindowWidth(fStatusBarAdapter->primitive( ),
        frame_clientwidth);

    int  scrollwidth = 0;
    if (isScrollbarEnabled(kVerticalScrollbar))
        scrollwidth = GetSystemMetrics(SM_CXHTHUMB);

    int  scrollheight = 0;
    if (isScrollbarEnabled(kHorizontalScrollbar))
        scrollheight = GetSystemMetrics(SM_CYVTHUMB);

    int  content_clientheight =
        Win32Tools::getClientAreaHeight(fContentPaneAdapter->primitive( ));
    int  content_clientwidth =
        Win32Tools::getClientAreaWidth(fContentPaneAdapter->primitive( ));

    Win32Tools::setWindowOrigin(fHorizScrollAdapter->primitive( ),
        0, content_clientheight - scrollheight);
    Win32Tools::setWindowWidth(fHorizScrollAdapter->primitive( ),
        content_clientwidth - scrollwidth);

    Win32Tools::setWindowOrigin(fVertScrollAdapter->primitive( ),
        content_clientwidth - scrollwidth, 0);
    Win32Tools::setWindowHeight(fVertScrollAdapter->primitive( ),
        content_clientheight - scrollheight);

    Win32Tools::setWindowWidth(fClientPaneAdapter->primitive( ),
        content_clientwidth - scrollwidth);
    Win32Tools::setWindowHeight(fClientPaneAdapter->primitive( ),
        content_clientheight - scrollheight);

    /* reset the scrollbar major track increments to 2/3rd of the dimension
       of the composable area */
    fHorizScrollMajorIncr = (2 * this->composableWidth( )) / 3;
    fVertScrollMajorIncr = (2 * this->composableHeight( )) / 3;
}



void  ApplicationWindowThingy::enableScrollbar(const ScrollbarSelector& bar)
{
    if (bar == kHorizontalScrollbar)
        uEnableHorizontalScrollbar( );
    else if (bar == kVerticalScrollbar)
        uEnableVerticalScrollbar( );
    else
        throw std::logic_error("ApplicationWindowThingy: "
            "enableScrollbar( ): invalid scrollbar selector");
}



void  ApplicationWindowThingy::disableScrollbar(const ScrollbarSelector& bar)
{
    if (bar == kHorizontalScrollbar)
        uDisableHorizontalScrollbar( );
    else if (bar == kVerticalScrollbar)
        uDisableVerticalScrollbar( );
    else
        throw std::logic_error("ApplicationWindowThingy: "
            "disableScrollbar( ): invalid scrollbar selector");
}



bool  ApplicationWindowThingy::isScrollbarEnabled(
    const ScrollbarSelector& bar) const
{
    if (bar == kHorizontalScrollbar)
        return uIsHorizontalScrollbarEnabled( );
    else if (bar == kVerticalScrollbar)
        return uIsVerticalScrollbarEnabled( );
    else
        throw std::logic_error("ApplicationWindowThingy: "
            "isScrollbarEnabled( ): invalid scrollbar selector");
}



void  ApplicationWindowThingy::uEnableHorizontalScrollbar( )
{
    ShowWindow(fHorizScrollAdapter->primitive( ), SW_SHOW);

    fIsHorizScrollEnabled = true;

    uLayoutAssembly( );
}



void  ApplicationWindowThingy::uDisableHorizontalScrollbar( )
{
    ShowWindow(fHorizScrollAdapter->primitive( ), SW_HIDE);

    fIsHorizScrollEnabled = false;

    uLayoutAssembly( );
}



bool  ApplicationWindowThingy::uIsHorizontalScrollbarEnabled( ) const
{
    return fIsHorizScrollEnabled;
}



void  ApplicationWindowThingy::uEnableVerticalScrollbar( )
{
    ShowWindow(fVertScrollAdapter->primitive( ), SW_SHOW);

    fIsVerticalScrollEnabled = true;

    uLayoutAssembly( );
}



void  ApplicationWindowThingy::uDisableVerticalScrollbar( )
{
    ShowWindow(fVertScrollAdapter->primitive( ), SW_HIDE);

    fIsVerticalScrollEnabled = false;

    uLayoutAssembly( );
}



bool  ApplicationWindowThingy::uIsVerticalScrollbarEnabled( ) const
{
    return fIsVerticalScrollEnabled;
}



bool  ApplicationWindowThingy::uIsToolbarButton(long buttonID)
{
    if (fToolbarButtKindDict.find(buttonID) != fToolbarButtKindDict.end( ))
        return true;
    else
        return false;
}



bool  ApplicationWindowThingy::uIsButtonSelected(long buttonID)
{
    LRESULT  msgresult = SendMessage(fToolbarAdapter->primitive( ),
        TB_GETSTATE, (WPARAM)(buttonID), 0);

    if (msgresult == -1)
        throw std::runtime_error("ApplicationWindowThingy: "
            "uIsButtonSelected( ): attempt to query button state failed");

    if (msgresult & TBSTATE_CHECKED)
        return true;
    else
        return false;
}



void  ApplicationWindowThingy::uProcessToolbarButtonClick(long buttonID)
{
    ToolbarButtonKind  kind =
        static_cast<ToolbarButtonKind>(fToolbarButtKindDict[buttonID]);

    if (kind == kSelectButtonKind) {

        if (uIsButtonSelected(buttonID))
            fController->thingyToolbarButtonSelected(*this, buttonID);
        else
            fController->thingyToolbarButtonDeselected(*this, buttonID);
    }
    else if (kind == kPushButtonKind) {

        fController->thingyToolbarButtonPushed(*this, buttonID);
    }
}



void  ApplicationWindowThingy::selectToolbarButton(
    ToolbarButtonToken buttonID)
{
    if (! uIsToolbarButton(buttonID))
        throw std::logic_error("ApplicationWindowThingy: "
            "selectToolbarButton( ): invalid button ID");

    long kind = fToolbarButtKindDict[buttonID];
    if (kind != kSelectButtonKind)
        throw std::logic_error("ApplicationWindowThingy: "
            "selectToolbarButton( ): button isn't a select button");

    SendMessage(fToolbarAdapter->primitive( ), TB_SETSTATE,
        (WPARAM)(buttonID),
        (LPARAM)(MAKELONG((TBSTATE_CHECKED | TBSTATE_ENABLED), 0)));
}



void  ApplicationWindowThingy::deselectToolbarButton(
    ToolbarButtonToken buttonID)
{
    if (! uIsToolbarButton(buttonID))
        throw std::logic_error("ApplicationWindowThingy: "
            "deselectToolbarButton( ): invalid button ID");

    long kind = fToolbarButtKindDict[buttonID];
    if (kind != kSelectButtonKind)
        throw std::logic_error("ApplicationWindowThingy: "
            "deselectToolbarButton( ): button isn't a select button");

    SendMessage(fToolbarAdapter->primitive( ), TB_SETSTATE,
        (WPARAM)(buttonID),
        (LPARAM)(MAKELONG(TBSTATE_ENABLED, 0)));
}



bool  ApplicationWindowThingy::isToolbarButtonSelected(
    ToolbarButtonToken buttonID) const
{
    ApplicationWindowThingy*  this_mutable =
        const_cast<ApplicationWindowThingy*>(this);

    if (! this_mutable->uIsToolbarButton(buttonID))
        throw std::logic_error("ApplicationWindowThingy: "
            "isToolbarButtonSelected( ): invalid button ID");

    long kind = this_mutable->fToolbarButtKindDict[buttonID];
    if (kind != kSelectButtonKind)
        throw std::logic_error("ApplicationWindowThingy: "
            "isToolbarButtonSelected( ): button isn't a select button");

    return this_mutable->uIsButtonSelected(buttonID);
}



void  ApplicationWindowThingy::setScrollbarPageSize(
    const ScrollbarSelector& which, int pagesize)
{
    HWND targetbar;
    if (which == kHorizontalScrollbar)
        targetbar = fHorizScrollAdapter->primitive( );
    else if (which == kVerticalScrollbar)
        targetbar = fVertScrollAdapter->primitive( );

    SCROLLINFO newinfo;
    newinfo.cbSize = sizeof(SCROLLINFO);
    newinfo.fMask = SIF_PAGE;
    newinfo.nMin = 0;
    newinfo.nMax = 0;
    newinfo.nPage = pagesize;
    newinfo.nPos = 0;
    newinfo.nTrackPos = 0;

    SetScrollInfo(targetbar, SB_CTL, &newinfo, TRUE);
}



int   ApplicationWindowThingy::scrollbarPageSize(
    const ScrollbarSelector& bar) const
{
    HWND barwin;
    if (bar == kHorizontalScrollbar)
        barwin = fHorizScrollAdapter->primitive( );
    else if (bar == kVerticalScrollbar)
        barwin = fVertScrollAdapter->primitive( );

    SCROLLINFO barinfo;
    barinfo.cbSize = sizeof(SCROLLINFO);
    barinfo.fMask = SIF_PAGE;
    barinfo.nMin = 0;
    barinfo.nMax = 0;
    barinfo.nPage = 0;
    barinfo.nPos = 0;
    barinfo.nTrackPos = 0;

    BOOL  result_ok = GetScrollInfo(barwin, SB_CTL, &barinfo);
    if (!result_ok)
        throw std::runtime_error("ApplicationWindowThingy: GetScrollInfo( ) "
            "failed in scrollbarPageSize( )");
    return barinfo.nPage;
}



void  ApplicationWindowThingy::setScrollbarPosition(
    const ScrollbarSelector& which, int pos)
{
    HWND targetbar;
    if (which == kHorizontalScrollbar)
        targetbar = fHorizScrollAdapter->primitive( );
    else if (which == kVerticalScrollbar)
        targetbar = fVertScrollAdapter->primitive( );

    SCROLLINFO newinfo;
    newinfo.cbSize = sizeof(SCROLLINFO);
    newinfo.fMask = SIF_POS;
    newinfo.nMin = 0;
    newinfo.nMax = 0;
    newinfo.nPage = 0;
    newinfo.nPos = pos;
    newinfo.nTrackPos = 0;

    SetScrollInfo(targetbar, SB_CTL, &newinfo, TRUE);
}



int   ApplicationWindowThingy::scrollbarPosition(
    const ScrollbarSelector& bar) const
{
    HWND barwin;
    if (bar == kHorizontalScrollbar)
        barwin = fHorizScrollAdapter->primitive( );
    else if (bar == kVerticalScrollbar)
        barwin = fVertScrollAdapter->primitive( );

    SCROLLINFO barinfo;
    barinfo.cbSize = sizeof(SCROLLINFO);
    barinfo.fMask = SIF_POS;
    barinfo.nMin = 0;
    barinfo.nMax = 0;
    barinfo.nPage = 0;
    barinfo.nPos = 0;
    barinfo.nTrackPos = 0;

    BOOL  result_ok = GetScrollInfo(barwin, SB_CTL, &barinfo);
    if (!result_ok)
        throw std::runtime_error("ApplicationWindowThingy: GetScrollInfo( ) "
            "failed in scrollbarPosition( )");

    return barinfo.nPos;
}



void  ApplicationWindowThingy::setScrollbarRange(
    const ScrollbarSelector& which, int rmin, int rmax)
{
    HWND targetbar;
    if (which == kHorizontalScrollbar)
        targetbar = fHorizScrollAdapter->primitive( );
    else if (which == kVerticalScrollbar)
        targetbar = fVertScrollAdapter->primitive( );

    SCROLLINFO newinfo;
    newinfo.cbSize = sizeof(SCROLLINFO);
    newinfo.fMask = SIF_RANGE;
    newinfo.nMin = rmin;
    newinfo.nMax = rmax;
    newinfo.nPage = 0;
    newinfo.nPos = 0;
    newinfo.nTrackPos = 0;

    SetScrollInfo(targetbar, SB_CTL, &newinfo, TRUE);
}



int   ApplicationWindowThingy::scrollbarRangeMin(
    const ScrollbarSelector& bar) const
{
    HWND barwin;
    if (bar == kHorizontalScrollbar)
        barwin = fHorizScrollAdapter->primitive( );
    else if (bar == kVerticalScrollbar)
        barwin = fVertScrollAdapter->primitive( );

    SCROLLINFO barinfo;
    barinfo.cbSize = sizeof(SCROLLINFO);
    barinfo.fMask = SIF_RANGE;
    barinfo.nMin = 0;
    barinfo.nMax = 0;
    barinfo.nPage = 0;
    barinfo.nPos = 0;
    barinfo.nTrackPos = 0;

    BOOL  result_ok = GetScrollInfo(barwin, SB_CTL, &barinfo);
    if (!result_ok)
        throw std::runtime_error("ApplicationWindowThingy: GetScrollInfo( ) "
            "failed in scrollbarRangeMin( )");

    return barinfo.nMin;
}



int   ApplicationWindowThingy::scrollbarRangeMax(
    const ScrollbarSelector& bar) const
{
    HWND barwin;
    if (bar == kHorizontalScrollbar)
        barwin = fHorizScrollAdapter->primitive( );
    else if (bar == kVerticalScrollbar)
        barwin = fVertScrollAdapter->primitive( );

    SCROLLINFO barinfo;
    barinfo.cbSize = sizeof(SCROLLINFO);
    barinfo.fMask = SIF_RANGE;
    barinfo.nMin = 0;
    barinfo.nMax = 0;
    barinfo.nPage = 0;
    barinfo.nPos = 0;
    barinfo.nTrackPos = 0;

    BOOL  result_ok = GetScrollInfo(barwin, SB_CTL, &barinfo);
    if (!result_ok)
        throw std::runtime_error("ApplicationWindowThingy: GetScrollInfo( ) "
            "failed in scrollbarRangeMax( )");

    return barinfo.nMax;
}



void  ApplicationWindowThingy::enableMenuItem(int itemID)
{
    MENUITEMINFO iteminfo;
    iteminfo.cbSize = sizeof(MENUITEMINFO);
    iteminfo.fMask = MIIM_STATE;
    iteminfo.fState = MFS_ENABLED;
    iteminfo.wID = 0;
    iteminfo.hSubMenu = 0;
    iteminfo.hbmpChecked = 0;
    iteminfo.hbmpUnchecked = 0;
    iteminfo.dwItemData = 0;
    iteminfo.dwTypeData = 0;
    iteminfo.cch = 0;
    iteminfo.hbmpItem = 0;

    HMENU  currmenu = GetMenu(fFrameAdapter->primitive( ));

    if (currmenu == NULL)
        throw std::logic_error("ApplicationWindowThingy: can't enable menu "
        "item: no menu is installed");

    BOOL call_ok = SetMenuItemInfo(currmenu, itemID, FALSE, &iteminfo);
    if (! call_ok)
        throw std::runtime_error("ApplicationWindowThingy: call to "
        "SetMenuItemInfo( ) failed when attempting to enable menu item");
}



void  ApplicationWindowThingy::disableMenuItem(int itemID)
{
    MENUITEMINFO iteminfo;
    iteminfo.cbSize = sizeof(MENUITEMINFO);
    iteminfo.fMask = MIIM_STATE;
    iteminfo.fState = MFS_DISABLED;
    iteminfo.wID = 0;
    iteminfo.hSubMenu = 0;
    iteminfo.hbmpChecked = 0;
    iteminfo.hbmpUnchecked = 0;
    iteminfo.dwItemData = 0;
    iteminfo.dwTypeData = 0;
    iteminfo.cch = 0;
    iteminfo.hbmpItem = 0;

    HMENU  currmenu = GetMenu(fFrameAdapter->primitive( ));

    if (currmenu == NULL)
        throw std::logic_error("ApplicationWindowThingy: can't disable menu "
        "item: no menu is installed");

    BOOL call_ok = SetMenuItemInfo(currmenu, itemID, FALSE, &iteminfo);
    if (! call_ok)
        throw std::runtime_error("ApplicationWindowThingy: call to "
        "SetMenuItemInfo( ) failed when attempting to disable menu item");
}




StatusBarPaneToken   ApplicationWindowThingy::appendStatusPane(int length)
{
    int  partinfo[256];

    int  numparts = SendMessage(fStatusBarAdapter->primitive( ), SB_GETPARTS,
        256, (LPARAM)(partinfo));

    if ((numparts == 1) && (partinfo[0] == -1))
        numparts = 0;

    if (numparts == 0)
        partinfo[numparts] = length;
    else
        partinfo[numparts] = length + partinfo[numparts - 1];

    numparts++;

    SendMessage(fStatusBarAdapter->primitive( ), SB_SETPARTS,
        (WPARAM)(numparts), (LPARAM)(partinfo));

    return (numparts - 1);
}



void  ApplicationWindowThingy::setStatusPaneText(StatusBarPaneToken pane,
    const std::string& text)
{
    std::string pad = "";
    if (pane == 0)
        pad = "   ";

    std::string  paddedtext = pad + text;

    BOOL  call_ok = SendMessage(fStatusBarAdapter->primitive( ), SB_SETTEXT,
        (WPARAM)(pane), (LPARAM)(paddedtext.c_str( )));

    if (! call_ok)
        throw std::runtime_error("ApplicationWindowThingy: attempt to send "
        "text-setting message to status bar control failed");
}



void  ApplicationWindowThingy::setScrollbarMinorIncrement(
    const ScrollbarSelector& bar, int incr)
{
    if (bar == kHorizontalScrollbar)
        fHorizScrollMinorIncr = incr;
    else if (bar == kVerticalScrollbar)
        fVertScrollMinorIncr = incr;
}




void  ApplicationWindowThingy::uInterpretScrollAction(
    const ScrollbarSelector& bar, int actioncode)
{
    int  currpos = this->scrollbarPosition(bar);
    int  newpos;
    int  trackpos;

    if (bar == kHorizontalScrollbar) {

        switch (actioncode) {

            case SB_LINELEFT:
                newpos = currpos - fHorizScrollMinorIncr;
                this->setScrollbarPosition(kHorizontalScrollbar, newpos);
                newpos = this->scrollbarPosition(kHorizontalScrollbar);
                if (newpos != currpos)
                    fController->thingyScrollbarPositioned(*this, bar);
            break;

            case SB_LINERIGHT:
                newpos = currpos + fHorizScrollMinorIncr;
                this->setScrollbarPosition(kHorizontalScrollbar, newpos);
                newpos = this->scrollbarPosition(kHorizontalScrollbar);
                if (newpos != currpos)
                    fController->thingyScrollbarPositioned(*this, bar);
            break;

            case SB_PAGELEFT:
                newpos = currpos - fHorizScrollMajorIncr;
                this->setScrollbarPosition(kHorizontalScrollbar, newpos);
                newpos = this->scrollbarPosition(kHorizontalScrollbar);
                if (newpos != currpos)
                    fController->thingyScrollbarPositioned(*this, bar);
            break;

            case SB_PAGERIGHT:
                newpos = currpos + fHorizScrollMajorIncr;
                this->setScrollbarPosition(kHorizontalScrollbar, newpos);
                newpos = this->scrollbarPosition(kHorizontalScrollbar);
                if (newpos != currpos)
                    fController->thingyScrollbarPositioned(*this, bar);
            break;

            case SB_THUMBTRACK:
                trackpos = uGetScrollTrackValue(kHorizontalScrollbar);
                fController->thingyScrollbarTracked(*this, bar, trackpos);
            break;

            case SB_THUMBPOSITION:
                newpos = uGetScrollTrackValue(kHorizontalScrollbar);
                this->setScrollbarPosition(kHorizontalScrollbar, newpos);
                fController->thingyScrollbarPositioned(*this, bar);
            break;
        } /* switch */
    } /* if horizontal scrollbar was actuated */
    else if (bar == kVerticalScrollbar) {

        switch (actioncode) {

            case SB_LINEUP:
                newpos = currpos - fVertScrollMinorIncr;
                this->setScrollbarPosition(bar, newpos);
                newpos = this->scrollbarPosition(bar);
                if (newpos != currpos)
                    fController->thingyScrollbarPositioned(*this, bar);
            break;

            case SB_LINEDOWN:
                newpos = currpos + fVertScrollMinorIncr;
                this->setScrollbarPosition(bar, newpos);
                newpos = this->scrollbarPosition(bar);
                if (newpos != currpos)
                    fController->thingyScrollbarPositioned(*this, bar);
            break;

            case SB_PAGEUP:
                newpos = currpos - fVertScrollMajorIncr;
                this->setScrollbarPosition(bar, newpos);
                newpos = this->scrollbarPosition(bar);
                if (newpos != currpos)
                    fController->thingyScrollbarPositioned(*this, bar);
            break;

            case SB_PAGEDOWN:
                newpos = currpos + fVertScrollMajorIncr;
                this->setScrollbarPosition(bar, newpos);
                newpos = this->scrollbarPosition(bar);
                if (newpos != currpos)
                    fController->thingyScrollbarPositioned(*this, bar);
            break;

            case SB_THUMBTRACK:
                trackpos = uGetScrollTrackValue(bar);
                fController->thingyScrollbarTracked(*this, bar, trackpos);
            break;

            case SB_THUMBPOSITION:
                newpos = uGetScrollTrackValue(bar);
                this->setScrollbarPosition(bar, newpos);
                fController->thingyScrollbarPositioned(*this, bar);
            break;
        } /* switch */
    } /* if vertical scrollbar was actuated */
}



int   ApplicationWindowThingy::uGetScrollTrackValue(
    const ScrollbarSelector& bar) const
{
    HWND barwin;
    if (bar == kHorizontalScrollbar)
        barwin = fHorizScrollAdapter->primitive( );
    else if (bar == kVerticalScrollbar)
        barwin = fVertScrollAdapter->primitive( );

    SCROLLINFO barinfo;
    barinfo.cbSize = sizeof(SCROLLINFO);
    barinfo.fMask = SIF_TRACKPOS;
    barinfo.nMin = 0;
    barinfo.nMax = 0;
    barinfo.nPage = 0;
    barinfo.nPos = 0;
    barinfo.nTrackPos = 0;

    BOOL  result_ok = GetScrollInfo(barwin, SB_CTL, &barinfo);
    if (!result_ok)
        throw std::runtime_error("ApplicationWindowThingy: GetScrollInfo( ) "
            "failed in uGetScrollTrackValue( )");

    return barinfo.nTrackPos;
}



void  ApplicationWindowThingy::setMinimumTrackSize(int x, int y)
{
    this->setMinimumTrackSize(Vector2Di(x, y));
}



void  ApplicationWindowThingy::setMinimumTrackSize(Vector2Di& dim)
{
    fMinTrackSize = dim;
}



const Vector2Di&  ApplicationWindowThingy::minimumTrackSize( ) const
{
    return fMinTrackSize;
}



void  ApplicationWindowThingy::installKeys(int accelID)
{
    HACCEL  acceltable =
        LoadAccelerators(Application::instance( ).systemModule( ),
        MAKEINTRESOURCE(accelID));

    if (acceltable == NULL)
        throw std::runtime_error("ApplicationWindowThingy: couldn't install "
            "keys: specified accelerator table not found in system module");

    fInstalledKeys = acceltable;
}



HACCEL  ApplicationWindowThingy::installedKeys( ) const
{
    return fInstalledKeys;
}



bool  ApplicationWindowThingy::areKeysInstalled( ) const
{
    if (fInstalledKeys)
        return true;
    else
        return false;
}



void  ApplicationWindowThingy::wait( )
{
    fIsWaiting = true;

    EnableWindow(adapter( ).primitive( ), FALSE);
}



void  ApplicationWindowThingy::unwait( )
{
    EnableWindow(adapter( ).primitive( ), TRUE);

    fIsWaiting = false;
}



bool  ApplicationWindowThingy::isWaiting( ) const
{
    return fIsWaiting;
}
//////////////////////////////////////////////////////////////////////////////
// END  ApplicationWindowThingy                                             //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  ApplicationWindowController                                       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class  NullApplicationWindowController :
    public virtual ApplicationWindowController {

public:

    NullApplicationWindowController( ) { }
    virtual  ~NullApplicationWindowController( ) { }

    void  thingyClosed(ApplicationWindowThingy&)  { }
    void  thingyMaximized(ApplicationWindowThingy&) { }
    void  thingyRestored(ApplicationWindowThingy&) { }
    void  thingyMenuItemChosen(ApplicationWindowThingy&, int) { }
    void  thingyToolbarButtonPushed(ApplicationWindowThingy&,
        ToolbarButtonToken) { }
    void  thingyToolbarButtonSelected(ApplicationWindowThingy&,
        ToolbarButtonToken) { }
    void  thingyToolbarButtonDeselected(ApplicationWindowThingy&,
        ToolbarButtonToken) { }
    void  thingySized(ApplicationWindowThingy&) { }
    void  layoutSubthingies(CompositeThingy&) { }
    void  enableSubthingies(CompositeThingy&) { }
    void  disableSubthingies(CompositeThingy&) { }
    void  thingyScrollbarPositioned(ApplicationWindowThingy&,
        const ScrollbarSelector&) { }
    void  thingyScrollbarTracked(ApplicationWindowThingy&,
        const ScrollbarSelector&, int) { }
};



ApplicationWindowController*
    ApplicationWindowController::nullController( )
{
    return new NullApplicationWindowController( );
}
//////////////////////////////////////////////////////////////////////////////
// END  ApplicationWindowController                                         //
//////////////////////////////////////////////////////////////////////////////










//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  PushButtonThingy                                                  //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
PushButtonThingy::PushButtonThingy(CompositeThingy& composite)
    : fController(PushButtonController::nullController( ))
{
    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );
    HWND w = Win32Tools::widgetFactory( ).makePushButton(parent.primitive( ),
        nextChildID( ));
    fAdapter.reset(new ThingyAdapter(*this, w));
}





LRESULT  PushButtonThingy::processParentMessage(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    if ((msgcode == WM_COMMAND) &&
        (((HWND) lparam) == fAdapter->primitive( ))) {

        fController->thingyClicked(*this);
        return 0;
    }
    else {

        return DefWindowProc(target, msgcode, wparam, lparam);
    }
}




ThingyAdapter&  PushButtonThingy::adapter( )
{
    return *fAdapter;
}




const ThingyAdapter&  PushButtonThingy::adapter( ) const
{
    return *fAdapter;
}




void  PushButtonThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  PushButtonThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  PushButtonThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  PushButtonThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  PushButtonThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  PushButtonThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  PushButtonThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  PushButtonThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}




unsigned short  PushButtonThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  PushButtonThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  PushButtonThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}




void  PushButtonThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}




bool  PushButtonThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}




PushButtonController&  PushButtonThingy::controller( )
{
    return *fController;
}




const PushButtonController&  PushButtonThingy::controller( ) const
{
    return *fController;
}




PushButtonController&  PushButtonThingy::installController(
        PushButtonController& newctlr)
{
    PushButtonController* oldctlr = fController;
    fController = & newctlr;
    return *oldctlr;
}




std::string  PushButtonThingy::title( ) const
{
    return Win32Tools::getWindowTextString(fAdapter->primitive( ));
}




void  PushButtonThingy::setTitle(const std::string& t)
{
    Win32Tools::setWindowTextString(fAdapter->primitive( ), t);
}
//////////////////////////////////////////////////////////////////////////////
// END  PushButtonThingy                                                    //
//////////////////////////////////////////////////////////////////////////////









//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  PushButtonController                                              //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class  NullPushButtonController :
    public virtual PushButtonController {

public:

    NullPushButtonController( ) { }
    virtual  ~NullPushButtonController( ) { }

    void  thingyClicked(PushButtonThingy& sender) { }
};

PushButtonController*
    PushButtonController::nullController( )
{
    return new NullPushButtonController( );
}
//////////////////////////////////////////////////////////////////////////////
// END  PushButtonController                                                //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  GroupBoxThingy                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
bool  GroupBoxThingy::sIsClassRegistered = false;


GroupBoxThingy::GroupBoxThingy(CompositeThingy& composite)
    : fController(CompositeThingyController::nullController( ))
{
    if (!sIsClassRegistered) {

        WNDCLASS    workPaneClass;

        workPaneClass.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
        workPaneClass.lpfnWndProc = ThingyAdapter::sUniversalProc;
        workPaneClass.cbClsExtra = 0;
        workPaneClass.cbWndExtra = 0;
        workPaneClass.hInstance = NULL;
        workPaneClass.hIcon = NULL;
        workPaneClass.hCursor = LoadCursor(NULL, IDC_ARROW);
        workPaneClass.hbrBackground = (HBRUSH)(COLOR_BTNFACE + 1);
        workPaneClass.lpszMenuName = NULL;
        workPaneClass.lpszClassName = "class_GroupBoxWorkPane";

        ATOM  registrationResult = RegisterClass(&workPaneClass);
        if (!registrationResult)
            throw std::runtime_error("GroupBoxThingy: failed to register "
                "work pane window class");
        sIsClassRegistered = true;
    }

    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );

    HWND ow = CreateWindow(TEXT("class_GroupBoxWorkPane"), TEXT(""),
        WS_CHILD, 2, 10, 64, 32, parent.primitive( ), (HMENU)(nextChildID( )),
        NULL, NULL);
    fOutsetPaneAdapter.reset(new ThingyAdapter(*this, ow));

    HWND gw = Win32Tools::widgetFactory( ).makeGroupBox(ow, nextChildID( ));
    fGroupFrameAdapter.reset(new ThingyAdapter(*this, gw));
    ShowWindow(gw, SW_SHOW);

    HWND iw = CreateWindow(TEXT("class_GroupBoxWorkPane"), TEXT(""),
        WS_CHILD | WS_VISIBLE, 0, 0, 64, 32, ow, (HMENU)(nextChildID( )),
        NULL, NULL);
    fInsetPaneAdapter.reset(new ThingyAdapter(*this, iw));

    uLayoutInnerPanes( );
}




LRESULT  GroupBoxThingy::processMessage(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    switch (msgcode) {

        case WM_SIZE:
            uLayoutInnerPanes( );
            fController->layoutSubthingies(*this);
            return 0;
        break;

        case WM_HSCROLL:
        case WM_COMMAND:
            if (lparam) /* only control msgs have non-null lparams */
                return fInsetPaneAdapter->propogateChildMessage(target,
                    msgcode, wparam, lparam);
            else
                return DefWindowProc(target, msgcode, wparam, lparam);
        break;

        case WM_PAINT:
            return DefWindowProc(target, msgcode, wparam, lparam);
        break;

        default:
            return DefWindowProc(target, msgcode, wparam, lparam);
        break;
    }
}




ThingyAdapter&  GroupBoxThingy::adapter( )
{
    return *fOutsetPaneAdapter;
}




const ThingyAdapter&  GroupBoxThingy::adapter( ) const
{
    return *fOutsetPaneAdapter;
}




ThingyAdapter&  GroupBoxThingy::composerAdapter( )
{
    return *fInsetPaneAdapter;
}




const ThingyAdapter&  GroupBoxThingy::composerAdapter( ) const
{
    return *fInsetPaneAdapter;
}




void  GroupBoxThingy::show( )
{
    ShowWindow(fOutsetPaneAdapter->primitive( ), SW_SHOW);
    ShowWindow(fInsetPaneAdapter->primitive( ), SW_SHOW);
}




void  GroupBoxThingy::hide( )
{
    ShowWindow(fOutsetPaneAdapter->primitive( ), SW_HIDE);
}




bool  GroupBoxThingy::isVisible( ) const
{
    if (IsWindowVisible(fOutsetPaneAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  GroupBoxThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fOutsetPaneAdapter->primitive( ));
}




void  GroupBoxThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fOutsetPaneAdapter->primitive( ),
        ogn.x, ogn.y);
}




void  GroupBoxThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  GroupBoxThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fOutsetPaneAdapter->primitive( ));
}




void  GroupBoxThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fOutsetPaneAdapter->primitive( ), w);
    uLayoutInnerPanes( );
}




unsigned short  GroupBoxThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fOutsetPaneAdapter->primitive( ));
}




void  GroupBoxThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fOutsetPaneAdapter->primitive( ), h);
    uLayoutInnerPanes( );
}




void  GroupBoxThingy::enable( )
{
    EnableWindow(fOutsetPaneAdapter->primitive( ), TRUE);
}




void  GroupBoxThingy::disable( )
{
    EnableWindow(fOutsetPaneAdapter->primitive( ), FALSE);
}




bool  GroupBoxThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fOutsetPaneAdapter->primitive( )))
        return true;
    else
        return false;
}




CompositeThingyController&  GroupBoxThingy::controller( )
{
    return *fController;
}




const CompositeThingyController&  GroupBoxThingy::controller( ) const
{
    return *fController;
}




CompositeThingyController&  GroupBoxThingy::installController(
        CompositeThingyController& newctlr)
{
    CompositeThingyController* oldctlr = fController;
    fController = & newctlr;
    return *oldctlr;
}




std::string  GroupBoxThingy::title( ) const
{
    return Win32Tools::getWindowTextString(fGroupFrameAdapter->primitive( ));
}




void  GroupBoxThingy::setTitle(const std::string& t)
{
    Win32Tools::setWindowTextString(fGroupFrameAdapter->primitive( ), t);
}




unsigned short  GroupBoxThingy::composableWidth( ) const
{
    GroupBoxThingy* thismut = const_cast<GroupBoxThingy*>(this);

    return Win32Tools::getClientAreaWidth(
        thismut->composerAdapter( ).primitive( ));
}




unsigned short  GroupBoxThingy::composableHeight( ) const
{
    GroupBoxThingy* thismut = const_cast<GroupBoxThingy*>(this);

    return Win32Tools::getClientAreaHeight(
        thismut->composerAdapter( ).primitive( ));
}



void  GroupBoxThingy::uLayoutInnerPanes( )
{
    static  const int kInsetTopMargin = 14;
    static  const int kInsetLeftMargin = 10;
    static  const int kInsetRightMargin = 10;
    static  const int kInsetBottonMargin = 8;

    const int kInsetUnivMargin = 5;

    int frame_wd = Win32Tools::getClientAreaWidth(adapter( ).primitive( ));
    int frame_ht = Win32Tools::getClientAreaHeight(adapter( ).primitive( ));

    Win32Tools::setWindowOrigin(fGroupFrameAdapter->primitive( ), 0, 0);
    Win32Tools::setWindowWidth(fGroupFrameAdapter->primitive( ), frame_wd);
    Win32Tools::setWindowHeight(fGroupFrameAdapter->primitive( ), frame_ht);

    Win32Tools::setWindowOrigin(composerAdapter( ).primitive( ),
        kInsetLeftMargin, kInsetTopMargin);

    int inner_wd = frame_wd - kInsetLeftMargin - kInsetRightMargin;
    int inner_ht = frame_ht - kInsetTopMargin - kInsetBottonMargin;

    Win32Tools::setWindowWidth(composerAdapter( ).primitive( ), inner_wd);
    Win32Tools::setWindowHeight(composerAdapter( ).primitive( ), inner_ht);
}
//////////////////////////////////////////////////////////////////////////////
// END  GroupBoxThingy                                                      //
//////////////////////////////////////////////////////////////////////////////










//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  TextLabelThingy                                                   //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
TextLabelThingy::TextLabelThingy(CompositeThingy& composite)
{
    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );
    HWND w = Win32Tools::widgetFactory( ).makeLabel(parent.primitive( ),
        nextChildID( ));
    fAdapter.reset(new ThingyAdapter(*this, w));
}




ThingyAdapter&  TextLabelThingy::adapter( )
{
    return *fAdapter;
}




const ThingyAdapter&  TextLabelThingy::adapter( ) const
{
    return *fAdapter;
}




void  TextLabelThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  TextLabelThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  TextLabelThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  TextLabelThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  TextLabelThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  TextLabelThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  TextLabelThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  TextLabelThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}




unsigned short  TextLabelThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  TextLabelThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  TextLabelThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}




void  TextLabelThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}




bool  TextLabelThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}




std::string  TextLabelThingy::title( ) const
{
    return Win32Tools::getWindowTextString(fAdapter->primitive( ));
}




void  TextLabelThingy::setTitle(const std::string& t)
{
    Win32Tools::setWindowTextString(fAdapter->primitive( ), t);
}




TextLabelThingy::TextAlignmentMode  TextLabelThingy::alignment( ) const
{
    LONG stylecode = GetWindowLong(fAdapter->primitive( ), GWL_STYLE);

    if (stylecode & SS_RIGHT)
        return kAlignRight;
    else
        return kAlignLeft;
}




void  TextLabelThingy::setAlignment(TextLabelThingy::TextAlignmentMode mode)
{
    LONG stylecode = GetWindowLong(fAdapter->primitive( ), GWL_STYLE);

    if (mode == kAlignLeft) {

        stylecode = stylecode & (~SS_RIGHT);
        stylecode = stylecode | SS_LEFT;
    }
    else {

        stylecode = stylecode & (~SS_LEFT);
        stylecode = stylecode | SS_RIGHT;
    }

    SetWindowLong(fAdapter->primitive( ), GWL_STYLE, stylecode);
}
//////////////////////////////////////////////////////////////////////////////
// END  TextLabelThingy                                                     //
//////////////////////////////////////////////////////////////////////////////









//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  CheckBoxThingy                                                    //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
CheckBoxThingy::CheckBoxThingy(CompositeThingy& composite)
{
    fController = CheckBoxController::nullController( );

    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );
    HWND w = Win32Tools::widgetFactory( ).makeCheckBox(parent.primitive( ),
        nextChildID( ));
    fAdapter.reset(new ThingyAdapter(*this, w));
}



ThingyAdapter&  CheckBoxThingy::adapter( )
{
    return *fAdapter;
}



const ThingyAdapter&  CheckBoxThingy::adapter( ) const
{
    return *fAdapter;
}




LRESULT  CheckBoxThingy::processParentMessage(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    switch (msgcode) {

        case WM_COMMAND:
            if (HIWORD(wparam) == BN_CLICKED) {

                if (isChecked( ))
                    setUnchecked( );
                else
                    setChecked( );

                if (isChecked( ))
                    fController->thingyChecked(*this);
                else
                    fController->thingyUnchecked(*this);

                return 0;
            }
            else {

                return DefWindowProc(target, msgcode, wparam, lparam);
            }
        break;

        default:
            return DefWindowProc(target, msgcode, wparam, lparam);
        break;
    }
}




void  CheckBoxThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  CheckBoxThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  CheckBoxThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  CheckBoxThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  CheckBoxThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  CheckBoxThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  CheckBoxThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  CheckBoxThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}




unsigned short  CheckBoxThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  CheckBoxThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  CheckBoxThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}




void  CheckBoxThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}




bool  CheckBoxThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}




std::string  CheckBoxThingy::title( ) const
{
    return Win32Tools::getWindowTextString(fAdapter->primitive( ));
}




void  CheckBoxThingy::setTitle(const std::string& t)
{
    Win32Tools::setWindowTextString(fAdapter->primitive( ), t);
}




bool  CheckBoxThingy::isChecked( ) const
{
    LRESULT  checkstatus = SendMessage(fAdapter->primitive( ), BM_GETCHECK,
        0, 0);

    if (checkstatus == BST_CHECKED)
        return true;
    else
        return false;
}




void  CheckBoxThingy::setChecked( )
{
    SendMessage(fAdapter->primitive( ), BM_SETCHECK, BST_CHECKED, 0);
}




void  CheckBoxThingy::setUnchecked( )
{
    SendMessage(fAdapter->primitive( ), BM_SETCHECK, BST_UNCHECKED, 0);
}




CheckBoxController&  CheckBoxThingy::controller( )
{
    return *fController;
}




const CheckBoxController&  CheckBoxThingy::controller( ) const
{
    return *fController;
}




CheckBoxController&  CheckBoxThingy::installController(CheckBoxController&
    newctlr)
{
    CheckBoxController* oldctlr = fController;
    fController = & newctlr;
    return *oldctlr;
}
//////////////////////////////////////////////////////////////////////////////
// END  CheckBoxThingy                                                      //
//////////////////////////////////////////////////////////////////////////////









//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  CheckBoxController                                                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
class NullCheckBoxController
    : public virtual CheckBoxController {

public:

    NullCheckBoxController( ) { }

    ~NullCheckBoxController( ) { }

    void  thingyChecked(CheckBoxThingy& sender) { }
    void  thingyUnchecked(CheckBoxThingy& sender) { }
};


CheckBoxController*  CheckBoxController::nullController( )
{
    return new NullCheckBoxController( );
}
//////////////////////////////////////////////////////////////////////////////
// END  CheckBoxController                                                  //
//////////////////////////////////////////////////////////////////////////////









//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  TextEntryThingy                                                   //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
TextEntryThingy::TextEntryThingy(CompositeThingy& composite)
{
    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(composite);
    ThingyAdapter& parent = composite_tac.composerAdapter( );
    HWND w = Win32Tools::widgetFactory( ).makeEditArea(parent.primitive( ),
        nextChildID( ));
    fAdapter.reset(new ThingyAdapter(*this, w));
}




ThingyAdapter&  TextEntryThingy::adapter( )
{
    return *fAdapter;
}




const ThingyAdapter&  TextEntryThingy::adapter( ) const
{
    return *fAdapter;
}




void  TextEntryThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}




void  TextEntryThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}




bool  TextEntryThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}




Vector2Di  TextEntryThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}




void  TextEntryThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}




void  TextEntryThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}




unsigned short  TextEntryThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}




void  TextEntryThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}





unsigned short  TextEntryThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}




void  TextEntryThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}




void  TextEntryThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}




void  TextEntryThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}




bool  TextEntryThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}




std::string  TextEntryThingy::contents( ) const
{
    return Win32Tools::getWindowTextString(fAdapter->primitive( ));
}




void  TextEntryThingy::setContents(const std::string& t)
{
    Win32Tools::setWindowTextString(fAdapter->primitive( ), t);
}
//////////////////////////////////////////////////////////////////////////////
// END  TextEntryThingy                                                     //
//////////////////////////////////////////////////////////////////////////////








/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  DialogThingy                                                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
DialogThingy*  DialogThingy::sObjAwaitingAssoc = 0;


INT_PTR  CALLBACK  DialogThingy::sDialogProc(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    DialogThingy*  target_obj = 0;

    switch (msgcode) {

        case WM_INITDIALOG:
            if (sObjAwaitingAssoc == 0)
                throw std::logic_error("DialogThingy: dialog procedure got "
                    "an init message, but no object needs association");
            sObjAwaitingAssoc->fAdapter.reset(new ThingyAdapter(
                *sObjAwaitingAssoc, target));
            target_obj = & sGetObject(target);
            sObjAwaitingAssoc = 0;
            target_obj->fController->thingyInteractionStarted(*target_obj);
            return TRUE;
        break;

        default:
            if (ThingyAdapter::isPrimitiveAssociated(target)) {

                target_obj = & sGetObject(target);

                if (target_obj->fIsInteractionStopping) {

                    target_obj->fController->thingyInteractionStopped(
                        *target_obj);
                    target_obj->fAdapter->release( );
                    target_obj->fAdapter.reset(0);
                    EndDialog(target, 1);
                    target_obj->fIsInteractionStopping = false;
                }
                else {

                    target_obj->processMessage(target, msgcode, wparam,
                        lparam);
                } /* else if the interaction isn't stopping */
            } /* if the primitive is associated */

            return FALSE;
        break;
    }
}



DialogThingy&  DialogThingy::sGetObject(HWND primitive)
{
    ThingyAdapter&  obj_adapter = ThingyAdapter::getObject(primitive);
    DialogThingy&  obj = dynamic_cast<DialogThingy&>(obj_adapter.client( ));
    return obj;
}



DialogThingy::DialogThingy(CompositeThingy& composite, int dialogID)
    : fController(DialogController::nullController( )),
      fParentWindow(&composite), fDialogResID(dialogID),
      fIsInteractionStopping(false)
{
}



DialogThingy::DialogThingy(int dialogID)
    : fController(DialogController::nullController( )),
      fParentWindow(0), fDialogResID(dialogID),
      fIsInteractionStopping(false)
{
}



DialogThingy::~DialogThingy( )
{
}



LRESULT  DialogThingy::processMessage(HWND target, UINT msgcode,
    WPARAM wparam, LPARAM lparam)
{
    HWND  controlhandle = 0;

    switch (msgcode) {

        case WM_CLOSE:
            fController->thingyClosed(*this);
        break;

        case WM_SIZE:
            fController->layoutSubthingies(*this);
        break;

        case WM_COMMAND:
            if (lparam) { /* control messages have non-null lparams */

                fAdapter->propogateChildMessage(target, msgcode,
                    wparam, lparam);
                fController->thingyItemActivated(*this, LOWORD(wparam));
            }
        break;

        case WM_HSCROLL:
            controlhandle = (HWND)(lparam);
            fController->thingyItemActivated(*this,
                GetDlgCtrlID(controlhandle));
        break;
    }

    return 0;
}



ThingyAdapter&  DialogThingy::adapter( )
{
    return *fAdapter;
}



const ThingyAdapter&  DialogThingy::adapter( ) const
{
    return *fAdapter;
}



ThingyAdapter&  DialogThingy::composerAdapter( )
{
    return adapter( );
}



const ThingyAdapter&  DialogThingy::composerAdapter( ) const
{
    return adapter( );
}



void  DialogThingy::show( )
{
    ShowWindow(fAdapter->primitive( ), SW_SHOW);
}



void  DialogThingy::hide( )
{
    ShowWindow(fAdapter->primitive( ), SW_HIDE);
}



bool  DialogThingy::isVisible( ) const
{
    if (IsWindowVisible(fAdapter->primitive( )))
        return true;
    else
        return false;
}



Vector2Di  DialogThingy::origin( ) const
{
    return Win32Tools::getWindowOrigin(fAdapter->primitive( ));
}



void  DialogThingy::setOrigin(const Vector2Di& ogn)
{
    Win32Tools::setWindowOrigin(fAdapter->primitive( ), ogn.x, ogn.y);
}



void  DialogThingy::setOrigin(unsigned short x, unsigned short y)
{
    setOrigin(Vector2Di(x, y));
}



unsigned short  DialogThingy::width( ) const
{
    return Win32Tools::getWindowWidth(fAdapter->primitive( ));
}



void  DialogThingy::setWidth(unsigned short w)
{
    Win32Tools::setWindowWidth(fAdapter->primitive( ), w);
}



unsigned short  DialogThingy::height( ) const
{
    return Win32Tools::getWindowHeight(fAdapter->primitive( ));
}



void  DialogThingy::setHeight(unsigned short h)
{
    Win32Tools::setWindowHeight(fAdapter->primitive( ), h);
}



void  DialogThingy::enable( )
{
    EnableWindow(fAdapter->primitive( ), TRUE);
}



void  DialogThingy::disable( )
{
    EnableWindow(fAdapter->primitive( ), FALSE);
}



bool  DialogThingy::isEnabled( ) const
{
    if (IsWindowEnabled(fAdapter->primitive( )))
        return true;
    else
        return false;
}



unsigned short  DialogThingy::composableWidth( ) const
{
    return Win32Tools::getClientAreaWidth(fAdapter->primitive( ));
}



unsigned short  DialogThingy::composableHeight( ) const
{
    return Win32Tools::getClientAreaHeight(fAdapter->primitive( ));
}



DialogController&  DialogThingy::controller( )
{
    return *fController;
}



const DialogController&  DialogThingy::controller( ) const
{
    return *fController;
}



DialogController&  DialogThingy::installController(DialogController& newctlr)
{
    DialogController* oldctlr = fController;
    fController = & newctlr;
    return *oldctlr;
}



std::string  DialogThingy::title( ) const
{
    return Win32Tools::getWindowTextString(fAdapter->primitive( ));
}



void  DialogThingy::setTitle(const std::string& t)
{
    Win32Tools::setWindowTextString(fAdapter->primitive( ), t);
}



void  DialogThingy::startInteraction( )
{
    CompositeThingyAdapterClient& composite_tac =
        dynamic_cast<CompositeThingyAdapterClient&>(*fParentWindow);

    sObjAwaitingAssoc = this;

    if (fParentWindow) {

        ThingyAdapter& parent_adapter = composite_tac.composerAdapter( );

        INT_PTR  errorcode =
            DialogBox(Application::instance( ).systemModule( ),
            MAKEINTRESOURCE(fDialogResID), parent_adapter.primitive( ),
            sDialogProc);
        if ((errorcode == 0) || (errorcode == -1))
            throw std::runtime_error("DialogThingy: couldn't start "
                "interaction: call to DialogBox( ) failed");
    }
    else {

        INT_PTR  errorcode =
            DialogBox(Application::instance( ).systemModule( ),
            MAKEINTRESOURCE(fDialogResID), NULL, sDialogProc);
        if ((errorcode == 0) || (errorcode == -1))
            throw std::runtime_error("DialogThingy: couldn't start "
                "interaction: call to DialogBox( ) failed");
    }
}



void  DialogThingy::stopInteraction( )
{
    if (! isInteractionRunning( ))
        throw std::logic_error("DialogThingy: can't stop interaction: "
            "interaction isn't running");

    fIsInteractionStopping = true;
    SendMessage(fAdapter->primitive( ), 0, 0, 0);
}



bool  DialogThingy::isInteractionRunning( ) const
{
    if (fAdapter.get( ))
        return true;
    else
        return false;
}



void  DialogThingy::themeItem(int itemID)
{
    Win32Tools::applyControlFaceProperties(GetDlgItem(fAdapter->primitive( ),
        itemID));
}



std::string  DialogThingy::itemText(int itemID) const
{
    return Win32Tools::getWindowTextString(GetDlgItem(fAdapter->primitive( ),
        itemID));
}



void  DialogThingy::setItemText(int itemID, const std::string& text)
{
    HWND  targetwin = GetDlgItem(fAdapter->primitive( ), itemID);

    Win32Tools::setWindowTextString(targetwin, text);
}



void  DialogThingy::enableItem(int itemID)
{
    HWND  itemhandle = GetDlgItem(fAdapter->primitive( ), itemID);

    EnableWindow(itemhandle, TRUE);
}



void  DialogThingy::disableItem(int itemID)
{
    HWND  itemhandle = GetDlgItem(fAdapter->primitive( ), itemID);

    EnableWindow(itemhandle, FALSE);
}



bool  DialogThingy::isItemEnabled(int itemID) const
{
    HWND  itemhandle = GetDlgItem(fAdapter->primitive( ), itemID);

    if (IsWindowEnabled(itemhandle))
        return true;
    else
        return false;
}
/////////////////////////////////////////////////////////////////////////////
// END  DialogThingy                                                       //
/////////////////////////////////////////////////////////////////////////////









/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// CLASS  DialogController                                                 //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
class  NullDialogController :
    public virtual DialogController {

public:

    NullDialogController( ) { }
    ~NullDialogController( ) { }

    void  thingyItemActivated(DialogThingy&, int) { }
    void  thingyInteractionStarted(DialogThingy&) { }
    void  thingyInteractionStopped(DialogThingy&) { }
    void  thingyClosed(DialogThingy& sender) { }
    void  layoutSubthingies(CompositeThingy&) { }
    void  enableSubthingies(CompositeThingy&) { }
    void  disableSubthingies(CompositeThingy&) { }
};


DialogController*  DialogController::nullController( )
{
    return new NullDialogController( );
}
/////////////////////////////////////////////////////////////////////////////
// END  DialogController                                                   //
/////////////////////////////////////////////////////////////////////////////
