//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ControllerScrapbook.cpp                                                  //
//                                                                          //
// Implements methods in classes SourceImageController and                  //
// PyramidLevelController.                                                  //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver. 1.4.1 of Mon 16 June 2008 @ 11:50pm EDT                             //
//                                                                          //
//    Fixed:  fixed hidden widget problems in class PyramidLevelController  //
//                                                                          //
// ver. 1.4.0 of Tue 20 May 2008 @ 2:25pm EDT                               //
//                                                                          //
//    New:    added new source image file path access functionality.        //
//    Fixed:  fixed hidden widget problem by adding ShowWindow( ) calls     //
//            immediately following calls to widget factory creation        //
//            routines; added a do-nothing cancelAction( ) method to        //
//            comply with changes in the InterstitialActionSender interface //
//                                                                          //
// ver.  1.3.2 of Wednesday 27 March 2008 @ 7:44pm EDT                      //
//                                                                          //
//    Added methods viewEnabled( ) and viewDisabled( ) to class             //
//    PyramidLevelController to make it compliant with its superclass       //
//    WorkPaneController.                                                   //
//                                                                          //
// older versions:                                                          //
//                                                                          //
//     modification log elided for versions several generations old;        //
//     checkout older revisions from CVS repository for complete changelog  //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2007-2008, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "ControllerScrapbook.h"
#include "ImageRepresentation.h"
#include "Application.h"
#include <cstdio>
#include <sstream>

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  SourceImageController                                             //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
const float  SourceImageController::kNoAspectRatioConstraint = -1.0f;

SourceImageController::SourceImageController(
    SourceImageControllerParent& par)
        : fParent(&par), fTitleLabel(0), fFilenameLabel(0), fSetButton(0),
          fSrcImgFileChooser(new OpenFileChooser( )), fSrcImgData(0),
          fReadoutLeftEdge(kTitleLabelWidth + kSpacePixels), fManagedView(0),
          fWidthMinConst(kImageMinWidth), fWidthMaxConst(kImageMaxWidth),
          fHeightMinConst(kImageMinHeight), fHeightMaxConst(kImageMaxHeight),
          fAspectRatioConst(kNoAspectRatioConstraint),
          fIsInspectionEnabled(false), fInspectButton(0),
          fButtonSpacing(kDefaultButtonSpacing), fImageInspector(0),
          fButtonWidth(kDefaultButtonWidth), fIsPowerOf2ConstEnabled(false)
{
    fSrcImgFileChooser->setTitle("Choose a Source Image");
    fSrcImgFileChooser->addFiletypeMask("Truevision TARGA (*.tga; *.tpic)",
        "*.tga");
    fSrcImgFileChooser->addFiletypeMask("Truevision TARGA (*.tga; *.tpic)",
        "*.tpic");
}




SourceImageController::~SourceImageController( )
{
    if (isInstalled( ))
        fManagedView->installController(
            * WorkPaneController::nullController( ));

    delete fSrcImgFileChooser;
    fSrcImgFileChooser = 0;
}




void  SourceImageController::installed(WorkPaneView& sender)
{
    if (isInstalled( ))
        throw std::logic_error("SourceImageController: source image "
            "controllers don't support multiple installation");

    sender.setHeight(kDefaultPaneHeight);
    sender.setWidth(kDefaultPaneWidth);

    int  idticker = WorkPaneView::kChildID0;

    fTitleLabel =
        Win32Tools::widgetFactory( ).makeBoldLabel(sender.peerHandle( ),
        idticker++);
    Win32Tools::setWindowTextString(fTitleLabel, "Source Image:");
    Win32Tools::setWindowWidth(fTitleLabel, kTitleLabelWidth);
    ShowWindow(fTitleLabel, SW_SHOW);

    fFilenameLabel =
        Win32Tools::widgetFactory( ).makeLabel(sender.peerHandle( ),
        idticker++);
    Win32Tools::setWindowTextString(fFilenameLabel, "none loaded.");
    Win32Tools::setWindowWidth(fFilenameLabel, kFilenameLabelWidth);
    ShowWindow(fFilenameLabel, SW_SHOW);

    fSetButton =
        Win32Tools::widgetFactory( ).makePushButton(sender.peerHandle( ),
        idticker++);
    Win32Tools::setWindowTextString(fSetButton, "Set...");
    Win32Tools::setWindowWidth(fSetButton, fButtonWidth);
    ShowWindow(fSetButton, SW_SHOW);

    if (isInspectionEnabled( )) {

        fInspectButton =
            Win32Tools::widgetFactory( ).makePushButton(sender.peerHandle( ),
            kInspectButtonChildID);
        Win32Tools::setWindowTextString(fInspectButton, "Inspect...");
        Win32Tools::setWindowWidth(fInspectButton, fButtonWidth);
        ShowWindow(fInspectButton, SW_SHOW);

        if (!isSourceImageLoaded( ))
            EnableWindow(fInspectButton, FALSE);
    }

    fManagedView = & sender;
}




void  SourceImageController::deinstalled(WorkPaneView& sender)
{
    fManagedView = 0;

    Win32Tools::destroyPrimitive(fTitleLabel);
    Win32Tools::destroyPrimitive(fFilenameLabel);

    if (fInspectButton)
        Win32Tools::destroyPrimitive(fInspectButton);

    if (fImageInspector) {

        delete fImageInspector;
        fImageInspector = 0;
    }

    Win32Tools::destroyPrimitive(fSetButton);

    if (fSrcImgData) {
        delete fSrcImgData;
        fSrcImgData = 0;
    }
}




void  SourceImageController::layout(WorkPaneView& sender)
{
    if (isInstalled( )) {

        int  pane_ht = sender.height( );
        int  pane_wd = sender.width( );
        int  setbut_ht = Win32Tools::getWindowHeight(fSetButton);
        int  setbut_wd = Win32Tools::getWindowWidth(fSetButton);
        int  title_ht = Win32Tools::getWindowHeight(fTitleLabel);
        int  title_wd = Win32Tools::getWindowWidth(fTitleLabel);
        int  fn_ht = Win32Tools::getWindowHeight(fFilenameLabel);
        int  insbut_wd = 0;
        if (isInspectionEnabled( ))
            insbut_wd = Win32Tools::getWindowWidth(fInspectButton);
        int  effspace = 0;
        if (isInspectionEnabled( ))
            effspace = fButtonSpacing;

        Win32Tools::setWindowOrigin(fTitleLabel, 0,
            (pane_ht - title_ht) / 2);

        Win32Tools::setWindowOrigin(fFilenameLabel, fReadoutLeftEdge,
            (pane_ht - fn_ht) / 2);
        Win32Tools::setWindowWidth(fFilenameLabel, pane_wd - setbut_wd -
            effspace - insbut_wd - fReadoutLeftEdge);

        Win32Tools::setWindowOrigin(fSetButton, pane_wd - setbut_wd -
            insbut_wd - effspace, (pane_ht - setbut_ht) / 2);

        if (isInspectionEnabled( )) {

            Win32Tools::setWindowOrigin(fInspectButton, pane_wd - insbut_wd,
                (pane_ht - setbut_ht) / 2);

        } /* if inspection is enabled */
    } /* if this controller is installed to manage a view */
}




void  SourceImageController::commandPropagated(WorkPaneView& sender,
    HWND commandSource, unsigned short commandID)
{
    if (commandSource == fSetButton) {

        uRunImageSetInteraction( );
    }
    else if (commandSource == fInspectButton) {

        if (!fImageInspector)
            fImageInspector = new ImageInspector(
                fParent->dialogOverlayPeer( ), *fSrcImgData);

        fImageInspector->runInspectionInteraction( );
    }
}




void  SourceImageController::viewEnabled(UIComponentView& sender)
{
    EnableWindow(fSetButton, TRUE);

    if (isInspectionEnabled( ) && isSourceImageLoaded( ))
        EnableWindow(fInspectButton, TRUE);
}




void  SourceImageController::viewDisabled(UIComponentView& sender)
{
    EnableWindow(fSetButton, FALSE);

    if (isInspectionEnabled( ))
        EnableWindow(fInspectButton, FALSE);
}




void  SourceImageController::uRunImageSetInteraction( )
{
    fParent->notifySourceImageInteractionStarted(*this);

    std::string  imgfn = fSrcImgFileChooser->runFileChoiceInteraction(
        fParent->dialogOverlayPeer( ));

    /* if the image filename 'imgfn' returned is the null string, then the
       user clicked 'Cancel' in the 'Open...' dialog box before choosing an
       image file to open, so just return */
    if (imgfn == "") {

        fParent->notifySourceImageInteractionFinished(*this);
        return;
    }

    /* try to load image data from the the file the user selected */
    RGBAImage*  userimg = 0;
    try {

        userimg = RGBAImage::createMappedImage(RGBAImage::kTargaImageFormat,
            imgfn, fReceiverSet, *this);
    }
    catch (std::runtime_error&) {

        const std::string&  appname = Application::instance( ).name( );
        MessageBox(NULL, TEXT("The file selected does not contain "
            "valid TARGA image data"), appname.c_str( ), MB_ICONERROR);

        delete userimg;
        fParent->notifySourceImageInteractionFinished(*this);
        return;
    }

    /* if execution reaches this point, then we were able read image data
       from the file chosen by the user sucessfully. But we're still not in
       the clear, since the image setting interaction can still fail if the
       image chosen by the user doesn't meet the constraints specified for
       *this, so let's check it against these constraints now */

    if ((userimg->width( ) < fWidthMinConst) ||
        (userimg->width( ) > fWidthMaxConst)) {

        const std::string&  appname = Application::instance( ).name( );
        std::ostringstream  msgstream;
        msgstream << "The image you've chosen is too wide.\n" <<
            "Valid source images have widths between " << fWidthMinConst <<
            " pixels and " << fWidthMaxConst << " pixels, inclusive.";

        MessageBox(NULL, msgstream.str( ).c_str( ), appname.c_str( ),
            MB_ICONERROR);

        delete userimg;
        fParent->notifySourceImageInteractionFinished(*this);
        return;
    }

    if ((userimg->height( ) < fHeightMinConst) ||
        (userimg->height( ) > fHeightMaxConst)) {

        const std::string&  appname = Application::instance( ).name( );
        std::ostringstream  msgstream;
        msgstream << "The image you've chosen is too tall.\n" <<
            "Valid source images have heights between " << fHeightMinConst <<
            " pixels and " << fHeightMaxConst << " pixels, inclusive.";

        MessageBox(NULL, msgstream.str( ).c_str( ), appname.c_str( ),
            MB_ICONERROR);

        delete userimg;
        fParent->notifySourceImageInteractionFinished(*this);
        return;
    }

    if (fAspectRatioConst != kNoAspectRatioConstraint) {

        float imgw = static_cast<float>(userimg->width( ));
        float imgh = static_cast<float>(userimg->height( ));
        float rat = (imgw / imgh);

        if (rat != fAspectRatioConst) {

            const std::string&  appname = Application::instance( ).name( );
            std::ostringstream  msgstream;
            msgstream << "The image you've chosen has the wrong aspect " <<
                "ratio.\nValid source images must have an aspect ratio (" <<
                "width / height) of " << fAspectRatioConst << ".";

            MessageBox(NULL, msgstream.str( ).c_str( ), appname.c_str( ),
                MB_ICONERROR);

            delete userimg;
            fParent->notifySourceImageInteractionFinished(*this);
            return;
        }
    }

    if (fIsPowerOf2ConstEnabled) {

        double lgwidth = std::log(static_cast<double>(userimg->width( ))) /
            std::log(2.0);
        double lgheight = std::log(static_cast<double>(userimg->height( ))) /
            std::log(2.0);

        double fracwidth;
        std::modf(lgwidth, &fracwidth);
        fracwidth = lgwidth - fracwidth;

        double fracheight;
        std::modf(lgheight, &fracheight);
        fracheight = lgheight - fracheight;

        if (fracwidth != 0.0) {

            const std::string&  appname = Application::instance( ).name( );
            std::ostringstream  msgstream;
            msgstream << "The image you've chosen has a width that is not "
                "a power of two.\nValid source images must have widths and "
                "heights that are powers of two.";

            MessageBox(NULL, msgstream.str( ).c_str( ), appname.c_str( ),
                MB_ICONERROR);

            delete userimg;
            fParent->notifySourceImageInteractionFinished(*this);
            return;
        }

        if (fracheight != 0.0) {

            const std::string&  appname = Application::instance( ).name( );
            std::ostringstream  msgstream;
            msgstream << "The image you've chosen has a height that is not "
                "a power of two.\nValid source images must have widths and "
                "heights that are powers of two.";

            MessageBox(NULL, msgstream.str( ).c_str( ), appname.c_str( ),
                MB_ICONERROR);

            delete userimg;
            fParent->notifySourceImageInteractionFinished(*this);
            return;
        }
    }

    /* if execution reaches this point, then none of the constraint tests
       failed, so lets go ahead and set the chosen filename readout and
       notify our parent that the user has chosen a new source image */

    int  lastPathDelimLoc = (int) imgfn.find_last_of("\\/");

    std::string  simpleFilename;
    if (lastPathDelimLoc != std::string::npos)
        simpleFilename = imgfn.substr(lastPathDelimLoc + 1,
            imgfn.length( ) - lastPathDelimLoc - 1);
    else
        simpleFilename = imgfn;

    Win32Tools::setWindowTextString(fFilenameLabel, simpleFilename);

    RGBAImage* oldsrc = fSrcImgData;
    fSrcImgData = userimg;
    if (oldsrc)
        delete oldsrc;
    fSrcImgPath = imgfn;

    if (isInspectionEnabled( )) {

        if (fManagedView->isEnabled( ))
            EnableWindow(fInspectButton, TRUE);

        if (fImageInspector)
            fImageInspector->setTargetImage(*fSrcImgData);
    }

    fParent->notifySourceImageChanged(*this);
    fParent->notifySourceImageInteractionFinished(*this);
}




RGBAImage&  SourceImageController::sourceImageData( )
{
    return *fSrcImgData;
}



bool  SourceImageController::isSourceImageLoaded( ) const
{
    if (fSrcImgData)
        return true;
    else
        return false;
}




bool  SourceImageController::isInstalled( ) const
{
    if (fManagedView)
        return true;
    else
        return false;
}




void  SourceImageController::setReadoutLeftEdge(int le)
{
    if (! isInstalled( )) {

        if ((le < kTitleLabelWidth) || (le > 2048))
            throw std::out_of_range("SourceImageController: label left edge "
                "out of range");

        fReadoutLeftEdge = le;
    }
    else {

        int  pane_wd = fManagedView->width( );
        int  title_wd = Win32Tools::getWindowWidth(fTitleLabel);

        if ((le < title_wd) || (le > pane_wd))
            throw std::out_of_range("SourceImageController: label left edge "
                "out of range");

        fReadoutLeftEdge = le;

        this->layout(*fManagedView);
    }
} 




int   SourceImageController::readoutLeftEdge( ) const
{
    return fReadoutLeftEdge;
}




void  SourceImageController::setWidthConstraint(int minwd, int maxwd)
{
    if (minwd > maxwd)
        throw std::invalid_argument("SourceImageController: minimum "
            "width constraint cannot exceed maximum width constraint");

    if (minwd < kImageMinWidth)
        throw std::out_of_range("SourceImageController: minimum "
            "width constraint out of range");

    if (maxwd > kImageMaxWidth)
        throw std::out_of_range("SourceImageController: maximum "
            "width constraint out of range");

    fWidthMinConst = minwd;
    fWidthMaxConst = maxwd;
}




void  SourceImageController::setHeightConstraint(int minht, int maxht)
{
    if (minht > maxht)
        throw std::invalid_argument("SourceImageController: minimum "
            "height constraint cannot exceed maximum height constraint");

    if (minht < kImageMinHeight)
        throw std::out_of_range("SourceImageController: minimum "
            "height constraint out of range");

    if (maxht > kImageMaxHeight)
        throw std::out_of_range("SourceImageController: maximum "
            "height constraint out of range");

    fHeightMinConst = minht;
    fHeightMaxConst = maxht;
}




void  SourceImageController::setAspectRatioConstraint(float ar)
{
    if (ar != kNoAspectRatioConstraint)
        if (ar <= 0.0)
            throw std::out_of_range("SourceImageController: aspect ratio "
                "constraint out of range");

    fAspectRatioConst = ar;
}




void  SourceImageController::enableInspection( )
{
    fIsInspectionEnabled = true;

    if (isInstalled( )) {

        fInspectButton = Win32Tools::widgetFactory( ).makePushButton(
            fManagedView->peerHandle( ), kInspectButtonChildID);
        ShowWindow(fInspectButton, SW_SHOW);
        Win32Tools::setWindowTextString(fInspectButton, "Inspect...");
        fManagedView->manageChild(fInspectButton);
        Win32Tools::setWindowWidth(fInspectButton, fButtonWidth);

        if (!isSourceImageLoaded( ))
            EnableWindow(fInspectButton, FALSE);

        layout(*fManagedView);
    }
}




void  SourceImageController::disableInspection( )
{
    fIsInspectionEnabled = false;

    if (isInstalled( )) {

        fManagedView->unmanageChild(fInspectButton);
        Win32Tools::destroyPrimitive(fInspectButton);

        layout(*fManagedView);
    }

    if (fImageInspector) {

        delete fImageInspector;
        fImageInspector = 0;
    }
}




bool  SourceImageController::isInspectionEnabled( )
{
    return fIsInspectionEnabled;
}




int   SourceImageController::buttonSpacing( ) const
{
    return fButtonSpacing;
}




void  SourceImageController::setButtonSpacing(int spacing)
{
    fButtonSpacing = spacing;

    if (isInstalled( ))
        layout(*fManagedView);
}




int   SourceImageController::buttonWidth( ) const
{
    return fButtonWidth;
}




void  SourceImageController::setButtonWidth(int bw)
{
    fButtonWidth = bw;

    if (isInstalled( )) {

        Win32Tools::setWindowWidth(fSetButton, bw);

        if (fInspectButton)
            Win32Tools::setWindowWidth(fInspectButton, bw);

        layout(*fManagedView);
    }
}




void  SourceImageController::enablePowerOf2Constraint( )
{
    fIsPowerOf2ConstEnabled = true;
}




void  SourceImageController::disablePowerOf2Constraint( )
{
    fIsPowerOf2ConstEnabled = false;
}




bool  SourceImageController::isPowerOf2ConstraintEnabled( )
{
    return fIsPowerOf2ConstEnabled;
}




void  SourceImageController::registerReceiver(
    InterstitialActionReceiver& recv)
{
    fReceiverSet.insert(&recv);
}



void  SourceImageController::deregisterReceiver(
    InterstitialActionReceiver& recv)
{
    fReceiverSet.erase(&recv);
}




void  SourceImageController::uDispatchIAEvents( )
{
    std::set<InterstitialActionReceiver*>::iterator  iter;
    for (iter = fReceiverSet.begin( ); iter != fReceiverSet.end( ); iter++) {

        InterstitialActionToken  iat("no status");
        (*iter)->interstitialActionEvent(*this, iat);
    }
}




void  SourceImageController::cancelAction(InterstitialActionReceiver& recv)
{
}




const std::string&  SourceImageController::sourceImagePath( ) const
{
    if (! isSourceImageLoaded( ))
        throw std::logic_error("SourceImageController: can't get image path: "
            "no image is loaded");
    else
        return fSrcImgPath;
}
//////////////////////////////////////////////////////////////////////////////
// END  SourceImageController                                               //
//////////////////////////////////////////////////////////////////////////////








//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  PyramidLevelController                                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
PyramidLevelController::PyramidLevelController(
    PyramidLevelControllerParent& par, unsigned short numlevels)
        : fParent(&par), fTitleLabel(0), fLevelLabel(0),
          fNextLevelButton(0), fPrevLevelButton(0), fPyramidLevel(0),
          fNumLevels(numlevels), fManagedView(0),
          fReadoutLeftEdge(kTitleLabelWidth),
          fButtonSpacing(kDefaultButtonSpacing),
          fButtonWidth(kDefaultButtonWidth)
{
    if (numlevels > kMaxNumLevels)
        throw  std::logic_error("PyramidLevelController: can't create new "
            "controller instance: the requested number of pyramid levels "
            "exceeds the maximum permissible number");
}




PyramidLevelController::~PyramidLevelController( )
{
    if (isInstalled( ))
        fManagedView->installController(
            * WorkPaneController::nullController( ));
}




void  PyramidLevelController::installed(WorkPaneView& sender)
{
    if (isInstalled( ))
        throw std::logic_error("PyramidLevelController: pyramid level "
            "controllers don't support multiple installation");

    int  idticker = WorkPaneView::kChildID0;

    fTitleLabel =
        Win32Tools::widgetFactory( ).makeBoldLabel(sender.peerHandle( ),
        idticker++);
    Win32Tools::setWindowTextString(fTitleLabel, "Pyramid Level:");
    Win32Tools::setWindowWidth(fTitleLabel, 100);
    ShowWindow(fTitleLabel, SW_SHOW);

    fLevelLabel =
        Win32Tools::widgetFactory( ).makeLabel(sender.peerHandle( ),
        idticker++);
    Win32Tools::setWindowWidth(fLevelLabel, 20);
    ShowWindow(fLevelLabel, SW_SHOW);

    fNextLevelButton =
        Win32Tools::widgetFactory( ).makePushButton(sender.peerHandle( ),
        idticker++);
    Win32Tools::setWindowTextString(fNextLevelButton, "");
    Win32Tools::setWindowWidth(fNextLevelButton, fButtonWidth);
    sender.manageChild(fNextLevelButton);
    ShowWindow(fNextLevelButton, SW_SHOW);

    fPrevLevelButton =
        Win32Tools::widgetFactory( ).makePushButton(sender.peerHandle( ),
        idticker++);
    Win32Tools::setWindowTextString(fPrevLevelButton, "");
    Win32Tools::setWindowWidth(fPrevLevelButton, fButtonWidth);
    sender.manageChild(fPrevLevelButton);
    ShowWindow(fPrevLevelButton, SW_SHOW);

    sender.setWidth(kDefaultPaneWidth);
    sender.setHeight(kDefaultPaneHeight);

    uWriteLevelToControls( );

    fManagedView = & sender;
}




void  PyramidLevelController::deinstalled(WorkPaneView& sender)
{
    fManagedView = 0;

    sender.unmanageChild(fNextLevelButton);
    sender.unmanageChild(fPrevLevelButton);

    Win32Tools::destroyPrimitive(fTitleLabel);
    Win32Tools::destroyPrimitive(fLevelLabel);
    Win32Tools::destroyPrimitive(fNextLevelButton);
    Win32Tools::destroyPrimitive(fPrevLevelButton);
}




void  PyramidLevelController::layout(WorkPaneView& sender)
{
    // run the layout routine only if we've created all the child
    // window controls -- otherise, don't bother
    if (fTitleLabel && fLevelLabel && fNextLevelButton && fPrevLevelButton) {

        int  pane_wd = sender.width( );
        int  pane_ht = sender.height( );
        int  title_wd = Win32Tools::getWindowWidth(fTitleLabel);
        int  title_ht = Win32Tools::getWindowHeight(fTitleLabel);
        int  level_wd = Win32Tools::getWindowWidth(fLevelLabel);
        int  level_ht = Win32Tools::getWindowHeight(fLevelLabel);
        int  btn_wd = Win32Tools::getWindowWidth(fNextLevelButton);
        int  btn_ht = Win32Tools::getWindowHeight(fNextLevelButton);

        Win32Tools::setWindowOrigin(fNextLevelButton, pane_wd - btn_wd,
            (pane_ht - btn_ht) / 2);

        Win32Tools::setWindowOrigin(fPrevLevelButton, pane_wd - btn_wd -
            fButtonSpacing - btn_wd, (pane_ht - btn_ht) / 2);

        Win32Tools::setWindowOrigin(fTitleLabel, 0, (pane_ht - title_ht)
            / 2);

        Win32Tools::setWindowOrigin(fLevelLabel, fReadoutLeftEdge, (pane_ht -
            level_ht) / 2);
    }
}




void  PyramidLevelController::commandPropagated(WorkPaneView& sender,
    HWND commandSource, unsigned short commandID)
{
    if (commandSource == fNextLevelButton)
        uIncrementPyramidLevel( );
    else if (commandSource == fPrevLevelButton)
        uDecrementPyramidLevel( );
}




void  PyramidLevelController::viewEnabled(UIComponentView& sender)
{
}




void  PyramidLevelController::viewDisabled(UIComponentView& sender)
{
}




int  PyramidLevelController::pyramidLevel( ) const
{
    return fPyramidLevel;
}




void  PyramidLevelController::uIncrementPyramidLevel( )
{
    if (fPyramidLevel < (fNumLevels - 1)) {
        fPyramidLevel++;
        uWriteLevelToControls( );
        fParent->notifyPyramidLevelChanged(*this);
    }
}




void  PyramidLevelController::uDecrementPyramidLevel( )
{
    if (fPyramidLevel > 0) {
        fPyramidLevel--;
        uWriteLevelToControls( );
        fParent->notifyPyramidLevelChanged(*this);
    }
}




void  PyramidLevelController::uWriteLevelToControls( )
{
    char  buffer[4];
    std::sprintf(buffer, "%d", fPyramidLevel);
    Win32Tools::setWindowTextString(fLevelLabel, buffer);

    if (fPyramidLevel == 0)
        EnableWindow(fPrevLevelButton, FALSE);
    else
        EnableWindow(fPrevLevelButton, TRUE);

    if (fPyramidLevel == (fNumLevels - 1))
        EnableWindow(fNextLevelButton, FALSE);
    else
        EnableWindow(fNextLevelButton, TRUE);
}





bool  PyramidLevelController::isInstalled( ) const
{
    if (fManagedView)
        return true;
    else
        return false;
}



void  PyramidLevelController::setPyramidLevel(int lev)
{
    if ((lev < 0) || (lev > (fNumLevels - 1)))
        throw std::out_of_range("PyramidLevelController: level specification "
            "out of range");

    fPyramidLevel = lev;

    uWriteLevelToControls( );
}




int   PyramidLevelController::numLevels( ) const
{
    return fNumLevels;
}




void  PyramidLevelController::setNumLevels(int levs)
{
    if ((levs < 1) || (levs > kMaxNumLevels))
        throw std::out_of_range("PyramidLevelController: number of levels "
            "out of range");

    fNumLevels = levs;

    if (fPyramidLevel > (fNumLevels - 1))
        setPyramidLevel(fNumLevels - 1);
}




int  PyramidLevelController::readoutLeftEdge( ) const
{
    return fReadoutLeftEdge;
}



void  PyramidLevelController::setReadoutLeftEdge(int le)
{
    if (le < kTitleLabelWidth)
        le = kTitleLabelWidth;

    fReadoutLeftEdge = le;

    if (isInstalled( ))
        this->layout(* fManagedView);
}



int   PyramidLevelController::buttonSpacing( ) const
{
    return fButtonSpacing;
}



void  PyramidLevelController::setButtonSpacing(int spacing)
{
    if (spacing < 1)
        spacing = 1;

    fButtonSpacing = spacing;

    if (isInstalled( ))
        this->layout(* fManagedView);
}



int   PyramidLevelController::buttonWidth( ) const
{
    return fButtonWidth;
}



void  PyramidLevelController::setButtonWidth(int bw)
{
    if (bw < kDefaultButtonWidth)
        bw = kDefaultButtonWidth;

    fButtonWidth = bw;

    if (isInstalled( )) {

        Win32Tools::setWindowWidth(fPrevLevelButton, fButtonWidth);
        Win32Tools::setWindowWidth(fNextLevelButton, fButtonWidth);

        this->layout(* fManagedView);
    }
}
//////////////////////////////////////////////////////////////////////////////
// END  PyramidLevelController                                              //
//////////////////////////////////////////////////////////////////////////////







//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// CLASS  TrackerThingyController                                           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

const std::string  TrackerThingyController::kDefaultTitleText
    = "Tracker Thingy:";

TrackerThingyController::TrackerThingyController(
    TrackerThingyControllerParent& par)
        : fParent(&par), fIsInstalled(false), fTitleLabel(0),
          fValueTrackbar(0), fValueEditArea(0),
          fTitleWidth(kDefaultTitleWidth), fTitleText(kDefaultTitleText),
          fEditAreaWidth(kDefaultEditAreaWidth),
          fMinTrackVal(kDefaultMinTrackVal),
          fMaxTrackVal(kDefaultMaxTrackVal),
          fTickIncrement(kDefaultTickIncrement), fManaged(0),
          fTrackerValue(kDefaultMinTrackVal)
{
}




TrackerThingyController::~TrackerThingyController( )
{
    if (isInstalled( ))
        fManaged->installController(* WorkPaneController::nullController( ));
}




void  TrackerThingyController::installed(WorkPaneView& sender)
{
    if (isInstalled( ))
        throw std::logic_error("TrackerThingyController: a controller can't "
            "be multiply installed.");

    int idticker = sender.kChildID0;

    fTitleLabel =
        Win32Tools::widgetFactory( ).makeLabel(sender.peerHandle( ),
        idticker++);
    Win32Tools::setWindowTextString(fTitleLabel, fTitleText);
    Win32Tools::setWindowWidth(fTitleLabel, fTitleWidth);

    fValueTrackbar =
        Win32Tools::widgetFactory( ).makeHorizontalTrackbar(
        sender.peerHandle( ), idticker++);
    SendMessage(fValueTrackbar, TBM_SETRANGE, TRUE,
        MAKELONG(static_cast<short>(fMinTrackVal),
        static_cast<short>(fMaxTrackVal)));
    SendMessage(fValueTrackbar, TBM_SETTICFREQ, 1, 0);

    fValueEditArea = CreateWindowEx(WS_EX_CLIENTEDGE, TEXT("EDIT"), NULL,
        WS_CHILDWINDOW | WS_VISIBLE | ES_LEFT | ES_AUTOVSCROLL |
        ES_MULTILINE | ES_WANTRETURN, 0, 0,
        Win32Tools::kDialogPushButtonWidth, Win32Tools::kEditAreaHeight,
        sender.peerHandle( ), (HMENU) idticker++, NULL, NULL);
    Win32Tools::applyControlFaceProperties(fValueEditArea);
    Win32Tools::setWindowWidth(fValueEditArea, fEditAreaWidth);

    uWriteValueToTrackbar( );
    uWriteValueToEditArea( );

    fManaged = & sender;
    fIsInstalled = true;
}




void  TrackerThingyController::deinstalled(WorkPaneView& sender)
{
    fIsInstalled = false;
    fManaged = 0;

    Win32Tools::destroyPrimitive(fTitleLabel);
    Win32Tools::destroyPrimitive(fValueTrackbar);
    Win32Tools::destroyPrimitive(fValueEditArea);
}




void  TrackerThingyController::layout(WorkPaneView& sender)
{
    static  const unsigned short  kSpacerPixels = 3;

    if (isInstalled( )) {

        int  pane_w = sender.width( );
        int  pane_h = sender.height( );
        int  trackbar_h = Win32Tools::getWindowHeight(fValueTrackbar);
        int  trackbar_top = (pane_h - trackbar_h) / 2;
        int  title_h = Win32Tools::getWindowHeight(fTitleLabel);
        int  title_top = trackbar_top + 4;
        int  edit_h = Win32Tools::getWindowHeight(fValueEditArea);
        int  edit_top = trackbar_top + 4;
        int  trackbar_left = fTitleWidth + kSpacerPixels;
        int  trackbar_right = pane_w - fEditAreaWidth - kSpacerPixels;

        Win32Tools::setWindowOrigin(fTitleLabel, 0, title_top);

        Win32Tools::setWindowOrigin(fValueEditArea, pane_w - fEditAreaWidth,
            edit_top);

        Win32Tools::setWindowOrigin(fValueTrackbar, trackbar_left,
            trackbar_top);
        Win32Tools::setWindowWidth(fValueTrackbar, trackbar_right -
            trackbar_left);
    }
}




void  TrackerThingyController::commandPropagated(WorkPaneView& sender,
    HWND commandSource, unsigned short commandID)
{
    if (commandSource == fValueTrackbar) {

        fTrackerValue = static_cast<int>(SendMessage(fValueTrackbar,
            TBM_GETPOS, 0, 0));

        fParent->notifyTrackerValueChanged(*this);

        uWriteValueToEditArea( );
    }
    else if (commandSource == fValueEditArea) {

        if (commandID == EN_UPDATE)
            uProcessEditAreaChange( );
    }
}




int   TrackerThingyController::trackerValue( ) const
{
    return fTrackerValue;
}




void  TrackerThingyController::setTrackerValue(int val)
{
    if (val < fMinTrackVal)
        fTrackerValue = fMinTrackVal;
    else if (val > fMaxTrackVal)
        fTrackerValue = fMaxTrackVal;
    else
        fTrackerValue = val;

    if (isInstalled( )) {

        uWriteValueToEditArea( );
        uWriteValueToTrackbar( );
    }
}




bool  TrackerThingyController::isInstalled( ) const
{
    return fIsInstalled;
}




unsigned short  TrackerThingyController::titleWidth( ) const
{
    return fTitleWidth;
}




void  TrackerThingyController::setTitleWidth(unsigned short titlew)
{
    fTitleWidth = titlew;

    if (isInstalled( )) {

        Win32Tools::setWindowWidth(fTitleLabel, fTitleWidth);
        layout(*fManaged);
    }
}




unsigned short  TrackerThingyController::editAreaWidth( ) const
{
    return fEditAreaWidth;
}




void  TrackerThingyController::setEditAreaWidth(unsigned short eaw)
{
    fEditAreaWidth = eaw;

    if (isInstalled( )) {

        Win32Tools::setWindowWidth(fValueEditArea, fEditAreaWidth);
        layout(*fManaged);
    }
}




int   TrackerThingyController::maxTrackValue( ) const
{
    return fMaxTrackVal;
}




void  TrackerThingyController::setMaxTrackValue(int maxtrack)
{
    fMaxTrackVal = maxtrack;

    if (isInstalled( )) {

        SendMessage(fValueTrackbar, TBM_SETRANGE, TRUE,
            MAKELONG(static_cast<short>(fMinTrackVal),
            static_cast<short>(fMaxTrackVal)));

        if (fTrackerValue > fMaxTrackVal) {

            fTrackerValue = fMaxTrackVal;
            uWriteValueToEditArea( );
            uWriteValueToTrackbar( );
        }
    }
}




int   TrackerThingyController::minTrackValue( ) const
{
    return fMinTrackVal;
}




void  TrackerThingyController::setMinTrackValue(int mintrack)
{
    fMinTrackVal = mintrack;

    if (isInstalled( )) {

        SendMessage(fValueTrackbar, TBM_SETRANGE, TRUE,
            MAKELONG(static_cast<short>(fMinTrackVal),
            static_cast<short>(fMaxTrackVal)));

        if (fTrackerValue < fMinTrackVal) {

            fTrackerValue = fMinTrackVal;
            uWriteValueToEditArea( );
            uWriteValueToTrackbar( );
        }
    }
}




unsigned short  TrackerThingyController::tickIncrement( ) const
{
    return fTickIncrement;
}




void  TrackerThingyController::setTickIncrement(unsigned short incr)
{
    fTickIncrement = incr;

    if (isInstalled( ))
        SendMessage(fValueTrackbar, TBM_SETTICFREQ, fTickIncrement, 0);
}





void  TrackerThingyController::setTitleText(const std::string& text)
{
    fTitleText = text;

    if (isInstalled( ))
        Win32Tools::setWindowTextString(fTitleLabel, fTitleText);
}




const std::string&  TrackerThingyController::titleText( ) const
{
    return fTitleText;
}




void  TrackerThingyController::uProcessEditAreaChange( )
{
    std::string usertext = Win32Tools::getWindowTextString(fValueEditArea);

    if (usertext.find_first_not_of("-1234567890\r\n") != std::string::npos) {

        Win32Tools::setWindowTextIntegral(fValueEditArea, fTrackerValue);
        SetFocus(0);
        return;
    }

    if (usertext.find_first_of("\r\n") != std::string::npos) {

        int  userval = Win32Tools::getWindowTextIntegral(fValueEditArea);

        if ((userval < fMinTrackVal) || (userval > fMaxTrackVal)) {

            std::ostringstream  msgstream;

            msgstream << "Please enter a value between " << fMinTrackVal <<
                " and " << fMaxTrackVal << ".";

            MessageBox(fManaged->peerHandle( ), msgstream.str( ).c_str( ), TEXT("Oops!"),
                MB_ICONWARNING);

            Win32Tools::setWindowTextIntegral(fValueEditArea, fTrackerValue);
            SetFocus(0);
            return;
        }

        fTrackerValue = userval;
        uWriteValueToTrackbar( );
        uWriteValueToEditArea( );
        SetFocus(0);
        fParent->notifyTrackerValueChanged(*this);
        return;
    }
}



void  TrackerThingyController::uWriteValueToEditArea( )
{
    Win32Tools::setWindowTextIntegral(fValueEditArea, fTrackerValue);
}




void  TrackerThingyController::uWriteValueToTrackbar( )
{
    SendMessage(fValueTrackbar, TBM_SETPOS, TRUE,
        static_cast<short>(fTrackerValue));
}
//////////////////////////////////////////////////////////////////////////////
// END  TrackerThingyController                                             //
//////////////////////////////////////////////////////////////////////////////
