//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// SynAgentF.glsl                                                           //
//                                                                          //
// Fragment shader used by the SynthesisAgent class of the LDXPowerTools    //
// (PT/CI) framework. Enables the most computationally-intensive parts      //
// of the texture synthesis algorithm of SynthesisAgents to be executed     //
// on the GPU. Note that this file must be present in imaging programs      //
// directory of any PT/CI application that uses SynthesisAgents.            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver 1.0.0 of Fri 09-Jan-2009 @ 4:43am EST                                //
//                                                                          //
//    initial revision                                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008-2009, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------//
// CONSTANTS                                                               //
//-------------------------------------------------------------------------//
const    float      kPositiveInfinity = 1.0e26;
const    vec2       kProxFrameLoBound = vec2(-1.0, -1.0);
const    vec2       kProxFrameHiBound = vec2(1.0, 1.0);
const    vec2       kNullShift = vec2(0.0, 0.0);
const    int        kMaxRefinementLevel = 8;
const    int        kUpsampleJitterMode = 0;
const    int        kCorrectionMode = 2;
const    int        kOutputMode = 3;
const    int        kNHRegVectorSize = 25;
const    vec2       kNHLoBound = vec2(-2.0, -2.0);
const    vec2       kNHHiBound = vec2(2.0, 2.0);
const    float      kColorDecodeMult = 255.0;
const    float      kColorEncodeMult = (1.0 / 255.0);
const    float      kEdgeExclusionPix = 2.0;
const    float      kTexture1DNormMult = (1.0 / 1024.0);
const    float      kTexture1DDecodeMult = 4095.0;
const    int        kMaxNumSimLevels = 5;
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// UNIFORMS                                                                //
//-------------------------------------------------------------------------//
uniform  sampler2D  WorkTex;
uniform  vec2       WorkTexExtent;
uniform  sampler2D  SourceTex;
uniform  vec2       SourceTexExtent;
uniform  vec2       SourceTileOrigins[kMaxRefinementLevel + 1];
uniform  float      SourceTileExtents[kMaxRefinementLevel + 1];
uniform  sampler2D  SimTex;
uniform  vec2       SimTexExtent;
uniform  float      SimNumEntries;
uniform  int        OperationMode;
uniform  int        RefinementLevel;
uniform  int        NumRefinementLevels;
uniform  float      PixelWeights[kNHRegVectorSize];
uniform  vec2       UpsampleEffectsGeometry[2 * (kMaxRefinementLevel + 1)];
uniform  vec2       Corr1EffectsGeometry[2 * (kMaxRefinementLevel - 2)];
uniform  vec2       Corr2EffectsGeometry[2 * (kMaxRefinementLevel - 2)];
uniform  vec2       MaxOutputEffectsGeometry[2];
uniform  int        CorrectionPassNumber;
uniform  sampler1D  StateTex;
uniform  float      JumpPenalizationParam;
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// GLOBALS                                                                 //
//-------------------------------------------------------------------------//
vec3  SourceNHRegister[kNHRegVectorSize];
vec3  WorkNHRegister[kNHRegVectorSize];
vec2  SimTileOrigin;
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// FUNCTION PROTOTYPES                                                     //
//-------------------------------------------------------------------------//
vec2   GetUpsampleJitterExtent(in int);
vec2   GetUpsampleJitterOrigin(in int);
vec2   GetOutputExtent(in int);
vec2   GetOutputOrigin(in int);
vec2   GetCorrectionExtent(in int, in int);
vec2   GetCorrectionOrigin(in int, in int);
void   PrefetchEncodedState( );
vec2   EdgeSourceLoc(in vec2);
vec2   SampleSimilarityMap(in vec2, in float);
float  ComputeRegisterDelta( );
vec4   SampleSourceImage(in int, in vec2);
vec2   SampleWorkImage(in vec2);
bool   IsNearSourceEdge(in vec2);
vec2   JitterPixelValue(in vec2, in vec2);
void   LoadSourceRegister(in vec2);
void   LoadWorkRegister(in vec2);
float  FindBestJumpMatch(in vec2, out vec2);
vec2   FindBestMatchNonEdge(in vec2);
vec2   FindBestMatchEdge(in vec2);
vec2   FindBestMatch(in vec2);
vec4   DoUpsampleJitter(in vec2);
vec4   DoCorrection(in vec2);
vec4   DoOutput(in vec2);
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// GetUpsampleJitterExtent ( )                                             //
//-------------------------------------------------------------------------//
// SYNOPSIS:       returns the extent (in whole pixels) of the shader      //
//                 effects region for the upsample & jitter operation at   //
//                 refinement level 'lev'                                  //
//                                                                         //
// PRECONDITIONS:  'lev' is between 0 and (NumRefinementLevels - 1)        //
//-------------------------------------------------------------------------//
vec2  GetUpsampleJitterExtent(in int lev)
{
    return UpsampleEffectsGeometry[(2 * lev) + 1];
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// GetUpsampleJitterOrigin ( )                                             //
//-------------------------------------------------------------------------//
// SYNOPSIS:       returns the logical origin (in whole work image         //
//                 universe coordinates) of the shader effects region for  //
//                 the upsample & jitter operation at refinement level     //
//                 'lev'                                                   //
//                                                                         //
// PRECONDITIONS:  'lev' is between 0 and (NumRefinementLevels - 1)        //
//-------------------------------------------------------------------------//
vec2  GetUpsampleJitterOrigin(in int lev)
{
    return UpsampleEffectsGeometry[2 * lev];
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// GetCorrectionExtent ( )                                                 //
//-------------------------------------------------------------------------//
// SYNOPSIS:       returns the extent (in whole pixels) of the shader      //
//                 effects region for the 'n'-th correction operation that //
//                 occurs on refinement level 'lev'                        //
//                                                                         //
// PRECONDITIONS:  'lev' is between 0 and (NumRefinementLevels - 1) and    //
//                 'n' is either 1 or 2                                    //
//-------------------------------------------------------------------------//
vec2  GetCorrectionExtent(in int lev, in int n)
{
    if (n == 1)
        return Corr1EffectsGeometry[(2 * (lev - 3)) + 1];
    else if (n == 2)
        return Corr2EffectsGeometry[(2 * (lev - 3)) + 1];
    else
        discard;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// GetCorrectionOrigin ( )                                                 //
//-------------------------------------------------------------------------//
// SYNOPSIS:       returns the logical origin (in whole work image         //
//                 universe coordinates) of the shader effects region for  //
//                 the 'n'-th correction operation that occurs on          //
//                 refinement level 'lev'                                  //
//                                                                         //
// PRECONDITIONS:  'lev' is between 0 and (NumRefinementLevels - 1) and    //
//                 'n' is either 1 or 2                                    //
//-------------------------------------------------------------------------//
vec2  GetCorrectionOrigin(in int lev, in int n)
{
    if (n == 1)
        return Corr1EffectsGeometry[2 * (lev - 3)];
    else if (n == 2)
        return Corr2EffectsGeometry[2 * (lev - 3)];
    else
        discard;
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// PrefetchEncodedState ( )                                                //
//-------------------------------------------------------------------------//
// SYNOPSIS:       decodes persistent state information encoded as         //
//                 luminance values in 1D textures into global             //
//                 variables; this is useful in that it reduces the number //
//                 of texture reads                                        //
//-------------------------------------------------------------------------//
void  PrefetchEncodedState( )
{
    int  simtilenum = NumRefinementLevels - RefinementLevel - 1;

    float  ogntexcoord_x = (2.0 * float(simtilenum) + 0.5) *
        kTexture1DNormMult;
    float  ogntexcoord_y = (2.0 * float(simtilenum) + 1.0 + 0.5) *
        kTexture1DNormMult;

    SimTileOrigin.x = texture1D(StateTex, ogntexcoord_x).r;
    SimTileOrigin.y = texture1D(StateTex, ogntexcoord_y).r;
    SimTileOrigin *= kTexture1DDecodeMult;    
}
//-------------------------------------------------------------------------//




//-------------------------------------------------------------------------//
// EdgeSourceLoc ( )                                                       //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given an unencoded source image location 'loc' that     //
//                 potentially falls outside of the edge boundaries of the //
//                 source image at the current level of refinement, apply  //
//                 a wrapping edge-handling strategy to map 'loc' to a     //
//                 new pixel location within the image edges and return    //
//                 the new location to the caller.  If 'loc' already falls //
//                 within the source image edge boundaries, then simply    //
//                 return 'loc'                                            //
//-------------------------------------------------------------------------//
vec2  EdgeSourceLoc(in vec2 loc)
{
    vec2  bounds = vec2(SourceTileExtents[RefinementLevel],
        SourceTileExtents[RefinementLevel]) - vec2(1.0, 1.0);

    if (loc.x < 0.0)
        loc.x = SourceTileExtents[RefinementLevel] + loc.x;
    else if (loc.x > bounds.x)
        loc.x = loc.x - SourceTileExtents[RefinementLevel];

    if (loc.y < 0.0)
        loc.y = SourceTileExtents[RefinementLevel] + loc.y;
    else if (loc.y > bounds.y)
        loc.y = loc.y - SourceTileExtents[RefinementLevel];
    
    return loc;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// SampleSimilarityMap ( )                                                 //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given an unencoded source image location 'samppt', look //
//                 up 'samppt' in the similarity map and return the        //
//                 unencoded source image location of the 'n'-th pixel     //
//                 whose surrounding neighborhood is perceptually similar  //
//                 to the neighborhood about 'samppt'                      //
//                                                                         //
// PRECONDITIONS:  'n' is a whole number between 0 and (SimNumEntries - 1) //
//-------------------------------------------------------------------------//
vec2  SampleSimilarityMap(in vec2 samppt, in float n)
{
    vec2 samppt_mosaic = samppt;
    samppt_mosaic.x *= SimNumEntries;
    samppt_mosaic.x += n;
    samppt_mosaic += SimTileOrigin;
    samppt_mosaic += vec2(0.5, 0.5);
    
    vec2 samppt_tex = samppt_mosaic / SimTexExtent;
    
    vec4 val = texture2D(SimTex, samppt_tex);
    
    return (val.rg * kColorDecodeMult);
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// ComputeRegisterDelta ( )                                                //
//-------------------------------------------------------------------------//
// SYNOPSIS:       calculate the value of the perceptual difference metric //
//                 between the currently loaded source and work            //
//                 neighborhoods and return it to the caller               //
//                                                                         //
// PRECONDITIONS:  caller must load the global source and work             //
//                 neighborhood registers appropriately before calling     //
//                 this function                                           //
//-------------------------------------------------------------------------//
float  ComputeRegisterDelta( )
{
    vec3 comp_accum = vec3(0.0, 0.0, 0.0);

    for (int i = 0; i < kNHRegVectorSize; i++) {
    
        vec3 thisdelta = WorkNHRegister[i] - SourceNHRegister[i];
        thisdelta *= thisdelta;
        thisdelta = sqrt(thisdelta);
        
        comp_accum += (thisdelta * PixelWeights[i]);
    }
    
    return (comp_accum.r + comp_accum.g + comp_accum.b);
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// SampleSourceImage ( )                                                   //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given an unencoded source image location 'sourceloc',   //
//                 look up the color value of the source image pixel at    //
//                 location 'sourceloc' at refinement level 'lev' and      //
//                 return it to the caller                                 //
//                                                                         //
// PRECONDITIONS:  caller must ensure that 'lev' falls within the range    //
//                 0..(NumRefinementLevels - 1), inclusive, and that       //
//                 that 'sourceloc' falls within the boundaries of the     //
//                 source image on refinement level 'lev'                  //
//-------------------------------------------------------------------------//
vec4  SampleSourceImage(in int lev, in vec2 sourceloc)
{
    vec2  tileogn_log = SourceTileOrigins[lev] + vec2(0.5, 0.5);
    
    vec2  log2tex = vec2((1.0 / SourceTexExtent.x),
        (1.0 / SourceTexExtent.y));
    
    vec2  tileoffset_tex = log2tex * sourceloc;
    vec2  tileogn_tex = log2tex * tileogn_log;
    
    return  texture2D(SourceTex, tileogn_tex + tileoffset_tex);
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// SampleWorkImage ( )                                                     //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given an unencoded work texture location 'workloc',     //
//                 look up the value of the work image pixel at location   //
//                 'workloc' and return this value to the caller.          //
//                 Note that since the work image is used for indirection  //
//                 into the source image, the value returned is not a      //
//                 color value but is instead an unencoded source image    //
//                 location                                                //
//                                                                         //
// PRECONDITIONS:  caller must ensure that 'workloc' falls within the      //
//                 boundaries of the work texture active subspace for      //
//                 the current operation at the current refinement         //
//                 level.                                                  //
//-------------------------------------------------------------------------//
vec2  SampleWorkImage(in vec2 workloc)
{
    vec2  workloc_tex = (workloc + vec2(0.5, 0.5)) / WorkTexExtent;
    
    vec2  val_tex = texture2D(WorkTex, workloc_tex).rg;
    
    vec2  val_srccoords = val_tex * kColorDecodeMult;
    
    return val_srccoords;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// IsNearSourceEdge ( )                                                    //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given an unencoded source image location 'sourceloc',   //
//                 return true if 'sourceloc' lies near an image edge,     //
//                 false otherwise                                         //
//-------------------------------------------------------------------------//
bool  IsNearSourceEdge(in vec2 sourceloc)
{
    float mincoord = min(sourceloc.x, sourceloc.y);
    float maxcoord = max(sourceloc.x, sourceloc.y);

    if (mincoord < kEdgeExclusionPix)
        return true;

    if (maxcoord >= (SourceTileExtents[RefinementLevel] - kEdgeExclusionPix))
        return true;

    return false;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// JitterPixelValue ( )                                                    //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given a whole work image universe location 'workloc'    //
//                 and an unencoded source image location 'srcloc' that    //
//                 is the value of the pixel at 'workloc' (recall that the //
//                 work image is used for indirection into the source      //
//                 image, so work image pixel values are source image      //
//                 coordinates), return a new value for the pixel at       //
//                 'workloc' computed by deterministically jittering the   //
//                 previous value. The value returned is in an unencoded   //
//                 source image location                                   //
//-------------------------------------------------------------------------//
vec2  JitterPixelValue(in vec2 workloc, in vec2 srcloc)
{
    vec2 jit_offset = vec2(
        2.0 * fract(sin(dot(workloc.xy, vec2(12.9898,78.233)))
        * 43758.5453) - 1.0, 2.0 * fract(sin(dot(workloc.xy,
        vec2(12.9898,78.233))) * 43758.5453) - 1.0);

    jit_offset = floor(jit_offset + vec2(0.5, 0.5));
        
    float levdim = SourceTileExtents[RefinementLevel];
    
    if ((srcloc.x == 0.0) && (jit_offset.x < 0.0))
        jit_offset.x = levdim + jit_offset.x;
    if ((srcloc.x == (levdim - 1.0)) && (jit_offset.x > 0.0))
        jit_offset.x = jit_offset.x - levdim;
    
    if ((srcloc.y == 0.0) && (jit_offset.y < 0.0))
        jit_offset.y = levdim + jit_offset.y;
    if ((srcloc.y == (levdim - 1.0)) && (jit_offset.y > 0.0))
        jit_offset.y = jit_offset.y - levdim;
            
   return clamp(srcloc + jit_offset, 0.0,
       SourceTileExtents[RefinementLevel] - 1.0);
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// LoadSourceRegister ( )                                                  //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given an unencoded source image location 'ctrloc', copy //
//                 the color values of the pixels in the neighborhood      //
//                 about 'ctrloc' into the source neighborhood register    //
//-------------------------------------------------------------------------//
void  LoadSourceRegister(in vec2 ctrloc)
{
    vec2 curloc = EdgeSourceLoc(ctrloc + vec2(-2.0, -2.0));
    SourceNHRegister[0] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-1.0, -2.0));
    SourceNHRegister[1] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(0.0, -2.0));
    SourceNHRegister[2] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(1.0, -2.0));
    SourceNHRegister[3] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(2.0, -2.0));
    SourceNHRegister[4] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-2.0, -1.0));
    SourceNHRegister[5] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-1.0, -1.0));
    SourceNHRegister[6] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(0.0, -1.0));
    SourceNHRegister[7] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(1.0, -1.0));
    SourceNHRegister[8] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(2.0, -1.0));
    SourceNHRegister[9] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-2.0, 0.0));
    SourceNHRegister[10] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-1.0, 0.0));
    SourceNHRegister[11] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(0.0, 0.0));
    SourceNHRegister[12] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(1.0, 0.0));
    SourceNHRegister[13] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(2.0, 0.0));
    SourceNHRegister[14] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-2.0, 1.0));
    SourceNHRegister[15] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-1.0, 1.0));
    SourceNHRegister[16] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(0.0, 1.0));
    SourceNHRegister[17] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(1.0, 1.0));
    SourceNHRegister[18] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(2.0, 1.0));
    SourceNHRegister[19] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-2.0, 2.0));
    SourceNHRegister[20] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(-1.0, 2.0));
    SourceNHRegister[21] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(0.0, 2.0));
    SourceNHRegister[22] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(1.0, 2.0));
    SourceNHRegister[23] = SampleSourceImage(RefinementLevel, curloc).rgb;

    curloc = EdgeSourceLoc(ctrloc + vec2(2.0, 2.0));
    SourceNHRegister[24] = SampleSourceImage(RefinementLevel, curloc).rgb;    
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// LoadWorkRegister ( )                                                    //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given a work image location 'ctrloc' (in whole work     //
//                 texture coordinates), copy the color values of the      //
//                 pixels in the neighborhood about 'ctrloc' into the work //
//                 neighborhood register. Note that since the pixels of    //
//                 the work image are references into the source image,    //
//                 two texture reads are performed to get work image color //
//                 vales. The first is a read of the work image to obtain  //
//                 the location of a corresponding source image pixel and  //
//                 the second is a read of the source image to look up the //
//                 color value at that location                            //
//-------------------------------------------------------------------------//
void  LoadWorkRegister(in vec2 ctrloc)
{
    vec2 curloc = ctrloc + vec2(-2.0, -2.0);
    WorkNHRegister[0] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-1.0, -2.0);
    WorkNHRegister[1] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(0.0, -2.0);
    WorkNHRegister[2] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(1.0, -2.0);
    WorkNHRegister[3] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(2.0, -2.0);
    WorkNHRegister[4] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-2.0, -1.0);
    WorkNHRegister[5] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-1.0, -1.0);
    WorkNHRegister[6] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(0.0, -1.0);
    WorkNHRegister[7] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(1.0, -1.0);
    WorkNHRegister[8] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(2.0, -1.0);
    WorkNHRegister[9] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-2.0, 0.0);
    WorkNHRegister[10] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-1.0, 0.0);
    WorkNHRegister[11] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(0.0, 0.0);
    WorkNHRegister[12] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(1.0, 0.0);
    WorkNHRegister[13] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(2.0, 0.0);
    WorkNHRegister[14] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-2.0, 1.0);
    WorkNHRegister[15] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-1.0, 1.0);
    WorkNHRegister[16] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(0.0, 1.0);
    WorkNHRegister[17] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(1.0, 1.0);
    WorkNHRegister[18] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(2.0, 1.0);
    WorkNHRegister[19] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-2.0, 2.0);
    WorkNHRegister[20] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(-1.0, 2.0);
    WorkNHRegister[21] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(0.0, 2.0);
    WorkNHRegister[22] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(1.0, 2.0);
    WorkNHRegister[23] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;

    curloc = ctrloc + vec2(2.0, 2.0);
    WorkNHRegister[24] = SampleSourceImage(RefinementLevel, SampleWorkImage(
        curloc)).rgb;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// FindBestJumpMatch ( )                                                   //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given an unencoded source image location 'jumpref',     //
//                 sample the similarity map to collect all the locations  //
//                 whose neighborhoods are perceptually similar to the     //
//                 neighborhood about 'jumpref'. Then compare the          //
//                 the neighborhoods about each of the collected locations //
//                 against the neighborhood currently loaded into the work //
//                 register. When the function returns, the unencoded      //
//                 coordinates of the source image location whose          //
//                 neighborhood best matches that loaded into the work     //
//                 register is returned in the output parameter            //
//                 'matchedloc', and the perceptual distance measured      //
//                 between 'matchedloc' and the neighborhood loaded into   //
//                 the work register is retured as the value of the        //
//                 function                                                //
//                                                                         //
// PRECONDITIONS:  caller must ensure that the the work register is loaded //
//                 appropriately before calling this function              //
//-------------------------------------------------------------------------//
float  FindBestJumpMatch(in vec2 jumpref, out vec2 matchedloc)
{
   float  bestdist = kPositiveInfinity;

   for (float i = 0.0; i < SimNumEntries; i += 1.0) {

      vec2  currpix = SampleSimilarityMap(jumpref, i);
      
      LoadSourceRegister(currpix);

      float  currdist = ComputeRegisterDelta( );
      if (currdist < bestdist) {

          matchedloc = currpix;
          bestdist = currdist;
      }
   } /* for all pixels with similar neighborhoods */

   return (JumpPenalizationParam * bestdist);
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// FindBestMatchNonEdge ( )                                                //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given a work image pixel location 'workloc' (in whole   //
//                 work texture coordinates) whose source mate does not    //
//                 lie near an image edge, return the unencoded source     //
//                 image location whose surrounding neighborhood is        //
//                 perceptually closest to the neighborhood around         //
//                 'workloc'                                               //
//                                                                         //
// PRECONDITIONS:  the function doesn't verify that the source image mate  //
//                 of 'workloc' fails to lie near an image edge. Caller    //
//                 must check for this in advance and invoke               //
//                 FindBestMatchEdge( ) if the mate of 'workloc' lies near //
//                 an image edge, or FindBestMatchNonEdge( ) if the mate   //
//                 of 'workloc' doesn't lie near han image edge            //
//-------------------------------------------------------------------------//
vec2  FindBestMatchNonEdge(in vec2 workloc)
{
    vec2   bestmatch;
    float  bestdist = kPositiveInfinity;

    LoadWorkRegister(workloc);

    vec2  backshift = kProxFrameLoBound;
    for (; backshift.y <= kProxFrameHiBound.y; backshift.y += 1.0) {
        for (; backshift.x <= kProxFrameHiBound.x; backshift.x += 1.0) {
    
            vec2  currproxloc = workloc + backshift;
            vec2  mate = EdgeSourceLoc(SampleWorkImage(currproxloc) -
                backshift);
        
            LoadSourceRegister(mate);
    
            float  proxdist = ComputeRegisterDelta( );
            if (proxdist < bestdist) {
    
                bestmatch = mate;
                bestdist = proxdist;
            }
    
            vec2   jumpmatch;
            float  jumpdist = FindBestJumpMatch(mate, jumpmatch);
            if (jumpdist < bestdist) {
    
                bestmatch = jumpmatch;
                bestdist = jumpdist;
            }
        } /* for all pixels in the current frame row */
    } /* for all rows in the proximity frame */

    return bestmatch;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// FindBestMatchEdge ( )                                                   //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given a work image pixel location 'workloc' (in whole   //
//                 work texture coordinates) whose source mate lies        //
//                 near an image edge, return the unencoded source image   //
//                 location whose surrounding neighborhood is perceptually //
//                 closest to the neighborhood around 'workloc'            //
//                                                                         //
// PRECONDITIONS:  the function doesn't verify that the source image mate  //
//                 of 'workloc' actually lies near an image edge.          //
//                 Caller must check for this in advance and invoke        //
//                 FindBestMatchEdge( ) if the mate of 'workloc' lies near //
//                 an image edge, or FindBestMatchNonEdge( ) if the mate   //
//                 of 'workloc' doesn't lie near an image edge             //
//-------------------------------------------------------------------------//
vec2  FindBestMatchEdge(in vec2 workloc)
{
    vec2   bestmatch;
    float  bestdist = kPositiveInfinity;
    
    LoadWorkRegister(workloc);

    vec2  backshift = kProxFrameLoBound;
    for (; backshift.y <= kProxFrameHiBound.y; backshift.y += 1.0) {
        for (; backshift.x <= kProxFrameHiBound.x; backshift.x += 1.0) {
    
            vec2  currproxloc = workloc + backshift;
            vec2  mate = EdgeSourceLoc(SampleWorkImage(currproxloc) -
                backshift);
            
            vec2   matematch;
            float  matchdist = FindBestJumpMatch(mate, matematch);
            if (matchdist < bestdist) {
        
                bestmatch = matematch;
                bestdist = matchdist;
            }
        } /* for all pixels in the current row */
    } /* for all rows in the proximity frame */

    return bestmatch;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// FindBestMatch ( )                                                       //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given as input a work image pixel location 'workloc'    //
//                 (in whole work image universe coordinates), return the  //
//                 unencoded source image location whose surrounding       //
//                 neighborhood is perceptually closest to the             //
//                 neighborhood about 'workloc'                            //
//-------------------------------------------------------------------------//
vec2  FindBestMatch(in vec2 workloc)
{
    /* we need to convert 'workloc' into work texture coordinates before
       we can sample the work texture; if this is the first correction pass,
       then the work texture is loaded with the output of the previous
       upsample/jitter pass; if this is the second correction pass,
       then the work texture is loaded with output of the first correction
       pass */
    vec2 workloc_tex;
    if (CorrectionPassNumber == 1)
        workloc_tex = workloc - GetUpsampleJitterOrigin(RefinementLevel);
    else if (CorrectionPassNumber == 2)
        workloc_tex = workloc - GetCorrectionOrigin(RefinementLevel, 1);
    else
        discard;
    
    /* sample the work image to get this pixel's value */
    vec2  srcmate = SampleWorkImage(workloc_tex);
    
    vec2 matched;
    if (IsNearSourceEdge(srcmate))
        matched = FindBestMatchEdge(workloc_tex);
    else
        matched = FindBestMatchNonEdge(workloc_tex);
    
    return matched;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// DoUpsampleJitter ( )                                                    //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given 'pixcoords', the coordinates of the pixel for     //
//                 which this shader was invoked, in normalized shader     //
//                 effects region (SXR) coordinates, run the upsample and  //
//                 jitter operations and return their output as an encoded //
//                 work image pixel value. If the host system has set the  //
//                 shader to run in upsample & jitter mode, the value      //
//                 returned by this function is the overall fragment       //
//                 shader output value (i.e. gl_FragColor).                //
//-------------------------------------------------------------------------//
vec4  DoUpsampleJitter(in vec2 pixcoords)
{
    if (RefinementLevel == 0) {
    
        return vec4(0.0, 0.0, 0.0, 1.0);
    }
    else {

        /* convert the input normalized shader effects region coordinates
           into whole pixel effects regions coordinates */
        vec2  pixcoords_whole = floor(pixcoords *
            GetUpsampleJitterExtent(RefinementLevel));

        /* convert the whole pixel effects region coordinates computed above
           into work image universe coordinates */
        vec2  pixcoords_univ = pixcoords_whole +
            GetUpsampleJitterOrigin(RefinementLevel);

        /* get the location of the current pixel's ancestor in work image
           universe coordinates */
        vec2  ancestor_univ = floor(pixcoords_univ * 0.5);
        
        /* convert the coordinates of the current pixel's ancestor from
           work image universe coordinates into whole work texture
           coordinates -- we can then use these to sample the ancestor's
           value from the work texture; depending on what level we're on,
           either the output of the previous level's second correction pass
           or the output of the previous level's upsample/jitter pass is
           loaded into the work texture */
        vec2  ancestor_tex;
        if (RefinementLevel > 3)
            ancestor_tex = ancestor_univ - GetCorrectionOrigin(
            RefinementLevel - 1, 2);
        else
            ancestor_tex = ancestor_univ - GetUpsampleJitterOrigin(
            RefinementLevel - 1);

        /* compute the current pixel's offset from its ancestor */
        vec2  pixoffset = pixcoords_univ - (2.0 * ancestor_univ);

        /* get the value of the current pixel's ancestor by using the
           ancestor's work texture coordinates to sample the work texture */
        vec2  ancestor_val = SampleWorkImage(ancestor_tex);

        /* upsample the the value of the ancestor */
        vec2  pixval_whole = ((2.0 * ancestor_val) + pixoffset);
        
        return vec4(JitterPixelValue(pixcoords_univ, pixval_whole) *
            kColorEncodeMult, 0.0, 1.0);
    }
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// DoCorrection ( )                                                        //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given 'pixcoords', the coordinates of the pixel on      //
//                 which this shader was invoked, in normalized shader     //
//                 effects region coordinates, run the correction          //
//                 operation and return the output of the correction       //
//                 phase as an encoded work image pixel value. If the host //
//                 system has set the shader to run in correction mode,    //
//                 the value returned by this function is the overall      //
//                 fragment shader output value (i.e. gl_FragColor).       //
//-------------------------------------------------------------------------//
vec4  DoCorrection(in vec2 pixcoords)
{
    /* prefetch the persistent state information from TRAM and cache it
       locally */
    PrefetchEncodedState( );

    /* convert the input normalized shader effects region coordinates into
       whole pixel effects region coordinates */
    vec2  pixcoords_whole = floor(pixcoords * GetCorrectionExtent(
        RefinementLevel, CorrectionPassNumber));

    /* convert the whole pixel effects region coordinates into work image
       universe coordinates */
    vec2  pixcoords_univ = pixcoords_whole + GetCorrectionOrigin(
        RefinementLevel, CorrectionPassNumber);

    return vec4(FindBestMatch(pixcoords_univ) * kColorEncodeMult, 0.0, 1.0);
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// DoOutput ( )                                                            //
//-------------------------------------------------------------------------//
// SYNOPSIS:       given 'pixcoords', the coordinates of the pixel for     //
//                 which this shader was invoked, in normalized shader     //
//                 effects region (SXR) coordinates, run the output scan   //
//                 operation and return its result as an (r,g,b,a) color   //
//                 value. If the host has set the shader to run in         //
//                 output mode, the value returned by this function is     //
//                 the overall fragment shader output value (i.e. the      //
//                 gl_FragColor).                                          //
//-------------------------------------------------------------------------//
vec4  DoOutput(in vec2 pixcoords)
{
    /* convert the input normalized shader effects region coordinates into
       whole pixel effects region coordinates */
    vec2  pixcoords_whole = floor(pixcoords * MaxOutputEffectsGeometry[1]);

    /* convert the whole pixel effects region coordinates into work image
       universe coordinates */
    vec2  pixcoords_univ = pixcoords_whole + MaxOutputEffectsGeometry[0];
    
    /* convert the work image universe coordinates into work texture
       coordinates so that we can sample the work texture */
    vec2  pixcoords_tex = pixcoords_univ - GetCorrectionOrigin(
        RefinementLevel, 2);

    vec2  pixval = SampleWorkImage(pixcoords_tex);

    return SampleSourceImage(RefinementLevel, pixval);
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// main ( )                                                                //
//-------------------------------------------------------------------------//
void main( )
{
    /* the location of the pixel for which this shader was invoked,
       in normalized shader effects region coordinates */
    vec2  thispix_sxrn = gl_TexCoord[0].st;
    
    if (OperationMode == kUpsampleJitterMode)
        gl_FragColor = DoUpsampleJitter(thispix_sxrn);
    else if (OperationMode == kCorrectionMode)
        gl_FragColor = DoCorrection(thispix_sxrn);
    else
        gl_FragColor = DoOutput(thispix_sxrn);
}
//-------------------------------------------------------------------------//
