//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// DecalStrokeF.glsl                                                        //
//                                                                          //
// Fragment shader that controls the appearance of the decal stroke.        //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// ver 1.0.0 of Fri 09-Jan-2009 @ 4:19am EST                                //
//                                                                          //
//    initial revision                                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (c) 2008-2009, Lucas Stephen Beeler. All Rights Reserved.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////



//-------------------------------------------------------------------------//
// CONSTANTS                                                               //
//-------------------------------------------------------------------------//
const    int    kLowBoundary = 0;
const    int    kHighBoundary = 1;
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// UNIFORMS                                                                //
//-------------------------------------------------------------------------//
uniform  float      BlendBoundaries[2];
uniform  sampler2D  DecalTex;
uniform  bool       IsNoiseEnabled;
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// FUNCTION PROTOTYPES                                                     //
//-------------------------------------------------------------------------//
float  FeatherTransverseEdge(float);
float  FeatherInlineEdge(float);
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// FeatherInlineEdge ( )                                                   //
//-------------------------------------------------------------------------//
// SYNOPSIS:   Alpha-blends and applies speckly noise to the inline edges  //
//             of the user's brush stroke. The inline edges of the stroke  //
//             are those parallel to the brush's direction of travel.      //
//             To use this function, pass the t texture coordinate as 't'. //
//             Using the values of the host-configured uniforms            //
//             'BlendBoundaries' and 'IsNoiseEnabled', an opacity (alpha)  //
//             value is computed for fragment on which this shader program //
//             was invoked. This alpha value is the return value of the    //
//             function, and should be used to set the alpha component of  //
//             gl_FragColor.                                               //
//-------------------------------------------------------------------------//
float  FeatherInlineEdge(float t)
{
    float alpha;
    if (t < BlendBoundaries[kLowBoundary]) {

        float  invalpha = (BlendBoundaries[kLowBoundary] - t) /
            BlendBoundaries[kLowBoundary];

        alpha = smoothstep(1.0, 0.0, invalpha);
        
        if (IsNoiseEnabled) {
                    
            float  noise_val = fract(sin(dot(gl_FragCoord.xy.xy,
                vec2(12.9898,78.233))) * 43758.5453) * 0.5;
            
            alpha = alpha - ((1.0 - alpha) * noise_val);
        }
    }
    else if (t > BlendBoundaries[kHighBoundary]) {

        float  invalpha = (t - BlendBoundaries[kHighBoundary]) /
            BlendBoundaries[kLowBoundary];

        alpha = smoothstep(1.0, 0.0, invalpha);
        
        if (IsNoiseEnabled) {

            float  noise_val = fract(sin(dot(gl_FragCoord.xy.xy,
                vec2(12.9898,78.233))) * 43758.5453) * 0.5;

            alpha = alpha - ((1.0 - alpha) * noise_val);
        }
    }
    else {
    
        alpha = 1.0;
    }

    return alpha;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// FeatherTransverseEdge ( )                                               //
//-------------------------------------------------------------------------//
// SYNOPSIS:   Alpha-blends and applies speckly noise to the transverse    //
//             edges of the user's brush stroke. The transverse edges      //
//             are those perpendicular to the brush's direction of travel. //
//             To use this function, pass the s texture coordinate as 's'. //
//             Using the values of the host-configured uniforms            //
//             'BlendBoundaries' and 'IsNoiseEnabled', an opacity (alpha)  //
//             value is computed for fragment on which this shader program //
//             was invoked. This alpha value is the return value of the    //
//             function, and should be used to set the alpha component of  //
//             gl_FragColor.                                               //
//-------------------------------------------------------------------------//
float  FeatherTransverseEdge(float s)
{
    float alpha;
    if (s < 1.0) {

        alpha = smoothstep(0.0, 1.0, s);
        
        if (IsNoiseEnabled) {
        
            float  noise_val = fract(sin(dot(gl_FragCoord.xy.xy,
                vec2(12.9898,78.233))) * 43758.5453) * 0.5;

            alpha = alpha - ((1.0 - alpha) * noise_val);
        }
    }
    else {
    
        alpha = 1.0;
    }

    return alpha;
}
//-------------------------------------------------------------------------//



//-------------------------------------------------------------------------//
// main ( )                                                                //
//-------------------------------------------------------------------------//
void main( )
{    
    float alpha_s = FeatherTransverseEdge(gl_TexCoord[1].s);
    float alpha_t = FeatherInlineEdge(gl_TexCoord[1].t);
    
    vec3   basecolor = texture2D(DecalTex, gl_TexCoord[0].st).rgb;
        
    gl_FragColor = vec4(basecolor, min(min(alpha_s, alpha_t), alpha_s *
        alpha_t));
}
//-------------------------------------------------------------------------//
