/*

 SceneGenerators.c

 Copyright (c) 2006-2007, Lucas Stephen Beeler. All Rights Reserved.

 */

#include "SceneGenerators.h"
#include "LightingIntrinsics.h"
#include "ObjectInput.h"
#include "SpatialPartitioning.h"

Scene*  ConstructBunnyDebuggingScene(void)
{
    Scene*                  result;

    Mesh*                   floorMesh;
    Gel                     floorGel;
    MaterialProps           floorMaterial;
    HomogeneousVector3D     tempVertex;
    int                     tempVertList[3];
    HomogeneousTransform3D  floorTrans;
    RGBColor                mainLightColor;
    Light                   mainLight;
    const char*             bunnyFilename = "busm.ob";
    Mesh*                   bunnyMesh;
    FILE*                   bunnyFilePointer = 0;
    Gel                     bunnyGel;
    MaterialProps           bunnyMaterial;
    HomogeneousTransform3D  bunnyTransform;
    HomogeneousTransform3D  workTransform;
    HomogeneousVector3D     rotationalAxis;

    result = ConstructBaseScene( );

    /* Create the two triangular floor tiles -- they should be a
       medium blue-green color */

    floorMesh = ConstructMesh(kFaceNormalMode);

    /* Append floor lower-left vertex */
    SetVertex3D(&tempVertex, -1.0, -0.5, 1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor lower-right vertex */
    SetVertex3D(&tempVertex, 1.0, -0.5, 1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor upper-right vertex */
    SetVertex3D(&tempVertex, 1.0, -0.5, -1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor upper-left vertex */
    SetVertex3D(&tempVertex, -1.0, -0.5, -1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append the floor lower-right triangle */
    tempVertList[0] = 0;
    tempVertList[1] = 1;
    tempVertList[2] = 2;
    AppendFace(floorMesh, tempVertList, 3);

    /* Append the floor upper-left triangle */
    tempVertList[0] = 0;
    tempVertList[1] = 2;
    tempVertList[2] = 3;
    AppendFace(floorMesh, tempVertList, 3);

    floorMesh->isTriangleMesh = TRUE;

    /* Set up the floor material's colors */
    floorMaterial.ambient = kRGB_SeaGreen;
    floorMaterial.diffuse = kRGB_SeaGreen;
    floorMaterial.specular = kRGB_Gray50;
    floorMaterial.shinyness = 1.0;
    floorMaterial.reflectionBlur = 0.15;

    /* set up the floor's transform */
    ConfigureIdentity(&floorTrans);

    /* Set up a Gel for the floor mesh */
    SetGel(&floorGel, floorMesh, &floorMaterial, &floorTrans);

    /* Add the floor Gel to the scene */
    AddGel(result, &floorGel);

    /* Add a light back and to the right */
    SetVertex3D(&tempVertex, 2.0, 2.0, 2.0);
    SetRGBColor(&mainLightColor, 0.40, 0.40, 0.35);
    ConfigureGlobeLight(&mainLight, &mainLightColor, &tempVertex, 0.5);
    AddLight(result, &mainLight);

    /* Read in the bunny */
    bunnyFilePointer = fopen(bunnyFilename, "r");
    if (!bunnyFilePointer) {

        RuntimeError("ConstructBunnyDebuggingScene( )",
            "couldn't open bunny object data file");
    }

    bunnyMesh = ReadObjectData(bunnyFilePointer);

    /* set up a material for the bunny */
    bunnyMaterial.ambient = kRGB_BunnyPink;
    bunnyMaterial.diffuse = kRGB_BunnyPink;
    bunnyMaterial.specular = kRGB_Gray90;
    bunnyMaterial.shinyness = 2.0;
    bunnyMaterial.reflectionBlur = 0.0;

    /* set up a transform for the bunny */
    ConfigureIdentity(&bunnyTransform);

    /* set up a Gel for the bunny */
    SetGel(&bunnyGel, bunnyMesh, &bunnyMaterial, &bunnyTransform);

    /* compute bunny's BVH */
    bunnyGel.gelBVH = ConstructBVH(bunnyGel.gelMesh, kSpliceOnX);

    /* add the bunny Gel to the scene */
    AddGel(result, &bunnyGel);

    /* Just set up the scene a bit more */
    ConfigureTranslation(&workTransform, 0.0, -0.65, 0.0);
    TransformGel(&result->gelData[1], &workTransform);
    SetDirection3D(&rotationalAxis, 0.0, 1.0, 0.0);
    ConfigureRotation(&workTransform, &rotationalAxis, 0.0);
    TransformGel(&result->gelData[1], &workTransform);
    ConfigureScaling(&workTransform, 4.0, 4.0, 4.0);
    TransformGel(&result->gelData[1], &workTransform);

    return result;
}




Scene*  ConstructBoxDebuggingScene(void)
{
    Scene*                  result;

    Mesh*                   floorMesh;
    Gel                     floorGel;
    MaterialProps           floorMaterial;
    HomogeneousVector3D     tempVertex;
    int                     tempVertList[3];
    HomogeneousTransform3D  floorTrans;
    RGBColor                mainLightColor;
    Light                   mainLight;

    Mesh*                   cubeMesh;
    Gel                     cubeGel;
    MaterialProps           cubeMaterial;
    HomogeneousTransform3D  cubeTransform;

    result = ConstructBaseScene( );

    /* Create the two triangular floor tiles -- they should be a
       medium blue-green color */

    floorMesh = ConstructMesh(kFaceNormalMode);

    /* Append floor lower-left vertex */
    SetVertex3D(&tempVertex, -1.0, -0.5, 1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor lower-right vertex */
    SetVertex3D(&tempVertex, 1.0, -0.5, 1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor upper-right vertex */
    SetVertex3D(&tempVertex, 1.0, -0.5, -1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor upper-left vertex */
    SetVertex3D(&tempVertex, -1.0, -0.5, -1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append the floor lower-right triangle */
    tempVertList[0] = 0;
    tempVertList[1] = 1;
    tempVertList[2] = 2;
    AppendFace(floorMesh, tempVertList, 3);

    /* Append the floor upper-left triangle */
    tempVertList[0] = 0;
    tempVertList[1] = 2;
    tempVertList[2] = 3;
    AppendFace(floorMesh, tempVertList, 3);

    /* Set up the floor material's colors */
    floorMaterial.ambient = kRGB_SeaGreen;
    floorMaterial.diffuse = kRGB_SeaGreen;
    floorMaterial.specular = kRGB_Gray50;
    floorMaterial.shinyness = 2.0;
    floorMaterial.reflectionBlur = 0.10;

    /* set up the floor's transform */
    ConfigureIdentity(&floorTrans);

    /* Set up a Gel for the floor mesh */
    SetGel(&floorGel, floorMesh, &floorMaterial, &floorTrans);

    /* Add the floor Gel to the scene */
    AddGel(result, &floorGel);


    /* Create the cube -- it's made up of 12 triangles */

    cubeMesh = ConstructMesh(kFaceNormalMode);
    cubeMesh->isTriangleMesh = TRUE;

    /* cubeMesh->vertices[0] : top, front, left */
    SetVertex3D(&tempVertex, -0.5, 0.5, 0.5);
    AppendVertex(cubeMesh, &tempVertex);

    /* cubeMesh->vertices[1] : top, front, right */
    SetVertex3D(&tempVertex, 0.5, 0.5, 0.5);
    AppendVertex(cubeMesh, &tempVertex);

    /* cubeMesh->vertices[2] : top, rear, right */
    SetVertex3D(&tempVertex, 0.5, 0.5, -0.5);
    AppendVertex(cubeMesh, &tempVertex);

    /* cubeMesh->vertices[3] : top, rear, left */
    SetVertex3D(&tempVertex, -0.5, 0.5, -0.5);
    AppendVertex(cubeMesh, &tempVertex);

    /* cubeMesh->vertices[4] : bottom, front, left */
    SetVertex3D(&tempVertex, -0.5, -0.5, 0.5);
    AppendVertex(cubeMesh, &tempVertex);

    /* cubeMesh->vertices[5] : bottom, front, right */
    SetVertex3D(&tempVertex, 0.5, -0.5, 0.5);
    AppendVertex(cubeMesh, &tempVertex);

    /* cubeMesh->vertices[6] : bottom, rear, right */
    SetVertex3D(&tempVertex, 0.5, -0.5, -0.5);
    AppendVertex(cubeMesh, &tempVertex);

    /* cubeMesh->vertices[7] : bottom, rear, left */
    SetVertex3D(&tempVertex, -0.5, -0.5, -0.5);
    AppendVertex(cubeMesh, &tempVertex);

    /* cubeMesh->faces[0]: top LL face */
    tempVertList[0] = 0;
    tempVertList[1] = 1;
    tempVertList[2] = 3;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[1]: top UR face */
    tempVertList[0] = 1;
    tempVertList[1] = 2;
    tempVertList[2] = 3;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[2]: bottom LL face */
    tempVertList[0] = 5;
    tempVertList[1] = 4;
    tempVertList[2] = 7;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[3]: bottom LL face */
    tempVertList[0] = 5;
    tempVertList[1] = 7;
    tempVertList[2] = 6;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[4]: front LL face */
    tempVertList[0] = 5;
    tempVertList[1] = 0;
    tempVertList[2] = 4;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[5]: front UR face */
    tempVertList[0] = 5;
    tempVertList[1] = 1;
    tempVertList[2] = 0;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[6]: back LL face */
    tempVertList[0] = 7;
    tempVertList[1] = 3;
    tempVertList[2] = 6;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[7]: back UR face */
    tempVertList[0] = 6;
    tempVertList[1] = 3;
    tempVertList[2] = 2;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[8]: left LL face */
    tempVertList[0] = 7;
    tempVertList[1] = 4;
    tempVertList[2] = 3;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[9]: left UR face */
    tempVertList[0] = 4;
    tempVertList[1] = 0;
    tempVertList[2] = 3;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[10]: left LL face */
    tempVertList[0] = 5;
    tempVertList[1] = 6;
    tempVertList[2] = 1;
    AppendFace(cubeMesh, tempVertList, 3);

    /* cubeMesh->faces[11]: left UR face */
    tempVertList[0] = 6;
    tempVertList[1] = 2;
    tempVertList[2] = 1;
    AppendFace(cubeMesh, tempVertList, 3);

    /* Set up the cube material's colors */
    cubeMaterial.ambient = kRGB_BunnyPink;
    cubeMaterial.diffuse = kRGB_BunnyPink;
    cubeMaterial.specular = kRGB_Gray50;
    cubeMaterial.shinyness = 2.0;
    cubeMaterial.reflectionBlur = 0.25;

    /* Set up the cube's transform */
    ConfigureIdentity(&cubeTransform);

    /* Set up a Gel to package the cube */
    SetGel(&cubeGel, cubeMesh, &cubeMaterial, &cubeTransform);

    /* Add the Gel to the scene */
    AddGel(result, &cubeGel);

    /* Add a light back and to the right */
    SetVertex3D(&tempVertex, 1.0, 2.0, 1.0);
    SetRGBColor(&mainLightColor, 0.45, 0.45, 0.40);
    ConfigurePointLight(&mainLight, &mainLightColor, &tempVertex);
    AddLight(result, &mainLight);

    return result;
}




Scene*  ConstructFelineDebuggingScene(void)
{
    Scene*                  result;

    Mesh*                   floorMesh;
    Gel                     floorGel;
    MaterialProps           floorMaterial;
    HomogeneousVector3D     tempVertex;
    int                     tempVertList[3];
    HomogeneousTransform3D  floorTrans;
    RGBColor                mainLightColor;
    Light                   mainLight;
    const char*             bunnyFilename = "feline100.ob";
    Mesh*                   bunnyMesh;
    FILE*                   bunnyFilePointer = 0;
    Gel                     bunnyGel;
    MaterialProps           bunnyMaterial;
    HomogeneousTransform3D  bunnyTransform;

    result = ConstructBaseScene( );

    /* Create the two triangular floor tiles -- they should be a
       medium blue-green color */

    floorMesh = ConstructMesh(kFaceNormalMode);

    /* Append floor lower-left vertex */
    SetVertex3D(&tempVertex, -1.0, -0.5, 1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor lower-right vertex */
    SetVertex3D(&tempVertex, 1.0, -0.5, 1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor upper-right vertex */
    SetVertex3D(&tempVertex, 1.0, -0.5, -1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append floor upper-left vertex */
    SetVertex3D(&tempVertex, -1.0, -0.5, -1.0);
    AppendVertex(floorMesh, &tempVertex);

    /* Append the floor lower-right triangle */
    tempVertList[0] = 0;
    tempVertList[1] = 1;
    tempVertList[2] = 2;
    AppendFace(floorMesh, tempVertList, 3);

    /* Append the floor upper-left triangle */
    tempVertList[0] = 0;
    tempVertList[1] = 2;
    tempVertList[2] = 3;
    AppendFace(floorMesh, tempVertList, 3);

    /* Set up the floor material's colors */
    floorMaterial.ambient = kRGB_SeaGreen;
    floorMaterial.diffuse = kRGB_SeaGreen;
    floorMaterial.specular = kRGB_Gray50;
    floorMaterial.shinyness = 1.0;
    floorMaterial.reflectionBlur = 0.0;

    /* set up the floor's transform */
    ConfigureIdentity(&floorTrans);

    /* Set up a Gel for the floor mesh */
    SetGel(&floorGel, floorMesh, &floorMaterial, &floorTrans);

    /* Add the floor Gel to the scene */
    AddGel(result, &floorGel);

    /* Add a light back and to the right */
    SetVertex3D(&tempVertex, 2.0, 2.0, 2.0);
    SetRGBColor(&mainLightColor, 0.40, 0.40, 0.35);
    ConfigureGlobeLight(&mainLight, &mainLightColor, &tempVertex, 0.5);
    AddLight(result, &mainLight);

    /* Read in the bunny */
    bunnyFilePointer = fopen(bunnyFilename, "r");
    if (!bunnyFilePointer) {

        RuntimeError("ConstructFelineDebuggingScene( )",
            "couldn't open feline object data file");
    }

    bunnyMesh = ReadObjectData(bunnyFilePointer);

    /* set up a material for the bunny */
    bunnyMaterial.ambient = kRGB_BunnyPink;
    bunnyMaterial.diffuse = kRGB_BunnyPink;
    bunnyMaterial.specular = kRGB_Gray90;
    bunnyMaterial.shinyness = 2.0;
    bunnyMaterial.reflectionBlur = 0.0;

    /* set up a transform for the bunny */
    ConfigureIdentity(&bunnyTransform);

    /* set up a Gel for the bunny */
    SetGel(&bunnyGel, bunnyMesh, &bunnyMaterial, &bunnyTransform);

    /* compute bunny's BVH */
    bunnyGel.gelBVH = ConstructBVH(bunnyGel.gelMesh, kSpliceOnX);

    /* add the bunny Gel to the scene */
    AddGel(result, &bunnyGel);

    return result;
}
