/*

 ManagerPersonality.c

 Copyright (c) 2006-2007, Lucas Stephen Beeler and Jeremy Wagner-Kaiser.
 All Rights Reserved.

 */

#include "ManagerPersonality.h"
#include "RayEngine.h"
#include "Imaging.h"
#include <stdlib.h>

static  WorkPacketQueue*  mOutboundPackets = 0;

void  QueueInitialWorkPackets(const Scene*, const Image*);
void  DispatchWorkPacket(const WorkPacket*);
void  AccumulateWork(WorkPacket*, const Ray3D*, unsigned);
void  ReceiveResultPacket(ResultPacket*);
int*  CreateShuffledPixelArray(int);

static  void  BroadcastWorkPackets(void);

void  RunManagerPersonalityLoop(void)
{
    ResultPacket            inboundPacket;
    MPI_Request             receiveProbe;
    boolean                 wasPacketReceived = FALSE;
    Scene*                  workScene;
    Image*                  outputImage;
    ResultBuffer*           resultBuffer;
    WorkerStatusArray       workerStatusArray;

    /* make our inbound packet empty and set its opcode to null */
    MakeResultPacketEmpty(&inboundPacket);
    inboundPacket.opcode = kNullOpcode;

    /* Construct the scene object, the output image object,
       the outbound work packet queue, the result buffer,
       and the worker status array */
    workScene = ConstructBunnyDebuggingScene( );
    outputImage = ConstructImage(kImageTrueRGBKind, 640, 480, 8);
    mOutboundPackets = CreateWorkPacketQueue( );
    resultBuffer = ConstructResultBufferForImage(outputImage);
    workerStatusArray = CreateWorkerStatusArray( );

    PrintWorkerStatusArray(workerStatusArray);

    /* Construct initial work packet objects to package all the rays to
       be fired from the eye-point into the scene and enqueue them into
       the mOutboundPackets queue */
    QueueInitialWorkPackets(workScene, outputImage);

    printf("manager node: queued initial work packets.\n");

    /* Broadcast all the work packets in the queue out to worker nodes */
    BroadcastWorkPackets( );

    /* post the initial receive we'll use to get results from
       the worker nodes */
    MPI_Irecv(&inboundPacket, sizeof(ResultPacket), MPI_CHAR, MPI_ANY_SOURCE,
        0, MPI_COMM_WORLD, &receiveProbe);

    while (TRUE) {

        /* see if any results have come in */
        MPI_Test(&receiveProbe, &wasPacketReceived, MPI_STATUS_IGNORE);

        if (wasPacketReceived) {

            int i;

            if (inboundPacket.opcode == kNotifyWorkComplete) {

                printf("manager node: received notification of work complete "
                    "from worker (%d).\n", inboundPacket.sourcePID);

                MarkWorkerDone(workerStatusArray, inboundPacket.sourcePID);

                PrintWorkerStatusArray(workerStatusArray);

                if (AreAllWorkersDone(workerStatusArray)) {

                    printf("manager node: writing image data to disk.\n");

                    FoldResultBufferToImage(outputImage, resultBuffer);

                    SaveImage(outputImage, "netimage_out.tga");

                    printf("manager node: breaking out of personality "
                        "loop.\n");

                    break;
                }
            }
            else {

                for (i = 0; i < inboundPacket.numColors; i++) {

                    CompositeToResultBuffer(resultBuffer, inboundPacket.pixelIDs[i],
                        &inboundPacket.colorData[i]);
                }
            }

            MPI_Irecv(&inboundPacket, sizeof(ResultPacket), MPI_CHAR,
                MPI_ANY_SOURCE, 0, MPI_COMM_WORLD, &receiveProbe);

        } /* if packets were received */

    } /* infinte loop */

} /* RunManagerPersonalityLoop( ) */




void  QueueInitialWorkPackets(const Scene* scene, const Image* image)
{
    double  aspectRatio =
        ((double)(image->pixelWidth)) / ((double)(image->pixelHeight));
    double  fovyRadians = (scene->eye.fovy) * (kPi / 180.0);
    double  imagePlaneHeight =
        tan(fovyRadians) * scene->eye.imagePlaneDistance;
    double  imagePlaneWidth = aspectRatio * imagePlaneHeight;
    HomogeneousTransform3D  eyeToWorld;
    HomogeneousVector3D     pixPointEye;
    HomogeneousVector3D     pixPointWorld;
    HomogeneousVector3D     currRayDir;
    Ray3D                   currentRay;
    int i, j, k;
    RGBColor                pixColor = kRGB_Black;
    RGBColor                currentSampleColor;
    double                  pixPointZ = -scene->eye.imagePlaneDistance;
    int                     currentPixel;
    int                     numPixels =
        (image->pixelWidth) * (image->pixelHeight);
    WorkPacket              currentWorkPacket;
    int*                    shuffledPixelIDs;
    int                     counter;

    MakeWorkPacketEmpty(&currentWorkPacket);
    currentWorkPacket.sourcePID = gNodeInfo.nodePID;

    eyeToWorld.matrix[0]  = scene->eye.eyeSpaceX.x;
    eyeToWorld.matrix[1]  = scene->eye.eyeSpaceY.x;
    eyeToWorld.matrix[2]  = scene->eye.eyeSpaceZ.x;
    eyeToWorld.matrix[3]  = scene->eye.origin.x;
    eyeToWorld.matrix[4]  = scene->eye.eyeSpaceX.y;
    eyeToWorld.matrix[5]  = scene->eye.eyeSpaceY.y;
    eyeToWorld.matrix[6]  = scene->eye.eyeSpaceZ.y;
    eyeToWorld.matrix[7]  = scene->eye.origin.y;
    eyeToWorld.matrix[8]  = scene->eye.eyeSpaceX.z;
    eyeToWorld.matrix[9]  = scene->eye.eyeSpaceY.z;
    eyeToWorld.matrix[10] = scene->eye.eyeSpaceZ.z;
    eyeToWorld.matrix[11] = scene->eye.origin.z;
    eyeToWorld.matrix[12] = 0.0;
    eyeToWorld.matrix[13] = 0.0;
    eyeToWorld.matrix[14] = 0.0;
    eyeToWorld.matrix[15] = 1.0;

    /* create the shuffled pixel IDs array */
    shuffledPixelIDs = CreateShuffledPixelArray(numPixels);

    for (counter = 0; counter < numPixels; counter++) {

        currentPixel = shuffledPixelIDs[counter];

        i = currentPixel % (image->pixelWidth);
        j = currentPixel / (image->pixelWidth);

        pixColor = kRGB_Black;

        for (k = 0; k < pSamplesPerPixel; k++) {

            int  subPixelGridDimension = (int)
                floor(sqrt((double)(pSamplesPerPixel)));

            double  subPixelGridSquareSize =
                1.0 / ((double)(subPixelGridDimension));

            int  currentSampleGridX = k % subPixelGridDimension;
            int  currentSampleGridY = k / subPixelGridDimension;

            double  samplePosX =
                i + (currentSampleGridX * subPixelGridSquareSize);
            double  samplePosY =
                j + (currentSampleGridY * subPixelGridSquareSize);

            samplePosX +=
                SampleUniform( ) * subPixelGridSquareSize;
            samplePosY +=
                SampleUniform( ) * subPixelGridSquareSize;

            samplePosX = -(imagePlaneWidth / 2.0) +
                ((imagePlaneWidth * (samplePosX + 0.5)) /
                ((double) image->pixelWidth));
            samplePosY = -(imagePlaneHeight / 2.0) +
                ((imagePlaneHeight * (samplePosY + 0.5)) / 
                ((double) image->pixelHeight));

            SetVertex3D(&pixPointEye, samplePosX, samplePosY, pixPointZ);

            pixPointWorld = ApplyTransform(&eyeToWorld, &pixPointEye);

            currRayDir =
                VectorSubtract(&pixPointWorld, &scene->eye.origin);

            SetRay(&currentRay, &scene->eye.origin, &currRayDir);

            AccumulateWork(&currentWorkPacket, &currentRay, currentPixel);

        } /* for all samples taken at the current pixel */

    } /* for all pixels in image */

    if (currentWorkPacket.numRays != 0) {

        WorkPacketQueueElem  wrapper;

        wrapper.workPack = currentWorkPacket;

        EnqueueWorkPacket(mOutboundPackets, &wrapper);

        MakeWorkPacketEmpty(&currentWorkPacket);
    }

    free(shuffledPixelIDs);

} /* QueueInitialWorkPackets( ) */




void  AccumulateWork(WorkPacket* work, const Ray3D* ray, unsigned pixelID)
{
    AddRayToWorkPacket(work, ray, pixelID);

    if (work->numRays == kMaxPacketSize) {

        WorkPacketQueueElem  wrapper;

        wrapper.workPack = (*work);

        EnqueueWorkPacket(mOutboundPackets, &wrapper);

        MakeWorkPacketEmpty(work);
    }
}




void  BroadcastWorkPackets(void)
{
    MPI_Request       throwAway;
    int               targetWorker = 1;
    int               i;

    /*  set up the packet that we'll send out to the worker nodes to
        them that the initial work broadcast is complete */
    WorkPacket   broadcastDonePacket;
    broadcastDonePacket.opcode = kNotifyWorkBroadcastComplete;
    broadcastDonePacket.sourcePID = 0;

    printf("manager node: initiating work queue broadcast.\n");

    /* dequeue all the work packets in the work packet queue, set their
       opcodes so that the receiving worker accumulates them, and
       send them on their way */
    while (!IsWorkPacketQueueEmpty(mOutboundPackets)) {

            WorkPacketQueueElem  packetWrapper
                = DequeueWorkPacket(mOutboundPackets);

            packetWrapper.workPack.opcode = kAccumulateWork;

            MPI_Send(&packetWrapper.workPack, sizeof(WorkPacket), MPI_CHAR,
                targetWorker, 0, MPI_COMM_WORLD);

            targetWorker = NextWorkerNode(targetWorker);
    }

    printf("manager node: completed work queue broadcast.\n");

    printf("manager node: notifying workers that broadcast is complete.\n");

    for (i = 1; i < gNodeInfo.numNodes; i++) {

        MPI_Send(&broadcastDonePacket, sizeof(WorkPacket), MPI_CHAR,
            i, 0, MPI_COMM_WORLD);
    }

    printf("manager node: notified workers.\n");
}




int*  CreateShuffledPixelArray(int n)
{
    int*   result;
    int    i;

    result = malloc(sizeof(int) * n);

    if (!result) {

        RuntimeError("CreateShuffledPixelArray( )", "out of memory");
    }

    for (i = 0; i < n; i++) {

        result[i] = i;
    }

    RandomPermutation(result, n);

    return result;
}
