/*

 LightingIntrinsics.c

 Copyright (c) 2006, Lucas Stephen Beeler. All Rights Reserved.

 */

#include <stdio.h>
#include "LightingIntrinsics.h"
#include <math.h>
#include <stdlib.h>

HomogeneousVector3D  SampleLightPosition(const Light* target,
    const HomogeneousVector3D* sampleSource)
{
    return (target->sampleFunction)(&target->origin, sampleSource,
        target->personality);
}




HomogeneousVector3D
    uPointLightSampleProc(const HomogeneousVector3D* lightOrigin,
        const HomogeneousVector3D* sampleSrc,
        LightPersonalityPointer personalityPtr)
{
    return *lightOrigin;
}




void  ConfigurePointLight(Light*  target, const RGBColor*  userColor,
    const HomogeneousVector3D*  userPosition)
{
    target->origin         = (*userPosition);
    target->kind           = kPointLight;
    target->color          = (*userColor);
    target->sampleFunction = uPointLightSampleProc;
    target->personality    = 0;
}



struct  GlobeLightPersonality_REP {

    double  radius;
};
typedef struct GlobeLightPersonality_REP GlobeLightPersonality;

HomogeneousVector3D
    uGlobeLightSampleProc(const HomogeneousVector3D* lightOrigin,
        const HomogeneousVector3D* sampleSrc,
        LightPersonalityPointer personalityPtr)
{
    CoordinateSystem3D       sampleSystem;
    HomogeneousVector3D      shineDirection;
    HomogeneousVector3D      resultInSampleSpace;
    HomogeneousVector3D      result;
    double                   sampleTheta;
    double                   sampleRadiusScale;
    HomogeneousTransform3D   sampleToWorld;

    GlobeLightPersonality*   litePersonality =
        (GlobeLightPersonality*) personalityPtr;

    shineDirection = VectorSubtract(sampleSrc, lightOrigin);

    sampleSystem = ComputeZDirectedSystem(&shineDirection);

    sampleTheta = 2 * kPi * SampleUniform( );
    sampleRadiusScale = SampleUniform( );

    resultInSampleSpace.x =
        cos(sampleTheta) * sampleRadiusScale * litePersonality->radius;
    resultInSampleSpace.y =
        -sin(sampleTheta) * sampleRadiusScale * litePersonality->radius;

    resultInSampleSpace.z = 0.0;

    resultInSampleSpace.w = 1.0;

    ConfigureIdentity(&sampleToWorld);

    sampleToWorld.matrix[0] = sampleSystem.basisX.x;
    sampleToWorld.matrix[1] = sampleSystem.basisY.x;
    sampleToWorld.matrix[2] = sampleSystem.basisZ.x;

    sampleToWorld.matrix[4] = sampleSystem.basisX.y;
    sampleToWorld.matrix[5] = sampleSystem.basisY.y;
    sampleToWorld.matrix[6] = sampleSystem.basisZ.y;

    sampleToWorld.matrix[8]  = sampleSystem.basisX.z;
    sampleToWorld.matrix[9]  = sampleSystem.basisY.z;
    sampleToWorld.matrix[10] = sampleSystem.basisZ.z;

    sampleToWorld.matrix[3] = lightOrigin->x;
    sampleToWorld.matrix[7] = lightOrigin->y;
    sampleToWorld.matrix[11] = lightOrigin->z;

    result = ApplyTransform(&sampleToWorld, &resultInSampleSpace);

    return result;
}




void  ConfigureGlobeLight(Light* target, const RGBColor* userColor,
    const HomogeneousVector3D* userPosition, double userRadius)
{
    GlobeLightPersonality*  userLightPersonality =
        malloc(sizeof(GlobeLightPersonality));

    if (!userLightPersonality) {

        RuntimeError("ConfigureGlobeLight( )", "out of memory");
    }

    userLightPersonality->radius = userRadius;

    target->kind = kGlobeLight;
    target->color = *userColor;
    target->origin = *userPosition;
    target->sampleFunction = uGlobeLightSampleProc;
    target->personality = userLightPersonality;
}




void  PrintLight(const Light* target)
{
    printf("\nLight:\n");
    printf("\tkind              = ");
    if (target->kind == kPointLight) {

        printf("kPointLight\n");
    }
    else if (target->kind == kGlobeLight) {

        printf("kGlobeLight\n");
    }
    else {

        printf("(unknown light kind)\n");
    }
    printf("\torigin            = ");
    PrintVector(&target->origin);
    printf("\tcolor             = ");
    PrintRGBColor(&target->color);
    printf("\tsampling function = %#x\n",
        (unsigned long) target->sampleFunction);
    if (target->kind == kGlobeLight) {

        GlobeLightPersonality*  lightPersona =
            (GlobeLightPersonality*) target->personality;

        printf("\tradius             = %f\n", lightPersona->radius);
    }
}




