/*

 GeometryIntrinsics.c

 Implements functions over types necessary to support most of the basic
 computational geometry operations required for 3D graphics, such as
 operations over vectors (add, scalar multiply, etc.), over transforms,
 etc.

 Copyright (c) 2006, Lucas Stephen Beeler. All Rights Reserved.

 */

#include "ApplicationServices.h"
#include "GeometryIntrinsics.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

const double  kPi         = 3.14159265358979323846;
const double  kBigDouble  = 1.0e34;

const RGBColor kRGB_Black = {0.0, 0.0, 0.0};
const RGBColor kRGB_White = {1.0, 1.0, 1.0};
const RGBColor kRGB_SeaGreen = {0.2, 0.6, 0.6};
const RGBColor kRGB_Gray90 = {0.9, 0.9, 0.9};
const RGBColor kRGB_BunnyPink = { 0.6, 0.5, 0.6 };
const RGBColor kRGB_Gray50 = { 0.5, 0.5, 0.5 };


void  SetVertex3D(HomogeneousVector3D* target, double userX, double userY,
    double userZ)
{
    target->x = userX;
    target->y = userY;
    target->z = userZ;
    target->w = 1.0;   /* we're configuring this vector as a vertex */
}




void  SetDirection3D(HomogeneousVector3D* target, double userX, double userY,
    double userZ)
{
    target->x = userX;
    target->y = userY;
    target->z = userZ;
    target->w = 0.0;    /* we're configuring this vector as a direction */
}




HomogeneousVector3D  ScalarMultiply(const HomogeneousVector3D* in, double s)
{
    HomogeneousVector3D  result = (*in);

    result.x = s * result.x;
    result.y = s * result.y;
    result.z = s * result.z;
    /* don't do anything with w -- it's the homogeneous coordinate */

    return result;
}




HomogeneousVector3D  VectorAdd(const HomogeneousVector3D* term1,
    const HomogeneousVector3D* term2)
{
    HomogeneousVector3D  result = (*term1);

    result.x += term2->x;
    result.y += term2->y;
    result.z += term2->z;
    result.w += term2->w;

    /* make sure the client programmer didn't make a logical error by
       trying to add two position vectors; this is, of course, undefined */
    if (result.w > 1)
        LogicError("VectorAdd( )", "attempted to add two vertex vectors");

    return result;
}




HomogeneousVector3D  VectorSubtract(const HomogeneousVector3D* subtractor,
    const HomogeneousVector3D* subtractee)
{
    HomogeneousVector3D  result = (*subtractor);

    result.x -= subtractee->x;
    result.y -= subtractee->y;
    result.z -= subtractee->z;
    result.w -= subtractee->w;

    return result;
}




double  VectorDot(const HomogeneousVector3D* v1, const HomogeneousVector3D* v2)
{
    double result = 0.0;

    result += v1->x * v2->x;
    result += v1->y * v2->y;
    result += v1->z * v2->z;

    return result;
}




HomogeneousVector3D  VectorCross(const HomogeneousVector3D* v1,
    const HomogeneousVector3D* v2)
{
    HomogeneousVector3D  result;

    result.x = (v1->y * v2->z) - (v1->z * v2->y);
    result.y = (v1->z * v2->x) - (v1->x * v2->z);
    result.z = (v1->x * v2->y) - (v1->y * v2->x);

    result.w = 0.0;

    return result;
}




double  VectorLength(const HomogeneousVector3D* in)
{
    return sqrt((in->x * in->x) + (in->y * in->y) + (in->z * in->z));
}




HomogeneousVector3D  VectorNormalize(const HomogeneousVector3D* in)
{
    double  invLength = (1.0 / VectorLength(in));

    return  ScalarMultiply(in, invLength);
}




void  PrintVector(const HomogeneousVector3D* in)
{
    printf("(%.4f, %.4f, %.4f, %.4f).\n", in->x, in->y, in->z, in->w);
}




HomogeneousVector3D  ZeroPositionVector( )
{
    HomogeneousVector3D  result;

    result.x = 0.0;
    result.y = 0.0;
    result.z = 0.0;
    result.w = 1.0;

    return result;
}




void  ConfigureTranslation(HomogeneousTransform3D* target, double xOffset,
    double yOffset, double zOffset)
{
    target->matrix[0]  = 1.0;
    target->matrix[1]  = 0.0;
    target->matrix[2]  = 0.0;
    target->matrix[3]  = xOffset;

    target->matrix[4]  = 0.0;
    target->matrix[5]  = 1.0;
    target->matrix[6]  = 0.0;
    target->matrix[7]  = yOffset;
    
    target->matrix[8]  = 0.0;
    target->matrix[9]  = 0.0;
    target->matrix[10] = 1.0;
    target->matrix[11] = zOffset;
    
    target->matrix[12] = 0.0;
    target->matrix[13] = 0.0;
    target->matrix[14] = 0.0;
    target->matrix[15] = 1.0;
}




void  ConfigureScaling(HomogeneousTransform3D* target, double xScale,
    double yScale, double zScale)
{
    target->matrix[0]  = xScale;
    target->matrix[1]  = 0.0;
    target->matrix[2]  = 0.0;
    target->matrix[3]  = 0.0;

    target->matrix[4]  = 0.0;
    target->matrix[5]  = yScale;
    target->matrix[6]  = 0.0;
    target->matrix[7]  = 0.0;
    
    target->matrix[8]  = 0.0;
    target->matrix[9]  = 0.0;
    target->matrix[10] = zScale;
    target->matrix[11] = 0.0;

    target->matrix[12] = 0.0;
    target->matrix[13] = 0.0;
    target->matrix[14] = 0.0;
    target->matrix[15] = 1.0;
}




void  ConfigureRotation(HomogeneousTransform3D* target,
    const HomogeneousVector3D* axis, double thetaDegrees)
{
    double thetaRadians = (thetaDegrees / 180.0) * kPi;

    CoordinateSystem3D      rotBasis;
    HomogeneousVector3D     rotationalBasisX;
    HomogeneousVector3D     rotationalBasisY;
    HomogeneousVector3D     rotationalBasisZ;
    HomogeneousTransform3D  composeElement;

#ifdef DIAGNOSTICS_ON
    HomogeneousTransform3D  diagnosticsMatrix1;
    HomogeneousTransform3D  diagnosticsMatrix2;
#endif

    if (axis->w != 0.0) {

        LogicError("ConfigureRotation( )", "rotational axis must be a "
            "direction, not a vertex");
    }

    /* (1) form an orthonormal basis with +z = axis */
    rotBasis = ComputeZDirectedSystem(axis);

    rotationalBasisX = rotBasis.basisX;
    rotationalBasisY = rotBasis.basisY;
    rotationalBasisZ = rotBasis.basisZ;

#ifdef DIAGNOSTICS_ON
    printf("\n");
    printf("rotational basis X = ");
    PrintVector(&rotationalBasisX);
    printf("rotational basis Y = ");
    PrintVector(&rotationalBasisY);
    printf("rotational basis Z = ");
    PrintVector(&rotationalBasisZ);
    printf("\n");
#endif


    /* (2) form the first elemental matrix */

    ConfigureIdentity(&composeElement);

    composeElement.matrix[0] = rotationalBasisX.x;
    composeElement.matrix[1] = rotationalBasisY.x;
    composeElement.matrix[2] = rotationalBasisZ.x;

    composeElement.matrix[4] = rotationalBasisX.y;
    composeElement.matrix[5] = rotationalBasisY.y;
    composeElement.matrix[6] = rotationalBasisZ.y;

    composeElement.matrix[8] = rotationalBasisX.z;
    composeElement.matrix[9] = rotationalBasisY.z;
    composeElement.matrix[10] = rotationalBasisZ.z;

#ifdef DIAGNOSTICS_ON
    printf("\n");
    diagnosticsMatrix1 = composeElement;
    printf("first elemental matrix = \n");
    PrintTransform(&diagnosticsMatrix1);
    printf("\n");
#endif

    /* (3) set the target equal to the first elemental matrix */

    (*target) = composeElement;


    /* (4) form the second elemental (rotation) matrix */

    ConfigureIdentity(&composeElement);

    composeElement.matrix[0] = cos(thetaRadians);
    composeElement.matrix[1] = -sin(thetaRadians);

    composeElement.matrix[4] = sin(thetaRadians);
    composeElement.matrix[5] = cos(thetaRadians);


    /* (5) compose the target against the second elemental matrix */

    ComposeTransformsTo(target, &composeElement);


    /* (6) form the third elemental matrix */

    ConfigureIdentity(&composeElement);

    composeElement.matrix[0] = rotationalBasisX.x;
    composeElement.matrix[1] = rotationalBasisX.y;
    composeElement.matrix[2] = rotationalBasisX.z;

    composeElement.matrix[4] = rotationalBasisY.x;
    composeElement.matrix[5] = rotationalBasisY.y;
    composeElement.matrix[6] = rotationalBasisY.z;

    composeElement.matrix[8] = rotationalBasisZ.x;
    composeElement.matrix[9] = rotationalBasisZ.y;
    composeElement.matrix[10] = rotationalBasisZ.z;

#ifdef DIAGNOSTICS_ON
    printf("\n");
    printf("third elemental matrix = \n");
    diagnosticsMatrix2 = composeElement;
    PrintTransform(&diagnosticsMatrix2);
    printf("\n");
    printf("first x third = \n");
    ComposeTransformsTo(&diagnosticsMatrix1, &diagnosticsMatrix2);
    PrintTransform(&diagnosticsMatrix1);
    printf("\n\n\n\n\n");
#endif

    /* (7) compose the target against the third elemental matrix */

    ComposeTransformsTo(target, &composeElement);

    /* (8) DONE -- target is fully composed */
}





void  ConfigureIdentity(HomogeneousTransform3D* target)
{
    target->matrix[0]  = 1.0;
    target->matrix[1]  = 0.0;
    target->matrix[2]  = 0.0;
    target->matrix[3]  = 0.0;

    target->matrix[4]  = 0.0;
    target->matrix[5]  = 1.0;
    target->matrix[6]  = 0.0;
    target->matrix[7]  = 0.0;
    
    target->matrix[8]  = 0.0;
    target->matrix[9]  = 0.0;
    target->matrix[10] = 1.0;
    target->matrix[11] = 0.0;

    target->matrix[12] = 0.0;
    target->matrix[13] = 0.0;
    target->matrix[14] = 0.0;
    target->matrix[15] = 1.0;
}




HomogeneousVector3D  ApplyTransform(const HomogeneousTransform3D* trans,
    const HomogeneousVector3D* vec)
{
    HomogeneousVector3D result;

    result.x = (trans->matrix[0] * vec->x) + (trans->matrix[1] * vec->y) +
        (trans->matrix[2] * vec->z) + (trans->matrix[3] * vec->w);

    result.y = (trans->matrix[4] * vec->x) + (trans->matrix[5] * vec->y) +
        (trans->matrix[6] * vec->z) + (trans->matrix[7] * vec->w);

    result.z = (trans->matrix[8] * vec->x) + (trans->matrix[9] * vec->y) +
        (trans->matrix[10] * vec->z) + (trans->matrix[11] * vec->w);

    result.w = vec->w;

    return result;
}




void  ComposeTransformsTo(HomogeneousTransform3D* target,
    const HomogeneousTransform3D* src)
{
    HomogeneousTransform3D  temp;

    temp.matrix[0]  = (target->matrix[0] * src->matrix[0]) +
        (target->matrix[1] * src->matrix[4]) + (target->matrix[2] *
        src->matrix[8]) + (target->matrix[3] * src->matrix[12]);

    temp.matrix[1]  = (target->matrix[0] * src->matrix[1]) +
        (target->matrix[1] * src->matrix[5]) + (target->matrix[2] *
        src->matrix[9]) + (target->matrix[3] * src->matrix[13]);

    temp.matrix[2]  = (target->matrix[0] * src->matrix[2]) +
        (target->matrix[1] * src->matrix[6]) + (target->matrix[2] *
        src->matrix[10]) + (target->matrix[3] * src->matrix[14]);

    temp.matrix[3]  = (target->matrix[0] * src->matrix[3]) +
        (target->matrix[1] * src->matrix[7]) + (target->matrix[2] *
        src->matrix[11]) + (target->matrix[3] * src->matrix[15]);

    temp.matrix[4]  = (target->matrix[4] * src->matrix[0]) +
        (target->matrix[5] * src->matrix[4]) + (target->matrix[6] *
        src->matrix[8]) + (target->matrix[7] * src->matrix[12]);

    temp.matrix[5]  = (target->matrix[4] * src->matrix[1]) +
        (target->matrix[5] * src->matrix[5]) + (target->matrix[6] *
        src->matrix[9]) + (target->matrix[7] * src->matrix[13]);

    temp.matrix[6]  = (target->matrix[4] * src->matrix[2]) +
        (target->matrix[5] * src->matrix[6]) + (target->matrix[6] *
        src->matrix[10]) + (target->matrix[7] * src->matrix[14]);

    temp.matrix[7]  = (target->matrix[4] * src->matrix[3]) +
        (target->matrix[5] * src->matrix[7]) + (target->matrix[6] *
        src->matrix[11]) + (target->matrix[7] * src->matrix[15]);
    
    temp.matrix[8]  = (target->matrix[8] * src->matrix[0]) +
        (target->matrix[9] * src->matrix[4]) + (target->matrix[10] *
        src->matrix[8]) + (target->matrix[11] * src->matrix[12]);

    temp.matrix[9]  = (target->matrix[8] * src->matrix[1]) +
        (target->matrix[9] * src->matrix[5]) + (target->matrix[10] *
        src->matrix[9]) + (target->matrix[11] * src->matrix[13]);

    temp.matrix[10]  = (target->matrix[8] * src->matrix[2]) +
        (target->matrix[9] * src->matrix[6]) + (target->matrix[10] *
        src->matrix[10]) + (target->matrix[11] * src->matrix[14]);

    temp.matrix[11]  = (target->matrix[8] * src->matrix[3]) +
        (target->matrix[9] * src->matrix[7]) + (target->matrix[10] *
        src->matrix[11]) + (target->matrix[11] * src->matrix[15]);

    temp.matrix[12]  = (target->matrix[12] * src->matrix[0]) +
        (target->matrix[13] * src->matrix[4]) + (target->matrix[14] *
        src->matrix[8]) + (target->matrix[15] * src->matrix[12]);

    temp.matrix[13]  = (target->matrix[12] * src->matrix[1]) +
        (target->matrix[13] * src->matrix[5]) + (target->matrix[14] *
        src->matrix[9]) + (target->matrix[15] * src->matrix[13]);

    temp.matrix[14]  = (target->matrix[12] * src->matrix[2]) +
        (target->matrix[13] * src->matrix[6]) + (target->matrix[14] *
        src->matrix[10]) + (target->matrix[15] * src->matrix[14]);

    temp.matrix[15]  = (target->matrix[12] * src->matrix[3]) +
        (target->matrix[13] * src->matrix[7]) + (target->matrix[14] *
        src->matrix[11]) + (target->matrix[15] * src->matrix[15]);

    (*target) = temp;
}




void  TransposeTransform(HomogeneousTransform3D* in)
{
    HomogeneousTransform3D  result;
    int i;

    for (i = 0; i < 4; i++) {

        result.matrix[i] = in->matrix[i*4];
        result.matrix[i + 4] = in->matrix[i*4 + 1];
        result.matrix[i + 8] = in->matrix[i*4 + 2];
        result.matrix[i + 12] = in->matrix[i*4 + 3];
    }

    (*in) = result;
}




void  InvertTransform(HomogeneousTransform3D* in)
{
    HomogeneousTransform3D  inTranspose = (*in);
    double                  cofactorPairs[12];
    double                  det;
    int                     j;

    TransposeTransform(&inTranspose);

    cofactorPairs[0] = inTranspose.matrix[10] * inTranspose.matrix[15];
    cofactorPairs[1] = inTranspose.matrix[11] * inTranspose.matrix[14];
    cofactorPairs[2] = inTranspose.matrix[9] * inTranspose.matrix[15];
    cofactorPairs[3] = inTranspose.matrix[11] * inTranspose.matrix[13];
    cofactorPairs[4] = inTranspose.matrix[9] * inTranspose.matrix[14];
    cofactorPairs[5] = inTranspose.matrix[10] * inTranspose.matrix[13];
    cofactorPairs[6] = inTranspose.matrix[8] * inTranspose.matrix[15];
    cofactorPairs[7] = inTranspose.matrix[11] * inTranspose.matrix[12];
    cofactorPairs[8] = inTranspose.matrix[8] * inTranspose.matrix[14];
    cofactorPairs[9] = inTranspose.matrix[10] * inTranspose.matrix[12];
    cofactorPairs[10] = inTranspose.matrix[8] * inTranspose.matrix[13];
    cofactorPairs[11] = inTranspose.matrix[9] * inTranspose.matrix[12];

    in->matrix[0] =  (cofactorPairs[0] * inTranspose.matrix[5]) +
        (cofactorPairs[3] * inTranspose.matrix[6]) +
        (cofactorPairs[4] * inTranspose.matrix[7]);
    
    in->matrix[0] -= (cofactorPairs[1] * inTranspose.matrix[5]) +
        (cofactorPairs[2] * inTranspose.matrix[6]) +
        (cofactorPairs[5] * inTranspose.matrix[7]);

    in->matrix[1] =  (cofactorPairs[1] * inTranspose.matrix[4]) +
        (cofactorPairs[6] * inTranspose.matrix[6]) +
        (cofactorPairs[9] * inTranspose.matrix[7]);

    in->matrix[1] -= (cofactorPairs[0] * inTranspose.matrix[4]) +
        (cofactorPairs[7] * inTranspose.matrix[6]) +
        (cofactorPairs[8] * inTranspose.matrix[7]);

    in->matrix[2] =  (cofactorPairs[2] * inTranspose.matrix[4]) +
        (cofactorPairs[7] * inTranspose.matrix[5]) +
        (cofactorPairs[10] * inTranspose.matrix[7]);

    in->matrix[2] -= (cofactorPairs[3] * inTranspose.matrix[4]) +
        (cofactorPairs[6] * inTranspose.matrix[5]) +
        (cofactorPairs[11] * inTranspose.matrix[7]);

    in->matrix[3] =  (cofactorPairs[5] * inTranspose.matrix[4]) +
        (cofactorPairs[8] * inTranspose.matrix[5]) +
        (cofactorPairs[11]*inTranspose.matrix[6]);

    in->matrix[3] -= (cofactorPairs[4] * inTranspose.matrix[4]) +
        (cofactorPairs[9] * inTranspose.matrix[5]) +
        (cofactorPairs[10] * inTranspose.matrix[6]);

    in->matrix[4] =  (cofactorPairs[1] * inTranspose.matrix[1]) +
        (cofactorPairs[2] * inTranspose.matrix[2]) +
        (cofactorPairs[5] * inTranspose.matrix[3]);

    in->matrix[4] -= (cofactorPairs[0] * inTranspose.matrix[1]) +
        (cofactorPairs[3] * inTranspose.matrix[2]) +
        (cofactorPairs[4]*inTranspose.matrix[3]);

    in->matrix[5] =  (cofactorPairs[0] * inTranspose.matrix[0]) +
        (cofactorPairs[7] * inTranspose.matrix[2]) +
        (cofactorPairs[8] * inTranspose.matrix[3]);

    in->matrix[5] -= (cofactorPairs[1] * inTranspose.matrix[0]) +
        (cofactorPairs[6] * inTranspose.matrix[2]) +
        (cofactorPairs[9] * inTranspose.matrix[3]);

    in->matrix[6] =  (cofactorPairs[3] * inTranspose.matrix[0]) +
        (cofactorPairs[6] * inTranspose.matrix[1]) +
        (cofactorPairs[11] * inTranspose.matrix[3]);

    in->matrix[6] -= (cofactorPairs[2] * inTranspose.matrix[0]) +
        (cofactorPairs[7] * inTranspose.matrix[1]) +
        (cofactorPairs[10] * inTranspose.matrix[3]);

    in->matrix[7] =  (cofactorPairs[4] * inTranspose.matrix[0]) +
        (cofactorPairs[9] * inTranspose.matrix[1]) +
        (cofactorPairs[10] * inTranspose.matrix[2]);

    in->matrix[7] -= (cofactorPairs[5] * inTranspose.matrix[0]) +
        (cofactorPairs[8] * inTranspose.matrix[1]) +
        (cofactorPairs[11]*inTranspose.matrix[2]);

    cofactorPairs[0] =  inTranspose.matrix[2] * inTranspose.matrix[7];
    cofactorPairs[1] =  inTranspose.matrix[3] * inTranspose.matrix[6];
    cofactorPairs[2] =  inTranspose.matrix[1] * inTranspose.matrix[7];
    cofactorPairs[3] =  inTranspose.matrix[3] * inTranspose.matrix[5];
    cofactorPairs[4] =  inTranspose.matrix[1] * inTranspose.matrix[6];
    cofactorPairs[5] =  inTranspose.matrix[2] * inTranspose.matrix[5];
    cofactorPairs[6] =  inTranspose.matrix[0] * inTranspose.matrix[7];
    cofactorPairs[7] =  inTranspose.matrix[3] * inTranspose.matrix[4];
    cofactorPairs[8] =  inTranspose.matrix[0] * inTranspose.matrix[6];
    cofactorPairs[9] =  inTranspose.matrix[2] * inTranspose.matrix[4];
    cofactorPairs[10] = inTranspose.matrix[0] * inTranspose.matrix[5];
    cofactorPairs[11] = inTranspose.matrix[1] * inTranspose.matrix[4];

    in->matrix[8]   =  (cofactorPairs[0] * inTranspose.matrix[13]) +
        (cofactorPairs[3] * inTranspose.matrix[14]) +
        (cofactorPairs[4] * inTranspose.matrix[15]);

    in->matrix[8]  -=  (cofactorPairs[1] * inTranspose.matrix[13]) +
        (cofactorPairs[2] * inTranspose.matrix[14]) +
        (cofactorPairs[5] * inTranspose.matrix[15]);

    in->matrix[9]   =  (cofactorPairs[1] * inTranspose.matrix[12]) +
        (cofactorPairs[6] * inTranspose.matrix[14]) +
        (cofactorPairs[9] * inTranspose.matrix[15]);

    in->matrix[9]  -=  (cofactorPairs[0] * inTranspose.matrix[12]) +
        (cofactorPairs[7] * inTranspose.matrix[14]) +
        (cofactorPairs[8] * inTranspose.matrix[15]);

    in->matrix[10]  =  (cofactorPairs[2] * inTranspose.matrix[12]) +
        (cofactorPairs[7] * inTranspose.matrix[13]) +
        (cofactorPairs[10] * inTranspose.matrix[15]);

    in->matrix[10] -=  (cofactorPairs[3] * inTranspose.matrix[12]) +
        (cofactorPairs[6] * inTranspose.matrix[13]) +
        (cofactorPairs[11] * inTranspose.matrix[15]);

    in->matrix[11]  =  (cofactorPairs[5] * inTranspose.matrix[12]) +
        (cofactorPairs[8] * inTranspose.matrix[13]) +
        (cofactorPairs[11] * inTranspose.matrix[14]);

    in->matrix[11] -=  (cofactorPairs[4] * inTranspose.matrix[12]) +
        (cofactorPairs[9] * inTranspose.matrix[13]) +
        (cofactorPairs[10] * inTranspose.matrix[14]);

    in->matrix[12]  =  (cofactorPairs[2] * inTranspose.matrix[10]) +
        (cofactorPairs[5] * inTranspose.matrix[11]) +
        (cofactorPairs[1] * inTranspose.matrix[9]);

    in->matrix[12] -=  (cofactorPairs[4] * inTranspose.matrix[11]) +
        (cofactorPairs[0] * inTranspose.matrix[9]) +
        (cofactorPairs[3] * inTranspose.matrix[10]);

    in->matrix[13]  =  (cofactorPairs[8] * inTranspose.matrix[11]) +
        (cofactorPairs[0] * inTranspose.matrix[8]) +
        (cofactorPairs[7] * inTranspose.matrix[10]);

    in->matrix[13] -=  (cofactorPairs[6] * inTranspose.matrix[10]) +
        (cofactorPairs[9] * inTranspose.matrix[11]) +
        (cofactorPairs[1] * inTranspose.matrix[8]);

    in->matrix[14]  =  (cofactorPairs[6] * inTranspose.matrix[9]) +
        (cofactorPairs[11] * inTranspose.matrix[11]) +
        (cofactorPairs[3] * inTranspose.matrix[8]);

    in->matrix[14] -=  (cofactorPairs[10] * inTranspose.matrix[11]) +
        (cofactorPairs[2] * inTranspose.matrix[8]) +
        (cofactorPairs[7] * inTranspose.matrix[9]);

    in->matrix[15]  =  (cofactorPairs[10] * inTranspose.matrix[10]) +
        (cofactorPairs[4] * inTranspose.matrix[8]) +
        (cofactorPairs[9] * inTranspose.matrix[9]);

    in->matrix[15] -=  (cofactorPairs[8] * inTranspose.matrix[9]) +
        (cofactorPairs[11] * inTranspose.matrix[10]) +
        (cofactorPairs[5] * inTranspose.matrix[8]);

    det = (inTranspose.matrix[0] * in->matrix[0]) + (inTranspose.matrix[1] * in->matrix[1])
        + (inTranspose.matrix[2] * in->matrix[2]) + (inTranspose.matrix[3] * in->matrix[3]);

    det = (1.0 / det);
    for (j = 0; j < 16; j++) {

        in->matrix[j] *= det;
    }
}




void  PrintTransform(const HomogeneousTransform3D* in)
{
    printf("[ %8.4f %8.4f %8.4f %8.4f ]\n", in->matrix[0], in->matrix[1],
        in->matrix[2], in->matrix[3]);

    printf("[ %8.4f %8.4f %8.4f %8.4f ]\n", in->matrix[4], in->matrix[5],
        in->matrix[6], in->matrix[7]);

    printf("[ %8.4f %8.4f %8.4f %8.4f ]\n", in->matrix[8], in->matrix[9],
        in->matrix[10], in->matrix[11]);

    printf("[ %8.4f %8.4f %8.4f %8.4f ]\n", in->matrix[12], in->matrix[13],
        in->matrix[14], in->matrix[15]);
}




void  RGBClamp(RGBColor* in)
{
    if (in->r < 0.0)
        in->r = 0.0;
    else if (in->r > 1.0)
        in->r = 1.0;

    if (in->g < 0.0)
        in->g = 0.0;
    else if (in->g > 1.0)
        in->g = 1.0;

    if (in->b < 0.0)
        in->b = 0.0;
    else if (in->b > 1.0)
        in->b = 1.0;
}




void      SetRGBColor(RGBColor* in, double userR, double userG, double userB)
{
    in->r = userR;
    in->g = userG;
    in->b = userB;

    RGBClamp(in);
}




RGBColor  RGBColorAdd(const RGBColor* in1, const RGBColor* in2)
{
    RGBColor  result;

    result.r = in1->r + in2->r;
    result.g = in1->g + in2->g;
    result.b = in1->b + in2->b;

    RGBClamp(&result);

    return result;
}



RGBColor  RGBColorMultiply(const RGBColor* in1, const RGBColor* in2)
{
    RGBColor  result;

    result.r = in1->r * in2->r;
    result.g = in1->g * in2->g;
    result.b = in1->b * in2->b;

    RGBClamp(&result);

    return result;
}



RGBColor  RGBColorScale(const RGBColor* in1, double s)
{
    RGBColor  result;

    result.r = in1->r * s;
    result.g = in1->g * s;
    result.b = in1->b * s;

    RGBClamp(&result);

    return result;
}




void  PrintRGBColor(const RGBColor* target)
{
    printf("(r = %.2f; g = %.2f; b = %.2f)\n", target->r, target->g,
        target->b);
}




void  SetRay(Ray3D* target, const HomogeneousVector3D* userOrigin,
    const HomogeneousVector3D* userDirection)
{
    HomogeneousVector3D  epsilon;

    epsilon = ScalarMultiply(userDirection, 0.001);
    target->origin = VectorAdd(userOrigin, &epsilon);
    target->direction = *userDirection;
}




Ray3D  TransformRay(const HomogeneousTransform3D* userTrans, const Ray3D* userRay)
{
    Ray3D  result;

    result.origin = ApplyTransform(userTrans, &userRay->origin);
    result.direction = ApplyTransform(userTrans, &userRay->direction);

    return result;
}




void   SetBoundingBox(BoundingBox3D* target, double userXMin, double userXMax,
    double userYMin, double userYMax, double userZMin, double userZMax)
{
    target->xMin = userXMin;
    target->xMax = userXMax;

    target->yMin = userYMin;
    target->yMax = userYMax;

    target->zMin = userZMin;
    target->zMax = userZMax;
}




boolean   BoundingBoxIntersect(const BoundingBox3D* box, const Ray3D* ray)
{
    double   tXMin;
    double   tXMax;
    double   tYMin;
    double   tYMax;
    double   tZMin;
    double   tZMax;
    double   excluderX = 1.0 / ray->direction.x;
    double   excluderY = 1.0 / ray->direction.y;
    double   excluderZ = 1.0 / ray->direction.z;

    if (excluderX >= 0) {

        tXMin = excluderX * (box->xMin - ray->origin.x);
        tXMax = excluderX * (box->xMax - ray->origin.x);
    }
    else {

        tXMin = excluderX * (box->xMax - ray->origin.x);
        tXMax = excluderX * (box->xMin - ray->origin.x);
    }

    if (excluderY >= 0) {

        tYMin = excluderY * (box->yMin - ray->origin.y);
        tYMax = excluderY * (box->yMax - ray->origin.y);
    }
    else {

        tYMin = excluderY * (box->yMax - ray->origin.y);
        tYMax = excluderY * (box->yMin - ray->origin.y);
    }

    if (excluderZ >= 0) {

        tZMin = excluderZ * (box->zMin - ray->origin.z);
        tZMax = excluderZ * (box->zMax - ray->origin.z);
    }
    else {

        tZMin = excluderZ * (box->zMax - ray->origin.z);
        tZMax = excluderZ * (box->zMin - ray->origin.z);
    }

    if ((tXMin > tYMax) || (tYMin > tXMax)) {

        return FALSE;
    }

    if ((tXMin > tZMax) || (tZMin > tXMax)) {

        return FALSE;
    }

    if ((tYMin > tZMax) || (tZMin > tYMax)) {

        return FALSE;
    }

    return TRUE;
}




void  PrintBoundingBox(const BoundingBox3D* target)
{
    printf("\nbounding box {\n");

    printf("\tx span = [%f, %f].\n", target->xMin, target->xMax);
    printf("\ty span = [%f, %f].\n", target->yMin, target->yMax);
    printf("\tz span = [%f, %f].\n", target->zMin, target->zMax);

    printf("}.\n");
}




BoundingBox3D  BoundingBoxUnion(const BoundingBox3D* b1,
    const BoundingBox3D* b2)
{
    BoundingBox3D  result;

    if (b2->xMin < b1->xMin) {

        result.xMin = b2->xMin;
    }
    else {

        result.xMin = b1->xMin;
    }

    if (b2->xMax > b1->xMax) {

        result.xMax = b2->xMax;
    }
    else {

        result.xMax = b1->xMax;
    }

    if (b2->yMin < b1->yMin) {

        result.yMin = b2->yMin;
    }
    else {

        result.yMin = b1->yMin;
    }

    if (b2->yMax > b1->yMax) {

        result.yMax = b2->yMax;
    }
    else {

        result.yMax = b1->yMax;
    }

    if (b2->zMin < b1->zMin) {

        result.zMin = b2->zMin;
    }
    else {

        result.zMin = b1->zMin;
    }

    if (b2->zMax > b1->zMax) {

        result.zMax = b2->zMax;
    }
    else {

        result.zMax = b1->zMax;
    }

    return result;
}




BoundingBox3D  TriangleBoundingBox(const HomogeneousVector3D* v1,
    const HomogeneousVector3D* v2, const HomogeneousVector3D* v3)
{
    BoundingBox3D               result;

    const HomogeneousVector3D*  vertices[3] = {v1, v2, v3};
    int i;

    result.xMin = v1->x;
    result.xMax = v1->x;
    result.yMin = v1->y;
    result.yMax = v1->y;
    result.zMin = v1->z;
    result.zMax = v1->z;

    for (i = 1; i < 3; i++) {

        if (vertices[i]->x < result.xMin) {

            result.xMin = vertices[i]->x;
        }

        if (vertices[i]->x > result.xMax) {

            result.xMax = vertices[i]->x;
        }

        if (vertices[i]->y < result.yMin) {

            result.yMin = vertices[i]->y;
        }

        if (vertices[i]->y > result.yMax) {

            result.yMax = vertices[i]->y;
        }

        if (vertices[i]->z < result.zMin) {

            result.zMin = vertices[i]->z;
        }

        if (vertices[i]->z > result.zMax) {

            result.zMax = vertices[i]->z;
        }
    }

    return result;
}




static boolean isRandomGeneratorSeeded = FALSE;

double  SampleUniform(void)
{
    double  result;
    int     iSample;

    if (!isRandomGeneratorSeeded) {

        srand(time(0));
        isRandomGeneratorSeeded = TRUE;
    }

    iSample = rand( );

    result = ((double)(iSample)) / ((double)(RAND_MAX));

    return result;
}




HomogeneousVector3D  TriangleCenter(const HomogeneousVector3D* v1,
    const HomogeneousVector3D* v2, const HomogeneousVector3D* v3)
{
    double               meanX;
    double               meanY;
    double               meanZ;
    HomogeneousVector3D  result;

    meanX = (v1->x + v2->x + v3->x) / 3.0;
    meanY = (v1->y + v2->y + v3->y) / 3.0;
    meanZ = (v1->z + v2->z + v3->z) / 3.0;

    SetVertex3D(&result, meanX, meanY, meanZ);

    return result;
}




boolean   PointInBoundingBox(const HomogeneousVector3D* point,
    const BoundingBox3D* box)
{
    if (point->w != 1.0) {

        LogicError("PointInBox( )", "needs a point vector, not a direction "
            "vector");
    }

    if ((point->x >= box->xMin) && (point->x < box->xMax)) {

        if ((point->y >= box->yMin) && (point->y < box->yMax)) {

            if ((point->z >= box->zMin) && (point->z < box->zMax)) {

                return TRUE;
            }
        }
    }

    return FALSE;
}



enum { kMinimalX, kMinimalY, kMinimalZ };
typedef int MinimalComponentCode;

CoordinateSystem3D  ComputeZDirectedSystem(const HomogeneousVector3D* axis)
{
    CoordinateSystem3D    result;
    HomogeneousVector3D   basisT;
    MinimalComponentCode  tCode;
    double                minTComponent;

    if (axis->w != 0.0) {

        LogicError("ComputeZDirectedSystem( )", "input axis vector was a "
            "vertex vector, not a direction vector");
    }

    result.basisZ = VectorNormalize(axis);

    basisT = result.basisZ;

    tCode = kMinimalX;
    minTComponent = fabs(basisT.x);
    if (fabs(basisT.y) < minTComponent) {

        tCode = kMinimalY;
        minTComponent = fabs(basisT.y);
    }
    if (fabs(basisT.z) < minTComponent) {

        tCode = kMinimalZ;
    }

    if (tCode == kMinimalX) {

        basisT.x = 1.0;
    }
    else if (tCode == kMinimalY) {

        basisT.y = 1.0;
    }
    else {

        basisT.z = 1.0;
    }

    result.basisX = VectorCross(&basisT, &result.basisZ);
    result.basisX = VectorNormalize(&result.basisX);

    result.basisY = VectorCross(&result.basisZ, &result.basisX);

    return result;
}




HomogeneousVector3D  VectorPerturb(const HomogeneousVector3D* vector,
    double magnitude)
{
    HomogeneousVector3D  result;

    double               xFactor;
    double               yFactor;
    CoordinateSystem3D   perturbBasis = ComputeZDirectedSystem(vector);
    double               inLength = VectorLength(vector);
    HomogeneousVector3D  temp;
    const double         kPerturbConstant = 0.85;


    if ((magnitude < 0.0) || (magnitude > 1.0)) {

        LogicError("VectorPerturb( )", "perturbation magnitude must be "
            "in the interval [0, 1]");
    }


    xFactor = kPerturbConstant * (SampleUniform( ) - 0.5) * magnitude;
    yFactor = kPerturbConstant * (SampleUniform( ) - 0.5) * magnitude;

    result = perturbBasis.basisZ;

    temp = ScalarMultiply(&perturbBasis.basisX, xFactor);
    result = VectorAdd(&result, &temp);

    temp = ScalarMultiply(&perturbBasis.basisY, yFactor);
    result = VectorAdd(&result, &temp);

    result = VectorNormalize(&result);

    result = ScalarMultiply(&result, inLength);

    return result;
}




void   PrintRay(const Ray3D* target)
{
    printf("\nRay3D := {\n");
    printf("\torigin    =");
    PrintVector(&target->origin);
    printf("\tdirection =");
    PrintVector(&target->direction);
    printf("}.\n");
}
