/*

 WorkerPersonality.c

 Copyright (c) 2006-2007, Lucas Stephen Beeler and Jeremy Wagner-Kaiser.
 All Rights Reserved.

 */

#include "WorkerPersonality.h"
#include "SceneIntrinsics.h"
#include "SceneGenerators.h"
#include "ConfigurableParameters.h"
#include "RayEngine.h"

/* #define DIAGNOSTICS */

/*---------------------------------------------------------------------------*/
/* MODULE-LOCAL TYPES                                                        */
/*---------------------------------------------------------------------------*/
typedef int  WorkerPersonalityState;
enum { kInitializationState,
       kRenderingState,
       kWorkGiverState,
       kWorkReceiverState,
       kTerminationState };
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* MODULE-LOCAL VARIABLES                                                    */
/*---------------------------------------------------------------------------*/
static  WorkerPersonalityState  mCurrentState = kInitializationState;
static  boolean                 mIsProbeRecvLive = FALSE;
static  MPI_Request             mInboundProbeQueryReq;
static  NodeProbeQueryPacket    mInboundProbeQueryData;
static  Scene*                  mWorkScene = 0;
static  ResultPacketQueue*      mOutResPackets = 0;
static  WorkQueue*              mWorkQueue = 0;
static  int                     mRaysInFlight = 0;
static  double                  mTimeOfLastTransmission = -1.0;
static  int                     mRebalancePartner = 0;
static  const int               kMinRebalanceThreshold = 40000;
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* PROTOTYPES FOR MODULE-LOCAL UTILITY ROUTINES                              */
/*---------------------------------------------------------------------------*/
static  WorkerPersonalityState   InitializationStateLoop(void);
static  WorkerPersonalityState   RenderingStateLoop(void);
static  WorkerPersonalityState   WorkReceiverStateLoop(void);
static  WorkerPersonalityState   WorkGiverStateLoop(void);

static  void      PostProbeQueryReceive(NodeProbeQueryPacket*, MPI_Request*);
static  boolean   TestProbeQuery(const MPI_Request*);
static  boolean   GetNextWorkPacket(WorkPacket*, const MPI_Request*);
static  int       ProbeNeighborhood(MPI_Request*);
static  void      RunTerminationSequence(void);
static  void      AccumulateResult(ResultPacket*, const RGBColor*, unsigned);
static  void      RejectRebalanceProbe(const NodeProbeQueryPacket*);
static  boolean   HandleRebalanceProbe(const NodeProbeQueryPacket*);
static  void      ReceiveRebalanceWork(int, int*, int);
/*---------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------*/
/* FUNCTION  RunWorkerPersonalityLoop( )                                     */
/*---------------------------------------------------------------------------*/
void  RunWorkerPersonalityLoop(void)
{
    while (TRUE) {

        switch  (mCurrentState) {

            case  kInitializationState:

                mCurrentState = InitializationStateLoop( );

            break;


            case  kRenderingState:

                mCurrentState = RenderingStateLoop( );

            break;


            case  kWorkReceiverState:

                mCurrentState = WorkReceiverStateLoop( );

            break;

            case  kWorkGiverState:

                mCurrentState = WorkGiverStateLoop( );

            break;

            case  kTerminationState:

                RunTerminationSequence( );
                return;

            break;

        } /* switch on current state */

    } /* personality loop */

} /* RunWorkerPersonalityLoop( ) */
/*---------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------*/
/* FUNCTION  PostProbeQueryReceive( )                                        */
/*---------------------------------------------------------------------------*/
void  PostProbeQueryReceive(NodeProbeQueryPacket* inPack,
    MPI_Request* inReq)
{
    MPI_Irecv(inPack, sizeof(NodeProbeQueryPacket), MPI_CHAR, MPI_ANY_SOURCE,
        kProbeQuery, MPI_COMM_WORLD, inReq);
}
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* FUNCTION  TestProbeQuery( )                                               */
/*---------------------------------------------------------------------------*/
boolean   TestProbeQuery(const MPI_Request* req)
{
    boolean  result;
    int      errorCode;

    errorCode = MPI_Test((MPI_Request*)(req), &result, MPI_STATUS_IGNORE);

    if (errorCode != MPI_SUCCESS) {

        RuntimeError("TestProbeQuery( )", "MPI_Test( ) reported error");
    }

    return result;
}
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* FUNCTION  GetNextWorkPacket( )                                            */
/*---------------------------------------------------------------------------*/
boolean   GetNextWorkPacket(WorkPacket* work, MPI_Request* req)
{
    boolean  result = FALSE;
    int      errorCode;

    errorCode = MPI_Test(req, &result, MPI_STATUS_IGNORE);

    if (errorCode != MPI_SUCCESS) {

        RuntimeError("GetNextWorkPacket( )", "MPI_Test( ) reported error");
    }

    return result;
}
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* FUNCTION  InitializationStateLoop( )                                      */
/*---------------------------------------------------------------------------*/
WorkerPersonalityState  InitializationStateLoop(void)
{
#ifdef DIAGNOSTICS
    boolean       userNotified = FALSE;
#endif
    WorkPacket    currentPacket;
    int           i;
    MPI_Request   inboundWorkRequest;
    boolean       gotPacket = FALSE;


    mWorkScene = ConstructBunnyDebuggingScene( );
    mOutResPackets = CreateResultPacketQueue( );
    mWorkQueue = CreateWorkQueue( );

    if (!mIsProbeRecvLive) {

        PostProbeQueryReceive(&mInboundProbeQueryData, &mInboundProbeQueryReq);

        mIsProbeRecvLive = TRUE;
    }

#ifdef DIAGNOSTICS
    printf("worker node (%d): waiting to receive intial work packets.\n",
        gNodeInfo.nodePID);
#endif

    currentPacket.opcode = kNullOpcode;

    MPI_Irecv(&currentPacket, sizeof(WorkPacket), MPI_CHAR, 0, 0,
        MPI_COMM_WORLD, &inboundWorkRequest);

    while (TRUE) {

        gotPacket = GetNextWorkPacket(&currentPacket, &inboundWorkRequest);

        if (gotPacket) {

            if (currentPacket.opcode == kNotifyWorkBroadcastComplete) {

#ifdef DIAGNOSTICS
                printf("worker node (%d): completed receipt of initial "
                    "work packets.\n", gNodeInfo.nodePID);

                printf("worker node (%d): breaking out of accumulation "
                    "loop.\n", gNodeInfo.nodePID);
#endif
                mRaysInFlight = WorkQueueSize(mWorkQueue);

                return kRenderingState;
            }
            else {

#ifdef DIAGNOSTICS
                if (!userNotified) {

                    printf("worker node (%d): began receiving initial work "
                        "packets.\n", gNodeInfo.nodePID);

                    userNotified = TRUE;
                }
#endif
                for (i = 0; i < currentPacket.numRays; i++) {

                    WorkQueueElement   currentElem;

                    currentElem.ray = currentPacket.rayData[i];
                    currentElem.pixelID = currentPacket.pixelIDs[i];

                    EnqueueWork(mWorkQueue, &currentElem);

                } /* for all rays in the received packet */

                /* post a new receive */
                MPI_Irecv(&currentPacket, sizeof(WorkPacket), MPI_CHAR, 0, 0,
                    MPI_COMM_WORLD, &inboundWorkRequest);

            } /* else we didn't get a "work broadcast complete notification" */

        } /* if we received a packet */


        /* if we should get a rebalance probe while in the initialization
           state, reject it */
        if (mIsProbeRecvLive) {

            if (TestProbeQuery(&mInboundProbeQueryReq)) {

                RejectRebalanceProbe(&mInboundProbeQueryData);

                PostProbeQueryReceive(&mInboundProbeQueryData,
                    &mInboundProbeQueryReq);
            }
        }

    } /* infinite loop */

} /* InitializationStateLoop( ) */
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* FUNCTION  RenderingStateLoop( )                                           */
/*---------------------------------------------------------------------------*/
WorkerPersonalityState  RenderingStateLoop(void)
{
    ResultPacket            resultPacket;
    RGBColor                currentResultColor;
    int                     currentResultPixelID;
    boolean                 rebalanceNow = FALSE;

    mRaysInFlight = WorkQueueSize(mWorkQueue);

    if (mIsProbeRecvLive) {

        if (TestProbeQuery(&mInboundProbeQueryReq)) {

            RejectRebalanceProbe(&mInboundProbeQueryData);

            PostProbeQueryReceive(&mInboundProbeQueryData,
                &mInboundProbeQueryReq);
        }
    }
    else {

        PostProbeQueryReceive(&mInboundProbeQueryData, &mInboundProbeQueryReq);

        mIsProbeRecvLive = TRUE;
    }

#ifdef DIAGNOSTICS
    printf("worker node (%d): preparing to render %d in-flight rays.\n",
        gNodeInfo.nodePID, mRaysInFlight);
#endif

    MakeResultPacketEmpty(&resultPacket);

    while (TRUE) {

        /* test for rebalance probe queries */
        if (mIsProbeRecvLive) {

            if (TestProbeQuery(&mInboundProbeQueryReq)) {

                rebalanceNow = HandleRebalanceProbe(&mInboundProbeQueryData);

                PostProbeQueryReceive(&mInboundProbeQueryData,
                    &mInboundProbeQueryReq);

                if (rebalanceNow) {

                    if (resultPacket.numColors > 0) {
#ifdef DIAGNOSTICS
                        printf("worker node (%d): preparing to act as "
                            "rebalance giver; flushing %d results to the "
                            "manager node.\n", gNodeInfo.nodePID,
                            resultPacket.numColors);
#endif
                        resultPacket.sourcePID = gNodeInfo.nodePID;

                        MPI_Send(&resultPacket, sizeof(ResultPacket), MPI_CHAR,
                            0, 0, MPI_COMM_WORLD);

                        mTimeOfLastTransmission = MPI_Wtime( );

                        mRaysInFlight -= resultPacket.numColors;

                        MakeResultPacketEmpty(&resultPacket);

                        return kWorkGiverState;
                    } /* if we've got a non-empty packet */
                } /* if we're going to rebalance now */
            } /* if we got a probe query */
        } /* if there's a live probe receive */


        /* if there are results to be sent out, send them */
        if (!IsResultPacketQueueEmpty(mOutResPackets)) {

            ResultPacketQueueElem  packetWrapper =
                DequeueResultPacket(mOutResPackets);

            MPI_Send(&packetWrapper.resultPack, sizeof(ResultPacket),
                MPI_CHAR, 0, 0, MPI_COMM_WORLD);

            mRaysInFlight -= packetWrapper.resultPack.numColors;

            mTimeOfLastTransmission = MPI_Wtime( );
        }

        /* if there's work to be done, do it */
        if (!IsWorkQueueEmpty(mWorkQueue)) {

#ifdef DIAGNOSTICS_MAX
            printf("\nworker node (%d): tracing a ray.\n");
#endif

            WorkQueueElement  currentElem = DequeueWork(mWorkQueue);

            currentResultColor = Trace(&currentElem.ray,
                mWorkScene, pRayRecursionDepth);

            currentResultPixelID = currentElem.pixelID;

            AccumulateResult(&resultPacket, &currentResultColor,
                currentResultPixelID);
#ifdef DIAGNOSTICS_MAX
            printf("worker node (%d): done tracing.\n");
#endif
        }

        /* if it's been more than a second since our last transmission, and we
           have only partially prepared a packet, go ahead and flush and send
           the partially prepared packet to the manager */
        if (mTimeOfLastTransmission != -1.0) {

            double  timeNow = MPI_Wtime( );

            if ((timeNow - mTimeOfLastTransmission) > 1.0) {

                if (resultPacket.numColors > 0) {
#ifdef DIAGNOSTICS
                    printf("worker node (%d): flushing a partial packet of %d "
                        "results to the manager node.\n", gNodeInfo.nodePID,
                        resultPacket.numColors);
#endif
                    resultPacket.sourcePID = gNodeInfo.nodePID;

                    MPI_Send(&resultPacket, sizeof(ResultPacket), MPI_CHAR,
                        0, 0, MPI_COMM_WORLD);

                    mTimeOfLastTransmission = MPI_Wtime( );

                    mRaysInFlight -= resultPacket.numColors;

                    MakeResultPacketEmpty(&resultPacket);
                }
            }
        } /* if it's been more than a second since last transmission */

        /* if we've completed all our work, attempt to perform local
           load re-balancing by entering the kWorkReceiverState */
        if (mRaysInFlight == 0) {

            return  kWorkReceiverState;

        } /* if we've completed all our work */

    } /* rendering loop */

} /* RenderingStateLoop( ) */
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* FUNCTION  WorkReceiverStateLoop( )                                        */
/*---------------------------------------------------------------------------*/
WorkerPersonalityState   WorkReceiverStateLoop(void)
{
    int                     outstandingProbes = 0;
    double                  stateStartTime = MPI_Wtime( );
    double                  currentTime = stateStartTime;
    double                  stateElapsedTimeNow = 0.0;
    double                  lastProbeTime = -1;
    MPI_Request             throwAway[4];
    NodeProbeStatusPacket   probedReplyData;
    MPI_Request             probedReplyReq;
    boolean                 probedReplyRecvd = FALSE;
    int                     highestWorkloadSeen = 0;
    int                     remoteNodesSeen = 0;
    int                     remoteNodeIDs[4];
    int                     remotePartner = 0;

#ifdef DIAGNOSTICS
    printf("worker node (%d): finished rendering in-flight rays; entering "
        "work receiver state.\n", gNodeInfo.nodePID);
#endif

    if (mIsProbeRecvLive) {

        if (TestProbeQuery(&mInboundProbeQueryReq)) {

            RejectRebalanceProbe(&mInboundProbeQueryData);

            PostProbeQueryReceive(&mInboundProbeQueryData,
                &mInboundProbeQueryReq);
        }
    }
    else {

        PostProbeQueryReceive(&mInboundProbeQueryData, &mInboundProbeQueryReq);

        mIsProbeRecvLive = TRUE;
    }

    while (TRUE) {

        /* test for rebalance probe queries and decline them */
        if (mIsProbeRecvLive) {

            if (TestProbeQuery(&mInboundProbeQueryReq)) {

                RejectRebalanceProbe(&mInboundProbeQueryData);

                PostProbeQueryReceive(&mInboundProbeQueryData,
                    &mInboundProbeQueryReq);
            }
        }

        currentTime = MPI_Wtime( );

        stateElapsedTimeNow = currentTime - stateStartTime;

        if (!outstandingProbes) {

            if (stateElapsedTimeNow > 15.0) {

                return kTerminationState;
            }

            if (remotePartner != 0) {

                ReceiveRebalanceWork(remotePartner, remoteNodeIDs,
                    remoteNodesSeen);

                return kRenderingState;
            }

            if (((currentTime - lastProbeTime) > 0.5) &&
                (stateElapsedTimeNow <= 7.0)) {

                outstandingProbes = ProbeNeighborhood(throwAway);

                MPI_Irecv(&probedReplyData, sizeof(NodeProbeStatusPacket),
                    MPI_CHAR, MPI_ANY_SOURCE, kProbeResultStatus,
                    MPI_COMM_WORLD, &probedReplyReq);

                lastProbeTime = MPI_Wtime( );
            }
        } /* if there aren't any outstanding probes */
        else { /* else there are some outstanding probes */

            MPI_Test(&probedReplyReq, &probedReplyRecvd, MPI_STATUS_IGNORE);

            if (probedReplyRecvd) {

                outstandingProbes--;

                if (probedReplyData.wantsRebalance == TRUE) {

                    remoteNodeIDs[remoteNodesSeen] = probedReplyData.sourcePID;
                    remoteNodesSeen++;

                    if (probedReplyData.raysInFlight > highestWorkloadSeen) {

                        highestWorkloadSeen = probedReplyData.raysInFlight;
                        remotePartner = probedReplyData.sourcePID;
                    }
                } /* if remote node wants rebalance */

                if (outstandingProbes) {

                    MPI_Irecv(&probedReplyData, sizeof(NodeProbeStatusPacket),
                        MPI_CHAR, MPI_ANY_SOURCE, kProbeResultStatus,
                        MPI_COMM_WORLD, &probedReplyReq);
                }
            }
        } /* else there are outstanding probes */

    } /* work receiver loop */

} /* WorkReceiverStateLoop( ) */
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* FUNCTION  WorkGiverStateLoop( )                                           */
/*---------------------------------------------------------------------------*/
WorkerPersonalityState   WorkGiverStateLoop(void)
{
    int               raysToSend = WorkQueueSize(mWorkQueue) / 2;
    int               i, j;
    WorkPacket        outPacket;
    WorkQueueElement  currentElem;
    WorkPacket        endPacket;

    endPacket.opcode = kNotifyWorkBroadcastComplete;
    endPacket.numRays = 0;

#ifdef DIAGNOSTICS
    printf("worker node (%d): initiating transfer of %d in-flight rays to "
        "node %d.\n", gNodeInfo.nodePID, raysToSend, mRebalancePartner);
#endif

    if (mIsProbeRecvLive) {

        if (TestProbeQuery(&mInboundProbeQueryReq)) {

            RejectRebalanceProbe(&mInboundProbeQueryData);

            PostProbeQueryReceive(&mInboundProbeQueryData,
                &mInboundProbeQueryReq);
        }
    }
    else {

        PostProbeQueryReceive(&mInboundProbeQueryData, &mInboundProbeQueryReq);

        mIsProbeRecvLive = TRUE;
    }

    MakeWorkPacketEmpty(&outPacket);
    outPacket.sourcePID = gNodeInfo.nodePID;
    outPacket.opcode = kAccumulateWork;

    for (i = 0; i < raysToSend; i++) {

        if (outPacket.numRays == kMaxPacketSize) {

            MPI_Send(&outPacket, sizeof(WorkPacket), MPI_CHAR,
                mRebalancePartner, kProbeWorkTransfer, MPI_COMM_WORLD);

            MakeWorkPacketEmpty(&outPacket);
        }

        currentElem = DequeueWork(mWorkQueue);

        j = i % kMaxPacketSize;

        outPacket.rayData[j] = currentElem.ray;
        outPacket.pixelIDs[j] = currentElem.pixelID;

        outPacket.numRays++;

        /* test for rebalance probe queries and decline them */
        if (mIsProbeRecvLive) {

            if (TestProbeQuery(&mInboundProbeQueryReq)) {

                RejectRebalanceProbe(&mInboundProbeQueryData);

                PostProbeQueryReceive(&mInboundProbeQueryData,
                    &mInboundProbeQueryReq);
            }
        }

    } /* for all the rays we have to send */

    if (outPacket.numRays > 0) {

        MPI_Send(&outPacket, sizeof(WorkPacket), MPI_CHAR,
            mRebalancePartner, kProbeWorkTransfer, MPI_COMM_WORLD);
    }

    mRaysInFlight = WorkQueueSize(mWorkQueue);

    MPI_Send(&endPacket, sizeof(WorkPacket), MPI_CHAR,
        mRebalancePartner, kProbeWorkTransfer, MPI_COMM_WORLD);

#ifdef DIAGNOSTICS
    printf("worker node (%d): completed ray transfer.\n", gNodeInfo.nodePID);
#endif

    return kRenderingState;
}
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* FUNCTION  ProbeNeighborhood( )                                            */
/*---------------------------------------------------------------------------*/
int  ProbeNeighborhood(MPI_Request* reqList)
{
    int                       probeList[4];
    int                       numToProbe = 0;
    NodeProbeQueryPacket      queryPacket;
    int                       i;

#ifdef DIAGNOSTICS
    printf("worker node (%d): preparing to probe nearby nodes "
        "for work.\n", gNodeInfo.nodePID);
#endif

    if ((gNodeInfo.numNodes - 1) == 1) {
#ifdef DIAGNOSTICS
        printf("worker node (%d): this node is sole worker node; no "
            "neighborhood to probe.\n", gNodeInfo.nodePID);
#endif
        numToProbe = 0;
    }
    else if ((gNodeInfo.numNodes - 1) == 2) {

        probeList[0] = NextWorkerNode(gNodeInfo.nodePID);

        numToProbe = 1;
#ifdef DIAGNOSTICS
        printf("worker node (%d): this node is one of only two worker nodes;"
            " probing counterpart node %d.\n", gNodeInfo.nodePID,
            probeList[0]);
#endif
    }
    else if ((gNodeInfo.numNodes - 1) == 3) {

        probeList[0] = PreviousWorkerNode(gNodeInfo.nodePID);
        probeList[1] = NextWorkerNode(gNodeInfo.nodePID);

        numToProbe = 2;
#ifdef DIAGNOSTICS
        printf("worker node (%d): probing nodes %d and %d.\n",
            gNodeInfo.nodePID, probeList[0], probeList[1]);
#endif
    }
    else if ((gNodeInfo.numNodes - 1) == 4) {

        probeList[0] = PreviousWorkerNode(gNodeInfo.nodePID);
        probeList[1] = NextWorkerNode(gNodeInfo.nodePID);
        probeList[2] = NextWorkerNode(NextWorkerNode(gNodeInfo.nodePID));

        numToProbe = 3;
#ifdef DIAGNOSTICS
        printf("worker node (%d): probing nodes %d, %d, and %d.\n",
            gNodeInfo.nodePID, probeList[0], probeList[1], probeList[2]);
#endif
    }
    else {

        probeList[0] = PreviousWorkerNode(gNodeInfo.nodePID);
        probeList[1] = NextWorkerNode(gNodeInfo.nodePID);
        probeList[2] = NextWorkerNode(NextWorkerNode(gNodeInfo.nodePID));
        probeList[3] =
            PreviousWorkerNode(PreviousWorkerNode(gNodeInfo.nodePID));

        numToProbe = 4;
#ifdef DIAGNOSTICS
        printf("worker node (%d): probing nodes %d, %d, %d, and %d.\n",
            gNodeInfo.nodePID, probeList[0], probeList[1], probeList[2],
            probeList[3]);
#endif
    }

    queryPacket.sourcePID = gNodeInfo.nodePID;

    for (i = 0; i < numToProbe; i++) {

        MPI_Ibsend(&queryPacket, sizeof(NodeProbeQueryPacket), MPI_CHAR,
            probeList[i], kProbeQuery, MPI_COMM_WORLD, &reqList[i]);
    }

    return numToProbe;
} /* ProbeNeighborhood( ) */
/*---------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------*/
/* FUNCTION  RunTerminationSequence( )                                       */
/*---------------------------------------------------------------------------*/
void   RunTerminationSequence(void)
{
    ResultPacket  completionNotifyPacket;

    completionNotifyPacket.opcode = kNotifyWorkComplete;
    completionNotifyPacket.sourcePID = gNodeInfo.nodePID;

#ifdef  DIAGNOSTICS
    printf("worker node (%d): entering termination sequence.\n",
        gNodeInfo.nodePID);

    printf("worker node (%d): sending work completion notify "
        "to manager node.\n", gNodeInfo.nodePID);
#endif

    MPI_Send(&completionNotifyPacket, sizeof(ResultPacket), MPI_CHAR,
        0, 0, MPI_COMM_WORLD);

#ifdef  DIAGNOSTICS
    printf("worker node (%d): transmission of completion notify "
        "done.\n", gNodeInfo.nodePID);

    printf("worker node (%d) : breaking out of personality loop.\n",
        gNodeInfo.nodePID);
#endif
} /* RunTerminationSequence( ) */
/*---------------------------------------------------------------------------*/







/*---------------------------------------------------------------------------*/
/* FUNCTION  AccumulateResult( )                                             */
/*---------------------------------------------------------------------------*/
void  AccumulateResult(ResultPacket* result, const RGBColor* color,
    unsigned pixelID)
{
    AddColorToResultPacket(result, color, pixelID);

    result->sourcePID = gNodeInfo.nodePID;

    if (result->numColors == kMaxPacketSize) {

        ResultPacketQueueElem  wrapper;

        wrapper.resultPack = (*result);

        EnqueueResultPacket(mOutResPackets, &wrapper);

        MakeResultPacketEmpty(result);

    } /* if we've got a full packet to send out */

} /* AccumulateResult( ) */
/*---------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------*/
/* FUNCTION  RejectRebalanceProbe( )                                         */
/*---------------------------------------------------------------------------*/
void  RejectRebalanceProbe(const NodeProbeQueryPacket* query)
{
    NodeProbeStatusPacket   localStat;

    localStat.sourcePID = gNodeInfo.nodePID;

    localStat.wantsRebalance = FALSE;

#ifdef DIAGNOSTICS
    printf("worker node (%d): rejecting rebalance probe query from node %d.\n",
        gNodeInfo.nodePID, query->sourcePID);
#endif

    MPI_Send(&localStat, sizeof(NodeProbeStatusPacket), MPI_CHAR,
        query->sourcePID, kProbeResultStatus, MPI_COMM_WORLD);
} /* RejectRebalanceProbe( ) */
/*---------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------*/
/* FUNCTION  HandleRebalanceProbe( )                                         */
/*---------------------------------------------------------------------------*/
boolean  HandleRebalanceProbe(const NodeProbeQueryPacket* query)
{
    int                     localRaysInFlight = WorkQueueSize(mWorkQueue);
    NodeProbeStatusPacket   localStat;
    boolean                 result;
    boolean                 remoteAccept = FALSE;

    localStat.sourcePID = gNodeInfo.nodePID;

    if (localRaysInFlight >= kMinRebalanceThreshold) {

        localStat.wantsRebalance = TRUE;
        localStat.raysInFlight = localRaysInFlight; 

        MPI_Send(&localStat, sizeof(NodeProbeStatusPacket), MPI_CHAR,
            query->sourcePID, kProbeResultStatus, MPI_COMM_WORLD);

        MPI_Recv(&remoteAccept, sizeof(boolean), MPI_CHAR, query->sourcePID,
            kProbePartnerAcknowledge, MPI_COMM_WORLD, MPI_STATUS_IGNORE);

        if (remoteAccept) {

            mRebalancePartner = query->sourcePID;

            result = TRUE;
        }
        else {

            result = FALSE;
        }
    }
    else {

        RejectRebalanceProbe(query);

        result = FALSE;
    }

    return result;
}
/*---------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------*/
/* FUNCTION  ReceiveRebalanceWork( )                                         */
/*---------------------------------------------------------------------------*/
void  ReceiveRebalanceWork(int partner, int* seenNodes, int numSeenNodes)
{
    int          i;
    boolean      falseBuffer = FALSE;
    boolean      trueBuffer = TRUE;
    WorkPacket   currentPacket;

    for (i = 0; i < numSeenNodes; i++) {

        if (seenNodes[i] != partner) {

            MPI_Send(&falseBuffer, sizeof(boolean), MPI_CHAR, seenNodes[i],
                kProbePartnerAcknowledge, MPI_COMM_WORLD);
        }
    }

#ifdef DIAGNOSTICS
    printf("worker node (%d): signaling node %d with work transfer accept.\n",
        gNodeInfo.nodePID, partner);
#endif

    MPI_Send(&trueBuffer, sizeof(boolean), MPI_CHAR, partner,
        kProbePartnerAcknowledge, MPI_COMM_WORLD);

#ifdef DIAGNOSTICS
    printf("worker node (%d): signaled work transfer accept; preparing to "
        "recieve work packets from node %d.\n",
        gNodeInfo.nodePID, partner);
#endif

    MPI_Recv(&currentPacket, sizeof(WorkPacket), MPI_CHAR, partner,
        kProbeWorkTransfer, MPI_COMM_WORLD, MPI_STATUS_IGNORE);

#ifdef DIAGNOSTICS
    printf("worker node (%d): received first work packet from node %d.\n",
        gNodeInfo.nodePID, partner);
#endif

    while (currentPacket.opcode != kNotifyWorkBroadcastComplete) {

        for (i = 0; i < currentPacket.numRays; i++) {

            WorkQueueElement   currentElem;

            currentElem.ray = currentPacket.rayData[i];
            currentElem.pixelID = currentPacket.pixelIDs[i];

            EnqueueWork(mWorkQueue, &currentElem);

        } /* for all rays in the received packet */

        MPI_Recv(&currentPacket, sizeof(WorkPacket), MPI_CHAR, partner,
            kProbeWorkTransfer, MPI_COMM_WORLD, MPI_STATUS_IGNORE);

        /* test for rebalance probe queries and decline them */
        if (mIsProbeRecvLive) {

            if (TestProbeQuery(&mInboundProbeQueryReq)) {

                RejectRebalanceProbe(&mInboundProbeQueryData);

                PostProbeQueryReceive(&mInboundProbeQueryData,
                    &mInboundProbeQueryReq);
            }
        }

    } /* while we haven't received broadcast complete */

#ifdef DIAGNOSTICS
    printf("worker node (%d): received work broadcast complete from node "
        "%d.\n", gNodeInfo.nodePID, partner);
#endif

} /* ReceiveRebalanceWork( ) */
/*---------------------------------------------------------------------------*/
