/*

 SpatialPartitioning.c

 Copyright (c) 2007, Lucas Stephen Beeler. All Rights Reserved.

 */

#include "SpatialPartitioning.h"
#include <stdlib.h>

static  void  ComputeSpliceInterval(SpliceAxisID axis, const BoundingBox3D*
    bounds, double* start, double* stop)
{
    if (axis == kSpliceOnX) {

        (*start) = bounds->xMin;
        (*stop) = bounds->xMax;
    }
    else if (axis == kSpliceOnY) {

        (*start) = bounds->yMin;
        (*stop) = bounds->yMax;
    }
    else {

        (*start) = bounds->zMin;
        (*stop) = bounds->zMax;
    }
}




static  SpliceAxisID  NextAxis(SpliceAxisID axis)
{
    return ((axis + 1) % 3);
}




static  BoundingBox3D  LowerSplicedPartition(const BoundingBox3D* source,
    SpliceAxisID spliceAxis)
{
    BoundingBox3D  result;

    if (spliceAxis == kSpliceOnX) {

        result.xMin = source->xMin;
        result.xMax = source->xMin + ((source->xMax - source->xMin) / 2.0);

        result.yMin = source->yMin;
        result.yMax = source->yMax;

        result.zMin = source->zMin;
        result.zMax = source->zMax;
    }
    else if (spliceAxis == kSpliceOnY) {

        result.yMin = source->yMin;
        result.yMax = source->yMin + ((source->yMax - source->yMin) / 2.0);

        result.xMin = source->xMin;
        result.xMax = source->xMax;

        result.zMin = source->zMin;
        result.zMax = source->zMax;
    }
    else {

        result.zMin = source->zMin;
        result.zMax = source->zMin + ((source->zMax - source->zMin) / 2.0);

        result.xMin = source->xMin;
        result.xMax = source->xMax;

        result.yMin = source->yMin;
        result.yMax = source->yMax;
    }

    return result;
}




static  BoundingBox3D  UpperSplicedPartition(const BoundingBox3D* source,
    SpliceAxisID spliceAxis)
{
    BoundingBox3D  result;

    if (spliceAxis == kSpliceOnX) {

        result.xMin = source->xMin + ((source->xMax - source->xMin) / 2.0);
        result.xMax = source->xMax;

        result.yMin = source->yMin;
        result.yMax = source->yMax;

        result.zMin = source->zMin;
        result.zMax = source->zMax;
    }
    else if (spliceAxis == kSpliceOnY) {

        result.yMin = source->yMin + ((source->yMax - source->yMin) / 2.0);
        result.yMax = source->yMax;

        result.xMin = source->xMin;
        result.xMax = source->xMax;

        result.zMin = source->zMin;
        result.zMax = source->zMax;
    }
    else {

        result.zMin = source->zMin + ((source->zMax - source->zMin) / 2.0);
        result.zMax = source->zMax;

        result.xMin = source->xMin;
        result.xMax = source->xMax;

        result.yMin = source->yMin;
        result.yMax = source->yMax;
    }

    return result;
}




static  BVHNode*  ConstructBVHLeafNode(const Mesh* mesh, int faceID)
{
    BVHNode*  result;

    result = malloc(sizeof(BVHNode));

    if (!result) {

        RuntimeError("ConstructBVHLeafNode( )", "out of memory");
    }

    result->bounds =
        TriangleBoundingBox(&mesh->vertices[mesh->faces[faceID].vindexes[0]],
            &mesh->vertices[mesh->faces[faceID].vindexes[1]],
            &mesh->vertices[mesh->faces[faceID].vindexes[2]]);

    result->faceID = faceID;

    result->hostMesh = (Mesh*)(mesh);

    result->leftChild = 0;

    result->rightChild = 0;

    return result;
}




BVHNode*  ConstructBVHNode(const Mesh* mesh, const BoundingBox3D* bounds,
    SpliceAxisID axis)
{
    HomogeneousVector3D    currTriCenter;
    int                    i;
    int                    numObjectsBounded = 0;
    int                    boundedFace;
    double                 start;
    double                 stop;
    BVHNode*               result;

    ComputeSpliceInterval(axis, bounds, &start, &stop);

    /* loop through all the faces in the mesh, and determine how many
       belong to this partition */
    for (i = 0; i < mesh->numFaces; i++) {

        currTriCenter =
            TriangleCenter(&mesh->vertices[mesh->faces[i].vindexes[0]],
                &mesh->vertices[mesh->faces[i].vindexes[1]],
                &mesh->vertices[mesh->faces[i].vindexes[2]]);

        if (PointInBoundingBox(&currTriCenter, bounds)) {

                numObjectsBounded++;

                boundedFace = i;
        }
    } /* for all faces in mesh */

    if (numObjectsBounded == 0) {

        return 0;
    }

    /* if we get here, the return statement above didn't execute, so
       our partition must contain at least one object */

    if (numObjectsBounded == 1) {

        result = ConstructBVHLeafNode(mesh, boundedFace);
    }
    else {

        /* prepare to make a recursive call to progressively partition
           space at finer levels of detail -- compute the next axis to
           partition on, bounding boxes for child nodes, etc. */

        SpliceAxisID   nextAxis = NextAxis(axis);

        BoundingBox3D  leftChildBounds =
            LowerSplicedPartition(bounds, axis);
        BoundingBox3D  rightChildBounds =
            UpperSplicedPartition(bounds, axis);

        result = malloc(sizeof(BVHNode));

        if (!result) {

            RuntimeError("ConstructBVHNode( )", "out of memory");
        }

        result->hostMesh = 0;
        result->faceID = kNullFace;

        result->leftChild =
            ConstructBVHNode(mesh, &leftChildBounds, nextAxis);
        result->rightChild =
            ConstructBVHNode(mesh, &rightChildBounds, nextAxis);

        if (!result->leftChild) {

            /* if there's no left child but there are objects, then
               the right child must be valid, so make the left
               child the right child */

            result->leftChild = result->rightChild;
            result->rightChild = 0;
            result->bounds = result->leftChild->bounds;
        }
        else {

            if (!result->rightChild) {

                result->bounds = result->leftChild->bounds;
            }
            else {

                result->bounds = BoundingBoxUnion(&result->leftChild->bounds,
                    &result->rightChild->bounds);
            }

        } /* else there is a left child present */

    } /* else more than object is contained in the current partition */

    return result;
}




BoundingVolumeHierarchy*  ConstructBVH(const Mesh* mesh, SpliceAxisID axis)
{
    BoundingVolumeHierarchy*  result;
    BoundingBox3D             meshBounds;

    result = malloc(sizeof(BoundingVolumeHierarchy));

    if (!result) {

        RuntimeError("ConstructBVH( )", "out of memory");
    }

    meshBounds = ComputeTriMeshBoundingBox(mesh);

    result->rootNode = ConstructBVHNode(mesh, &meshBounds, axis);

    return result;
}
