/*

 SceneIntrinsics.c

 Copyright (c) 2006, Lucas Stephen Beeler. All Rights Reserved.

 */

#include <stdlib.h>
#include <memory.h>
#include "SceneIntrinsics.h"

BoundingBox3D  ComputeGelBoundingBox(const Gel*);

void    SetGel(Gel* target, const Mesh* userMesh, const MaterialProps* userProps,
        const HomogeneousTransform3D* userTrans)
{
    HomogeneousTransform3D  inverseUserTransform = (*userTrans);
    InvertTransform(&inverseUserTransform);

    target->gelMesh = (Mesh*)(userMesh);
    target->gelProps = (*userProps);
    target->gelTrans = (*userTrans);
    target->invGelTrans = inverseUserTransform;
    target->boundingBox = ComputeGelBoundingBox(target);
    target->gelBVH = 0;
}




void   TransformGel(Gel* target, const HomogeneousTransform3D* trans)
{
    HomogeneousTransform3D  newGelTrans;
    HomogeneousTransform3D  inverseNewGelTrans;

    newGelTrans = target->gelTrans;
    ComposeTransformsTo(&newGelTrans, trans);

    inverseNewGelTrans = newGelTrans;

    InvertTransform(&inverseNewGelTrans);

    target->gelTrans = newGelTrans;
    target->invGelTrans = inverseNewGelTrans;

    target->boundingBox = ComputeGelBoundingBox(target);
}




Scene*   ConstructBaseScene(void)
{
    const int initGelSize = 8;
    const int initLightSize = 8;

    Scene*  result;

    HomogeneousVector3D  canonCamOrigin;
    HomogeneousVector3D  canonCamGaze;
    HomogeneousVector3D  canonCamUp;

    result = malloc(sizeof(Scene));

    if (!result) {

        RuntimeError("ConstructBaseScene( )", "out of memory");
    }

    SetVertex3D(&canonCamOrigin, 0.0, 0.0, 3.0);
    SetDirection3D(&canonCamGaze, 0.0, 0.0, -1.0);
    SetDirection3D(&canonCamUp, 0.0, 1.0, 0.0);

    SetCamera(&result->eye, &canonCamOrigin, &canonCamGaze, &canonCamUp, 45,
        640, 480, 1.0);

    result->numGels = 0;
    result->gelsAllocd = initGelSize;
    result->gelData = malloc(sizeof(Gel) * initGelSize);

    if (!result->gelData) {

        RuntimeError("ConstructBaseScene( )", "out of memory");
    }

    result->numLights = 0;
    result->lightsAllocd = initLightSize;
    result->lightData = malloc(sizeof(Light) * initLightSize);

    if (!result->lightData) {

        RuntimeError("ConstructBaseScene( )", "out of memory");
    }

    SetRGBColor(&result->ambientIntensity, 0.3, 0.3, 0.3);

    return result;
}




void    GrowGelHeap(Scene* target)
{
    int  newHeapSize = 2 * target->numGels;

    Gel*  tempGelHeap = malloc(sizeof(Gel) * newHeapSize);

    if (!tempGelHeap) {

        RuntimeError("GrowGelHeap( )", "out of memory");
    }

    memcpy(tempGelHeap, target->gelData, ((target->numGels) * sizeof(Gel)));

    free(target->gelData);

    target->gelData = tempGelHeap;

    target->gelsAllocd = newHeapSize;
}




void    AddGel(Scene* target, const Gel* newGel)
{
    if (target->numGels == target->gelsAllocd)
        GrowGelHeap(target);

    target->gelData[target->numGels] = *newGel;

    target->numGels++;
}




void     GrowLightHeap(Scene* target)
{
    int  newHeapSize = 2 * target->numLights;

    Light*  tempLightHeap = malloc(sizeof(Light) * newHeapSize);

    if (!tempLightHeap) {

        RuntimeError("GrowLightHeap( )", "out of memory");
    }

    memcpy(tempLightHeap, target->lightData,
        ((target->numLights) * sizeof(Light)));

    free(target->lightData);

    target->lightData = tempLightHeap;

    target->lightsAllocd = newHeapSize;
}




void     AddLight(Scene* target, const Light* newLight)
{
    if (target->numLights == target->lightsAllocd)
        GrowLightHeap(target);

    target->lightData[target->numLights] = *newLight;

    target->numLights++;
}




BoundingBox3D  ComputeGelBoundingBox(const Gel* gel)
{
    double                 currentXMin = kBigDouble;
    double                 currentXMax = -kBigDouble;
    double                 currentYMin = kBigDouble;
    double                 currentYMax = -kBigDouble;
    double                 currentZMin = kBigDouble;
    double                 currentZMax = -kBigDouble;
    int                    i;
    HomogeneousVector3D    currentVertex;
    BoundingBox3D          result;

    for (i = 0; i < gel->gelMesh->numVertices; i++) {

        currentVertex = gel->gelMesh->vertices[i];

        currentVertex = ApplyTransform(&gel->gelTrans, &currentVertex);

        if (currentVertex.x < currentXMin) {

            currentXMin = currentVertex.x;
        }
        if (currentVertex.x > currentXMax) {

            currentXMax = currentVertex.x;
        }

        if (currentVertex.y < currentYMin) {

            currentYMin = currentVertex.y;
        }
        if (currentVertex.y > currentYMax) {

            currentYMax = currentVertex.y;
        }
        if (currentVertex.z < currentZMin) {

            currentZMin = currentVertex.z;
        }
        if (currentVertex.z > currentZMax) {

            currentZMax = currentVertex.z;
        }
    };

    currentXMin -= 0.0005;
    currentXMax += 0.0005;

    currentYMin -= 0.0005;
    currentYMax += 0.0005;

    currentZMin -= 0.0005;
    currentZMax += 0.0005;

    result.xMin = currentXMin;
    result.xMax = currentXMax;
    result.yMin = currentYMin;
    result.yMax = currentYMax;
    result.zMin = currentZMin;
    result.zMax = currentZMax;

    return result;
}

