/*

 Mesh.c

 Copyright (c) 2006-2007, Lucas Stephen Beeler. All Rights Reserved.

 */

#include "Mesh.h"
#include <stdio.h>
#include <memory.h>
#include <stdlib.h>


Mesh*  ConstructMesh(MeshNormalMode userNormMode)
{
    Mesh*  result = malloc(sizeof(Mesh));

    const int  numVertexSlotsInitAllocd = 8;
    const int  numFaceSlotsInitAllocd = 8;

    if (!result) {

        RuntimeError("ConstructMesh( )", "out of memory");
    }

    result->vertices = malloc(sizeof(HomogeneousVector3D) *
        numVertexSlotsInitAllocd);

    if (!(result->vertices)) {

        RuntimeError("ConstructMesh( )", "out of memory");
    }

    if (userNormMode == kVertexNormalMode) {

        result->normals = malloc(sizeof(HomogeneousVector3D) *
            numVertexSlotsInitAllocd);

        if (!(result->normals)) {

            RuntimeError("ConstructMesh( )", "out of memory");
        }
    }
    else {

        result->normals = 0;
    }

    result->numVertices = 0;
    result->vertexSlotsAllocd = numVertexSlotsInitAllocd;

    result->faces = malloc(sizeof(MeshFace) * numFaceSlotsInitAllocd);

    if (!(result->faces)) {

        RuntimeError("ConstructMesh( )", "out of memory");
    }

    result->numFaces = 0;
    result->faceSlotsAllocd = numFaceSlotsInitAllocd;

    result->normalMode = userNormMode;

    return result;
}




void   GrowVertexNormalHeaps(Mesh* target)
{
    int  newHeapSize = 2 * target->numVertices;

    HomogeneousVector3D*  tempVertexHeap = malloc(sizeof(HomogeneousVector3D) *
        newHeapSize);

    if (!tempVertexHeap) {

        RuntimeError("GrowVertexNormalHeaps( )", "out of memory");
    }

    memcpy(tempVertexHeap, target->vertices,
        ((target->numVertices) * sizeof(HomogeneousVector3D)));

    free(target->vertices);

    target->vertices = tempVertexHeap;

    target->vertexSlotsAllocd = newHeapSize;

    if (target->normalMode == kVertexNormalMode) {

        HomogeneousVector3D*  tempNormalHeap =
            malloc(sizeof(HomogeneousVector3D) * newHeapSize);

        if (!tempNormalHeap) {

            RuntimeError("GrowVertexNormalHeaps( )", "out of memory");
        }

        memcpy(tempNormalHeap, target->normals,
            ((target->numVertices) * sizeof(HomogeneousVector3D)));

        free(target->normals);

        target->normals = tempNormalHeap;
    }
}




void   AppendVertex(Mesh* target, const HomogeneousVector3D* newVtx)
{
    if (target->numVertices == target->vertexSlotsAllocd) {

        GrowVertexNormalHeaps(target);
    }

    target->vertices[target->numVertices] = *newVtx;

    target->numVertices++;
}




void   SetVertexNormal(Mesh* target, int i, const HomogeneousVector3D* newNorm)
{
    if (i >= target->numVertices) {

        LogicError("SetVertexNormal( )", "vertex index out of range");
    }

    if (target->normalMode == kFaceNormalMode) {

        LogicError("SetVertexNormal( )", "parent mesh is in face normal mode");
    }

    target->normals[i] = *newNorm;
}





void   GrowFaceHeap(Mesh*  target)
{
    int  newHeapSize = 2 * target->numFaces;

    MeshFace*  tempFaceHeap =
        malloc(sizeof(MeshFace) * newHeapSize);

    if (!tempFaceHeap) {

        RuntimeError("GrowFaceHeap( )", "out of memory");
    }

    memcpy(tempFaceHeap, target->faces,
        ((target->numFaces) * sizeof(MeshFace)) );

    free(target->faces);

    target->faces = tempFaceHeap;

    target->faceSlotsAllocd = newHeapSize;
}




void   AppendFace(Mesh* target, const int* userIndexes, int numUserIndexes)
{
#ifdef DIAGNOSTICS_ON
    int k;
#endif
    if (target->numFaces == target->faceSlotsAllocd) {

        GrowFaceHeap(target);
    }

    target->faces[target->numFaces].vindexes =
        malloc(sizeof(int) * numUserIndexes);

    if (!(target->faces[target->numFaces].vindexes)) {

        RuntimeError("AppendFace( )", "out of memory");
    }

    target->faces[target->numFaces].numVindexes = numUserIndexes;

    if (target->normalMode == kFaceNormalMode) {

        HomogeneousVector3D  norm;
        int                  i;

        SetDirection3D(&norm, 0.0, 0.0, 0.0);

        for (i = 0; i < numUserIndexes; i++) {

            unsigned  currVertexBaseAddr = i;
            unsigned  nextVertexBaseAddr = (i + 1) % numUserIndexes;
            double  x_i =    target->vertices[currVertexBaseAddr].x;
            double  x_next = target->vertices[nextVertexBaseAddr].x;
            double  y_i =    target->vertices[currVertexBaseAddr].y;
            double  y_next = target->vertices[nextVertexBaseAddr].y;
            double  z_i =    target->vertices[currVertexBaseAddr].z;
            double  z_next = target->vertices[nextVertexBaseAddr].z;  
        
            norm.x += (y_i - y_next) * (z_i + z_next);
            norm.y += (x_i + x_next) * (z_i - z_next);
            norm.z += (x_i - x_next) * (y_i + y_next);
        }
        
        norm = VectorNormalize(&norm);

        target->faces[target->numFaces].normal = malloc(sizeof(HomogeneousVector3D));

        *(target->faces[target->numFaces].normal) = norm;
    }

#ifdef DIAGNOSTICS_ON
    printf("user face data = ");
    for (k = 0; k < numUserIndexes; k++) {

        printf("%d ", userIndexes[k]);
    }
    printf("\n");
#endif

    memcpy(target->faces[target->numFaces].vindexes, userIndexes,
        numUserIndexes * sizeof(int));

    target->numFaces++;
}




void   PrintMesh(const Mesh* target)
{
    int i, j;

    printf("\nMesh:\n");
    printf("        mesh normal mode =    ");

    if (target->normalMode == kVertexNormalMode) {

        printf("per-vertex mode (smooth)\n");
    }
    else {

        printf("per-face mode (flat)\n");
    }

    printf("        vertex slots filled = %d\n", target->numVertices);
    printf("        vertex slots allocd = %d\n", target->vertexSlotsAllocd);
    printf("        face slots filled =   %d\n", target->numFaces);
    printf("        face slots allocd =   %d\n", target->faceSlotsAllocd);

    if (target->isTriangleMesh) {

        printf("        is triangular mesh =  TRUE\n");
    }
    else {

        printf("        is triangular mesh =  FALSE\n");
    }

    printf("<<<<<<  VERTEX DATA\n");

    for (i = 0; i < target->numVertices; i++) {

        printf("    ");
        PrintVector(&target->vertices[i]);
    }

    if (target->normalMode == kVertexNormalMode) {

        printf("<<<<<<  NORMAL DATA\n");

        for (i = 0; i < target->numVertices; i++) {

            printf("    ");
            PrintVector(&target->normals[i]);
        }
    }


    printf("\n<<<<<<  FACE DATA\n");

    for (i = 0; i < target->numFaces; i++) {

        printf("    [%d] ", i);

        for (j = 0; j < target->faces[i].numVindexes; j++) {

            printf("%d ", target->faces[i].vindexes[j]);
        }

        printf("\n");
    }

    printf("\n\n\n\n");
}




BoundingBox3D  ComputeTriMeshBoundingBox(const Mesh* mesh)
{
    BoundingBox3D  result;
    BoundingBox3D  currentFaceBB;
    boolean        resultInitialized = FALSE;

    int           i;

    if (!mesh->isTriangleMesh) {

        LogicError("ComputeTriMeshBoundingBox( )", "input mesh is not "
            "triangular");
    }

    for (i = 0; i < mesh->numFaces; i++) {

        HomogeneousVector3D  v1 = mesh->vertices[mesh->faces[i].vindexes[0]];
        HomogeneousVector3D  v2 = mesh->vertices[mesh->faces[i].vindexes[1]];
        HomogeneousVector3D  v3 = mesh->vertices[mesh->faces[i].vindexes[2]];

        currentFaceBB = TriangleBoundingBox(&v1, &v2, &v3);

        if (!resultInitialized) {

            result = currentFaceBB;
            resultInitialized = TRUE;
        }
        else {

            result = BoundingBoxUnion(&result, &currentFaceBB);
        }
    }

    return result;
}
