/*

 GeometryIntrinsics.h

 Defines types and functions to support most of the basic computational
 geometry objects and operations required for 3D graphics, such as
 vectors (example operations: scalar multiply, dot product, etc.),
 transforms, and more.

 Copyright (c) 2006, Lucas Stephen Beeler. All Rights Reserved.

 */

#ifndef GEOMETRY_INTRINSICS__H
#define GEOMETRY_INTRINSICS__H

#include "ApplicationServices.h"

extern const double  kPi;
extern const double  kBigDouble;


/*
 TYPE  HomogeneousVector3D
 */
struct  HomogeneousVector3D_Rep {

    double  x;
    double  y;
    double  z;
    double  w;
};

typedef struct HomogeneousVector3D_Rep HomogeneousVector3D;

void  SetVertex3D(HomogeneousVector3D*, double, double, double);
void  SetDirection3D(HomogeneousVector3D*, double, double, double);
HomogeneousVector3D  ScalarMultiply(const HomogeneousVector3D*, double);
HomogeneousVector3D  VectorAdd(const HomogeneousVector3D*,
    const HomogeneousVector3D*);
HomogeneousVector3D  VectorSubtract(const HomogeneousVector3D*,
    const HomogeneousVector3D*);
double  VectorDot(const HomogeneousVector3D*, const HomogeneousVector3D*);
HomogeneousVector3D  VectorCross(const HomogeneousVector3D*,
    const HomogeneousVector3D*);
double  VectorLength(const HomogeneousVector3D*);
HomogeneousVector3D  VectorNormalize(const HomogeneousVector3D*);
void  PrintVector(const HomogeneousVector3D*);
HomogeneousVector3D  ZeroPositionVector( );
HomogeneousVector3D  VectorPerturb(const HomogeneousVector3D*, double);





/*
 TYPE  HomogeneousTransform3D
 */
struct  HomogeneousTransform3D_Rep {

    double  matrix[16]; /* matrix entries in row-major order */
};

typedef struct HomogeneousTransform3D_Rep HomogeneousTransform3D;

void  ConfigureTranslation(HomogeneousTransform3D*, double, double, double);
void  ConfigureScaling(HomogeneousTransform3D*, double, double, double);
void  ConfigureRotation(HomogeneousTransform3D*, const HomogeneousVector3D*,
    double);
void  ConfigureIdentity(HomogeneousTransform3D*);
HomogeneousVector3D  ApplyTransform(const HomogeneousTransform3D*,
    const HomogeneousVector3D*);
void  ComposeTransformsTo(HomogeneousTransform3D*,
    const HomogeneousTransform3D*);
void  TransposeTransform(HomogeneousTransform3D*);
void  InvertTransform(HomogeneousTransform3D*);
void  PrintTransform(const HomogeneousTransform3D*);





/*
 TYPE  RGBColor
 */
struct RGBColor_Rep {

    double r;
    double g;
    double b;
};

typedef struct RGBColor_Rep RGBColor;

void      SetRGBColor(RGBColor*, double, double, double);
RGBColor  RGBColorAdd(const RGBColor*, const RGBColor*);
RGBColor  RGBColorMultiply(const RGBColor*, const RGBColor*);
RGBColor  RGBColorScale(const RGBColor*, double);
void      PrintRGBColor(const RGBColor*);

extern const RGBColor kRGB_Black;
extern const RGBColor kRGB_White;
extern const RGBColor kRGB_SeaGreen;
extern const RGBColor kRGB_Gray90;
extern const RGBColor kRGB_Gray50;
extern const RGBColor kRGB_BunnyPink;

/*
 TYPE  Ray3D
 */
struct  Ray3D_rep {

    HomogeneousVector3D  origin;
    HomogeneousVector3D  direction;
};
typedef struct Ray3D_rep  Ray3D;

void   SetRay(Ray3D*, const HomogeneousVector3D*, const HomogeneousVector3D*);
Ray3D  TransformRay(const HomogeneousTransform3D*, const Ray3D*);
void   PrintRay(const Ray3D*);




/*
 TYPE  BoundingBox3D
 */
struct  BoundingBox3D_Rep {

    double  xMin;
    double  xMax;
    double  yMin;
    double  yMax;
    double  zMin;
    double  zMax;
};
typedef struct BoundingBox3D_Rep BoundingBox3D;

void   SetBoundingBox(BoundingBox3D*, double, double, double, double,
    double, double);
boolean   BoundingBoxIntersect(const BoundingBox3D*, const Ray3D*);
void  PrintBoundingBox(const BoundingBox3D*);
BoundingBox3D  BoundingBoxUnion(const BoundingBox3D*, const BoundingBox3D*);
BoundingBox3D  TriangleBoundingBox(const HomogeneousVector3D*,
    const HomogeneousVector3D*, const HomogeneousVector3D*);
boolean   PointInBoundingBox(const HomogeneousVector3D*,
    const BoundingBox3D*);




/*
 TYPE  CoordinateSystem3D
 */
struct  CoordinateSystem3D_REP {

    HomogeneousVector3D         basisX;
    HomogeneousVector3D         basisY;
    HomogeneousVector3D         basisZ;
};
typedef struct CoordinateSystem3D_REP CoordinateSystem3D;

CoordinateSystem3D  ComputeZDirectedSystem(const HomogeneousVector3D*);




/*
  FUNCTIONS  Miscellaneous Global Functions
 */
double  SampleUniform(void);
HomogeneousVector3D  TriangleCenter(const HomogeneousVector3D*,
    const HomogeneousVector3D*, const HomogeneousVector3D*);


#endif
